/*
 * Copyright (c) 2019, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;

import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;

/**
 * @test
 * @bug 8146239
 * @summary Tests that AU_FLOAT and AU_DOUBLE formats are supported
 */
public final class RecognizeAuFloat {
    // Create the audio file
    //   $sox  -c 2 -n -b 32 -e floating-point -r 32000 --comment '' tmp.au synth 0.01 sine 1-90000 sine 90000-1
    // sndfile will remove the optional information chunk
    //   $sndfile-convert tmp.au data.au
    private static final byte[] data32 =
            new byte[]{46, 115, 110, 100, 0, 0, 0, 24, 0, 0, 10, 0, 0, 0, 0, 6,
                    0, 0, 125, 0, 0, 0, 0, 2, 55, 30, 0, 0, -66, 83, 22, 72, 57,
                    20, 0, 0, -66, -22, -69, 4, 57, -100, -96, 0, 62, -119, 4,
                    -42, 57, -15, -80, 0, -67, -102, -15, -24, 58, 39, -84, 0,
                    -68, 77, 29, 64, 58, 88, -104, 0, 61, 39, -13, -64, 58,
                    -122, -16, 0, -69, -74, -27, -128, 58, -94, -26, 0, -66, 42,
                    16, 116, 58, -63, 38, 0, 63, 5, 27, -95, 58, -32, -6, 0,
                    -65, 25, 104, -34, 59, 1, -98, 0, -66, -67, -43, 98, 59, 19,
                    -85, 0, 62, -37, 121, -120, 59, 39, 4, 0, 63, 21, 30, 50,
                    59, 59, 110, 0, 63, 65, 109, 87, 59, 81, 60, 0, 63, 11, 74,
                    112, 59, 104, 67, 0, 62, -36, 123, -48, 59, -128, 100, 0,
                    -66, -50, -18, -128, 59, -115, 88, -128, -65, 20, -62, 90,
                    59, -101, 27, -128, -67, -122, 39, -88, 59, -87, -89, -128,
                    63, 81, 60, -32, 59, -71, 18, -128, -65, 52, -14, 17, 59,
                    -55, 95, -128, 62, 91, 110, -100, 59, -38, -97, 0, 62, 36,
                    28, -120, 59, -20, -38, 0, -66, -97, 28, -24, 60, 0, 15, 0,
                    62, -95, 14, 112, 60, 10, 60, -128, -66, -125, -38, 104, 60,
                    20, -5, 0, 62, 64, -85, -72, 60, 32, 83, -64, -66, 0, -88,
                    -124, 60, 44, 75, -128, 61, -104, -68, -104, 60, 56, -19,
                    64, -67, 11, -80, 16, 60, 70, 62, 0, 59, 12, 0, 0, 60, 84,
                    73, 64, 60, -72, 108, -128, 60, 99, 21, 0, -67, 6, -91, 32,
                    60, 114, -83, -128, 61, 110, -11, 32, 60, -127, -116, -64,
                    -67, 73, -87, -64, 60, -118, 51, 64, 61, -71, 28, -96, 60,
                    -109, 77, -32, -67, -66, 26, -72, 60, -100, -29, -96, 61,
                    -101, 24, -112, 60, -90, -7, 32, -67, 110, -4, -64, 60, -79,
                    -107, -96, 61, -85, -41, 112, 60, -68, -66, 64, -67, -117,
                    85, 8, 60, -56, 122, -96, 61, 119, 33, -128, 60, -44, -48,
                    -32, -67, -123, -116, 104, 60, -31, -56, -64, 60, -101, -65,
                    -128, 60, -17, 105, 96, 60, 71, 29, -128, 60, -3, -70, -32,
                    -67, -121, -72, 32, 61, 6, 98, -80, 62, 17, 12, -52, 61, 14,
                    72, -96, -66, 98, -7, 120, 61, 22, -109, -48, 62, -89, -123,
                    -114, 61, 31, 72, -64, -66, -46, -27, -126, 61, 40, 108,
                    112, 62, -24, 13, -98, 61, 50, 3, -64, -66, -49, -111, -118,
                    61, 60, 20, 32, 62, 65, -50, 0, 61, 70, -94, -48, 62, 7,
                    -53, -92, 61, 81, -75, -80, -65, 15, 22, 51, 61, 93, 82,
                    -112, 63, 75, 122, 6, 61, 105, 127, -64, -65, 39, -103, 108,
                    61, 118, 67, -128, 61, 102, -22, 32, 61, -127, -46, 80, 63,
                    3, -8, 3, 61, -120, -44, -16, -65, 13, 31, -78, 61, -112,
                    45, 88, -66, 26, 95, 88, 61, -105, -33, 40, 63, 56, 1, -51,
                    61, -97, -18, 104, -66, -87, 96, -96, 61, -88, 94, -24, -65,
                    14, -12, 116, 61, -79, 53, 8, 62, -28, 8, -68, 61, -70, 116,
                    -16, 62, -11, -35, 8, 61, -60, 35, 48, -65, 7, -110, 47, 61,
                    -50, 68, 96, -65, 4, 62, -13, 61, -40, -35, 48, 62, -54,
                    -54, 70, 61, -29, -14, -64, 63, 28, -19, -10, 61, -17, -119,
                    -80, -66, 74, 117, -88, 61, -5, -89, -24, -65, 55, 82, 2,
                    62, 4, 40, -40, -66, 119, -21, -24, 62, 10, -58, -4, 62, -8,
                    108, 110, 62, 17, -80, 40, 63, 38, 48, 49, 62, 24, -24,
                    -112, -67, 110, -90, 32, 62, 32, 113, 92, -65, 29, -111,
                    -11, 62, 40, 79, -92, -65, 31, 111, 119, 62, 48, -125, -68,
                    -67, -99, -66, -80, 62, 57, 19, -60, 62, -13, -69, 100, 62,
                    65, -1, 16, 63, 53, 109, -123, 62, 75, 76, -64, 62, -112,
                    45, 8, 62, 84, -5, 36, -66, -117, -58, -90, 62, 95, 18, 24,
                    -65, 43, -15, -111, 62, 105, -113, 36, -65, 37, -126, 17,
                    62, 116, 122, 72, -66, -71, 7, 124, 62, 127, -48, -48, 61,
                    -80, 58, 104, 62, -123, -52, -22, 63, 10, -42, -96, 62,
                    -117, -23, -88, 63, 53, 17, -15, 62, -110, 64, -8, 63, 4,
                    -33, -105, 62, -104, -46, -42, 62, 14, -16, 16, 62, -97,
                    -97, 98, -66, -120, -100, 48, 62, -90, -88, 106, -65, 16,
                    -55, 65, 62, -83, -22, -104, -65, 50, 111, 94, 62, -75, 106,
                    126, -65, 43, -76, -4, 62, -67, 31, -16, -65, 5, 60, 84, 62,
                    -59, 19, 34, -66, -101, 116, -118, 62, -51, 53, -64, -67,
                    -114, -112, 112, 62, -43, -108, 62, 62, 124, 61, 4, 62, -34,
                    25, 4, 63, 12, -46, 66, 62, -26, -44, -28, 63, 46, -89, -62,
                    62, -17, -86, 94, 63, 49, -118, -31, 62, -8, -83, -6, 63,
                    18, 74, -70, 63, 0, -35, 49, 62, -57, -13, -10, 63, 5, 115,
                    21, 62, 39, 28, -108, 63, 10, 2, -109, -67, 91, 2, -48, 63,
                    14, -106, 126, -66, -125, -114, 6, 63, 19, 22, 108, -66,
                    -43, -29, 52, 63, 23, -118, -117, -65, 11, 119, -38, 63, 27,
                    -39, -51, -65, 32, -43, 72, 63, 32, 6, -41, -65, 46, 64,
                    -64, 63, 35, -6, 65, -65, 51, -67, -118, 63, 39, -83, -78,
                    -65, 51, -12, -67, 63, 43, 14, 38, -65, 47, -33, -17, 63,
                    46, 8, 121, -65, 41, 19, 46, 63, 48, -111, 23, -65, 32, -60,
                    -18, 63, 50, -124, 76, -65, 23, -122, -109, 63, 51, -31, 5,
                    -65, 14, -121, 38, 63, 52, 111, -48, -65, 5, -96, -7, 63,
                    52, 60, -30, -66, -5, -52, -60, 63, 50, -5, 32, -66, -19,
                    -110, -102, 63, 48, -58, 58, -66, -30, 112, -76, 63, 45, 59,
                    -112, -66, -40, -12, 36, 63, 40, -121, -76, -66, -46, -85,
                    120, 63, 34, 53, -105, -66, -50, 29, 12, 63, 26, -125, 46,
                    -66, -52, -124, 104, 63, 16, -17, -11, -66, -52, -114, 12,
                    63, 5, -54, -26, -66, -49, 34, 40, 62, -15, 39, -124, -66,
                    -45, 38, -34, 62, -45, 83, 6, -66, -39, 53, 112, 62, -79,
                    58, -88, -66, -32, 113, 10, 62, -117, -71, 118, -66, -23,
                    44, -72, 62, 68, -124, 88, -66, -14, -61, 36, 61, -41, 35,
                    -48, -66, -3, 76, -16, 60, 73, 21, 64, -65, 4, 41, 121, -67,
                    -84, -32, 56, -65, 9, -32, 97, -66, 59, -16, -68, -65, 15,
                    -95, -73, -66, -112, -19, 104, -65, 21, 83, -106, -66, -62,
                    84, 28, -65, 26, -38, 66, -66, -15, 61, -70, -65, 32, 18,
                    -104, -65, 13, 91, 117, -65, 36, -21, 35, -65, 31, 24, -23,
                    -65, 41, 61, 53, -65, 43, -67, -83, -65, 44, -1, 47, -65,
                    51, 118, 20, -65, 48, 11, 126, -65, 51, -106, -70, -65, 50,
                    94, -68, -65, 45, 55, 12, -65, 51, -40, 0, -65, 29, 54, 57,
                    -65, 52, 121, 12, -65, 6, 103, 50, -65, 52, 40, 15, -66,
                    -54, 55, 44, -65, 50, -21, 51, -66, 126, 66, -44, -65, 48,
                    -79, 26, -67, 117, 95, -48, -65, 45, -125, -61, 61, -106,
                    52, -96, -65, 41, 90, -97, 61, -57, 113, -120, -65, 36, 66,
                    -15, 62, 12, -23, 24, -65, 30, 60, -115, 62, 48, 125, 20,
                    -65, 23, 87, 95, 62, 90, -59, 68, -65, 15, -102, -111, 62,
                    -127, -1, 112, -65, 7, 24, 12, 62, -104, 94, -106, -66, -5,
                    -71, -60, 62, -82, -25, -76, -66, -25, -8, -128, 62, -58,
                    53, -84, -66, -45, 14, -22, 62, -35, -81, -96, -66, -67, 36,
                    -120, 62, -11, 6, -66, -66, -90, 97, 4, 63, 6, 3, 127, -66,
                    -114, -21, -84, 63, 16, -20, 40, -66, 109, -36, 56, 63, 27,
                    40, 41, -66, 61, 28, 72, 63, 36, 33, -103, -66, 11, -23, 72,
                    63, 43, -83, 65, -67, -75, 19, 40, 63, 49, 27, 71, -67, 37,
                    37, 48, 63, 52, 32, -92, 59, -13, 9, 0, 63, 51, -1, -36, 61,
                    95, 94, -112, 63, 48, 88, -114, 61, -50, 123, -8, 63, 40,
                    111, -36, 62, 21, -110, 52, 63, 27, -12, -26, 62, 66, -90,
                    -48, 63, 10, 76, 35, 62, 110, 78, 0, 62, -26, -13, -12, 62,
                    -116, 49, -14, 62, -82, 100, -112, 62, -96, 99, 6, 62, 88,
                    -35, 60, 62, -77, -83, 18, 61, -122, 105, 8, 62, -58, 4, 54,
                    -67, -72, 110, -32, 62, -41, 95, -40, -66, -128, -100, 68,
                    62, -25, -72, -8, -66, -50, -95, -10, 62, -9, 11, 32, -65,
                    10, 90, -74, 63, 2, -87, -56, -65, 36, 41, 100, 63, 9, 72,
                    -63, -65, 51, 48, -12, 63, 15, 98, -30, -65, 49, 114, 53,
                    63, 20, -7, 86, -65, 31, -51, 3, 63, 26, 13, -7, -66, -14,
                    -93, 4, 63, 30, -93, 56, -66, -118, -69, 76, 63, 34, -68,
                    16, -70, -82, 44, 0, 63, 38, 91, -23, 62, 94, 113, -52, 63,
                    41, -122, -105, 62, -89, -122, -42, 63, 44, 64, 56, 62, -28,
                    76, -10, 63, 46, -115, 50, 63, 12, 66, 105, 63, 48, 114, 28,
                    63, 34, 21, 2, 63, 49, -13, -73, 63, 48, 66, -126, 63, 51,
                    22, -34, 63, 52, 69, -91, 63, 51, -32, 123, 63, 44, 33, -96,
                    63, 52, 85, -126, 63, 21, 55, -35, 63, 52, 122, -32, 62,
                    -33, -42, 72, 63, 52, 85, 123, 62, 113, 54, 96, 63, 51, -22,
                    37, -69, 84, -27, 0, 63, 51, 61, -98, -66, -125, -34, -80,
                    63, 50, 84, -124, -66, -7, -59, 56, 63, 49, 51, 90, -65, 38,
                    -69, 12, 63, 47, -34, 124, -65, 52, 120, -13, 63, 46, 90,
                    32, -65, 28, -48, -75, 63, 44, -86, 84, -66, -62, -122, 2,
                    63, 42, -46, -4, -68, -66, 18, -64, 63, 40, -41, -49, 62,
                    -125, 39, -32, 63, 38, -68, 89, 62, -23, 125, -60, 63, 36,
                    -125, -7, 63, 32, 56, 64, 63, 34, 49, -32, 63, 51, -95, -58,
                    63, 31, -55, 21, 63, 41, -75, 70, 63, 29, 76, 114, 62, -7,
                    -33, -98, 63, 26, -66, -91, 62, 65, -30, 76, 63, 24, 34, 52,
                    -66, 48, 71, -116, 63, 21, 121, 124, -65, 2, -67, 114, 63,
                    18, -58, -80, -65, 49, 29, -36, 63, 16, 11, -32, -65, 34,
                    -68, -19, 63, 13, 74, -12, -66, -104, 24, -26, 63, 10, -123,
                    -76, 62, 51, 28, -80, 63, 7, -67, -61, 63, 2, 89, -28, 63,
                    4, -12, -89, 63, 47, -73, -62, 63, 2, 43, -61, 63, 39, 28,
                    20, 62, -2, -56, -64, 62, -77, 8, -104, 62, -7, 63, 86, -67,
                    -19, -18, -96, 62, -13, -67, 110, -65, 14, -31, -41, 62,
                    -18, 68, -10, -65, 50, -108, 124, 62, -24, -41, -72, -66,
                    -51, -101, 100, 62, -29, 119, 74, 62, 81, -23, -92, 62, -34,
                    37, 28, 63, 35, 19, -90, 62, -40, -30, 118, 63, 42, 75, 0,
                    62, -45, -80, -126, 62, -101, 23, -2, 62, -50, -112, 60,
                    -66, -92, 113, 88, 62, -55, -126, -110, -65, 48, 57, -75,
                    62, -60, -120, 62, -66, -18, 109, -36, 62, -65, -95, -2, 62,
                    -122, 46, -94, 62, -70, -48, 80, 63, 43, -78, 92, 62, -74,
                    19, -56, 63, 8, -56, 59, 62, -79, 108, -82, -66, 53, 9, -12,
                    62, -84, -37, 124, -65, 45, -38, -43, 62, -88, 96, 68, -66,
                    -64, 66, -114, 62, -93, -5, 104, 62, -5, -24, -122, 62, -97,
                    -84, -46, 63, 37, -56, 106, 62, -101, 116, -52, -69, -51,
                    74, 0, 62, -105, 83, 38, -65, 52, 39, 12, 62, -109, 72, 12,
                    -66, 105, -56, -88, 62, -113, 83, 68, 63, 34, 28, -23, 62,
                    -117, 116, -54, 62, -41, 27, -96, 62, -121, -84, 124, -65,
                    6, 56, 36, 62, -125, -6, 18, -66, -24, -78, 120, 62, -128,
                    93, -98, 63, 26, -115, 55, 62, 121, -84, -36, 62, -72, -90,
                    88, 62, 114, -54, 0, -65, 24, -101, -10, 62, 108, 16, 20,
                    -66, 52, -39, -8, 62, 101, -127, 68, 63, 69, 45, 6, 62, 95,
                    25, 120, -66, -68, 112, 34, 62, 88, -36, 112, -66, -63,
                    -116, 94, 62, 82, -60, 72, 63, 28, 61, 0, 62, 76, -42, -116,
                    -66, 27, 72, 116, 62, 71, 11, 104, -66, -7, -1, -124, 62,
                    65, 106, 68, 63, 84, 48, 84, 62, 59, -23, 120, -65, 42, 107,
                    50, 62, 54, -110, 16, 62, -125, -115, -8, 62, 49, 89, 0, 62,
                    57, -24, -4, 62, 44, 72, 60, -66, -35, -117, 4, 62, 39, 84,
                    100, 63, 1, -103, 34, 62, 34, -121, 12, -66, -17, 116, 4,
                    62, 29, -42, 0, 62, -65, -115, 24, 62, 25, 72, -72, -66,
                    120, -42, -28, 62, 20, -40, 64, 62, 26, 47, 8, 62, 16, -121,
                    -112, -67, 77, -2, 32, 62, 12, 85, -124, 59, 64, 93, 0, 62,
                    8, 61, -16, 61, 73, 86, -80, 62, 4, 72, 80, -67, -102, 14,
                    0, 62, 0, 102, 100, 61, -76, -35, 112, 61, -7, 86, 72, -67,
                    121, 104, -96, 61, -15, -9, 56, 61, 123, -75, 16, 61, -22,
                    -15, 64, -67, 115, -85, 64, 61, -29, -15, 8, 61, -125, -102,
                    40, 61, -35, 86, -16, -68, -17, 52, 32, 61, -42, -80, -128,
                    61, 30, -101, 112, 61, -48, 124, -16, 60, 14, -37, 0, 61,
                    -54, 44, 88, -67, 38, 97, 112, 61, -60, 89, 8, 61, 79, -37,
                    -96, 61, -66, 91, -64, -67, 76, 109, 80, 61, -72, -31, 72,
                    61, -73, -94, -88, 61, -77, 54, 72, -67, -4, 114, -24, 61,
                    -82, 12, 32, 62, 5, 28, 48, 61, -88, -77, -8, -66, 37, 7,
                    20, 61, -93, -48, 88, 62, 36, -101, 8, 61, -98, -52, -16,
                    -67, -40, 80, 112, 61, -102, 37, 64, -67, -91, -60, 104, 61,
                    -107, 121, -88, 62, -47, 16, 34, 61, -111, 2, -72, -65, 69,
                    50, -82, 61, -116, -78, -96, 63, 20, -70, 1, 61, -120, 97,
                    56, 62, -127, -67, -36, 61, -124, 112, 96, -65, 33, -9, -43,
                    61, -128, 57, -16, -66, -7, -70, 72, 61, 121, 86, -32, 62,
                    41, -78, 116, 61, 113, 13, -80, 63, 18, -75, -76, 61, 106,
                    -72, 112, 63, 48, 90, 88, 61, 98, -123, 48, 63, 52, 13, -69,
                    61, 92, -10, 16, 63, 45, 31, -18, 61, 84, -47, 48, 62, -32,
                    12, 26, 61, 80, 0, 48, -66, 38, -81, 88, 61, 71, -23, -16,
                    -65, 63, -24, 61, 61, 67, -57, 0, 62, -106, -40, 110, 61,
                    59, -55, 48, 62, 116, 6, 44, 61, 56, 58, 80, -66, -67, 31,
                    42, 61, 48, 106, 64, 62, -88, 121, 104, 61, 45, 73, 16, -66,
                    -115, -11, -110, 61, 37, -53, 48, 62, -128, -110, -42, 61,
                    34, -33, -80, -66, 121, 84, -44, 61, 27, -17, 48, 62, 79,
                    -38, 28, 61, 24, -30, -16, 61, 91, 108, 16, 61, 18, -23, 32,
                    -65, 38, 12, 43, 61, 15, 26, -64, 62, 59, -92, -104, 61, 11,
                    11, 96, 62, -37, 45, 18, 61, 4, -81, -48, 61, 74, 69, -16,
                    61, 6, -67, 80,};

    // Create the audio file
    //   $sox  -c 2 -n -b 64 -e floating-point -r 32000 --comment '' tmp.au synth 0.01 sine 1-90000 sine 90000-1
    // sndfile will remove the optional information chunk
    //   $sndfile-convert tmp.au data.au
    private static final byte[] data64 =
            new byte[]{46, 115, 110, 100, 0, 0, 0, 24, 0, 0, 20, 0, 0, 0, 0, 7,
                    0, 0, 125, 0, 0, 0, 0, 2, 62, -29, -76, -128, 0, 0, 0, 0,
                    -65, -54, 98, -55, 10, 0, 0, 0, 63, 34, 127, 32, 0, 0, 0, 0,
                    -65, -35, 87, 96, -84, -128, 0, 0, 63, 51, -108, 34, 0, 0,
                    0, 0, 63, -47, 32, -102, -49, -128, 0, 0, 63, 62, 54, 74, 0,
                    0, 0, 0, -65, -77, 94, 60, -84, 0, 0, 0, 63, 68, -11, 79, 0,
                    0, 0, 0, -65, -119, -93, -86, 48, 0, 0, 0, 63, 75, 19, 0, 0,
                    0, 0, 0, 63, -92, -2, 120, 56, 0, 0, 0, 63, 80, -34, 23,
                    -128, 0, 0, 0, -65, 118, -36, -82, 0, 0, 0, 0, 63, 84, 92,
                    -63, -128, 0, 0, 0, -65, -59, 66, 14, 115, 0, 0, 0, 63, 88,
                    36, -38, 0, 0, 0, 0, 63, -32, -93, 116, 32, 0, 0, 0, 63, 92,
                    31, 46, -128, 0, 0, 0, -65, -29, 45, 27, -52, 64, 0, 0, 63,
                    96, 51, -72, -64, 0, 0, 0, -65, -41, -70, -84, 58, -128, 0,
                    0, 63, 98, 117, 82, -128, 0, 0, 0, 63, -37, 111, 49, 18, 0,
                    0, 0, 63, 100, -32, 114, -128, 0, 0, 0, 63, -30, -93, -58,
                    66, -64, 0, 0, 63, 103, 109, -69, -64, 0, 0, 0, 63, -24, 45,
                    -86, -27, -128, 0, 0, 63, 106, 39, 113, 0, 0, 0, 0, 63, -31,
                    105, 78, 14, 0, 0, 0, 63, 109, 8, 89, 0, 0, 0, 0, 63, -37,
                    -113, 121, -3, -128, 0, 0, 63, 112, 12, 124, 96, 0, 0, 0,
                    -65, -39, -35, -48, 29, -128, 0, 0, 63, 113, -85, 22, -32,
                    0, 0, 0, -65, -30, -104, 75, 72, -64, 0, 0, 63, 115, 99,
                    111, 64, 0, 0, 0, -65, -80, -60, -11, 82, 0, 0, 0, 63, 117,
                    52, -17, -128, 0, 0, 0, 63, -22, 39, -100, 28, 64, 0, 0, 63,
                    119, 34, 84, -32, 0, 0, 0, -65, -26, -98, 66, 30, 0, 0, 0,
                    63, 121, 43, -20, -96, 0, 0, 0, 63, -53, 109, -45, 127, 0,
                    0, 0, 63, 123, 83, -34, 0, 0, 0, 0, 63, -60, -125, -111, 35,
                    0, 0, 0, 63, 125, -101, 60, 64, 0, 0, 0, -65, -45, -29, -99,
                    28, -128, 0, 0, 63, -128, 1, -33, 0, 0, 0, 0, 63, -44, 33,
                    -50, 29, 0, 0, 0, 63, -127, 71, -115, -80, 0, 0, 0, -65,
                    -48, 123, 77, 30, 0, 0, 0, 63, -126, -97, 96, -96, 0, 0, 0,
                    63, -56, 21, 119, 58, 0, 0, 0, 63, -124, 10, 118, 80, 0, 0,
                    0, -65, -64, 21, 16, -60, 0, 0, 0, 63, -123, -119, 115, -96,
                    0, 0, 0, 63, -77, 23, -109, 98, 0, 0, 0, 63, -121, 29, -87,
                    -16, 0, 0, 0, -65, -95, 118, 1, 44, 0, 0, 0, 63, -120, -57,
                    -66, -128, 0, 0, 0, 63, 97, 127, -7, 0, 0, 0, 0, 63, -118,
                    -119, 40, -96, 0, 0, 0, 63, -105, 13, -111, 56, 0, 0, 0, 63,
                    -116, 98, -99, -96, 0, 0, 0, -65, -96, -44, -92, 100, 0, 0,
                    0, 63, -114, 85, -80, 32, 0, 0, 0, 63, -83, -34, -93, -24,
                    0, 0, 0, 63, -112, 49, -103, 80, 0, 0, 0, -65, -87, 53, 56,
                    100, 0, 0, 0, 63, -111, 70, 102, 32, 0, 0, 0, 63, -73, 35,
                    -108, 44, 0, 0, 0, 63, -110, 105, -69, 40, 0, 0, 0, -65,
                    -73, -61, 87, -118, 0, 0, 0, 63, -109, -100, 116, 88, 0, 0,
                    0, 63, -77, 99, 17, -86, 0, 0, 0, 63, -108, -33, 37, 80, 0,
                    0, 0, -65, -83, -33, -104, -20, 0, 0, 0, 63, -106, 50, -78,
                    8, 0, 0, 0, 63, -75, 122, -19, -18, 0, 0, 0, 63, -105, -105,
                    -57, 104, 0, 0, 0, -65, -79, 106, -95, 80, 0, 0, 0, 63,
                    -103, 15, 82, 104, 0, 0, 0, 63, -82, -28, 47, -68, 0, 0, 0,
                    63, -102, -102, 26, 96, 0, 0, 0, -65, -80, -79, -115, -122,
                    0, 0, 0, 63, -100, 57, 23, 96, 0, 0, 0, 63, -109, 119, -17,
                    0, 0, 0, 0, 63, -99, -19, 43, -8, 0, 0, 0, 63, -120, -29,
                    -81, -32, 0, 0, 0, 63, -97, -73, 93, -96, 0, 0, 0, -65, -80,
                    -9, 3, -100, 0, 0, 0, 63, -96, -52, 85, 76, 0, 0, 0, 63,
                    -62, 33, -103, -109, 0, 0, 0, 63, -95, -55, 20, 116, 0, 0,
                    0, -65, -52, 95, 46, -25, 0, 0, 0, 63, -94, -46, 121, 84, 0,
                    0, 0, 63, -44, -16, -79, -41, -128, 0, 0, 63, -93, -23, 24,
                    -80, 0, 0, 0, -65, -38, 92, -80, 77, 0, 0, 0, 63, -91, 13,
                    -115, -32, 0, 0, 0, 63, -35, 1, -77, -25, 0, 0, 0, 63, -90,
                    64, 120, -56, 0, 0, 0, -65, -39, -14, 49, 104, -128, 0, 0,
                    63, -89, -126, -125, -120, 0, 0, 0, 63, -56, 57, -65, -19,
                    0, 0, 0, 63, -88, -44, 90, 108, 0, 0, 0, 63, -64, -7, 116,
                    100, 0, 0, 0, 63, -86, 54, -74, -12, 0, 0, 0, -65, -31, -30,
                    -58, 100, -64, 0, 0, 63, -85, -86, 83, 0, 0, 0, 0, 63, -23,
                    111, 64, -60, -64, 0, 0, 63, -83, 47, -8, 120, 0, 0, 0, -65,
                    -28, -13, 45, -110, -128, 0, 0, 63, -82, -56, 111, -100, 0,
                    0, 0, 63, -84, -35, 68, 16, 0, 0, 0, 63, -80, 58, 73, -36,
                    0, 0, 0, 63, -32, 127, 0, 107, 64, 0, 0, 63, -79, 26, -98,
                    4, 0, 0, 0, -65, -31, -93, -10, 83, 64, 0, 0, 63, -78, 5,
                    -85, 102, 0, 0, 0, -65, -61, 75, -21, 38, 0, 0, 0, 63, -78,
                    -5, -27, 50, 0, 0, 0, 63, -25, 0, 57, -88, -128, 0, 0, 63,
                    -77, -3, -52, -102, 0, 0, 0, -65, -43, 44, 20, 8, -128, 0,
                    0, 63, -75, 11, -36, -46, 0, 0, 0, -65, -31, -34, -114,
                    -106, -128, 0, 0, 63, -74, 38, -95, 8, 0, 0, 0, 63, -36,
                    -127, 23, -107, 0, 0, 0, 63, -73, 78, -99, -84, 0, 0, 0, 63,
                    -34, -69, -95, 14, 0, 0, 0, 63, -72, -124, 102, 120, 0, 0,
                    0, -65, -32, -14, 69, -38, 64, 0, 0, 63, -71, -56, -117,
                    -40, 0, 0, 0, -65, -32, -121, -34, 103, -64, 0, 0, 63, -69,
                    27, -90, 118, 0, 0, 0, 63, -39, 89, 72, -38, 0, 0, 0, 63,
                    -68, 126, 87, -94, 0, 0, 0, 63, -29, -99, -66, -42, 0, 0, 0,
                    63, -67, -15, 54, 116, 0, 0, 0, -65, -55, 78, -76, -34, 0,
                    0, 0, 63, -65, 116, -4, -82, 0, 0, 0, -65, -26, -22, 64, 89,
                    -64, 0, 0, 63, -64, -123, 27, 3, 0, 0, 0, -65, -50, -3, 125,
                    21, 0, 0, 0, 63, -63, 88, -33, 109, 0, 0, 0, 63, -33, 13,
                    -115, -60, 0, 0, 0, 63, -62, 54, 5, 67, 0, 0, 0, 63, -28,
                    -58, 6, 53, 64, 0, 0, 63, -61, 29, 17, -18, 0, 0, 0, -65,
                    -83, -44, -61, -116, 0, 0, 0, 63, -60, 14, 43, 126, 0, 0, 0,
                    -65, -29, -78, 62, -76, -64, 0, 0, 63, -59, 9, -12, 83, 0,
                    0, 0, -65, -29, -19, -18, -10, -64, 0, 0, 63, -58, 16, 119,
                    -96, 0, 0, 0, -65, -77, -73, -42, 14, 0, 0, 0, 63, -57, 34,
                    120, -123, 0, 0, 0, 63, -34, 119, 108, -99, 0, 0, 0, 63,
                    -56, 63, -30, 64, 0, 0, 0, 63, -26, -83, -80, -100, -64, 0,
                    0, 63, -55, 105, -105, -2, 0, 0, 0, 63, -46, 5, -96, -15,
                    -128, 0, 0, 63, -54, -97, 100, -116, 0, 0, 0, -65, -47, 120,
                    -44, -83, -128, 0, 0, 63, -53, -30, 66, -15, 0, 0, 0, -65,
                    -27, 126, 50, 30, 0, 0, 0, 63, -51, 49, -28, -110, 0, 0, 0,
                    -65, -28, -80, 66, 35, 0, 0, 0, 63, -50, -113, 73, 47, 0, 0,
                    0, -65, -41, 32, -17, 119, -128, 0, 0, 63, -49, -6, 26, 31,
                    0, 0, 0, 63, -74, 7, 77, 102, 0, 0, 0, 63, -48, -71, -99,
                    98, 0, 0, 0, 63, -31, 90, -44, 17, 64, 0, 0, 63, -47, 125,
                    53, 14, -128, 0, 0, 63, -26, -94, 62, 29, 0, 0, 0, 63, -46,
                    72, 31, 11, 0, 0, 0, 63, -32, -101, -14, -11, 64, 0, 0, 63,
                    -45, 26, 90, -23, 0, 0, 0, 63, -63, -34, 1, -36, 0, 0, 0,
                    63, -45, -13, -20, 70, -128, 0, 0, -65, -47, 19, -123, -14,
                    -128, 0, 0, 63, -44, -43, 13, 101, -128, 0, 0, -65, -30, 25,
                    40, 47, -128, 0, 0, 63, -43, -67, 83, 25, 0, 0, 0, -65, -26,
                    77, -21, -50, -64, 0, 0, 63, -42, -83, 79, -58, 0, 0, 0,
                    -65, -27, 118, -97, -120, -64, 0, 0, 63, -41, -93, -2, 15,
                    0, 0, 0, -65, -32, -89, -118, 122, -128, 0, 0, 63, -40, -94,
                    100, 65, 0, 0, 0, -65, -45, 110, -111, 82, -128, 0, 0, 63,
                    -39, -90, -72, 34, -128, 0, 0, -65, -79, -46, 14, 48, 0, 0,
                    0, 63, -38, -78, -121, -43, -128, 0, 0, 63, -49, -121, -96,
                    108, 0, 0, 0, 63, -37, -61, 32, -94, -128, 0, 0, 63, -31,
                    -102, 72, 77, 0, 0, 0, 63, -36, -38, -100, -100, 0, 0, 0,
                    63, -27, -44, -8, 75, 64, 0, 0, 63, -35, -11, 75, -73, -128,
                    0, 0, 63, -26, 49, 92, 55, 0, 0, 0, 63, -33, 21, -65, 93,
                    -128, 0, 0, 63, -30, 73, 87, 61, 64, 0, 0, 63, -32, 27, -90,
                    45, -64, 0, 0, 63, -40, -2, 126, -69, 0, 0, 0, 63, -32, -82,
                    98, -88, 0, 0, 0, 63, -60, -29, -110, 102, 0, 0, 0, 63, -31,
                    64, 82, 94, -64, 0, 0, -65, -85, 96, 91, 4, 0, 0, 0, 63,
                    -31, -46, -49, -65, -64, 0, 0, -65, -48, 113, -64, -42, 0,
                    0, 0, 63, -30, 98, -51, -112, 64, 0, 0, -65, -38, -68, 102,
                    -91, -128, 0, 0, 63, -30, -15, 81, 95, 0, 0, 0, -65, -31,
                    110, -5, 67, 0, 0, 0, 63, -29, 123, 57, -90, -128, 0, 0,
                    -65, -28, 26, -87, 15, -64, 0, 0, 63, -28, 0, -38, -22,
                    -128, 0, 0, -65, -27, -56, 24, 7, 0, 0, 0, 63, -28, 127, 72,
                    54, -128, 0, 0, -65, -26, 119, -79, 75, 0, 0, 0, 63, -28,
                    -11, -74, 71, 64, 0, 0, -65, -26, 126, -105, -79, 64, 0, 0,
                    63, -27, 97, -60, -41, -64, 0, 0, -65, -27, -5, -3, -16, 64,
                    0, 0, 63, -27, -63, 15, 40, -64, 0, 0, -65, -27, 34, 101,
                    -55, 64, 0, 0, 63, -26, 18, 34, -27, -64, 0, 0, -65, -28,
                    24, -99, -49, 0, 0, 0, 63, -26, 80, -119, -121, 0, 0, 0,
                    -65, -30, -16, -46, 103, 0, 0, 0, 63, -26, 124, 32, -98,
                    -64, 0, 0, -65, -31, -48, -28, -55, -64, 0, 0, 63, -26,
                    -115, -6, 10, 0, 0, 0, -65, -32, -76, 31, 26, -64, 0, 0, 63,
                    -26, -121, -100, 75, 64, 0, 0, -65, -33, 121, -104, -99, 0,
                    0, 0, 63, -26, 95, 100, 7, -128, 0, 0, -65, -35, -78, 83,
                    60, 0, 0, 0, 63, -26, 24, -57, 81, 0, 0, 0, -65, -36, 78,
                    22, -86, -128, 0, 0, 63, -27, -89, 113, -4, -64, 0, 0, -65,
                    -37, 30, -124, -87, -128, 0, 0, 63, -27, 16, -10, 127, -64,
                    0, 0, -65, -38, 85, 111, 36, -128, 0, 0, 63, -28, 70, -78,
                    -26, -128, 0, 0, -65, -39, -61, -95, 119, 0, 0, 0, 63, -29,
                    80, 101, -46, 64, 0, 0, -65, -39, -112, -115, 43, -128, 0,
                    0, 63, -30, 29, -2, -81, 64, 0, 0, -65, -39, -111, -63, 125,
                    0, 0, 0, 63, -32, -71, 92, -46, 0, 0, 0, -65, -39, -28, 69,
                    16, 0, 0, 0, 63, -34, 36, -16, 122, -128, 0, 0, -65, -38,
                    100, -37, -65, 0, 0, 0, 63, -38, 106, 96, -55, 0, 0, 0, -65,
                    -37, 38, -82, 17, 0, 0, 0, 63, -42, 39, 85, 33, 0, 0, 0,
                    -65, -36, 14, 33, 82, 0, 0, 0, 63, -47, 119, 46, -80, -128,
                    0, 0, -65, -35, 37, -105, 5, -128, 0, 0, 63, -56, -112,
                    -118, -42, 0, 0, 0, -65, -34, 88, 100, -116, -128, 0, 0, 63,
                    -70, -28, 121, -10, 0, 0, 0, -65, -33, -87, -98, 26, -128,
                    0, 0, 63, -119, 34, -92, 64, 0, 0, 0, -65, -32, -123, 47,
                    53, 64, 0, 0, -65, -75, -100, 7, 12, 0, 0, 0, -65, -31, 60,
                    12, 44, 0, 0, 0, -65, -57, 126, 23, -88, 0, 0, 0, -65, -31,
                    -12, 54, -30, -128, 0, 0, -65, -46, 29, -83, 7, -128, 0, 0,
                    -65, -30, -86, 114, -44, 64, 0, 0, -65, -40, 74, -125, 121,
                    -128, 0, 0, -65, -29, 91, 72, 77, 64, 0, 0, -65, -34, 39,
                    -73, 65, -128, 0, 0, -65, -28, 2, 83, 7, 64, 0, 0, -65, -31,
                    -85, 110, -101, -64, 0, 0, -65, -28, -99, 100, 118, 64, 0,
                    0, -65, -29, -29, 29, 57, 64, 0, 0, -65, -27, 39, -90, -96,
                    0, 0, 0, -65, -27, 119, -75, -92, 0, 0, 0, -65, -27, -97,
                    -27, -34, -128, 0, 0, -65, -26, 110, -62, -112, 64, 0, 0,
                    -65, -26, 1, 111, -67, 64, 0, 0, -65, -26, 114, -41, 64,
                    -128, 0, 0, -65, -26, 75, -41, -118, -64, 0, 0, -65, -27,
                    -90, -31, -123, -64, 0, 0, -65, -26, 123, 0, 25, 64, 0, 0,
                    -65, -29, -90, -57, 30, 64, 0, 0, -65, -26, -113, 33, -103,
                    0, 0, 0, -65, -32, -52, -26, 84, -128, 0, 0, -65, -26, -123,
                    1, -28, 64, 0, 0, -65, -39, 70, -27, 117, 0, 0, 0, -65, -26,
                    93, 102, 118, 0, 0, 0, -65, -49, -56, 90, 97, 0, 0, 0, -65,
                    -26, 22, 35, 81, -64, 0, 0, -65, -82, -85, -7, -92, 0, 0, 0,
                    -65, -27, -80, 120, 104, 0, 0, 0, 63, -78, -58, -109, -68,
                    0, 0, 0, -65, -27, 43, 83, -31, 64, 0, 0, 63, -72, -18, 49,
                    104, 0, 0, 0, -65, -28, -120, 94, 42, 0, 0, 0, 63, -63, -99,
                    35, 2, 0, 0, 0, -65, -29, -57, -111, -85, 64, 0, 0, 63, -58,
                    15, -94, 118, 0, 0, 0, -65, -30, -22, -21, -37, 64, 0, 0,
                    63, -53, 88, -88, -60, 0, 0, 0, -65, -31, -13, 82, 55, 64,
                    0, 0, 63, -48, 63, -18, 34, 0, 0, 0, -65, -32, -29, 1, 122,
                    -64, 0, 0, 63, -45, 11, -46, -68, 0, 0, 0, -65, -33, 119,
                    56, 126, 0, 0, 0, 63, -43, -36, -10, -87, -128, 0, 0, -65,
                    -36, -1, 15, -10, 0, 0, 0, 63, -40, -58, -75, -91, 0, 0, 0,
                    -65, -38, 97, -35, 81, -128, 0, 0, 63, -37, -75, -12, 36, 0,
                    0, 0, -65, -41, -92, -111, 21, -128, 0, 0, 63, -34, -96,
                    -41, -42, 0, 0, 0, -65, -44, -52, 32, -107, -128, 0, 0, 63,
                    -32, -64, 111, -10, 64, 0, 0, -65, -47, -35, 117, 127, 0, 0,
                    0, 63, -30, 29, -123, 22, -128, 0, 0, -65, -51, -69, -121,
                    60, 0, 0, 0, 63, -29, 101, 5, 56, -128, 0, 0, -65, -57, -93,
                    -119, 56, 0, 0, 0, 63, -28, -124, 51, 40, 0, 0, 0, -65, -63,
                    125, 41, 21, 0, 0, 0, 63, -27, 117, -88, 29, -128, 0, 0,
                    -65, -74, -94, 100, -80, 0, 0, 0, 63, -26, 35, 104, -15,
                    -128, 0, 0, -65, -92, -92, -91, -16, 0, 0, 0, 63, -26, -124,
                    20, -119, -64, 0, 0, 63, 126, 97, 28, 32, 0, 0, 0, 63, -26,
                    127, -5, 123, -128, 0, 0, 63, -85, -21, -46, -32, 0, 0, 0,
                    63, -26, 11, 17, -44, -128, 0, 0, 63, -71, -49, 126, -70, 0,
                    0, 0, 63, -27, 13, -5, -122, 64, 0, 0, 63, -62, -78, 70,
                    -80, 0, 0, 0, 63, -29, 126, -100, -51, -64, 0, 0, 63, -56,
                    84, -39, -32, 0, 0, 0, 63, -31, 73, -124, 104, -64, 0, 0,
                    63, -51, -55, -65, -14, 0, 0, 0, 63, -36, -34, 126, -95, 0,
                    0, 0, 63, -47, -122, 62, 92, 0, 0, 0, 63, -43, -52, -110,
                    13, -128, 0, 0, 63, -44, 12, 96, -44, -128, 0, 0, 63, -53,
                    27, -89, -108, 0, 0, 0, 63, -42, 117, -94, 70, 0, 0, 0, 63,
                    -80, -51, 33, 86, 0, 0, 0, 63, -40, -64, -122, -40, 0, 0, 0,
                    -65, -73, 13, -37, -72, 0, 0, 0, 63, -38, -21, -5, 36, -128,
                    0, 0, -65, -48, 19, -120, -98, -128, 0, 0, 63, -36, -9, 31,
                    1, 0, 0, 0, -65, -39, -44, 62, -40, 0, 0, 0, 63, -34, -31,
                    100, 1, 0, 0, 0, -65, -31, 75, 86, -55, 64, 0, 0, 63, -32,
                    85, 57, 23, 64, 0, 0, -65, -28, -123, 44, -111, 0, 0, 0, 63,
                    -31, 41, 24, 30, -64, 0, 0, -65, -26, 102, 30, -108, 64, 0,
                    0, 63, -31, -20, 92, 76, 64, 0, 0, -65, -26, 46, 70, -87,
                    -128, 0, 0, 63, -30, -97, 42, -49, -128, 0, 0, -65, -29, -7,
                    -96, 121, -128, 0, 0, 63, -29, 65, -65, 33, 0, 0, 0, -65,
                    -34, 84, 96, -90, -128, 0, 0, 63, -29, -44, 103, 5, 0, 0, 0,
                    -65, -47, 87, 105, 113, -128, 0, 0, 63, -28, 87, -126, 5,
                    -128, 0, 0, -65, 85, -59, 122, 0, 0, 0, 0, 63, -28, -53,
                    125, 53, -128, 0, 0, 63, -53, -50, 57, 82, 0, 0, 0, 63, -27,
                    48, -46, -10, 64, 0, 0, 63, -44, -16, -38, -63, 0, 0, 0, 63,
                    -27, -120, 7, 2, -64, 0, 0, 63, -36, -119, -98, -76, -128,
                    0, 0, 63, -27, -47, -90, 68, 64, 0, 0, 63, -31, -120, 77,
                    53, 0, 0, 0, 63, -26, 14, 67, 126, -128, 0, 0, 63, -28, 66,
                    -96, 73, 64, 0, 0, 63, -26, 62, 118, -24, 64, 0, 0, 63, -26,
                    8, 80, 66, -64, 0, 0, 63, -26, 98, -37, -67, -64, 0, 0, 63,
                    -26, -120, -76, -79, 0, 0, 0, 63, -26, 124, 15, 117, -128,
                    0, 0, 63, -27, -124, 51, -4, 0, 0, 0, 63, -26, -118, -80,
                    71, 0, 0, 0, 63, -30, -90, -5, -78, 0, 0, 0, 63, -26, -113,
                    92, 0, -64, 0, 0, 63, -37, -6, -56, -2, -128, 0, 0, 63, -26,
                    -118, -81, 92, -64, 0, 0, 63, -50, 38, -52, 64, 0, 0, 0, 63,
                    -26, 125, 68, -75, 64, 0, 0, -65, 106, -100, -111, -64, 0,
                    0, 0, 63, -26, 103, -77, -59, 0, 0, 0, -65, -48, 123, -43,
                    -12, 0, 0, 0, 63, -26, 74, -112, -108, 0, 0, 0, -65, -33,
                    56, -89, 1, -128, 0, 0, 63, -26, 38, 107, 71, 64, 0, 0, -65,
                    -28, -41, 97, -115, -128, 0, 0, 63, -27, -5, -49, 123, -64,
                    0, 0, -65, -26, -113, 30, 103, -64, 0, 0, 63, -27, -53, 67,
                    -4, -64, 0, 0, -65, -29, -102, 22, -80, -64, 0, 0, 63, -27,
                    -107, 74, -114, -64, 0, 0, -65, -40, 80, -64, 70, 0, 0, 0,
                    63, -27, 90, 95, -120, -64, 0, 0, -65, -105, -62, 88, -64,
                    0, 0, 0, 63, -27, 26, -7, -6, 0, 0, 0, 63, -48, 100, -4, 29,
                    0, 0, 0, 63, -28, -41, -117, 54, 0, 0, 0, 63, -35, 47, -72,
                    -117, 0, 0, 0, 63, -28, -112, 127, 49, -128, 0, 0, 63, -28,
                    7, 8, 0, -64, 0, 0, 63, -28, 70, 60, 21, 0, 0, 0, 63, -26,
                    116, 56, -39, -128, 0, 0, 63, -29, -7, 34, -72, 0, 0, 0, 63,
                    -27, 54, -88, -53, -128, 0, 0, 63, -29, -87, -114, 65, -128,
                    0, 0, 63, -33, 59, -13, -32, 0, 0, 0, 63, -29, 87, -44, -80,
                    -64, 0, 0, 63, -56, 60, 73, -112, 0, 0, 0, 63, -29, 4, 70,
                    -111, 64, 0, 0, -65, -58, 8, -15, 84, 0, 0, 0, 63, -30, -81,
                    47, -124, 0, 0, 0, -65, -32, 87, -82, 61, 64, 0, 0, 63, -30,
                    88, -42, 9, 0, 0, 0, -65, -26, 35, -69, -119, -64, 0, 0, 63,
                    -30, 1, 124, 2, -64, 0, 0, -65, -28, 87, -99, -98, 64, 0, 0,
                    63, -31, -87, 94, -111, 64, 0, 0, -65, -45, 3, 28, -55, 0,
                    0, 0, 63, -31, 80, -74, -114, -128, 0, 0, 63, -58, 99, -107,
                    -26, 0, 0, 0, 63, -32, -9, -72, 118, -128, 0, 0, 63, -32,
                    75, 60, -115, 64, 0, 0, 63, -32, -98, -108, -34, 64, 0, 0,
                    63, -27, -10, -8, 88, -64, 0, 0, 63, -32, 69, 120, 96, -128,
                    0, 0, 63, -28, -29, -126, 123, 64, 0, 0, 63, -33, -39, 24,
                    37, -128, 0, 0, 63, -42, 97, 18, -1, 0, 0, 0, 63, -33, 39,
                    -22, -25, 0, 0, 0, -65, -67, -67, -45, -14, 0, 0, 0, 63,
                    -34, 119, -83, -70, 0, 0, 0, -65, -31, -36, 58, -38, -64, 0,
                    0, 63, -35, -56, -98, -35, -128, 0, 0, -65, -26, 82, -113,
                    -127, 0, 0, 0, 63, -35, 26, -9, 16, -128, 0, 0, -65, -39,
                    -77, 108, 121, 0, 0, 0, 63, -36, 110, -23, 49, -128, 0, 0,
                    63, -54, 61, 52, -56, 0, 0, 0, 63, -37, -60, -93, -119, 0,
                    0, 0, 63, -28, 98, 116, -68, -64, 0, 0, 63, -37, 28, 78,
                    -53, -128, 0, 0, 63, -27, 73, 96, 4, -64, 0, 0, 63, -38,
                    118, 16, 65, -128, 0, 0, 63, -45, 98, -1, -37, -128, 0, 0,
                    63, -39, -46, 7, -128, 0, 0, 0, -65, -44, -114, 42, -15,
                    -128, 0, 0, 63, -39, 48, 82, 66, -128, 0, 0, -65, -26, 7,
                    54, -101, -64, 0, 0, 63, -40, -111, 7, -40, 0, 0, 0, -65,
                    -35, -51, -69, -109, 0, 0, 0, 63, -41, -12, 63, -39, -128,
                    0, 0, 63, -48, -59, -44, 99, -128, 0, 0, 63, -41, 90, 10, 1,
                    -128, 0, 0, 63, -27, 118, 75, -116, 0, 0, 0, 63, -42, -62,
                    121, 24, -128, 0, 0, 63, -31, 25, 7, 100, 64, 0, 0, 63, -42,
                    45, -107, -23, 0, 0, 0, -65, -58, -95, 62, 125, 0, 0, 0, 63,
                    -43, -101, 111, -122, -128, 0, 0, -65, -27, -69, 90, -76,
                    -128, 0, 0, 63, -43, 12, 8, -126, -128, 0, 0, -65, -40, 8,
                    81, -41, -128, 0, 0, 63, -44, 127, 108, -11, 0, 0, 0, 63,
                    -33, 125, 16, -56, -128, 0, 0, 63, -45, -11, -102, 68, 0, 0,
                    0, 63, -28, -71, 13, 81, 64, 0, 0, 63, -45, 110, -103, -94,
                    0, 0, 0, -65, 121, -87, 70, -96, 0, 0, 0, 63, -46, -22, 100,
                    -25, -128, 0, 0, -65, -26, -124, -31, -123, 0, 0, 0, 63,
                    -46, 105, 1, -108, -128, 0, 0, -65, -51, 57, 21, 0, 0, 0, 0,
                    63, -47, -22, 104, 115, -128, 0, 0, 63, -28, 67, -99, 49,
                    -64, 0, 0, 63, -47, 110, -103, 96, 0, 0, 0, 63, -38, -29,
                    115, -19, -128, 0, 0, 63, -48, -11, -113, -88, 0, 0, 0, -65,
                    -32, -57, 4, -111, -64, 0, 0, 63, -48, 127, 66, 73, -128, 0,
                    0, -65, -35, 22, 79, 22, -128, 0, 0, 63, -48, 11, -77, -49,
                    -128, 0, 0, 63, -29, 81, -90, -35, 0, 0, 0, 63, -49, 53,
                    -101, -54, 0, 0, 0, 63, -41, 20, -53, 34, 0, 0, 0, 63, -50,
                    89, 64, 25, 0, 0, 0, -65, -29, 19, 126, -68, 64, 0, 0, 63,
                    -51, -126, 2, 123, 0, 0, 0, -65, -58, -101, 63, 23, 0, 0, 0,
                    63, -52, -80, 40, 91, 0, 0, 0, 63, -24, -91, -96, -50, 0, 0,
                    0, 63, -53, -29, 47, 22, 0, 0, 0, -65, -41, -114, 4, 95,
                    -128, 0, 0, 63, -53, 27, -114, 46, 0, 0, 0, -65, -40, 49,
                    -117, -81, 0, 0, 0, 63, -54, 88, -119, 16, 0, 0, 0, 63, -29,
                    -121, -96, 15, -128, 0, 0, 63, -55, -102, -47, -87, 0, 0, 0,
                    -65, -61, 105, 14, -83, 0, 0, 0, 63, -56, -31, 108, -33, 0,
                    0, 0, -65, -33, 63, -16, -111, -128, 0, 0, 63, -56, 45, 72,
                    -84, 0, 0, 0, 63, -22, -122, 10, -109, -128, 0, 0, 63, -57,
                    125, 47, 67, 0, 0, 0, -65, -27, 77, 102, 63, -64, 0, 0, 63,
                    -58, -46, 65, -16, 0, 0, 0, 63, -48, 113, -66, -14, -128, 0,
                    0, 63, -58, 43, 32, 18, 0, 0, 0, 63, -57, 61, 31, -100, 0,
                    0, 0, 63, -59, -119, 7, -105, 0, 0, 0, -65, -37, -79, 96,
                    -95, -128, 0, 0, 63, -60, -22, -116, 96, 0, 0, 0, 63, -32,
                    51, 36, 61, -64, 0, 0, 63, -60, 80, -31, 125, 0, 0, 0, -65,
                    -35, -18, -128, -112, -128, 0, 0, 63, -61, -70, -64, 56, 0,
                    0, 0, 63, -41, -15, -93, 33, -128, 0, 0, 63, -61, 41, 23,
                    61, 0, 0, 0, -65, -49, 26, -36, -127, 0, 0, 0, 63, -62,
                    -101, 7, -36, 0, 0, 0, 63, -61, 69, -32, -25, 0, 0, 0, 63,
                    -62, 16, -14, 0, 0, 0, 0, -65, -87, -65, -61, -96, 0, 0, 0,
                    63, -63, -118, -80, -78, 0, 0, 0, 63, 104, 11, -91, 64, 0,
                    0, 0, 63, -63, 7, -66, 20, 0, 0, 0, 63, -87, 42, -43, -36,
                    0, 0, 0, 63, -64, -119, 9, -46, 0, 0, 0, -65, -77, 65, -65,
                    -116, 0, 0, 0, 63, -64, 12, -52, 89, 0, 0, 0, 63, -74, -101,
                    -83, -58, 0, 0, 0, 63, -65, 42, -56, -30, 0, 0, 0, -65, -81,
                    45, 20, -48, 0, 0, 0, 63, -66, 62, -26, -48, 0, 0, 0, 63,
                    -81, 118, -95, -108, 0, 0, 0, 63, -67, 94, 40, 80, 0, 0, 0,
                    -65, -82, 117, 104, -28, 0, 0, 0, 63, -68, 126, 32, -26, 0,
                    0, 0, 63, -80, 115, 69, 50, 0, 0, 0, 63, -69, -86, -34, 124,
                    0, 0, 0, -65, -99, -26, -124, 32, 0, 0, 0, 63, -70, -42, 15,
                    -78, 0, 0, 0, 63, -93, -45, 109, -44, 0, 0, 0, 63, -70, 15,
                    -98, 2, 0, 0, 0, 63, -127, -37, 95, 16, 0, 0, 0, 63, -71,
                    69, -118, -82, 0, 0, 0, -65, -92, -52, 45, 36, 0, 0, 0, 63,
                    -72, -117, 32, -84, 0, 0, 0, 63, -87, -5, 115, 84, 0, 0, 0,
                    63, -73, -53, 119, -96, 0, 0, 0, -65, -87, -115, -86, -80,
                    0, 0, 0, 63, -73, 28, 40, -26, 0, 0, 0, 63, -74, -12, 85,
                    22, 0, 0, 0, 63, -74, 102, -55, -126, 0, 0, 0, -65, -65,
                    -114, 92, -36, 0, 0, 0, 63, -75, -63, -125, -78, 0, 0, 0,
                    63, -64, -93, -123, -55, 0, 0, 0, 63, -75, 22, 126, -110, 0,
                    0, 0, -65, -60, -96, -30, 79, 0, 0, 0, 63, -76, 122, 11, 0,
                    0, 0, 0, 63, -60, -109, 97, 47, 0, 0, 0, 63, -77, -39, -99,
                    -18, 0, 0, 0, -65, -69, 10, 14, 2, 0, 0, 0, 63, -77, 68,
                    -88, 54, 0, 0, 0, -65, -76, -72, -116, -102, 0, 0, 0, 63,
                    -78, -81, 52, -44, 0, 0, 0, 63, -38, 34, 4, 75, -128, 0, 0,
                    63, -78, 32, 86, -46, 0, 0, 0, -65, -24, -90, 85, -58, 0, 0,
                    0, 63, -79, -106, 83, -40, 0, 0, 0, 63, -30, -105, 64, 26,
                    -64, 0, 0, 63, -79, 12, 38, -62, 0, 0, 0, 63, -48, 55, -69,
                    121, -128, 0, 0, 63, -80, -114, 12, 80, 0, 0, 0, -65, -28,
                    62, -6, -92, -64, 0, 0, 63, -80, 7, 62, 66, 0, 0, 0, -65,
                    -33, 55, 73, 0, 0, 0, 0, 63, -81, 42, -36, -104, 0, 0, 0,
                    63, -59, 54, 78, 79, 0, 0, 0, 63, -82, 33, -74, 44, 0, 0, 0,
                    63, -30, 86, -74, -109, 0, 0, 0, 63, -83, 87, 14, 100, 0, 0,
                    0, 63, -26, 11, 75, 15, 64, 0, 0, 63, -84, 80, -91, -68, 0,
                    0, 0, 63, -26, -127, -73, 103, -128, 0, 0, 63, -85, -98,
                    -62, 84, 0, 0, 0, 63, -27, -93, -3, -41, 64, 0, 0, 63, -86,
                    -102, 37, -108, 0, 0, 0, 63, -36, 1, -125, 75, -128, 0, 0,
                    63, -86, 0, 5, -4, 0, 0, 0, -65, -60, -43, -22, -40, 0, 0,
                    0, 63, -88, -3, 61, -8, 0, 0, 0, -65, -25, -3, 7, -90, -64,
                    0, 0, 63, -88, 120, -33, 124, 0, 0, 0, 63, -46, -37, 13,
                    -62, -128, 0, 0, 63, -89, 121, 37, -104, 0, 0, 0, 63, -50,
                    -128, -59, -125, 0, 0, 0, 63, -89, 7, 73, 52, 0, 0, 0, -65,
                    -41, -93, -27, 70, 0, 0, 0, 63, -90, 13, 72, -32, 0, 0, 0,
                    63, -43, 15, 45, 20, 0, 0, 0, 63, -91, -87, 33, 104, 0, 0,
                    0, -65, -47, -66, -78, 59, 0, 0, 0, 63, -92, -71, 101, -36,
                    0, 0, 0, 63, -48, 18, 90, -72, -128, 0, 0, 63, -92, 91, -10,
                    92, 0, 0, 0, -65, -49, 42, -102, -49, 0, 0, 0, 63, -93, 125,
                    -26, -36, 0, 0, 0, 63, -55, -5, 67, 105, 0, 0, 0, 63, -93,
                    28, 93, -20, 0, 0, 0, 63, -85, 109, -126, -76, 0, 0, 0, 63,
                    -94, 93, 36, -20, 0, 0, 0, -65, -28, -63, -123, 118, 0, 0,
                    0, 63, -95, -29, 88, 16, 0, 0, 0, 63, -57, 116, -110, -20,
                    0, 0, 0, 63, -95, 97, 107, 88, 0, 0, 0, 63, -37, 101, -94,
                    50, 0, 0, 0, 63, -96, -107, -6, 36, 0, 0, 0, 63, -87, 72,
                    -66, 4, 0, 0, 0, 63, -96, -41, -86, 72, 0, 0, 0};

    public static void main(final String[] args) throws Exception {
        test(data32, 32);
        test(data64, 64);
    }

    private static void test(final byte[] data, final int bits)
            throws Exception {

        // Audio data should contain the same format as a command lines above
        checkAudioStream(new ByteArrayInputStream(data), data, bits);

        // Checks that the data will be preserved when we save audio data to the
        // file and read it again
        final File file = File.createTempFile("sound", "test");
        try (final InputStream fis = new FileInputStream(file);
             final InputStream is = new ByteArrayInputStream(data);
             final AudioInputStream ais = AudioSystem.getAudioInputStream(is)) {
            AudioSystem.write(ais, AudioFileFormat.Type.AU, file);
            if (!Arrays.equals(data, fis.readAllBytes())) {
                throw new RuntimeException("Wrong data in the file");
            }
        } finally {
            Files.delete(Paths.get(file.getAbsolutePath()));
        }

        // Checks that the data will be preserved when we save audio data to the
        // stream and read it again
        try (final ByteArrayOutputStream baos = new ByteArrayOutputStream();
             final InputStream is = new ByteArrayInputStream(data);
             final AudioInputStream ais = AudioSystem.getAudioInputStream(is)) {
            AudioSystem.write(ais, AudioFileFormat.Type.AU, baos);
            if (!Arrays.equals(data, baos.toByteArray())) {
                throw new RuntimeException("Wrong data in the stream");
            }
        }
    }

    /**
     * Tests that audio data is the same format as a command lines above.
     */
    private static void checkAudioStream(final InputStream is,
                                         final byte[] data, final int bits)
            throws Exception {
        final AudioFileFormat aff = AudioSystem.getAudioFileFormat(is);
        final int byteLength = aff.getByteLength();
        if (byteLength != data.length) {
            throw new RuntimeException("Wrong byteLength: " + byteLength);
        }
        final AudioFileFormat.Type type = aff.getType();
        if (!type.equals(AudioFileFormat.Type.AU)) {
            throw new RuntimeException("Wrong type: " + type);
        }
        final int frameLength = aff.getFrameLength();
        if (frameLength != 0.01f * aff.getFormat().getFrameRate()) {
            throw new RuntimeException("Wrong frameLength: " + frameLength);
        }
        checkAudioFormat(aff.getFormat(), bits);
        try (AudioInputStream ais = AudioSystem.getAudioInputStream(is)) {
            final AudioFormat format = ais.getFormat();
            if (ais.getFrameLength() != 0.01f * format.getFrameRate()) {
                throw new RuntimeException(
                        "Wrong frameLength: " + ais.getFrameLength());
            }
            checkAudioFormat(format, bits);
        }
    }

    /**
     * Tests that audio data is the same format as a command lines above.
     */
    private static void checkAudioFormat(final AudioFormat audioFormat,
                                         final int bits) {
        final int channels = audioFormat.getChannels();
        if (channels != 2) {
            throw new RuntimeException("Wrong channels: " + channels);
        }
        final int sizeInBits = audioFormat.getSampleSizeInBits();
        if (sizeInBits != bits) {
            throw new RuntimeException("Wrong sizeInBits: " + sizeInBits);
        }
        final int frameSize = audioFormat.getFrameSize();
        if (frameSize != sizeInBits * channels / 8) {
            throw new RuntimeException("Wrong frameSize: " + frameSize);
        }
        final AudioFormat.Encoding encoding = audioFormat.getEncoding();
        if (!encoding.equals(AudioFormat.Encoding.PCM_FLOAT)) {
            throw new RuntimeException("Wrong encoding: " + encoding);
        }
        final float frameRate = audioFormat.getFrameRate();
        if (frameRate != 32000) {
            throw new RuntimeException("Wrong frameRate: " + frameRate);
        }
        final float sampleRate = audioFormat.getSampleRate();
        if (sampleRate != 32000) {
            throw new RuntimeException("Wrong sampleRate: " + sampleRate);
        }
    }
}
