/* Copyright (c) 2001 Ben Woodard
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public 
 * Licence as published by the Free Software Foundation; either 
 * version 2 of the Licence, or (at your option) any later 
 * version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRENTY; without even the implied warrenty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public Licence in the COPYING file for more
 * details.
 *
 * You should have received a copy of the GNU Library General 
 * Public License along with the GNU C Library; see the file 
 * COPYING.LIB.  If not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* This is intended to be the external interface presented to the
   client programs who wish to print. */
#ifndef __PRINTSYS_H__
#define __PRINTSYS_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>
#include <wchar.h>

#define LPS_SOCK_PATHNAME "/var/run/gnulpr/master"
#define LPS_RELAYSOCK_PATH "/var/run/gnulpr"
#define LPS_DOCDBPATH "/var/run/gnulpr/docattr.tdb"

// these are for job aggregation
#define LPS_AGG_PREV  L"aggregate_previous"
#define LPS_AGG_SEQNO L"aggregate_seqno"
#define LPS_AGG_LAST  L"aggregate_last"
#define LPS_JOB_ATTR  L"job_attributes"

#define LPS_DOC_UID   L"docuid" 
#define LPS_DOC_GID   L"docgid"
#define LPS_DOC_PID   L"docpid"
#define LPS_JOBNAME   L"jobname"
#define LPS_JOBCLASS  L"jobclass"
#define LPS_DOCNAME   L"docname"
#define LPS_JOBBANNER L"banner"

// These are used by all filters
#define EXIT_OK 0
#define EXIT_NOSYSTEM 1
#define EXIT_BADPARAMS 2
#define EXIT_NOMEM 3

typedef enum {LPS_OK=0,LPS_ACCESS,LPS_UNIMP,LPS_CONNTERM,LPS_ATTRSHRT,
	      LPS_NOMEM,LPS_TDBOPENFAIL,LPS_TDBSTOREFAIL,LPS_NODOC,
	      LPS_BADMAGIC,LPS_NOSYS,LPS_BADNAME,
	      // These have errno set as well
	      LPS_SOCKET,LPS_CONNECT,LPS_SENDPR,LPS_READPR,LPS_SENDATTR,
	      LPS_READDOCID,LPS_NOCONFIG,LPS_INSANE,LPS_PRINTCAP,
	      // Have error message sent from server
	      LPS_PRNOTOK,LPS_BADATTR
} LPS_Error_t;

typedef struct {
  int magic;
  LPS_Error_t lps_errno;
  void *data;
} LPS_System_t;

extern LPS_System_t *lps_init(const void *initdata);
extern LPS_Error_t lps_end (LPS_System_t *system); 

/***** Pair data structure *****/
typedef struct {
  wchar_t *key;
  wchar_t *value;
} LPS_Pair_t;

extern wchar_t *lps_pair_lookup(LPS_Pair_t * pairs, const wchar_t *key);
extern LPS_Pair_t *lps_pair_update(LPS_Pair_t *pairs,const wchar_t *key,
				   const wchar_t *value);
extern unsigned int lps_num_pairs(LPS_Pair_t *pairs);
extern void lps_free_pairs(LPS_Pair_t *doomed);

/***** Printer data structures *****/
extern const wchar_t **lps_get_printer_list(LPS_System_t *sys,
					    const char *ns);
extern int lps_request_printer_list(LPS_System_t *sys,const char *ns);

extern wchar_t **lps_decode_printer_list(int fd, wchar_t **dest, 
					 unsigned int dest_len, int *more);
extern void lps_free_printer_list(wchar_t **doomed);

typedef struct {
  wchar_t **names;
  LPS_Pair_t *fields;
  wchar_t *comments;
  LPS_System_t *system;
} LPS_Printer_t;

extern LPS_Printer_t *lps_get_printer (LPS_System_t *sys,
				       const wchar_t *prname,
				       const char *ns);
extern int lps_request_printer(LPS_System_t *sys,const wchar_t *prname,
			       const char *ns);
extern LPS_Printer_t *lps_decode_printer(LPS_System_t *sys,int fd);
extern void lps_free_printer(LPS_Printer_t *doomed);

// create_printer takes ownership of prnames fields and comment
extern LPS_Printer_t *lps_create_printer(LPS_System_t *sys,
					 wchar_t **prnames, LPS_Pair_t *fields,
					 wchar_t *comments);
extern int lps_pr_update_field( LPS_Printer_t *printer, 
				const wchar_t *key , const wchar_t *value);

#define lps_pr_lookup_field( x_printer, x_key ) \
  lps_pair_lookup( (x_printer)->fields, (x_key) )
#define lps_pr_name( x_printer) ((x_printer)->names[0])
#define lps_pr_num_fields(x_printer) lps_num_pairs((x_printer)->fields)

/* apply to backend store */
extern LPS_Error_t lps_commit_printer(LPS_Printer_t *printer, 
				      const char *ns);
extern LPS_Error_t lps_destroy_printer(LPS_Printer_t *printer, 
				       const char *ns);

/***** Job data structure ****/
typedef struct {
  LPS_Printer_t *printer;
  unsigned int last_docid;
  unsigned int seqno;
  LPS_Pair_t *attributes;
} LPS_Job_t;

extern LPS_Job_t *lps_create_job(LPS_Printer_t *printer,
				 LPS_Pair_t *attributes);
extern void lps_free_job(LPS_Job_t *doomed);


/***** Document submission functions *****/
extern int lps_get_connection(LPS_Job_t *job,LPS_Pair_t *attributes,
			      unsigned int *docid,int errfd,int last);
extern LPS_Error_t lps_send_ref(LPS_Job_t *job,LPS_Pair_t *attributes,
				unsigned int *docid,int srcfd, 
				int errfd, int last);

extern LPS_Pair_t *lps_lookup_doc_attrib(unsigned int docid);
extern LPS_Error_t lps_commit_doc_attrib(unsigned int docid, 
					 LPS_Pair_t *attribs);

/***** Utility functions *****/
/* These functions are not really part of the print system and 
   they have virtually nothing to do with printing but 
   they end up being useful in more than one place and so I 
   decided to move them into here. At some point, I'll probably 
   move them into another library or header file. */

extern wchar_t *lps_promote(const char *str);
extern wchar_t **lps_promote_vec(const char **strs);

extern char *lps_demote(const wchar_t *str);
extern char **lps_demote_vec(const wchar_t **str);

extern ssize_t lps_block_write(int fd, const char *buf, size_t len);
extern int lps_send_data(int destfd,int srcfd);

extern void lps_fail_doc(unsigned int docid, LPS_Pair_t *docattr);
extern unsigned int lps_filter_opts(LPS_System_t *sys,int argc, 
				    char **argv, LPS_Printer_t **printer, 
				    LPS_Pair_t **docattr);
extern char *lps_chdir_sd(LPS_Printer_t *printer, int *err);
extern void _lps_dump_printer(LPS_Printer_t *printer);
extern char *_lps_encode_pairs(LPS_Pair_t *pairs, size_t *length);
extern LPS_Pair_t *_lps_decode_pairs(const char *attr_str, 
				     size_t attrib_size, LPS_Pair_t *attrib);
extern LPS_Printer_t *_lps_process_entry(char *buf);
extern char *_lps_write_printcap_entry(LPS_Printer_t *curprinter);

#ifdef __cplusplus
}
#endif

#endif /* __PRINTSYS_H__ */
