/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
} from "../harness/environments.js";
import * as fs from "node:fs";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runTermOfServiceFormatTest(t: GlobalTestState) {
  // Set up test environment
  const tosDir = t.testDir + `/tos/`;
  const langs = ["es", "en", "de"]

  langs.forEach(l => {
    const langDir = tosDir + l + "/"
    fs.mkdirSync(langDir, { recursive: true });
    fs.writeFileSync(langDir + "v1.txt", "text content");
    fs.writeFileSync(langDir + "v1.md", "markdown content");
    fs.writeFileSync(langDir + "v1.html", "html content");
  });

  const { walletClient, exchange, } =
    await createSimpleTestkudosEnvironmentV2(t, undefined, {
      additionalExchangeConfig: (ex) => {
        ex.changeConfig((cfg) => {
          cfg.setString("exchange", "terms_etag", "v1")
          cfg.setString("exchange", "terms_dir", tosDir)
        })
      }
    });


  {
    const tos = await walletClient.client.call(WalletApiOperation.GetExchangeTos, {
      exchangeBaseUrl: exchange.baseUrl,
    })

    t.assertDeepEqual(tos.content, "text content");
  }

  {
    const tos = await walletClient.client.call(WalletApiOperation.GetExchangeTos, {
      exchangeBaseUrl: exchange.baseUrl,
      acceptedFormat: ["text/html"]
    })

    t.assertDeepEqual(tos.content, "html content");
  }

  {
    const tos = await walletClient.client.call(WalletApiOperation.GetExchangeTos, {
      exchangeBaseUrl: exchange.baseUrl,
      acceptedFormat: ["text/markdown"]
    })

    t.assertDeepEqual(tos.content, "markdown content");
  }

  {
    const tos = await walletClient.client.call(WalletApiOperation.GetExchangeTos, {
      exchangeBaseUrl: exchange.baseUrl,
      acceptedFormat: ["text/markdown", "text/html"]
    })

    // prefer markdown since its the first one in the list
    t.assertDeepEqual(tos.content, "markdown content");
  }

  {
    const tos = await walletClient.client.call(WalletApiOperation.GetExchangeTos, {
      exchangeBaseUrl: exchange.baseUrl,
      acceptedFormat: ["text/pdf", "text/html"]
    })

    // there is no pdf so fallback in html
    t.assertDeepEqual(tos.content, "html content");
  }
}

runTermOfServiceFormatTest.suites = ["wallet"];
