#include <kdevice.h>
#include <kdevice.moc>

#include <stdio.h>
#include <math.h>
#include <time.h>

#include <dsp56utils.h>
#include <d56ioctl.h>

#include <kmsgbox.h>
#include <kconfig.h>
#include <kapp.h>

/* constants */
//#define DEBUG_TEST
#define POINTS	1024

#define DSP_DRIVER	"/dev/dsp"

extern const char* chWarning;
const char* chOpenDeviceError = "Can't open device!";
const char* chCapsError = "Error on ioctl SNDCTL_DSP_GETiCaps!";
const char* chISpaceError = "Error on ioctl SNDCTL_DSP_GETISPACE!";
const char* chMMapError = "Cannot mmap, Arrgh!";
const char* chNoMappingError = "Sound device does not support memory mapping I/O!";
const char* chSetFMTError = "Error on ioctl SNDCTL_DSP_SETFMT";
const char* chChannelsError = "Error on ioctl SNDCTL_DSP_CHANNELS";
const char* chSpeedError = "Error on ioctl SNDCTL_DSP_SPEED";
const char* chSetTriggerError = "Error on ioctl SNDCTL_DSP_SETTRIGGER";
const char* chGetIPTRError = "Error on ioctl SNDCTL_DSP_GETIPTR";
const char* chLoaderError = "Error on loading Bootloader";
const char* chApplicationError = "Error on loading Application";
const char* chRecordFile = "/kanalyzer/asm/record.lod";
const char* chSpectrumFile = "/kanalyzer/asm/spectrum.lod";
const char* chLoaderFile = "/kanalyzer/asm/loader.lod";
const char* chFormatError = "Sorry! Only 8kHz, 16kHz, 32kHz and 48kHz are supported!";
const char* chFormatSetupError = "Format setup failed!";

extern const char* chDeviceFFTEntry;
extern const char* chAbort;

extern KConfig* config;
extern KApplication* app;
extern char* appInitFile;

static short int device_buffer[BUFFERSIZE];
WaveInfoStruct WaveInfo;

/****************************************************************/
/*																*/
/*							KDevice								*/
/*																*/
/****************************************************************/

KDevice::KDevice(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	iDataPtr = 0;
	bActive = false;
	iFrequency = 400;
}

KDevice::~KDevice()
{
}

int KDevice::open(void)
{
	int intervall;
	iDataPtr = iTimePtr = 0;
	
	intervall = (int)(1024000.0/(double)WaveInfo.iSampleFrequency);
	startTimer(intervall);

	return 0;
}

int KDevice::close(void)
{
	killTimers();
	return 0;
}

short int* KDevice::getbufadr(void)
{
	return &device_buffer[0];
}

short int* KDevice::getcurbufptr(void)
{
	return &device_buffer[iDataPtr];
}

/****************************************************************/
/*																*/
/*							KSine								*/
/*																*/
/****************************************************************/

KSine::KSine(QWidget *parent, const char *name)
	: KDevice(parent, name)
{
}

KSine::~KSine()
{
}

void KSine::timerEvent(QTimerEvent*)
{
	clock_t acttime;
	int i, period;
	acttime = clock();
	period = WaveInfo.iSampleFrequency / iFrequency;
	
	for(i = 0; i < POINTS; i++)
	{
		device_buffer[iDataPtr] = (short int) (32767 * sin( 2 * M_PI * (float)iTimePtr / (float)period));
		iDataPtr = (iDataPtr + 1) % BUFFERSIZE;
		iTimePtr = (iTimePtr + 1) % period;
	}
}

/****************************************************************/
/*																*/
/*							KWaveIn								*/
/*																*/
/****************************************************************/

KWaveIn::KWaveIn(QWidget *parent, const char *name)
	: KDevice(parent, name)
{
	iwavein = 0;
	iChannels = 1;
	iBytesPerSample = 2;
	iFormat = AFMT_S16_LE;
}

KWaveIn::~KWaveIn()
{
}

int KWaveIn::open(void)
{
	int intervall;

    iDataPtr = iPrevPtr = iCurrentPtr = 0;
	
    audio_buf_info sBuffInfo;

    if((iwavein = ::open(DSP_DRIVER, O_RDONLY)) < 0)
	{
		KMsgBox::message(this,  chWarning, chOpenDeviceError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}
    /* query iCapsabilites */
    if(ioctl(iwavein, SNDCTL_DSP_GETCAPS, &iCaps) != 0)
	{
		KMsgBox::message(this, chWarning, chOpenDeviceError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}

#ifdef DEBUG_TEST
    if(iCaps & DSP_CAP_DUPLEX)		printf ("Debug:\tSound device support full duplex mode\n");
    if(iCaps & DSP_CAP_REALTIME)	printf ("Debug:\tSound device has realtime iCapsabilities\n");
    if(iCaps & DSP_CAP_MMAP)		printf ("Debug:\tSound device support memory mapping I/O\n");
    if(iCaps & DSP_CAP_TRIGGER)		printf ("Debug:\tSound device support triggering\n");
#endif
    /* we must obtain mmaping */
    if((iCaps & DSP_CAP_MMAP) && (iCaps & DSP_CAP_TRIGGER))
    {
        if(ioctl(iwavein, SNDCTL_DSP_GETISPACE, &sBuffInfo) != 0)
		{
			KMsgBox::message(this,  chWarning, chISpaceError,
						KMsgBox::EXCLAMATION, chAbort);
			return -1;
		}

        iBuffLen = sBuffInfo.fragsize * sBuffInfo.fragstotal; 

#ifdef DEBUG_TEST
        printf("Debug:\tDMA Buffer size is %d bytes\n", sBuffInfo.fragsize * sBuffInfo.fragstotal);
        printf("Debug:\tNumber of available fragments is %d\n", sBuffInfo.fragments);
        printf("Debug:\tNumber of fragments allocated is %d\n", sBuffInfo.fragstotal);
        printf("Debug:\tSize of a fragment is %d bytes\n", sBuffInfo.fragsize);
        printf("Debug:\tAvailable size is %d bytes\n", sBuffInfo.bytes);
        printf("Debug:\tWe use %d bytes\n", iBuffLen);
#endif

        /* map the DMA buffer in the proc vm */
        if((pBuffer = (caddr_t)mmap( NULL,					/* hint adress for kernel, free if NULL */
				                     iBuffLen,				/* length of buffer */
                				     PROT_READ,				/* reading access only */
				                     MAP_FILE|MAP_SHARED,	/* map to a file or a char device */
                				     iwavein,					/* file descriptor */
				                     0))					/* offset inside of file */
        				== (void *) -1)
		{
			KMsgBox::message(this, chWarning, chMMapError,
						KMsgBox::EXCLAMATION, chAbort);
			return -1;
		}

#ifdef DEBUG_TEST
        printf("Debug:\tDMA buffer adress is %x\n", (unsigned int) pBuffer);
#endif

    }
    else
	{
		KMsgBox::message(this,  chWarning, chNoMappingError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}

    /* set number of channels */
    if(ioctl(iwavein, SNDCTL_DSP_CHANNELS, &iChannels))
	{
		KMsgBox::message(this, chWarning, chChannelsError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}

	(iBytesPerSample == 1) ? (iFormat = AFMT_U8) : (iFormat = AFMT_S16_LE);
    /* set the format */
    if(ioctl(iwavein, SNDCTL_DSP_SETFMT, &iFormat))
	{
		KMsgBox::message(this, chWarning, chSetFMTError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}

    /* set the sampling rate */
    if(ioctl(iwavein, SNDCTL_DSP_SPEED, &WaveInfo.iSampleFrequency))
	{
		KMsgBox::message(this, chWarning, chSpeedError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}

#ifdef DEBUG_TEST
    /* write on stdout how the sound driver has done things for us */
    printf ("Debug:\tNumber of channels is %d\n", iChannels);
    printf ("Debug:\tSample rate is %d Hz\n", WaveInfo.iSampleFrequency);
    printf ("Debug:\tFormat is AFMT_S16_LE\n");
#endif

    iEnableInput = 0;

    /* toggle bit to zero */
    if(ioctl(iwavein, SOUND_PCM_SETTRIGGER, &iEnableInput))
	{
		KMsgBox::message(this, chWarning, chSetTriggerError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}
	
    iEnableInput = PCM_ENABLE_INPUT;

    /* toggle enable bit to one, start recording */
    if (ioctl (iwavein, SOUND_PCM_SETTRIGGER, &iEnableInput))
	{
		KMsgBox::message(this, chWarning, chSetTriggerError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}
	
	intervall = (int)1024000/(WaveInfo.iSampleFrequency*iChannels);	
	startTimer(intervall);
	
	return 0;
}

int KWaveIn::close(void)
{

	killTimers();
	
    iEnableInput = 0;

    /* toggle enable bit to zero */
    if(ioctl(iwavein, SOUND_PCM_SETTRIGGER, &iEnableInput))
	{
		KMsgBox::message(this, chWarning, chSetTriggerError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}

#ifdef DEBUG_TEST
	printf("\nDebug:\tfreeing DMA buffer at %x\n", (unsigned int) pBuffer);
#endif /* DEBUG */
	munmap(pBuffer, iBuffLen);
	::close(iwavein);
	return 0;
}

void KWaveIn::timerEvent(QTimerEvent*)
{
    static count_info sCountInfo;
    int iBuffSize;
	volatile short int *pLocalPtr;
	int i;
	
#ifdef DEBUG_TEST
    static int iDenom;
    static float fFps;
#endif

    iBuffSize = iBuffLen / iBytesPerSample;
    /* convenient way to convert data format. pBuffer is of type caddr_t	*/
    /* which is a typedef of char *, data returned by most soundcards is in	*/
    /* 16 bits little endian, which hoppefully is the short on i386 linux	*/
    pLocalPtr = (volatile short int *) pBuffer;
    
    if(ioctl(iwavein, SNDCTL_DSP_GETIPTR, &sCountInfo) != 0)
	{
		killTimers();
		KMsgBox::message(this, chWarning, chGetIPTRError,
					KMsgBox::EXCLAMATION, chAbort);
		return;
	}

    iPrevPtr = iCurrentPtr;
    iCurrentPtr = sCountInfo.ptr/iBytesPerSample;
	
#ifdef DEBUG_TEST
    /* calculate the number of frame per second */
    iDenom = (iCurrentPtr - iPrevPtr + iBuffSize) % iBuffSize;
    if (iDenom != 0)
        fFps = WaveInfo.iSampleFrequency / (float) iDenom;
    else
       fFps = 0.0;
 
    printf("\rDebug:\tDMA pointer %05d, Refresh rate %08.1f fps", sCountInfo.ptr, fFps);
    fflush(stdout);
#endif

    /* circular adressing of DMA buffer */
    for(i = iPrevPtr; i != iCurrentPtr; )
    {
		device_buffer[iDataPtr] = pLocalPtr[i];
		iDataPtr = (iDataPtr + 1) % BUFFERSIZE;
		i = (i + 1) % iBuffSize;
    }

}

/****************************************************************/
/*																*/
/*							KDSP56								*/
/*																*/
/****************************************************************/

KDSP56::KDSP56(QWidget *parent, const char *name)
	: KDevice(parent, name)
{
}

KDSP56::~KDSP56()
{
}

struct 
{
	int fmt;
	int dspfmt;
} dsp56setup[5] = { 8000, 0, 16000, 0x800, 32000, 0x1800, 48000, 0x3000, -1, -1};


int KDSP56::open(void)
{
	int intervall, fmt;
	DSP56STRUCT dsp56struct;
	QString appfile;
	
	iDataPtr = 0;
	
	fmt = 0;
	while(dsp56setup[fmt].fmt != -1)
	{
		if(WaveInfo.iSampleFrequency == dsp56setup[fmt].fmt)
			break;
		fmt = fmt + 1;
	}
	if(dsp56setup[fmt].fmt < 0)
	{
		KMsgBox::message(this,  chWarning, chFormatError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}
    if((idsp56 = ::open(DSP56_DRIVER, O_RDONLY)) < 0)
	{
		KMsgBox::message(this,  chWarning, chOpenDeviceError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}
	

	if(!load_bootloader(idsp56, QString(app->kde_datadir() + QString(chLoaderFile))))
	{
		KMsgBox::message(this,  chWarning, chLoaderError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}
	config->readNumEntry(chDeviceFFTEntry, 0) ? appfile = QString(chSpectrumFile)
							: appfile = QString(chRecordFile);
							
	if(!load_application(idsp56, QString(app->kde_datadir() + appfile)))
	{
		KMsgBox::message(this,  chWarning, chApplicationError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}
	dsp56struct.HostPort = dsp56setup[fmt].dspfmt;
	if(ioctl(idsp56, IOCTL_D56_HP_WRITE, (char *)&dsp56struct)<0)
	{
		KMsgBox::message(this,  chWarning, chFormatSetupError,
					KMsgBox::EXCLAMATION, chAbort);
		return -1;
	}

	intervall = (int)(1024000/WaveInfo.iSampleFrequency);
	startTimer(intervall);

	return 0;
}

int KDSP56::close(void)
{

	killTimers();

	::close(idsp56);
	return 0;
}

void KDSP56::timerEvent(QTimerEvent*)
{
	short int buf[2*POINTS];
	int bytes, i;
	
	// the following should be realized like above via mmap
	bytes = ::read(idsp56, &buf[0], sizeof(buf));
	if(bytes > 0)
	{
		for(i = 0; i < bytes; i++)
		{
			device_buffer[iDataPtr] = buf[i];
			iDataPtr = (iDataPtr + 1) % BUFFERSIZE;
		}
	}
}
