/*
 * KMLOCfg
 *
 * A utility to configure modems of the ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <termios.h>
#include <stdlib.h>

#include <qglobal.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qtooltip.h>
#include <qstring.h>
#include <qsize.h>
#include <qbuttongroup.h>

#include <kapp.h>
#include <kiconloader.h>
#include <klocale.h>

#include "configure.h"
#include "global.h"



Configure::Configure(QWidget *parent, const char *name): KDialogBase (IconList, i18n("Configure"), Help|Default|Ok|Apply|Cancel, Ok, parent, name, true, true)
{
	QBoxLayout *vbox, *svbox, *shbox;
	QLabel *label1, *label2, *label3;
	QSize size;
	QGroupBox *gb;
	QFrame *page;
	QButtonGroup *bg;

	config = kapp->config();

	setHelp("configure-help");
	setIconListAllVisible(true);
	
	/* Page 1: General */
	page = addPage(i18n("General"), i18n("General Settings"), UserIcon("general"));
	
	vbox = new QVBoxLayout(page, 12);

	gb = new QGroupBox(i18n("User Interface"), page);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	gen_tooltips = new QCheckBox(i18n("Show tooltips"), gb);
	gen_tooltips->setMinimumSize(gen_tooltips->sizeHint());
	svbox->addWidget(gen_tooltips);
	svbox->activate();
	
	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();

	/* Page 2: Modem */
	page = addPage(i18n("Modem"), i18n("Settings for the Modem"), UserIcon("modem"));
	
	vbox = new QVBoxLayout(page, 12);

	bg = new QButtonGroup(page);
	bg->setTitle(i18n("Modem Device"));
	svbox = new QVBoxLayout(bg, 10);
	svbox->addSpacing(8);
	
	mod_serial = new QRadioButton(i18n("Serial Port"), bg);
	mod_serial->setChecked(true);
	svbox->addWidget(mod_serial);

	shbox = new QHBoxLayout();
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	
	label1 = new QLabel(i18n("Device:"), bg);
	shbox->addWidget(label1);
	
	mod_ser_device = new QComboBox(false, bg);
	mod_ser_device->insertItem("/dev/modem", 0);
	mod_ser_device->insertItem("/dev/ttyS0", 1);
	mod_ser_device->insertItem("/dev/ttyS1", 2);
	mod_ser_device->insertItem("/dev/ttyS2", 3);
	mod_ser_device->insertItem("/dev/ttyS3", 4);
	mod_ser_device->insertItem("/dev/tts/0", 5);
	mod_ser_device->insertItem("/dev/tts/1", 6);
	mod_ser_device->insertItem("/dev/tts/2", 7);
	mod_ser_device->insertItem("/dev/tts/3", 8);
	shbox->addWidget(mod_ser_device);
	shbox->addStretch();
	
	shbox = new QHBoxLayout();
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	
	label2 = new QLabel(i18n("Speed:"), bg);
	shbox->addWidget(label2);

	mod_ser_speed = new QComboBox(false, bg);
	mod_ser_speed->insertItem("19200 Bps", 0);
	mod_ser_speed->insertItem("38400 Bps", 1);
#ifdef B57600
	mod_ser_speed->insertItem("57600 Bps", 2);
#endif
#ifdef B115200
	mod_ser_speed->insertItem("115200 Bps", 3);
#endif
#ifdef B230400
	mod_ser_speed->insertItem("230400 Bps", 4);
#endif
	shbox->addWidget(mod_ser_speed);
	shbox->addStretch();

	mod_usb = new QRadioButton(i18n("USB Port"), bg);
	svbox->addWidget(mod_usb);

	shbox = new QHBoxLayout();
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	
	label3 = new QLabel(i18n("Device:"), bg);
	shbox->addWidget(label3);
	
	mod_usb_device = new QComboBox(false, bg);
	mod_usb_device->insertItem("/dev/usb/ttyACM0", 0);
	mod_usb_device->insertItem("/dev/usb/ttyACM1", 1);
	mod_usb_device->insertItem("/dev/usb/ttyACM2", 2);
	mod_usb_device->insertItem("/dev/usb/ttyACM3", 3);
	mod_usb_device->insertItem("/dev/usb/acm/0", 4);
	mod_usb_device->insertItem("/dev/usb/acm/1", 5);
	mod_usb_device->insertItem("/dev/usb/acm/2", 6);
	mod_usb_device->insertItem("/dev/usb/acm/3", 7);
	shbox->addWidget(mod_usb_device);
	shbox->addStretch();

	size = label1->sizeHint().expandedTo(label2->sizeHint()).expandedTo(label3->sizeHint());
	label1->setFixedSize(size);
	label2->setFixedSize(size);
	label3->setFixedSize(size);
	
	size = mod_ser_device->sizeHint().expandedTo(mod_ser_speed->sizeHint()).expandedTo(mod_usb_device->sizeHint());
	mod_ser_device->setFixedSize(size);
	mod_ser_speed->setFixedSize(size);
	mod_usb_device->setFixedSize(size);

	svbox->activate();

	vbox->addWidget(bg);
	vbox->addStretch(1);
	vbox->activate();

	config->setGroup(ID_PREF_GROUP_GENERAL);
	if (config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS)) {
		QToolTip::add(gen_tooltips, i18n("Show tooltips in dialogs"));
		QToolTip::add(mod_ser_device, i18n("Serial device of the modem"));
		QToolTip::add(mod_ser_speed, i18n("Speed for serial communication"));
		QToolTip::add(mod_usb_device, i18n("USB device of the modem"));
	}

	connect(bg, SIGNAL(clicked(int)), SLOT(deviceSelected(int)));
	
	reset();
}


Configure::~Configure()
{
}


void Configure::slotDefault()
{
	gen_tooltips->setChecked(PREF_GEN_TOOLTIPS);
#if PREF_MOD_USB == false
	deviceSelected(0);
	mod_serial->setChecked(true);
#endif
	mod_ser_device->setCurrentItem(PREF_MOD_SER_DEVICE);
	mod_ser_speed->setCurrentItem(PREF_MOD_SER_SPEED);
#if PREF_MOD_USB == true
	deviceSelected(1);
	mod_usb->setChecked(true);
#endif
	mod_usb_device->setCurrentItem(PREF_MOD_USB_DEVICE);
}


void Configure::slotApply()
{
	config->setGroup(ID_PREF_GROUP_GENERAL);
	config->writeEntry(ID_PREF_GEN_TOOLTIPS, gen_tooltips->isChecked());
	config->setGroup(ID_PREF_GROUP_MODEM);
	config->writeEntry(ID_PREF_MOD_USB, mod_usb->isChecked());
	config->writeEntry(ID_PREF_MOD_SER_DEVICE, mod_ser_device->currentItem());
	config->writeEntry(ID_PREF_MOD_SER_SPEED, mod_ser_speed->currentItem());
	config->writeEntry(ID_PREF_MOD_USB_DEVICE, mod_usb_device->currentItem());
	config->sync();
}


void Configure::accept()
{
	slotApply();

	done(Accepted);
}


void Configure::reset()
{
	config->setGroup(ID_PREF_GROUP_GENERAL);
	gen_tooltips->setChecked(config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS));

	config->setGroup(ID_PREF_GROUP_MODEM);
	if (config->readBoolEntry(ID_PREF_MOD_USB, PREF_MOD_USB)) {
		deviceSelected(1);
		mod_usb->setChecked(true);
	} else {
		deviceSelected(0);
		mod_serial->setChecked(true);
	}
	mod_ser_device->setCurrentItem(config->readNumEntry(ID_PREF_MOD_SER_DEVICE, PREF_MOD_SER_DEVICE));
	mod_ser_speed->setCurrentItem(config->readNumEntry(ID_PREF_MOD_SER_SPEED, PREF_MOD_SER_SPEED));
	mod_usb_device->setCurrentItem(config->readNumEntry(ID_PREF_MOD_USB_DEVICE, PREF_MOD_USB_DEVICE));
}


void Configure::deviceSelected(int id_)
{
	mod_ser_device->setEnabled(id_ == 0);
	mod_ser_speed->setEnabled(id_ == 0);
	mod_usb_device->setEnabled(id_ == 1);
}
