/*
 * KMLOCfg
 *
 * A utility to configure modems of the ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <qlayout.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qsize.h>
#include <qbuttongroup.h>
#include <qfile.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kapp.h>
#include <kmessagebox.h>

#include "tamsetupdlg.h"
#include "global.h"
#include "mloloader.h"


	
	
TAMConfig::TAMConfig()
{
	mNumber = 0;
	entry_map = new TAMEntryMap();
}


TAMConfig::~TAMConfig()
{
	delete entry_map;
}


void TAMConfig::writeEntry(const QCString &key, const QCString &value)
{
	TAMEntryKey entry_key(mGroup, key);

	if (entry_map->contains(entry_key))
		entry_map->replace(entry_key, value);
	else
		entry_map->insert(entry_key, value);
}	



void TAMConfig::writeStrEntry(const QCString &key, const QString &value)
{
	writeEntry(key, QCString("\"") + QCString(value.latin1()) + QCString("\""));
}


void TAMConfig::writeNumEntry(const QCString &key, int value)
{
	QCString s;

	s.setNum(value);
	writeEntry(key, s);
}


void TAMConfig::writeBoolEntry(const QCString &key, bool value)
{
	if (value)
		writeEntry(key, "1");
	else
		writeEntry(key, "0");
}


QCString TAMConfig::readEntry(const QCString &key)
{
	TAMEntryKey entry_key(mGroup, key);

	if (entry_map->contains(entry_key))
		return (*entry_map)[entry_key];

	return 0;
}


QString TAMConfig::readStrEntry(const QCString &key)
{
	int i;
	QCString v = readEntry(key);

	if (v.isNull())
		return QString::null;
		
  if (v[0] != '"')
		return QString::null;
	
	i = v.find('"', 1);
	
  if (i == -1)
		return QString::null;
	
	return QString(v.mid(1, i-1));
}


int TAMConfig::readNumEntry(const QCString &key)
{
	int i;
	QCString v = readEntry(key);

	if (v.isNull())
		return 0;

	if ((i = v.find(';')) != -1)
  	v = v.left(i);

	v = v.stripWhiteSpace();

	return v.toInt();
}


bool TAMConfig::readBoolEntry(const QCString &key)
{
	int i = readNumEntry(key);

	return (i != 0);
}


void TAMConfig::readFile(const QString &fname)
{
	char buffer[81];
	QCString s;
	int i, j;
	QFile file(fname);

	entry_map->clear();
	mNumber = 0;

	if (file.open(IO_ReadOnly)) {
		while (!file.atEnd()) {
			file.readLine(buffer, 80);
			s = QCString(buffer);
			i = s.length();
			if ((i > 0) && (s[i-1] == 0x0A))
				i--;
			if ((i > 0) && (s[i-1] == 0x0D))
				i--;
			s = s.left(i).stripWhiteSpace();
	
			i = s.length();
		
			if (s[0] == '[') {
				j = s.find(']', 1);
				if (j != -1) {
					QCString g = s.mid(1, j-1);
					if ((g.length() == 6) && (g.left(4) == "TAM_")) // && g[4].isDigit() && g[5].isDigit())
						mNumber++;
					setGroup(g);
				}
			} else if ((j = s.find('=', 1)) != -1) {
				writeEntry(s.left(j).stripWhiteSpace(), s.mid(j+1, i-j).stripWhiteSpace());
			}
		}
		file.close();
	} else {
		// Create new config.cfg
		setGroup("COMMON");
		writeEntry("TamActive", "1");
		writeEntry("CountryConfigured", "1");
		writeFile(fname);
	}
}


void TAMConfig::writeFile(const QString &fname)
{
	QFile file(fname);
	TAMEntryMapIterator it;
	QCString s;

	file.open(IO_WriteOnly | IO_Truncate);
	
	setGroup(0);
	for (it = entry_map->begin(); it != entry_map->end(); it++) {
		if (it.key().mGroup != mGroup) {
			mGroup = it.key().mGroup;
			s.sprintf("[%s]\r\n", mGroup.data());
			file.writeBlock(s.data(), s.length());
		}
		s.sprintf("%s=%s\r\n", it.key().mKey.data(), it.data().data());
		file.writeBlock(s.data(), s.length());
	}
	file.close();
}


void TAMConfig::deleteTAM(int num)
{
	TAMEntryMap *new_map;
	TAMEntryMapIterator it;
	QCString group;
	int t;
	bool copy, ok;

	new_map = new TAMEntryMap();

	for (it = entry_map->begin(); it != entry_map->end(); it++) {
		group = it.key().mGroup;
		copy = true;
		if (group.left(4) == "TAM_") {
			t = group.mid(4).toInt(&ok);
			if (ok && (t >= num)) {
				copy = false;
				if (t > num) {
					group.sprintf("TAM_%02d", t-1);
					TAMEntryKey entry_key(group, it.key().mKey);
					new_map->insert(entry_key, it.data());
				}
			}
		}
		if (copy)
			new_map->insert(it.key(), it.data());
	}

	delete entry_map;
	entry_map = new_map;

	mNumber--;
}


void TAMConfig::addTAM(int model, bool fax)
{
	mGroup.sprintf("TAM_%02d", ++mNumber);

	if (model == MODEL_OFFICE_II) {
		writeNumEntry("GreetingNo", 1);
		writeStrEntry("Greeting_01", "");
		writeStrEntry("Greeting_02", "");
		writeStrEntry("Greeting_03", "");
		writeStrEntry("Greeting_04", "");
		writeStrEntry("Greeting_05", "");
		writeBoolEntry("Active", true);
		writeNumEntry("VrxFormat", 132);
		writeBoolEntry("M3_OnRing", false);
		writeBoolEntry("LoudHearEnable", true);
		writeNumEntry("NumberOfRing", 3);
		writeBoolEntry("EarlyAnswer", false);
		writeBoolEntry("BeepOnRing", false);
		writeBoolEntry("RemoteConfig", false);
		writeBoolEntry("SpeakerOnRecord", false);
		writeBoolEntry("RestCut", false);
		writeNumEntry("MinRecTime", 5);
		writeNumEntry("MaxRecTime", 60);
		writeBoolEntry("TAMOnS1", false);
		writeBoolEntry("SpeakCaller", true);
		writeBoolEntry("SpeakTime", true);
		writeBoolEntry("SpeakCallerID", true);
		writeNumEntry("SpeakSystemInfo", 2);
		writeBoolEntry("AnswerIfFull", false);
		writeBoolEntry("ControlByKeyboard", true);
		writeBoolEntry("AnswerIfVoice", true);
		writeBoolEntry("AnswerIfFax", true);
		writeBoolEntry("DistinctiveRing", false); // ?
		writeStrEntry("MSN", "");
		writeStrEntry("Name", i18n("Answering Machine and Fax"));					// !
		writeStrEntry("Language", "english");
		writeStrEntry("LocalFaxID", "");
	} else if (model == MODEL_ISDN_OFFICE) {
		if (fax) {
			writeStrEntry("Greeting_Default", "");
			writeStrEntry("Greeting_01", "");
			writeStrEntry("Greeting_02", "");
			writeStrEntry("Greeting_03", "");
			writeStrEntry("Greeting_04", "");
			writeStrEntry("Greeting_05", "");
			writeBoolEntry("Active", true);
			writeNumEntry("VrxFormat", 149);
			writeBoolEntry("M3_OnRing", false);
			writeBoolEntry("LoudHearEnable", true);
			writeNumEntry("NumberOfRing", 2);
			writeBoolEntry("EarlyAnswer", false);
			writeBoolEntry("BeepOnRing", true);
			writeBoolEntry("RemoteConfig", false);
			writeBoolEntry("SpeakerOnRecord", false);
			writeBoolEntry("RestCut", true);
			writeNumEntry("MinRecTime", 3);
			writeNumEntry("MaxRecTime", 60);
			writeBoolEntry("TAMOnS1", true);
			writeBoolEntry("SpeakCaller", false);
			writeBoolEntry("SpeakTime", true);
			writeBoolEntry("SpeakCallerID", false);			// !
			writeNumEntry("SpeakSystemInfo", 1);
			writeBoolEntry("AnswerIfFull", true);
			writeBoolEntry("ControlByKeyboard", true);
			writeBoolEntry("AnswerIfVoice", false);			// !
			writeBoolEntry("AnswerIfFax", true);					// !
			writeStrEntry("MSN", "");
			writeStrEntry("Name", i18n("Fax"));					// !
			writeStrEntry("Language", "english");
			writeStrEntry("LocalFaxID", "");							// !
			writeStrEntry("Password", "");								// !
			writeStrEntry("VIP_Number", "");							// !
		} else {
			writeStrEntry("Greeting_Default", "");
			writeStrEntry("Greeting_01", "");
			writeStrEntry("Greeting_02", "");
			writeStrEntry("Greeting_03", "");
			writeStrEntry("Greeting_04", "");
			writeStrEntry("Greeting_05", "");
			writeBoolEntry("Active", true);
			writeNumEntry("VrxFormat", 149);
			writeBoolEntry("M3_OnRing", false);
			writeBoolEntry("LoudHearEnable", true);
			writeNumEntry("NumberOfRing", 3);
			writeBoolEntry("EarlyAnswer", false);
			writeBoolEntry("BeepOnRing", true);
			writeBoolEntry("RemoteConfig", false);
			writeBoolEntry("SpeakerOnRecord", false);
			writeBoolEntry("RestCut", true);
			writeNumEntry("MinRecTime", 3);
			writeNumEntry("MaxRecTime", 60);
			writeBoolEntry("TAMOnS1", true);
			writeBoolEntry("SpeakCaller", false);
			writeBoolEntry("SpeakTime", true);
			writeBoolEntry("SpeakCallerID", true);							// !
			writeNumEntry("SpeakSystemInfo", 1);
			writeBoolEntry("AnswerIfFull", true);
			writeBoolEntry("ControlByKeyboard", true);
			writeBoolEntry("AnswerIfVoice", true);							// !
			writeBoolEntry("AnswerIfFax", false);							// !
			writeStrEntry("MSN", "");
			writeStrEntry("Name", i18n("Answering Machine"));	// !
			writeStrEntry("Language", "english");
		}
	}
}



PageGeneral::PageGeneral(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *hbox;
	QLabel *label, *label1, *label2;
	QSize size;

	vbox = new QVBoxLayout(this, 10);

	call_accept = new QCheckBox(i18n("Call accepting"), this);
	call_accept->setMinimumSize(call_accept->sizeHint());
	vbox->addWidget(call_accept);
	
	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	label1 = new QLabel(i18n("Name:"), this);
	hbox->addWidget(label1);

	tam_name = new QLineEdit(this);
	tam_name->setMinimumSize(tam_name->sizeHint());
	hbox->addWidget(tam_name);

	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	label2 = new QLabel(i18n("MSN:"), this);
	hbox->addWidget(label2);

	size = label1->sizeHint().expandedTo(label2->sizeHint());
	label1->setFixedSize(size);
	label2->setFixedSize(size);

	msn = new QLineEdit(this);
	msn->setMinimumSize(msn->sizeHint());
	hbox->addWidget(msn);

	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	label = new QLabel(i18n("Rings until call accept:"), this);
	label->setFixedSize(label->sizeHint());
	hbox->addWidget(label);

	rings = new QSpinBox(1, 8, 1, this);
	rings->setFixedSize(rings->sizeHint());
	hbox->addWidget(rings);
	hbox->addStretch();

	vbox->activate();
}


PageGeneral::~PageGeneral()
{
}


void PageGeneral::setConfig(TAMConfig &conf)
{
	call_accept->setChecked(conf.readBoolEntry("Active"));
	tam_name->setText(conf.readStrEntry("Name"));
	msn->setText(conf.readStrEntry("MSN"));
	rings->setValue(conf.readNumEntry("NumberOfRing"));
}


void PageGeneral::getConfig(TAMConfig &conf)
{
	conf.writeBoolEntry("Active", call_accept->isChecked());
	conf.writeStrEntry("Name", tam_name->text());
	conf.writeStrEntry("MSN", msn->text());
	conf.writeNumEntry("NumberOfRing", rings->value());
}



PageVoiceOutput::PageVoiceOutput(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *hbox;
	QLabel *label;
	QButtonGroup *bg;

	vbox = new QVBoxLayout(this, 10);

	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	label = new QLabel(i18n("Language:"), this);
	label->setFixedHeight(label->sizeHint().height());
	hbox->addWidget(label);

	language = new QComboBox(this);
	language->insertItem(i18n("English"), 0);
	language->insertItem(i18n("German"), 1);
	language->setMinimumHeight(language->sizeHint().height());
	hbox->addWidget(language);
	
	time_date = new QCheckBox(i18n("Time/Date"), this);
	time_date->setMinimumSize(time_date->sizeHint());
	vbox->addWidget(time_date);
	
	caller_id = new QCheckBox(i18n("Caller ID"), this);
	caller_id->setMinimumSize(caller_id->sizeHint());
	vbox->addWidget(caller_id);
	
	sys_info = new QCheckBox(i18n("System information"), this);
	sys_info->setMinimumSize(sys_info->sizeHint());
	vbox->addWidget(sys_info);
	
	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	hbox->addSpacing(19);
	
	comprehensive = new QRadioButton(this);
	comprehensive->setText(i18n("Comprehensive"));
	comprehensive->setFixedSize(comprehensive->sizeHint());
	hbox->addWidget(comprehensive);
	
	hbox->addStretch();
	
	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	hbox->addSpacing(19);
	
	reduced = new QRadioButton(this);
	reduced->setText(i18n("Reduced"));
	reduced->setFixedSize(reduced->sizeHint());
	hbox->addWidget(reduced);
	
	hbox->addStretch();
	
	bg = new QButtonGroup(this);
	bg->hide();
	bg->insert(comprehensive, 0);
	bg->insert(reduced, 1);
	
	vbox->activate();

	connect(sys_info, SIGNAL(toggled(bool)), SLOT(sysInfoToggled(bool)));
}


PageVoiceOutput::~PageVoiceOutput()
{
}


void PageVoiceOutput::setConfig(TAMConfig &conf)
{
	static const char *languages[] = { "english", "german" };
	int i;
	QString s;

	s = conf.readStrEntry("Language");
	for (i=0; i < 2; i++) {
		if (s == languages[i]) {
			language->setCurrentItem(i);
			break;
		}
	}
	time_date->setChecked(conf.readBoolEntry("SpeakTime"));
	caller_id->setChecked(conf.readBoolEntry("SpeakCallerID"));
	i = conf.readNumEntry("SpeakSystemInfo");
	sys_info->setChecked(i > 0);
	if (i < 2)
		reduced->setChecked(true);
	else
		comprehensive->setChecked(true);
	comprehensive->setEnabled(i > 0);
	reduced->setEnabled(i > 0);
}


void PageVoiceOutput::getConfig(TAMConfig &conf)
{
	static const char *languages[] = { "english", "german" };
	int i;

	conf.writeStrEntry("Language", languages[language->currentItem()]);
	conf.writeBoolEntry("SpeakTime", time_date->isChecked());
	conf.writeBoolEntry("SpeakCallerID", caller_id->isChecked());
	
	if (sys_info->isChecked())
		i = reduced->isChecked() ? 1 : 2;
	else
		i = 0;
	conf.writeNumEntry("SpeakSystemInfo", i);
}


void PageVoiceOutput::sysInfoToggled(bool on)
{
	comprehensive->setEnabled(on);
	reduced->setEnabled(on);
}



PageAnsweringMachine::PageAnsweringMachine(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *hbox, *hbox2, *vbox2;
	QLabel *label1, *label2;
	QSize size;
	QButtonGroup *bg;

	vbox = new QVBoxLayout(this, 10);

	record = new QCheckBox(i18n("Record incoming messages"), this);
	record->setMinimumSize(record->sizeHint());
	vbox->addWidget(record);
	
	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	hbox->addSpacing(19);
	label1 = new QLabel(i18n("Maximum recording duration [seconds]:"), this);
	hbox->addWidget(label1);

	rec_max = new QSpinBox(0, 60, 1, this);
	rec_max->setFixedSize(rec_max->sizeHint());
	hbox->addWidget(rec_max);
	hbox->addStretch();

	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	hbox->addSpacing(19);
	label2 = new QLabel(i18n("Minimum recording duration [seconds]:"), this);
	hbox->addWidget(label2);

	size = label1->sizeHint().expandedTo(label2->sizeHint());
	label1->setFixedSize(size);
	label2->setFixedSize(size);

	rec_min = new QSpinBox(0, 60, 1, this);
	rec_min->setFixedSize(rec_min->sizeHint());
	hbox->addWidget(rec_min);
	hbox->addStretch();

	enable_speaker = new QCheckBox(i18n("Enable speaker during recording"), this);
	enable_speaker->setMinimumSize(enable_speaker->sizeHint());
	vbox->addWidget(enable_speaker);
	
	early_accept = new QCheckBox(i18n("Accept calls early"), this);
	early_accept->setMinimumSize(early_accept->sizeHint());
	vbox->addWidget(early_accept);
	
	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	bg = new QButtonGroup(i18n("Recording Quality"), this);

	vbox2 = new QVBoxLayout(bg, 6);
	vbox2->addSpacing(12);

	hbox2 = new QHBoxLayout(10);
	vbox2->addLayout(hbox2);

	low_quality = new QRadioButton(bg);
	low_quality->setText(i18n("Low"));
	low_quality->setFixedSize(low_quality->sizeHint());
	hbox2->addWidget(low_quality);

	hbox2->addStretch();
	
	hbox2 = new QHBoxLayout(10);
	vbox2->addLayout(hbox2);

	high_quality = new QRadioButton(bg);
	high_quality->setText(i18n("High"));
	high_quality->setFixedSize(high_quality->sizeHint());
	hbox2->addWidget(high_quality);
	
	hbox2->addStretch();
  	
	vbox2->activate();
	
	bg->setFixedSize(bg->sizeHint());
	hbox->addWidget(bg);

	hbox->addStretch();	

	vbox->activate();

	connect(record, SIGNAL(toggled(bool)), SLOT(recordToggled(bool)));
}


PageAnsweringMachine::~PageAnsweringMachine()
{
}


void PageAnsweringMachine::setConfig(TAMConfig &conf)
{
	int i;
	QString s;

	i = conf.readNumEntry("MaxRecTime");
	record->setChecked(i != 0);
	rec_max->setValue(i);
	rec_min->setValue((i) ? conf.readNumEntry("MinRecTime") : 0);
	enable_speaker->setChecked(conf.readBoolEntry("SpeakerOnRecord"));
	early_accept->setChecked(conf.readBoolEntry("EarlyAnswer"));
	i = conf.readNumEntry("VrxFormat");
	if (i == 149)
		low_quality->setChecked(true);
	else
		high_quality->setChecked(true);
}


void PageAnsweringMachine::getConfig(TAMConfig &conf)
{
	conf.writeNumEntry("MaxRecTime", record->isChecked() ? rec_max->value() : 0);
	conf.writeNumEntry("MinRecTime", record->isChecked() ? rec_min->value() : 0);
	conf.writeBoolEntry("SpeakerOnRecord", enable_speaker->isChecked());
	conf.writeBoolEntry("EarlyAnswer", early_accept->isChecked());
	conf.writeNumEntry("VrxFormat", low_quality->isChecked() ? 149 : 132);
}


void PageAnsweringMachine::recordToggled(bool on)
{
	rec_max->setEnabled(on);
	rec_min->setEnabled(on);

	rec_max->setValue(on ? 60 : 0);
	rec_min->setValue(on ? 3 : 0);
}



PageFax::PageFax(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *hbox;
	QLabel *label;

	vbox = new QVBoxLayout(this, 10);

	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	label = new QLabel(i18n("Fax ID:"), this);
	hbox->addWidget(label);

	fax_id = new QLineEdit(this);
	fax_id->setMinimumSize(fax_id->sizeHint());
	hbox->addWidget(fax_id);

	vbox->activate();
}


PageFax::~PageFax()
{
}


void PageFax::setConfig(TAMConfig &conf)
{
	fax_id->setText(conf.readStrEntry("LocalFaxID"));
}


void PageFax::getConfig(TAMConfig &conf)
{
	conf.writeStrEntry("LocalFaxID", fax_id->text());
}



PageSecurity::PageSecurity(QWidget *parent, const char *name): QWidget(parent, name)
{
	QBoxLayout *vbox, *hbox;
	QLabel *label1, *label2;
	QSize size;

	vbox = new QVBoxLayout(this, 10);

	remote_config = new QCheckBox(i18n("Allow remote configuration"), this);
	remote_config->setMinimumSize(remote_config->sizeHint());
	vbox->addWidget(remote_config);
	
	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	hbox->addSpacing(19);

	label1 = new QLabel(i18n("Password:"), this);
	hbox->addWidget(label1);

	password = new QLineEdit(this);
	password->setMinimumSize(password->sizeHint());
	password->setEchoMode(QLineEdit::Password);
	hbox->addWidget(password);

	hbox = new QHBoxLayout(10);
	vbox->addLayout(hbox);

	hbox->addSpacing(19);

	label2 = new QLabel(i18n("VIP number:"), this);
	hbox->addWidget(label2);

	size = label1->sizeHint().expandedTo(label2->sizeHint());
	label1->setFixedSize(size);
	label2->setFixedSize(size);

	vip_number = new QLineEdit(this);
	vip_number->setMinimumSize(vip_number->sizeHint());
	hbox->addWidget(vip_number);

	monitor = new QCheckBox(i18n("Enable monitoring of connected telephones"), this);
	monitor->setMinimumSize(monitor->sizeHint());
	vbox->addWidget(monitor);
	
	keyboard = new QCheckBox(i18n("Connected to device keyboard"), this);
	keyboard->setMinimumSize(keyboard->sizeHint());
	vbox->addWidget(keyboard);
	
	vbox->activate();

	connect(remote_config, SIGNAL(toggled(bool)), SLOT(remoteConfigToggled(bool)));
}


PageSecurity::~PageSecurity()
{
}


void PageSecurity::setConfig(TAMConfig &conf)
{
	bool remote = conf.readBoolEntry("RemoteConfig");
	remote_config->setChecked(remote);
	password->setText(conf.readStrEntry("Password"));
	vip_number->setText(conf.readStrEntry("VIP_Number"));
	monitor->setChecked(conf.readBoolEntry("LoudHearEnable"));
	keyboard->setChecked(conf.readBoolEntry("ControlByKeyboard"));

	password->setEnabled(remote);
	vip_number->setEnabled(remote);
}


void PageSecurity::getConfig(TAMConfig &conf)
{
	conf.writeBoolEntry("RemoteConfig", remote_config->isChecked());
	conf.writeStrEntry("Password", password->text());
	conf.writeStrEntry("VIP_Number", vip_number->text());
	conf.writeBoolEntry("LoudHearEnable", monitor->isChecked());
	conf.writeBoolEntry("ControlByKeyboard", keyboard->isChecked());
}


void PageSecurity::remoteConfigToggled(bool on)
{
	password->setEnabled(on);
	vip_number->setEnabled(on);
}	



TAMEditDlg::TAMEditDlg(int model, bool fax, QWidget *parent, const char *name): KDialogBase(IconList, 0, Help|Ok|Cancel, Ok, parent, name, true, true)
{
	QBoxLayout *vbox;
	QFrame *page;

  modem_model = model;
  isdn_fax = fax;
	
	if (modem_model == MODEL_ISDN_OFFICE) {
		page = addPage(i18n("General"), i18n("General"), UserIcon("tam_general"));
		vbox = new QVBoxLayout(page, 2);
		page1 = new PageGeneral(page);
		vbox->addWidget(page1);
		vbox->addStretch(1);
		vbox->activate();
	
		page = addPage(i18n("Voice Output"), i18n("Voice Output"), UserIcon("tam_voice"));
		vbox = new QVBoxLayout(page, 2);
		page2 = new PageVoiceOutput(page);
		vbox->addWidget(page2);
		vbox->addStretch(1);
		vbox->activate();

		if (isdn_fax) {
			page = addPage(i18n("Fax"), i18n("Fax"), UserIcon("tam_fax"));
			vbox = new QVBoxLayout(page, 2);
			page3 = new PageFax(page);
		} else {
			page = addPage(i18n("Answering Machine"), i18n("Answering Machine"), UserIcon("tam_am"));
			vbox = new QVBoxLayout(page, 2);
			page3 = new PageAnsweringMachine(page);
		}
		vbox->addWidget(page3);
		vbox->addStretch(1);
		vbox->activate();

		page = addPage(i18n("Security"), i18n("Security"), UserIcon("tam_security"));
		vbox = new QVBoxLayout(page, 2);
		page4 = new PageSecurity(page);
		vbox->addWidget(page4);
		vbox->addStretch(1);
		vbox->activate();
	}

	setHelp("deviceedit-help");

	setCaption(fax ? i18n("Configure Fax") : i18n("Configure Answering Machine"));
}


TAMEditDlg::~TAMEditDlg()
{
}


void TAMEditDlg::setConfig(TAMConfig &conf)
{
	if (modem_model == MODEL_ISDN_OFFICE) {
		((PageGeneral *)page1)->setConfig(conf);
		((PageVoiceOutput *)page2)->setConfig(conf);
		if (isdn_fax)
			((PageFax *)page3)->setConfig(conf);
		else
			((PageAnsweringMachine *)page3)->setConfig(conf);
		((PageSecurity *)page4)->setConfig(conf);
	}
}


void TAMEditDlg::getConfig(TAMConfig &conf)
{
	if (modem_model == MODEL_ISDN_OFFICE) {
		((PageGeneral *)page1)->getConfig(conf);
		((PageVoiceOutput *)page2)->getConfig(conf);
		if (isdn_fax)
			((PageFax *)page3)->getConfig(conf);
		else
			((PageAnsweringMachine *)page3)->getConfig(conf);
		((PageSecurity *)page4)->getConfig(conf);
	}
}


TAMSetupDlg::TAMSetupDlg(int model, QWidget *parent, const char *name): KDialog(parent, name, true)
{
	QVBoxLayout *vbox, *gvbox;
	QHBoxLayout *hbox, *ghbox;
	QGroupBox *gb;
	QSize size;
	QPushButton *help, *bcancel;

	modified = false;
	modem_model = model;

	vbox = new QVBoxLayout(this, marginHint());

	gb = new QGroupBox(this);

	gvbox = new QVBoxLayout(gb, marginHint());
	
	list = new QListView(gb, "tamlist");
	CHECK_PTR(list);
	list->addColumn(i18n("No."));
	list->addColumn(i18n("Name"));
	list->addColumn(i18n("Type"));
	list->setColumnWidth(0, 40);
	list->setColumnAlignment(0, AlignVCenter | AlignHCenter);
	list->setColumnWidth(1, 200);
	list->setColumnWidth(2, 100);
	
	list->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	list->setAllColumnsShowFocus(true);
	list->setShowSortIndicator(true);

	gvbox->addWidget(list, 1);

	ghbox = new QHBoxLayout();
	CHECK_PTR(ghbox);
	gvbox->addLayout(ghbox);

	add_am = new QPushButton(i18n("New &AM"), gb);
	add_fax = new QPushButton(i18n("New &Fax"), gb);
	edit_item = new QPushButton(i18n("&Edit"), gb);
	edit_item->setEnabled(false);
	del_item = new QPushButton(i18n("&Delete"), gb);
	del_item->setEnabled(false);

	size = add_am->sizeHint().expandedTo(add_fax->sizeHint()).expandedTo(edit_item->sizeHint()).expandedTo(del_item->sizeHint());

	add_am->setFixedSize(size);
	add_fax->setFixedSize(size);
	edit_item->setFixedSize(size);
	del_item->setFixedSize(size);

	ghbox->addWidget(add_am);
	ghbox->addStretch();
	ghbox->addWidget(add_fax);
	ghbox->addStretch();
	ghbox->addWidget(edit_item);
	ghbox->addStretch();
	ghbox->addWidget(del_item);

	gvbox->activate();

	vbox->addWidget(gb, 1);
	
	hbox = new QHBoxLayout(8);
	vbox->addLayout(hbox);

	help = new QPushButton(i18n("&Help"), this);
	ok = new QPushButton(i18n("&OK"), this);
	ok->setDefault(true);
	ok->setEnabled(false);
	bcancel = new QPushButton(i18n("&Cancel"), this);
	size = help->sizeHint().expandedTo(ok->sizeHint()).expandedTo(bcancel->sizeHint());
	help->setFixedSize(size);
	ok->setFixedSize(size);
	bcancel->setFixedSize(size);

	hbox->addWidget(help);
	hbox->addStretch();
	hbox->addWidget(ok);
	hbox->addSpacing(spacingHint());
	hbox->addWidget(bcancel);

	vbox->activate();

	connect(list, SIGNAL(selectionChanged(QListViewItem *)), SLOT(selectionChangedSlot(QListViewItem *)));
	connect(list, SIGNAL(doubleClicked(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(list, SIGNAL(returnPressed(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(add_am, SIGNAL(clicked()), SLOT(addAMItem()));
	connect(add_fax, SIGNAL(clicked()), SLOT(addFaxItem()));
	connect(edit_item, SIGNAL(clicked()), SLOT(editItem()));
	connect(del_item, SIGNAL(clicked()), SLOT(deleteItem()));
	connect(help, SIGNAL(clicked()), SLOT(slotHelp()));
	connect(ok, SIGNAL(clicked()), SLOT(accept()));
	connect(bcancel, SIGNAL(clicked()), SLOT(reject()));

	tam_config.readFile(expandPath("config.cfg"));
	loadList();
	
	add_am->setEnabled(tamsAvailable());
	add_fax->setEnabled(tamsAvailable());
	
	setCaption(i18n("Configure Devices"));
}


TAMSetupDlg::~TAMSetupDlg()
{
}


void TAMSetupDlg::loadList()
{
	QListViewItem *item;
	QString s;
	int no;

	list->clear();
	for (no=0; no < tam_config.numberOfTAMs(); no++) {
		tam_config.setTAM(no+1);
		item = new QListViewItem(list);
		s.setNum(no+1);
		item->setText(0, s);
		item->setText(1, tam_config.readTAMName());
		if (tam_config.isFax()) {
			if (tam_config.isTAM())
				item->setText(2, i18n("Answering Machine / Fax"));
			else
				item->setText(2, i18n("Fax"));
		} else
			item->setText(2, i18n("Answering Machine"));
	}
}
	
	
void TAMSetupDlg::accept()
{
	tam_config.writeFile(expandPath("config.cfg"));
	KDialog::accept();
}


void TAMSetupDlg::reject()
{
	if (modified) {
		if (KMessageBox::warningYesNo(this, i18n("You have made some changes to the configuration.\nDiscard those changes?"), i18n("Warning"), i18n("&Discard"), i18n("&Cancel")) == KMessageBox::Yes) {
			KDialog::reject();
		}
	} else
		KDialog::reject();
}


void TAMSetupDlg::selectionChangedSlot(QListViewItem *item)
{
	if (item) {
		edit_item->setEnabled(true);
		del_item->setEnabled(true);
	}
}


void TAMSetupDlg::executedSlot(QListViewItem *)
{
	editItem();
}


void TAMSetupDlg::addItem(bool fax)
{
	int no;

	if (!tamsAvailable())
		return;

	tam_config.addTAM(modem_model, fax);
	no = tam_config.numberOfTAMs();

	TAMEditDlg edit(modem_model, fax, this);
	edit.setConfig(tam_config);
	if (edit.exec()) {
		QListViewItem *item;
		QString s;

		edit.getConfig(tam_config);
		item = new QListViewItem(list);
		s.setNum(no);
		item->setText(0, s);
		item->setText(1, tam_config.readTAMName());
		if (tam_config.isFax()) {
			if (tam_config.isTAM())
				item->setText(2, i18n("Answering Machine / Fax"));
			else
				item->setText(2, i18n("Fax"));
		} else
			item->setText(2, i18n("Answering Machine"));
		modified = true;
		ok->setEnabled(modified);
		add_am->setEnabled(tamsAvailable());
		add_fax->setEnabled(tamsAvailable());
	} else
		tam_config.deleteTAM(no);
}


void TAMSetupDlg::addAMItem()
{
	addItem(false);
}


void TAMSetupDlg::addFaxItem()
{
	addItem(true);
}


void TAMSetupDlg::editItem()
{
	QListViewItem *item;

	if ((item = list->currentItem())) {
		tam_config.setTAM(item->text(0).toInt());
		TAMEditDlg edit(modem_model, tam_config.isFax(), this);
		edit.setConfig(tam_config);
		if (edit.exec()) {
			edit.getConfig(tam_config);
			item->setText(1, tam_config.readTAMName());
			modified = true;
			ok->setEnabled(modified);
		}
	}
}


void TAMSetupDlg::deleteItem()
{
	QListViewItem *item;
	QString s;
	int num, num2;

	if ((item = list->currentItem())) {
		if (KMessageBox::warningYesNo(this, i18n("Really delete entry\n\"%1\"?").arg(item->text(1)), i18n("Question"), i18n("&Delete"), i18n("&Cancel")) == KMessageBox::Yes) {
			num = item->text(0).toInt();
			tam_config.deleteTAM(num);
			delete item;
			for (item = list->firstChild(); (item); item = item->nextSibling()) {
				num2 = item->text(0).toInt();
				if (num2 > num) {
					s.setNum(num2-1);
					item->setText(0, s);
				}
			}
			selectionChangedSlot(0);
			modified = true;
			ok->setEnabled(modified);
			add_am->setEnabled(tamsAvailable());
			add_fax->setEnabled(tamsAvailable());
		}
	}
}


void TAMSetupDlg::slotHelp()
{
	kapp->invokeHelp("deviceconfig-help");
}


bool TAMSetupDlg::tamsAvailable()
{
	if (modem_model == MODEL_OFFICE_II)
		return tam_config.numberOfTAMs() < 1;

	if (modem_model == MODEL_ISDN_OFFICE)
		return tam_config.numberOfTAMs() < 9;
	
	return false;
}
