/**
 * kcfgentryeditor.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#include "kcfgentryeditor.h"

#include "items.h"
#include "applicationtoken.h"
#include "grouptoken.h"
#include "entrytoken.h"

#include <ktextedit.h>
#include <klineedit.h>
#include <keditlistbox.h>
#include <klistview.h>
#include <kcombobox.h>
#include <kpushbutton.h>
#include <knuminput.h>

#include <qcheckbox.h>

using namespace KCfgCreator;

EntryEditor::EntryEditor( QWidget *parent )
    : KCFGEditUI( parent )
{
    initConnections();
}

void
EntryEditor::initConnections()
{
    connect( m_keyEdit, SIGNAL(textChanged(const QString&)),
             SLOT(slotKeyChanged(const QString&)) );
    connect( m_nameEdit, SIGNAL(textChanged(const QString&)),
             SLOT(slotNameChanged(const QString&)) );

    connect( m_typeCombo, SIGNAL(activated(const QString&)),
             SLOT(slotTypeChanged(const QString&)) );
    connect( m_hidden, SIGNAL(toggled(bool)),
             SLOT(slotHiddenToggled(bool)) );
    connect( m_labelEdit, SIGNAL(textChanged(const QString&)),
             SLOT(slotLabelChanged(const QString&)) );
    connect( m_whatsEdit, SIGNAL(textChanged()),
             SLOT(slotWhatsThisChanged()) );
    connect( m_defaultValueEdit, SIGNAL(textChanged(const QString&)),
             SLOT(slotDefaultValueChanged(const QString&)) );
    connect( m_defaultCodeCB, SIGNAL(toggled(bool)),
             SLOT(slotDefaultCodeToggled(bool)) );
    connect( m_minEdit, SIGNAL(textChanged(const QString&)),
             SLOT(slotMinChanged(const QString&)) );
    connect( m_minCB, SIGNAL(toggled(bool)),
             SLOT(slotMinCodeToggled(bool)) );
    connect( m_maxEdit, SIGNAL(textChanged(const QString&)),
             SLOT(slotMaxChanged(const QString&)) );
    connect( m_maxCB, SIGNAL(toggled(bool)),
             SLOT(slotMaxCodeToggled(bool)) );

    connect( m_valuesBox, SIGNAL(added(const QString&)),
             SLOT(slotValueAdded(const QString&)) );
    connect( m_valuesBox, SIGNAL(removed(const QString&)),
             SLOT(slotValueRemoved(const QString&)) );

    connect( m_addChoice, SIGNAL(clicked()),
             SLOT(slotAddChoiceClicked()) );
    connect( m_modifyChoice, SIGNAL(clicked()),
             SLOT(slotModifyChoiceClicked()) );
    connect( m_removeChoice, SIGNAL(clicked()),
             SLOT(slotRemoveChoiceClicked()) );

    connect( m_paramName, SIGNAL(textChanged(const QString&)),
             SLOT(slotParamNameChanged(const QString&)) );
    connect( m_paramType, SIGNAL(textChanged(const QString&)),
             SLOT(slotParamTypeChanged(const QString&)) );
    connect( m_paramMax, SIGNAL(valueChanged(int)),
             SLOT(slotParamMaxChanged(int)) );
    connect( m_paramValues, SIGNAL(added(const QString&)),
             SLOT(slotParamValueAdded(const QString&)) );
    connect( m_paramValues, SIGNAL(removed(const QString&)),
             SLOT(slotParamValueRemoved(const QString&)) );
}

void EntryEditor::slotKeyChanged( const QString& str )
{
    m_item->token()->setKey( str );
    m_item->update();
}

void EntryEditor::slotNameChanged( const QString& str )
{
    QString name = str;
    name = name.remove( ' ' );
    m_item->token()->setName( name );
    m_item->update();
    m_nameEdit->setText( name );
}

void EntryEditor::slotTypeChanged( const QString& type )
{
    if ( Utils::stringToType( type ) != EntryToken::Invalid ) {
        GroupItem* groupitem = dynamic_cast<GroupItem*>( m_item->parent() );
        if ( groupitem ) {
            EntryToken *nToken = groupitem->group()->changeType( m_item->name(), type );
            m_item->switchTokens( nToken );
        }
    }
}

void EntryEditor::slotHiddenToggled( bool b )
{
    m_item->token()->setHidden( b );
}

void EntryEditor::slotLabelChanged( const QString& lab )
{
    m_item->token()->setLabel( lab );
}

void EntryEditor::slotWhatsThisChanged()
{
    m_item->token()->setWhatsThis( m_whatsEdit->text() );
}

void EntryEditor::slotDefaultValueChanged( const QString& val )
{
    EntryToken::DefaultValue d = m_item->token()->defaultValue();
    d.value = val;
    m_item->token()->setDefaultValue( d );
}

void EntryEditor::slotDefaultCodeToggled( bool code )
{
    EntryToken::DefaultValue d = m_item->token()->defaultValue();
    d.isCode = code;
    m_item->token()->setDefaultValue( d );
}

void EntryEditor::slotMinChanged( const QString& str )
{
    EntryToken::Value val = m_item->token()->min();
    val.value = str;
    m_item->token()->setMin( val );
}

void EntryEditor::slotMinCodeToggled( bool code )
{
    EntryToken::Value val = m_item->token()->min();
    val.isCode = code;
    m_item->token()->setMin( val );
}

void EntryEditor::slotMaxChanged( const QString& str )
{
    EntryToken::Value val = m_item->token()->max();
    val.value = str;
    m_item->token()->setMax( val );
}

void EntryEditor::slotMaxCodeToggled( bool code )
{
    EntryToken::Value val = m_item->token()->max();
    val.isCode = code;
    m_item->token()->setMax( val );
}

void EntryEditor::slotValueAdded( const QString& val )
{
    QStringList vals = m_item->token()->values();
    vals.append( val );
    m_item->token()->setValues( vals );
}

void EntryEditor::slotValueRemoved( const QString& val )
{
    QStringList vals = m_item->token()->values();
    vals.remove( val );
    m_item->token()->setValues( vals );
}

void EntryEditor::slotAddChoiceClicked()
{
}

void EntryEditor::slotModifyChoiceClicked()
{
}

void EntryEditor::slotRemoveChoiceClicked()
{
}

void EntryEditor::slotParamNameChanged( const QString& name )
{
    EntryToken::Parameter param = m_item->token()->parameter();
    param.name = name;
    m_item->token()->setParameter( param );
}

void EntryEditor::slotParamTypeChanged( const QString& type )
{
    EntryToken::Parameter param = m_item->token()->parameter();
    param.type = type;
    m_item->token()->setParameter( param );
}

void EntryEditor::slotParamMaxChanged( int max )
{
    EntryToken::Parameter param = m_item->token()->parameter();
    param.max = max;
    m_item->token()->setParameter( param );
}

void EntryEditor::slotParamValueAdded( const QString& val )
{
    EntryToken::Parameter param = m_item->token()->parameter();
    param.values.append( val );
    m_item->token()->setParameter( param );
}

void EntryEditor::slotParamValueRemoved( const QString& val )
{
    EntryToken::Parameter param = m_item->token()->parameter();
    param.values.remove( val );
    m_item->token()->setParameter( param );
}

void EntryEditor::updateEntry( EntryItem* item )
{
    if ( item ) {
        m_item = item;

        EntryToken *token = item->token();
        m_keyEdit->setText( token->key() );
        m_nameEdit->setText( token->name() );
        m_typeCombo->setCurrentText( token->typeName() );
        m_hidden->setChecked( token->hidden() );
        m_labelEdit->setText( token->label() );
        m_whatsEdit->setText( token->whatsThis() );

        m_defaultValueEdit->setText( token->defaultValue().value );
        m_defaultCodeCB->setChecked( token->defaultValue().isCode );

        m_minEdit->setText( token->min().value );
        m_minCB->setChecked( token->min().isCode );

        m_maxEdit->setText( token->max().value );
        m_maxCB->setChecked( token->max().isCode );

        m_valuesBox->clear();
        m_valuesBox->insertStringList( token->values() );

        m_choicesView->clear();
        QValueList<EntryToken::Choice> l = token->choices();
        for ( QValueListIterator<EntryToken::Choice> itr = l.begin(); itr != l.end(); ++itr ) {
            new QListViewItem( m_choicesView, (*itr).name, (*itr).label,
                               (*itr).whatsThis );
        }

        m_paramName->setText( token->parameter().name );
        m_paramType->setText( token->parameter().type );
        m_paramMax->setValue( token->parameter().max );
        m_paramValues->listBox()->clear();
        m_paramValues->insertStringList( token->parameter().values );
    }
}

#include "kcfgentryeditor.moc"
