/**
 * gnomeapp.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */
#include "gnomeapp.h"

#include "gnomesubsystem.h"

#include "grouptoken.h"
#include "entries.h"
#include "entrytoken.h"

#include <qapplication.h>

#ifdef signals
#define qtsignals signals
#undef signals
#endif
#include <gconf/gconf-client.h>
#include <glib.h>
#ifdef qtsignals
#define signals qtsignals
#endif

#include <kdebug.h>

using namespace KConfigEditor;

#define KC(x) (static_cast<GConfClient*>(x))

static void printDirs( gpointer data, gpointer user_data )
{
    GNOMEApp *app = static_cast<GNOMEApp*>( user_data );
    const char *dir = ( const char * )( data );
    app->newTopLevelItem( dir );
}

static GConfValue *tokenToValue( EntryToken *token )
{
    GConfValue *value = 0;
    switch( token->type() ) {
    case Token::String:
    {
        value = gconf_value_new( GCONF_VALUE_STRING );
        QVariant var = token->variant();
        gconf_value_set_string( value, var.toString().ascii() );
    }
        break;
    case Token::Int:
    {
        value = gconf_value_new( GCONF_VALUE_INT );
        QVariant var = token->variant();
        gconf_value_set_int( value, var.toInt() );
    }
        break;
    case Token::Double:
    {
        value = gconf_value_new( GCONF_VALUE_FLOAT );
        QVariant var = token->variant();
        gconf_value_set_float( value, var.toDouble() );
    }
        break;
    case Token::Bool:
    {
        value = gconf_value_new( GCONF_VALUE_BOOL );
        QVariant var = token->variant();
        gconf_value_set_bool( value, var.toBool() );
    }
        break;
    default:
        kdWarning()<<"Type "<< token->typeName() << " is not supported by GNOME subsystem"<<endl;
        break;
    }
    return value;
}

GNOMEApp::GNOMEApp( GNOMESubsystem *parent )
    : m_receiver( parent ), m_operation( Parse )
{
}

void GNOMEApp::run()
{
    g_type_init();

    m_client = gconf_client_get_default();

    if ( !m_client ) {
        kdWarning()<<"No default GConf object!"<<endl;
        return;
    }

    while ( 1 ) {
        switch( m_operation ) {
        case Parse:
        {
            GSList *dirs = gconf_client_all_dirs( KC( m_client ), "/", 0);
            g_slist_foreach( dirs, printDirs, static_cast<gpointer>( this )  );
            DoneParsingEvent *pe = new DoneParsingEvent();
            QApplication::postEvent( m_receiver, pe );
        }
            break;
        case Save:
        {
            saveEntries();
            DoneSavingEvent *pe = new DoneSavingEvent();
            QApplication::postEvent( m_receiver, pe );
        }
            break;
        default:
            break;
        }
        m_operation = None;
        m_condition.wait();
    }
}

void GNOMEApp::newTopLevelItem( const QString &name )
{
    GroupToken *top = m_receiver->createTopLevelToken( name );
    addDir( top );
}

void GNOMEApp::addDir( GroupToken *token )
{
    KURL url = token->url();
    const char *path = url.path().latin1();
    GSList *dirs = gconf_client_all_dirs( KC( m_client ), path, 0 );

    for ( GSList *it = dirs; it; it = g_slist_next( it ) ) {
        GroupToken *nGroup = token->createGroup( ( const char * ) ( it->data ) );
        nGroup->propagateChanges( false );
        addDir( nGroup );
    }

    GSList *entries = gconf_client_all_entries( KC( m_client ),
                                                path,
                                                0 );
    for ( GSList *it = entries; it; it = g_slist_next( it ) ) {
        GConfEntry *gconfEntry = static_cast<GConfEntry*>( it->data );
        QString key( gconf_entry_get_key( gconfEntry ) );

        GConfValue *gconfValue = gconf_entry_get_value( gconfEntry );

        if ( gconfValue ) {
            EntryToken *entry = 0;
            switch( gconfValue->type ) {
            case GCONF_VALUE_STRING:
                entry = token->createEntry( key, "String" );
                entry->setValueFromString( gconf_value_get_string( gconfValue ) );
                break;
            case GCONF_VALUE_INT:
                entry = token->createEntry( key, "Int" );
                entry->setVariant( QVariant( gconf_value_get_int( gconfValue ) ) );
                break;
            case GCONF_VALUE_FLOAT:
                entry = token->createEntry( key, "Double" );
                entry->setVariant( QVariant( gconf_value_get_float( gconfValue ) ) );
                break;
            case GCONF_VALUE_BOOL:
                entry = token->createEntry( key, "Bool" );
                entry->setVariant( QVariant( gconf_value_get_bool( gconfValue ) ) );
                break;
            case GCONF_VALUE_SCHEMA:
            {
                static bool warned = false;
                if ( !warned ) {
                    kdWarning()<<"Schema not yet supported ("<< key <<")"<<endl;
                    warned = true; //to avoid billion warnings
                }
            }
                break;
            case GCONF_VALUE_LIST:
            {
                static bool warned = false;
                if ( !warned ) {
                    kdWarning()<<"Value list not yet supported ("<< key <<")"<<endl;
                    warned = true; //to avoid billion warnings
                }
            }
                break;
            case GCONF_VALUE_PAIR:
            {
                static bool warned = false;
                if ( !warned ) {
                    kdWarning()<<"Value pair not yet supported ("<< key <<")"<<endl;
                    warned = true; //to avoid billion warnings
                }
            }
                break;
            case GCONF_VALUE_INVALID:
                kdWarning()<<"Invalid entry at ("<< key <<")"<<endl;
                break;
            default:
                break;
            }
            if ( entry ) {
                const char *schemaName = gconf_entry_get_schema_name( gconfEntry );
                if ( schemaName ) {
                    GConfValue *schemaValue = gconf_client_get( KC( m_client ), schemaName, 0 );

                    GConfSchema *schema = gconf_value_get_schema( schemaValue );
                    if ( schema ) {
                        const char *descr = gconf_schema_get_short_desc( schema );
                        if ( descr )
                            entry->setLabel( descr );

                        descr = gconf_schema_get_long_desc( schema );
                        if ( descr )
                            entry->setWhatsThis( descr );
                    }
                }
            }
        }
    }
    token->propagateChanges( true );
}

void GNOMEApp::parse()
{
    m_operation = Parse;
    m_condition.wakeAll();
}

void GNOMEApp::save( const QDict<EntryToken> &entries )
{
    m_operation = Save;
    m_entries = entries;
    m_condition.wakeAll();
}

void GNOMEApp::saveEntries()
{
    QDictIterator<EntryToken> itr( m_entries );
    EntryToken *token = 0;
    while ( (token=itr.current() ) ) {
        GConfValue *value = tokenToValue( token );
        gconf_client_set( KC( m_client ), token->url().path().ascii(), value, 0 );
        gconf_value_free( value );
        ++itr;
    }
}

void GNOMEApp::saveToken( EntryToken *token )
{
    m_operation = Save;
    m_entries.clear();
    m_entries.insert( token->url().path(), token );
    m_condition.wakeAll();
}
