/***************************************************************************
                          table.h  -  description
                             -------------------
    begin                : Tue Sep 7 1999
    copyright            : (C) 1999 by Jan Mueller
    email                : janmueller7@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef TABLE_H
#define TABLE_H
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <sys/time.h>
#include <sys/types.h>
#include "kaspabase.h"
// #include <kapp.h>

/** Thrown by @ref Table. Table cannot process more than one tuple.*/
class TblMoreThanOneRec: public KaspaErr {
 public:
  TblMoreThanOneRec(char *s=0): KaspaErr(s) {};
};

class TblIsReadOnly: public KaspaErr {
 public:
  TblIsReadOnly(char *s=0): KaspaErr(s) {};
};

class TblIsInvalid: public KaspaErr {
 public:
  TblIsInvalid(char *s=0): KaspaErr(s) {};
};

typedef bool CallBackType(void);


/** Base class of classes which are holding one tuple in a specific table,
    e.g. @ref AuthorBase, @ref NoteBase. This class implements the handling
    of locking, deleting, rereading and saving. Classes inheriting Table
    have to reimplement the pure virtual methods @ref read, @ref write and @ref del.
    The members of class table do not change the underlying @ref KaspaRec record
    except freeing it in the destructor.
    Allocating and other manipulations are on the derived classes (@see AuthorBase).
    The method init() has to be called before any other method is called.
    @ref init and @ref reset throw @ref TblMoreThanOneRec.
    @ref flush and @ref delete throw @ref TblIsReadOnly and TblIsInvalid.
*/
class Table: protected KaspaBase {
 private:
  Table(const Table&);
	Table& operator=(const Table&);	


  char *table;
  bool readonly;
  struct timeval idletimeout;

 protected:		
  KaspaRec *rec;
	CallBackType *callBack;

 public:
  /** Constructor. Arguments are the name of the table, the database name and
      (optional) a pointer to a callback function.
      @see Sql @see KaspaBase */
  Table(const char *tbl, const char *dbname);
  /** Same as above, initializes connection from an established one. */
  Table(const char *tbl, Sql *conn);
  /** Destructor. Deletes the underlying @ref KaspaRec record. */
  virtual ~Table() throw();
	inline bool idle() throw();
//	void cancelExec() { KaspaBase::cancel(); }
	virtual void setCallBack(CallBackType *cb) { callBack=cb; }
  /** Initializes the instance with r */
  virtual void init(KaspaRec *r);
  /** Rereads the tuple from DBMS. Calls @ref read */
  virtual void reset();
  /** Saves the record to DBMS. Calls @ref write. */
  virtual void flush(bool block=false);
  /** Returns TRUE, if a query is processed. */
  virtual bool isWorking() { return KaspaBase::isWorking(); }
  /** Returns TRUE, if the tuple is read only (not locked) */
  virtual bool isReadOnly();
  /** Returns TRUE, if the current record is valid. */
  virtual bool isValid();
  /** Deletes the current tuple. The method calls @ref del. */
  virtual void deleteRec();
  /** Get the links associated with the current object. */
  virtual Title *getLinks();
  /** Inserts a new link between the current object and another. */
  virtual void insertLink(Oid note);
  /** Removes a link between the current object and another one. */
  void removeLink(Oid o);
  /** Returns the object id of the current tuple. */
  virtual Oid id() { if (rec) return rec->id.get(); else return InvalidOid; }
  /** Returns the where clause, from which the current object was determined. Abstract. */
  virtual const char *getWhereClause()=0;
	/** Returns @ref AuthorName with Authors */
	virtual AuthorName *getAuthors();
	/** Returns @ref PublTitle with Publications */
	virtual PublTitle *getPubls();
	/** Returns @ref NoteTitle with Notes */
	virtual NoteTitle *getNotes();	
	/** Returns @ref PartTitle with Parts */
	virtual PartTitle *getParts();
	/** Returns @ref PartTitle with Publishers */
	virtual PublisherName *getPublishers();
	/** Returns @ref PartTitle with Journals */
	virtual JournalName *getJournals();
	
 protected:
  /** Pure virtual method. Derived classes have to implement reading and reserving
      the records specified by the where clause. E.g. use @ref KaspaBase::getAuthor
      to read an author. The members of class table do not change the underlying
      @ref KaspaRec record. Allocating, deleting etc. is on the derived classes.
   */
  virtual KaspaRec *read(const char *where)=0;
  /** Same as above, use @ref KaspaBase::updateAuthor for updating. @see read */
  virtual void write(const char *where, bool block=false)=0;
  /** Same as above, use @ref KaspaBase::deleteAuthor to delete records. @see read */
  virtual void del(const Oid o)=0;
	/** Returns the settings if an error occurs. */
  virtual KaspaRec *invalid()=0;
};

#endif
