/***************************************************************************
                          ktransferkio.cpp  -  description
                             -------------------
    begin                : Thu Aug 9 2001
    copyright            : (C) 2001 by Jonathon Sim
    email                : jonathonsim@iname.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ktransferkio.h"
#include <kio/job.h>
#include <kio/netaccess.h>
#include <kdebug.h>
#include <qvaluelist.h>
#include <qdict.h>
#include "global.h"
#include <klocale.h>
#include <kmessagebox.h>


#define EXIST_AND_NOT_EQUAL(A,B) A && B && (A != B)
#define EXIST_AND_GREATER(A,B) A && B && (A > B)


KTransferKIO::KTransferKIO(const KSynkConnection & connection)
    : KTransfer(connection)
{
    finished=false;
    transferqueue=0;
    dirlister = 0L;
    transferjob = 0L;
    localfilelist.setAutoDelete( true );
    remotefilelist.setAutoDelete( true );
}

KTransferKIO::~KTransferKIO()
{
    kdDebug(DEBUG_AREA) << "KTransferKIO:: destructor..." << endl;
    delete dirlister;
    delete transferqueue;
}

/** Self explanatory really.  The public method used to start the transfer   */
void KTransferKIO::transfer()
{
    /*The actual process of performing this transfer is in fact more complicated
      than the name of this function might indicate.  It involves many other methods
      signals and slots as indicated schematicaly below below
      transfer->startListing (local)->KRecursiveLister (local)
      -> slotListingResult (local)->startListing (remote)->RecursiveLister (Remote)->slotListingResult (remote)
      ->findChanged
      This gets us to the point where we are ready to start the actual transfer.
      To do that, we use a CopyJob for each directory, with a queue of (dest, sourcelist) pairs,
      like so:
      ->transferFiles->makeTransferQueue->Nexttransfer->SlotTransferResult->nextTransfer->slotTransferResult ....
      and so on.
    */
	
    emit signalMessage(i18n("Finding files..."));
    emit signalPercent(0, 0);
    startListing(Connection->getLocalURL(),"local");
}

/** handles completion of a transfer */
void KTransferKIO::slotTransferResult(KIO::Job* job)
{
    KURL::List urls = static_cast<KIO::CopyJob*>(job)->srcURLs();

    if ( job->error() )
    {
        if ( job->error() == KIO::ERR_USER_CANCELED )
        {
            emit signalFilesCanceled( urls );
            if ( KMessageBox::questionYesNo( 0L,
                                             i18n("Do want to continue the synchronization?"),
                                             i18n("Continue Synchronization?"),
                                             KStdGuiItem::cont(),
                                             KGuiItem("Abort") )
                 == KMessageBox::No )
            {
                emit signalMessage(i18n("Transfer aborted!"));
                transferjob = 0L;
                kill(); // AIII!
                return;
            }
        }
        else {
            emit signalFilesError( urls );
            job->showErrorDialog(  0L  );
        }
    }
    else // files completed
        emit signalFilesComplete( urls );

    nextTransfer();
    if (finished)	{
        emit signalMessage(i18n("Synchronisation complete."));
        KMessageBox::information(0L,
                                 i18n("The Synchronization has finished."),
                                 i18n("Finished"), "syncFinished");
        delete this;		//Now that we are finished, lets delete ourselves
    }
}

/** Updates a percentage bar (somewhere) */
void KTransferKIO::slotPercent(KIO::Job*, unsigned long percent)
{
    emit signalPercent(percent, transferqueue->count());
}

/** Starts finding a listing of all directories matching the current connections filter.
    @param	url		The url to list
    @param	name 	A name used to identify which connection we are listing
*/
void KTransferKIO::startListing(KURL url,const QString & name)
{
    if (!dirlister)	{
        dirlister=new KRecursiveLister();
        connect(dirlister,SIGNAL(completed()), this, SLOT(slotListingResult ()) );
    }
    dirlister->setName(name);
    kdDebug(DEBUG_AREA) << " KTransferKIO::startListing:Begin Listing: "<< name << endl;
    dirlister->openURL( url ,true);
}

/** Handle the completion of the listing */
void KTransferKIO::slotListingResult()
{
    kdDebug(DEBUG_AREA) << "KTransferKIO::slotListingResult()"<<endl;
    if (QString(dirlister->name()) == "local" )	{
        //Handles completion of the local listing
        kdDebug(DEBUG_AREA) << "Local listing complete" << endl;
        KFileItemList templist=dirlister->items();
        KFileItem * item = 0L;
        KFileItem * newitem = 0L;
        localfilelist.clear();
        kdDebug(DEBUG_AREA) << "KTransferKIO::slotListingResult() --local filecheck" << endl;		
        for( item = templist.first(); item != 0; item=templist.next() ) {
            newitem= new KFileItem(*item);
            kdDebug(DEBUG_AREA) << "\t" << newitem->url().url() <<endl;
            localfilelist.append(newitem);
        }
        startListing(Connection->getRemoteURL(), "remote" );
        //This is silly, but it will have to do:
        emit signalPercent(50, 0);
    }
    else if (QString(dirlister->name())=="remote" )	{
        //Handles completion of the remote listing
        kdDebug(DEBUG_AREA) << "remote listing complete" << endl;
        KFileItemList templist=dirlister->items();
        KFileItem * item = 0L;
        KFileItem * newitem = 0L;
        remotefilelist.clear();
        kdDebug(DEBUG_AREA) << "KTransferKIO::slotListingResult() --remote filecheck" << endl;	
        //Get the remote files
        for( item = templist.first(); item != 0; item=templist.next() ) {
            newitem= new KFileItem(*item);
            kdDebug(DEBUG_AREA) << "\t" << newitem->url().url() <<endl;
            remotefilelist.append(newitem);
        }	
        //Get the remote dirs
        kdDebug(DEBUG_AREA) << "KTransferKIO::slotListingResult() --remote dir structure" << endl;
        templist=dirlister->dirs();
        for( item = templist.first(); item != 0; item=templist.next() ) {
            newitem= new KFileItem(*item);
            kdDebug(DEBUG_AREA) << "\t" << newitem->url().url() <<endl;
            remotedirtree.append(newitem);
        }	
        emit signalPercent(100, 0);
        emit signalMessage(i18n("Finding changed files..."));
        emit signalPercent(0, 0);
        KFileItemList * filestoget=findChanged();
        KFileItem * file = 0L;
        KURL::List urlstoget;
        emit signalPercent(100, 0);
        for( file = filestoget->first(); file != 0; file=filestoget->next() ) {
            urlstoget.append(file->url() );	
        }
        delete filestoget;
        
        emit signalPercent(0, 0);
        emit signalGettingFiles(urlstoget);
        //Start the actual transfer
        transferFiles(urlstoget,Connection->getLocalURL());
    }
    else	{
        kdDebug(DEBUG_AREA) << "KTransferKIO::ListingResult() : Unknown DirLister:" << endl;
        kdDebug(DEBUG_AREA) << dirlister->name() << endl;
    }
}

/** Determines which files in the lists need to be updated
    @return	A KFileItemList of the files to get.  Note that the caller is responsible for deleteing it.
*/
KFileItemList * KTransferKIO::findChanged()
{
    //We do this in the simplest way I can think of.
    kdDebug(DEBUG_AREA) << "KTransferKIO::findChanged()" <<endl;
    QDict<KFileItem> localdict ;
    KFileItemList * itemstoget=new KFileItemList;
    KFileItem * remotefile ;
    KFileItem * localfile;
    QString relurl;
	
    //First we put all the local files into a dictionary, indexed by their relative url...
    for( localfile = localfilelist.first(); localfile != 0; localfile=localfilelist.next() ) 	{
        localdict.insert(makeRelative(localfile->url(),Connection->getLocalURL() ),localfile);
    }
	
    //...then for each file on the remote server we look up its relative url to get
    //the corresponding local file
    for( remotefile = remotefilelist.first(); remotefile != 0; remotefile=remotefilelist.next() ) 	{
        relurl=makeRelative(remotefile->url(),Connection->getRemoteURL()) ;
        localfile= localdict[relurl];	
        if ( localfile ) {
            //The current file is already here
            //KFileItems return 0 if an attribute can't be determined, so we use macros to make this prettier
            if ( EXIST_AND_NOT_EQUAL(localfile->size(),remotefile->size()) ) {
                itemstoget->append(remotefile);
                localdict.remove(makeRelative(localfile->url(),Connection->getLocalURL() ));	
                kdDebug(DEBUG_AREA) << "file has different size:" << remotefile->url().url() << "(" << (ulong) localfile->size() << ", " << (ulong) remotefile->size() << ")" <<endl;
            }
            else if (!getConnection().ignoreModifcationTime() &&
                     EXIST_AND_GREATER(remotefile->time(KIO::UDS_MODIFICATION_TIME), \
                                       localfile->time(KIO::UDS_MODIFICATION_TIME)) ){
                itemstoget->append(remotefile);
                localdict.remove(makeRelative(localfile->url(),Connection->getLocalURL() ));
                kdDebug(DEBUG_AREA) << "file has newer modification time:" << remotefile->url().url() <<endl;
            }	
            else
                kdDebug(DEBUG_AREA) << "file is unchanged:" << remotefile->url().url() <<endl;
        }
        else {
            //The current file is not here
            kdDebug(DEBUG_AREA) << "file is not present:" << remotefile->url().url() <<endl;
            itemstoget->append(remotefile);
        }
    }
	
    //Emit a signal so that gui elements monitoring this transfer have something to say.
	
    //### Anything still present in the localdict is not present on the remote server, so should be deleted?
    return  itemstoget;
}

/** Helper function that will convert a url to a relative url */
QString KTransferKIO::makeRelative(const KURL&url,const KURL & relativeto)
{
    //TODO : This without strings.
    QString tempstring=url.url();
    if ( tempstring.startsWith( relativeto.url() ) )
        tempstring = tempstring.mid( relativeto.url().length() );

    return tempstring;
}

/** The function that actually begins the transfer. */
void KTransferKIO::transferFiles(const KURL::List& source_list,
                                 const KURL& dest)
{
    kdDebug(DEBUG_AREA) << "KTransferKIO::transferFiles: source file listing:" <<endl;
    if (!source_list.isEmpty()) {
        emit signalMessage(i18n("Transfering %1 changed files:").arg(source_list.count()) );
        KURL::List::ConstIterator it;
        for( it = source_list.begin(); it != source_list.end(); ++it ) {
            kdDebug(DEBUG_AREA) << "\t"<<(*it).url()<<endl;
        }
        kdDebug(DEBUG_AREA) << "KTransferKIO::transferFiles: destination:"<<dest.url() <<endl;
        createDirs();	
        //Constructs a queue of destination/sourcelist pairs
        transferqueue=makeTransferQueue(source_list);
        nextTransfer();
    }
    else {
        kdDebug(DEBUG_AREA) << "No files to transfer"<<endl;
        emit signalMessage(i18n("Complete - All files were already up-to-date."));
        emit signalComplete();
    }	
}

/** Creates the queue of transfers for the copy job. */
KTransferKIO::TransferQueue * KTransferKIO::makeTransferQueue(const KURL::List & urllist)
{
    //We are attempting to create a queue of (destination url, source url list) pairs.
    QDict<TransferDescriptor> dict;
    TransferQueue * queue=new TransferQueue;
    KURL::List::ConstIterator it;
    TransferDescriptor * transfer;
    QString localdir;
    for (it = urllist.begin(); it != urllist.end(); ++it ) {
        //Construct a QString representing the local directoy corresponding to this file.
        localdir=remoteToLocal(*it).directory(false);
        //Then use it as an index into a dictionary...
        if (!dict[makeRelative(*it,Connection->getRemoteURL())]) {
            transfer=new TransferDescriptor;	
            //Construct the local url corresponding to the remote file, and find out what dir its in.
            transfer->dest=remoteToLocal(*it).directory(false);
            dict.insert( localdir, transfer);
            queue->enqueue(transfer);
        }	
        else	{
            transfer=dict[localdir];
        }	
        transfer->sources.append(*it);
    }
    return queue;
}

/** Starts the next transfer in the queue */
void KTransferKIO::nextTransfer()
{
    if (!transferqueue->isEmpty())
    {
        TransferDescriptor * nexttransfer=transferqueue->dequeue();
        kdDebug(DEBUG_AREA) << "KTransferKIO::nextTransfer() :  dest" << nexttransfer->dest.url() << endl;
        DEBUG_KURLLIST("KTransferKIO::nextTransfer() : sources :",nexttransfer->sources);
        transferjob = new  KIO::CopyJob( nexttransfer->sources,
                                         nexttransfer->dest,
                                         KIO::CopyJob::Copy,false,false);
        delete nexttransfer;
        connect(transferjob, SIGNAL(result( KIO::Job * )),
                this, SLOT(slotTransferResult( KIO::Job * )) );
        connect(transferjob, SIGNAL(percent( KIO::Job *, unsigned long  ) ),
                this, SLOT(slotPercent( KIO::Job *, unsigned long)) );
        connect(transferjob, SIGNAL( copying( KIO::Job *,
                                              const KURL&, const KURL& )),
                this, SLOT( slotCopying( KIO::Job *,
                                         const KURL&, const KURL& )) );
    }
    else {
	//We are done!
        finished=true;
        emit signalComplete();
    }
}

/** Duplicates the sub-directory structure of the remote url locally, as a preliminary to copying files into it */
void KTransferKIO::createDirs()
{
    KFileItem * dir;
    KURL dirurl;
    KURL::List createlist;  //A list of directories to create
    KURL::List::Iterator it;
    kdDebug(DEBUG_AREA) << "KTransferKIO::createDirs()" << endl;
    for( dir = remotedirtree.first(); dir != 0; dir=remotedirtree.next() ) {
        //see if the directory exists.  If not, create it.
        createlist.clear();
        dirurl=remoteToLocal(dir->url());
        if (!KIO::NetAccess::exists(dirurl))
        {
            kdDebug(DEBUG_AREA) << "KTransferKIO::createDirs(): pending create: " << dirurl.url() << endl;
            createlist.append(dirurl);
            dirurl=dirurl.upURL();
            //Keep changing up a directory, until we find a parent that does exist
            while (!KIO::NetAccess::exists(dirurl)) {
                kdDebug(DEBUG_AREA) << "KTransferKIO::createDirs(): pending create: " << dirurl.url() << endl;
                createlist.append(dirurl);
                dirurl=dirurl.upURL();
            }

            //Now create all the directories between the original dir and this existing parent
            while ( (it = createlist.fromLast()) != createlist.end() )
            {
                if (!KIO::NetAccess::mkdir(*it))
                {
                    kdDebug(DEBUG_AREA) << "mkdir: "<<
                        KIO::NetAccess::lastErrorString() << " "
                                        << (*it).url()<< endl;
                    emit signalTransferError(i18n("Unable to create directory: %1\n").arg( (*it).url()) + KIO::NetAccess::lastErrorString() );
                    //TODO : better error handling
                }

                createlist.remove( it );
            }
        }
    }
}

/** Converts the specified remote url to the corresponding local url */
KURL KTransferKIO::remoteToLocal(const KURL & url)
{
    //Simple its true, but we do this a lot.
    const QString relativeurl=makeRelative(url,Connection->getRemoteURL());
    return KURL(Connection->getLocalURL(),relativeurl);
}

/** Stops the current transfer.  */
void KTransferKIO::kill()
{
    kdDebug(DEBUG_AREA) << "Killing transfer" << endl;
    if (transferjob) {
        disconnect( transferjob, 0L, this, 0L );
        transferjob->kill();
        transferjob = 0L;
    }
    else if (dirlister) {
        dirlister->stop();
    }
    finished=true;
    emit signalMessage(i18n("Transfer Cancelled"));
    emit signalComplete();
    delete this;
}

void KTransferKIO::slotCopying( KIO::Job *,
                                const KURL& from, const KURL& /* to */ )
{
    if ( !transferqueue ) // eeh?
        return;

    if ( transferqueue->isEmpty() )
        emit signalMessage(i18n("<qt>Copying last file: <br><b>%1</b></qt>").arg( from.fileName() ) );
    else
        emit signalMessage(i18n("<qt>%1 files to copy...<br><b>%2</b></qt>").arg(transferqueue->count() + 1).arg( from.fileName() ));
}

#include "ktransferkio.moc"
