﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/comprehend/ComprehendRequest.h>
#include <aws/comprehend/Comprehend_EXPORTS.h>
#include <aws/comprehend/model/Tag.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Comprehend {
namespace Model {

/**
 */
class ImportModelRequest : public ComprehendRequest {
 public:
  AWS_COMPREHEND_API ImportModelRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ImportModel"; }

  AWS_COMPREHEND_API Aws::String SerializePayload() const override;

  AWS_COMPREHEND_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the custom model to import.</p>
   */
  inline const Aws::String& GetSourceModelArn() const { return m_sourceModelArn; }
  inline bool SourceModelArnHasBeenSet() const { return m_sourceModelArnHasBeenSet; }
  template <typename SourceModelArnT = Aws::String>
  void SetSourceModelArn(SourceModelArnT&& value) {
    m_sourceModelArnHasBeenSet = true;
    m_sourceModelArn = std::forward<SourceModelArnT>(value);
  }
  template <typename SourceModelArnT = Aws::String>
  ImportModelRequest& WithSourceModelArn(SourceModelArnT&& value) {
    SetSourceModelArn(std::forward<SourceModelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name to assign to the custom model that is created in Amazon Comprehend
   * by this import.</p>
   */
  inline const Aws::String& GetModelName() const { return m_modelName; }
  inline bool ModelNameHasBeenSet() const { return m_modelNameHasBeenSet; }
  template <typename ModelNameT = Aws::String>
  void SetModelName(ModelNameT&& value) {
    m_modelNameHasBeenSet = true;
    m_modelName = std::forward<ModelNameT>(value);
  }
  template <typename ModelNameT = Aws::String>
  ImportModelRequest& WithModelName(ModelNameT&& value) {
    SetModelName(std::forward<ModelNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version name given to the custom model that is created by this import.
   * Version names can have a maximum of 256 characters. Alphanumeric characters,
   * hyphens (-) and underscores (_) are allowed. The version name must be unique
   * among all models with the same classifier name in the account/Region.</p>
   */
  inline const Aws::String& GetVersionName() const { return m_versionName; }
  inline bool VersionNameHasBeenSet() const { return m_versionNameHasBeenSet; }
  template <typename VersionNameT = Aws::String>
  void SetVersionName(VersionNameT&& value) {
    m_versionNameHasBeenSet = true;
    m_versionName = std::forward<VersionNameT>(value);
  }
  template <typename VersionNameT = Aws::String>
  ImportModelRequest& WithVersionName(VersionNameT&& value) {
    SetVersionName(std::forward<VersionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ID for the KMS key that Amazon Comprehend uses to encrypt trained custom
   * models. The ModelKmsKeyId can be either of the following formats:</p> <ul> <li>
   * <p>KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code> </p> </li>
   * <li> <p>Amazon Resource Name (ARN) of a KMS Key:
   * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
   * </p> </li> </ul>
   */
  inline const Aws::String& GetModelKmsKeyId() const { return m_modelKmsKeyId; }
  inline bool ModelKmsKeyIdHasBeenSet() const { return m_modelKmsKeyIdHasBeenSet; }
  template <typename ModelKmsKeyIdT = Aws::String>
  void SetModelKmsKeyId(ModelKmsKeyIdT&& value) {
    m_modelKmsKeyIdHasBeenSet = true;
    m_modelKmsKeyId = std::forward<ModelKmsKeyIdT>(value);
  }
  template <typename ModelKmsKeyIdT = Aws::String>
  ImportModelRequest& WithModelKmsKeyId(ModelKmsKeyIdT&& value) {
    SetModelKmsKeyId(std::forward<ModelKmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role that grants Amazon Comprehend
   * permission to use Amazon Key Management Service (KMS) to encrypt or decrypt the
   * custom model.</p>
   */
  inline const Aws::String& GetDataAccessRoleArn() const { return m_dataAccessRoleArn; }
  inline bool DataAccessRoleArnHasBeenSet() const { return m_dataAccessRoleArnHasBeenSet; }
  template <typename DataAccessRoleArnT = Aws::String>
  void SetDataAccessRoleArn(DataAccessRoleArnT&& value) {
    m_dataAccessRoleArnHasBeenSet = true;
    m_dataAccessRoleArn = std::forward<DataAccessRoleArnT>(value);
  }
  template <typename DataAccessRoleArnT = Aws::String>
  ImportModelRequest& WithDataAccessRoleArn(DataAccessRoleArnT&& value) {
    SetDataAccessRoleArn(std::forward<DataAccessRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags to associate with the custom model that is created by this import. A tag
   * is a key-value pair that adds as a metadata to a resource used by Amazon
   * Comprehend. For example, a tag with "Sales" as the key might be added to a
   * resource to indicate its use by the sales department.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  ImportModelRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  ImportModelRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sourceModelArn;

  Aws::String m_modelName;

  Aws::String m_versionName;

  Aws::String m_modelKmsKeyId;

  Aws::String m_dataAccessRoleArn;

  Aws::Vector<Tag> m_tags;
  bool m_sourceModelArnHasBeenSet = false;
  bool m_modelNameHasBeenSet = false;
  bool m_versionNameHasBeenSet = false;
  bool m_modelKmsKeyIdHasBeenSet = false;
  bool m_dataAccessRoleArnHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Comprehend
}  // namespace Aws
