﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class EnableImageDeprecationRequest : public EC2Request {
 public:
  AWS_EC2_API EnableImageDeprecationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "EnableImageDeprecation"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the AMI.</p>
   */
  inline const Aws::String& GetImageId() const { return m_imageId; }
  inline bool ImageIdHasBeenSet() const { return m_imageIdHasBeenSet; }
  template <typename ImageIdT = Aws::String>
  void SetImageId(ImageIdT&& value) {
    m_imageIdHasBeenSet = true;
    m_imageId = std::forward<ImageIdT>(value);
  }
  template <typename ImageIdT = Aws::String>
  EnableImageDeprecationRequest& WithImageId(ImageIdT&& value) {
    SetImageId(std::forward<ImageIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time to deprecate the AMI, in UTC, in the following format:
   * <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z. If you specify a
   * value for seconds, Amazon EC2 rounds the seconds to the nearest minute.</p>
   * <p>You can’t specify a date in the past. The upper limit for
   * <code>DeprecateAt</code> is 10 years from now, except for public AMIs, where the
   * upper limit is 2 years from the creation date.</p>
   */
  inline const Aws::Utils::DateTime& GetDeprecateAt() const { return m_deprecateAt; }
  inline bool DeprecateAtHasBeenSet() const { return m_deprecateAtHasBeenSet; }
  template <typename DeprecateAtT = Aws::Utils::DateTime>
  void SetDeprecateAt(DeprecateAtT&& value) {
    m_deprecateAtHasBeenSet = true;
    m_deprecateAt = std::forward<DeprecateAtT>(value);
  }
  template <typename DeprecateAtT = Aws::Utils::DateTime>
  EnableImageDeprecationRequest& WithDeprecateAt(DeprecateAtT&& value) {
    SetDeprecateAt(std::forward<DeprecateAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline EnableImageDeprecationRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_imageId;

  Aws::Utils::DateTime m_deprecateAt{};

  bool m_dryRun{false};
  bool m_imageIdHasBeenSet = false;
  bool m_deprecateAtHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
