// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { GetItemCommand } from '@aws-sdk/client-dynamodb';
import { needs, NodeBranchKeyMaterial, } from '@aws-crypto/material-management';
import { unmarshall } from '@aws-sdk/util-dynamodb';
import { DecryptCommand } from '@aws-sdk/client-kms';
import { PARTITION_KEY, SORT_KEY, TABLE_FIELD, CUSTOM_ENCRYPTION_CONTEXT_FIELD_PREFIX, BRANCH_KEY_IDENTIFIER_FIELD, TYPE_FIELD, KEY_CREATE_TIME_FIELD, HIERARCHY_VERSION_FIELD, KMS_FIELD, BRANCH_KEY_FIELD, BRANCH_KEY_ACTIVE_VERSION_FIELD, BRANCH_KEY_TYPE_PREFIX, BRANCH_KEY_ACTIVE_TYPE, BEACON_KEY_TYPE_VALUE, } from './constants';
/**
 * This utility function uses a partition and sort key to query for a single branch
 * keystore record
 * @param ddbClient
 * @param ddbTableName
 * @param partitionValue
 * @param sortValue
 * @returns A DDB response item representing the branch keystore record
 * @throws 'Record not found in DynamoDB' if the query yields no hits for a
 * branch key record
 */
export async function getBranchKeyItem({ ddbClient, ddbTableName, }, partitionValue, sortValue) {
    // create a getItem command with the querying partition and sort keys
    // send the query for DDB to run
    // get the response
    const response = await ddbClient.send(new GetItemCommand({
        TableName: ddbTableName,
        Key: {
            [PARTITION_KEY]: { S: partitionValue },
            [SORT_KEY]: { S: sortValue },
        },
    }));
    // the response has an Item field if the branch keystore record was found
    const responseItem = response.Item;
    // error out if there is not Item field (record not found)
    needs(responseItem, `A branch key record with ${PARTITION_KEY}=${partitionValue} and ${SORT_KEY}=${sortValue} was not found in the DynamoDB table ${ddbTableName}.`);
    // at this point, we got back a record so convert the DDB response item into
    // a more JS-friendly object
    return unmarshall(responseItem);
}
/**
 * This utility function validates the DDB response item against the required
 * record fromat and transforms the item into a branch key record
 * @param item is the DDB response item representing a branch keystore record
 * @returns a validated branch key record abiding by the proper record format
 * @throws `Branch keystore record does not contain a ${BRANCH_KEY_IDENTIFIER_FIELD} field of type string`
 * @throws `Branch keystore record does not contain a valid ${TYPE_FIELD} field of type string`
 * @throws `Branch keystore record does not contain a ${BRANCH_KEY_ACTIVE_VERSION_FIELD} field of type string`
 * if the type field is "branch:ACTIVE" but there is no version field in the DDB
 * response item
 * @throws `Branch keystore record does not contain ${BRANCH_KEY_FIELD} field of type Uint8Array`
 * @throws `Branch keystore record does not contain ${KMS_FIELD} field of type string`
 * @throws `Branch keystore record does not contain ${KEY_CREATE_TIME_FIELD} field of type string`
 * @throws `Branch keystore record does not contain ${HIERARCHY_VERSION_FIELD} field of type number`
 * @throws `Custom encryption context key ${field} should be prefixed with ${CUSTOM_ENCRYPTION_CONTEXT_FIELD_PREFIX}`
 * if there are additional fields within the response item that
 * don't follow the proper custom encryption context key naming convention
 */
//= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
//# A branch key record MUST include the following key-value pairs:
export function validateBranchKeyRecord(item) {
    //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
    //# 1. `branch-key-id` : Unique identifier for a branch key; represented as [AWS DDB String](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes)
    needs(BRANCH_KEY_IDENTIFIER_FIELD in item &&
        typeof item[BRANCH_KEY_IDENTIFIER_FIELD] === 'string', `Branch keystore record does not contain a ${BRANCH_KEY_IDENTIFIER_FIELD} field of type string`);
    //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
    //# 1. `type` : One of the following; represented as [AWS DDB String](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes)
    //#    - The string literal `"beacon:ACTIVE"`. Then `enc` is the wrapped beacon key.
    //#    - The string `"branch:version:"` + `version`, where `version` is the Branch Key Version. Then `enc` is the wrapped branch key.
    //#    - The string literal `"branch:ACTIVE"`. Then `enc` is the wrapped beacon key of the active version. Then
    needs(TYPE_FIELD in item &&
        typeof item[TYPE_FIELD] === 'string' &&
        (item[TYPE_FIELD] === BRANCH_KEY_ACTIVE_TYPE ||
            item[TYPE_FIELD].startsWith(BRANCH_KEY_TYPE_PREFIX) ||
            item[TYPE_FIELD] === BEACON_KEY_TYPE_VALUE), `Branch keystore record does not contain a valid ${TYPE_FIELD} field of type string`);
    //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
    //# 1. `version` : Only exists if `type` is the string literal `"branch:ACTIVE"`.
    //#   Then it is the Branch Key Version. represented as [AWS DDB String](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes)
    if (item[TYPE_FIELD] === BRANCH_KEY_ACTIVE_TYPE) {
        needs(BRANCH_KEY_ACTIVE_VERSION_FIELD in item &&
            typeof item[BRANCH_KEY_ACTIVE_VERSION_FIELD] === 'string', `Branch keystore record does not contain a ${BRANCH_KEY_ACTIVE_VERSION_FIELD} field of type string`);
    }
    //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
    //# 1. `enc` : Encrypted version of the key;
    //#   represented as [AWS DDB Binary](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes)
    needs(BRANCH_KEY_FIELD in item && item[BRANCH_KEY_FIELD] instanceof Uint8Array, `Branch keystore record does not contain ${BRANCH_KEY_FIELD} field of type Uint8Array`);
    //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
    //# 1. `kms-arn`: The AWS KMS Key ARN used to generate the `enc` value.
    //#   represented as [AWS DDB String](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes)
    needs(KMS_FIELD in item && typeof item[KMS_FIELD] === 'string', `Branch keystore record does not contain ${KMS_FIELD} field of type string`);
    //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
    //# 1. `create-time`: Timestamp in ISO 8601 format in UTC, to microsecond precision.
    //#   Represented as [AWS DDB String](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes)
    needs(KEY_CREATE_TIME_FIELD in item &&
        typeof item[KEY_CREATE_TIME_FIELD] === 'string', `Branch keystore record does not contain ${KEY_CREATE_TIME_FIELD} field of type string`);
    //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
    //# 1. `hierarchy-version`: Version of the hierarchical keyring;
    //#   represented as [AWS DDB Number](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes)
    needs(HIERARCHY_VERSION_FIELD in item &&
        typeof item[HIERARCHY_VERSION_FIELD] === 'number', `Branch keystore record does not contain ${HIERARCHY_VERSION_FIELD} field of type number`);
    // This requirement is around the construction of the encryption context.
    // It is possible that customers will have constructed their own branch keys
    // with a custom creation method.
    // In this case encryption context may not be prefixed.
    // The Dafny version of this code does not enforce
    // that additional encryption context keys MUST be prefixed,
    // therefore the JS release does not as well.
    //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#record-format
    //# A branch key record MAY include [custom encryption context](../branch-key-store.md#custom-encryption-context) key-value pairs.
    //# These attributes should be prefixed with `aws-crypto-ec:` the same way they are for [AWS KMS encryption context](../branch-key-store.md#encryption-context).
    // serialize the DDB response item as a more well-defined and validated branch
    // key record object
    return Object.assign({}, item);
}
/**
 * This utility function builds an authenticated encryption context from the DDB
 * response item
 * @param logicalKeyStoreName
 * @param branchKeyRecord
 * @returns authenticated encryption context
 */
export function constructAuthenticatedEncryptionContext(
//= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#logical-keystore-name
//# It is not stored on the items in the so it MUST be added
//# to items retrieved from the table.
{ logicalKeyStoreName }, branchKeyRecord) {
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#encryption-context
    //# This section describes how the AWS KMS encryption context is built
    //# from an [encrypted hierarchical key](./key-store/key-storage.md#encryptedhierarchicalkey).
    //#
    //# The following encryption context keys are shared:
    //#
    //# - MUST have a `branch-key-id` attribute
    //# - The `branch-key-id` field MUST not be an empty string
    //# - MUST have a `type` attribute
    //# - The `type` field MUST not be an empty string
    //# - MUST have a `create-time` attribute
    //# - MUST have a `tablename` attribute to store the logicalKeyStoreName
    //# - MUST have a `kms-arn` attribute
    //# - MUST have a `hierarchy-version`
    //# - MUST NOT have a `enc` attribute
    //#
    //# Any additionally attributes in the EncryptionContext
    //# of the [encrypted hierarchical key](./key-store/key-storage.md#encryptedhierarchicalkey)
    //# MUST be added to the encryption context.
    //#
    // the encryption context is a string to string map, so serialize the branch
    // key record to this form
    // filter out the enc field
    // add in the tablename key-value pair
    const encryptionContext = {
        ...Object.fromEntries(Object.entries(branchKeyRecord)
            .map(([key, value]) => [key, value.toString()])
            .filter(([key]) => key !== BRANCH_KEY_FIELD)),
        [TABLE_FIELD]: logicalKeyStoreName,
    };
    return encryptionContext;
}
/**
 * This utility function decrypts a branch key via KMS
 * @param kmsConfiguration
 * @param grantTokens
 * @param kmsClient
 * @param branchKeyRecord
 * @param authenticatedEncryptionContext
 * @returns the unencrypted branch key
 * @throws 'KMS ARN from DDB response item MUST be compatible with the configured KMS Key in the AWS KMS Configuration for this keystore'
 * @throws 'KMS branch key decryption failed' if the KMS response does not
 * contain a plaintext field representing the plaintext branch data key
 */
export async function decryptBranchKey({ kmsConfiguration, grantTokens, 
//= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
//# The operation MUST use the configured `KMS SDK Client` to decrypt the value of the branch key field.
kmsClient, }, encryptedHierarchicalKey) {
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#discovery
    //# The Keystore MAY use the KMS Key ARNs already
    //# persisted to the backing DynamoDB table,
    //# provided they are in records created
    //# with an identical Logical Keystore Name.
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#mrdiscovery
    //# The Keystore MAY use the KMS Key ARNs already
    //# persisted to the backing DynamoDB table,
    //# provided they are in records created
    //# with an identical Logical Keystore Name.
    const KeyId = kmsConfiguration.getCompatibleArnArn(encryptedHierarchicalKey.kmsArn);
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
    //# When calling [AWS KMS Decrypt](https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html),
    //# the keystore operation MUST call with a request constructed as follows:
    const response = await kmsClient.send(new DecryptCommand({
        KeyId,
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
        //# - `CiphertextBlob` MUST be the `CiphertextBlob` attribute value on the provided EncryptedHierarchicalKey
        CiphertextBlob: encryptedHierarchicalKey.ciphertextBlob,
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
        //# - `EncryptionContext` MUST be the [encryption context](#encryption-context) of the provided EncryptedHierarchicalKey
        EncryptionContext: encryptedHierarchicalKey.encryptionContext,
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
        //# - `GrantTokens` MUST be this keystore's [grant tokens](https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token).
        GrantTokens: grantTokens ? grantTokens.slice() : grantTokens,
    }));
    // error out if for some reason the KMS response does not contain the
    // plaintext branch data key
    needs(response.Plaintext, 'KMS branch key decryption failed');
    // convert the unencrypted branch key into a Node Buffer
    return Buffer.from(response.Plaintext);
}
/**
 * This utility function constructs branch key materials from the authenticated
 * encryption context
 * @param branchKey
 * @param branchKeyId
 * @param authenticatedEncryptionContext
 * @returns branch key materials
 * @throws 'Unable to get branch key version to construct branch key materials from authenticated encryption context'
 * if the type in the EC is invalid
 */
export function constructBranchKeyMaterials(branchKey, encryptedHierarchicalKey) {
    return new NodeBranchKeyMaterial(
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
    //# - [Branch Key](./structures.md#branch-key) MUST be the [decrypted branch key material](#aws-kms-branch-key-decryption)
    branchKey, 
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
    //# - [Branch Key Id](./structures.md#branch-key-id) MUST be the `branch-key-id`
    encryptedHierarchicalKey.branchKeyId, 
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
    //# - [Branch Key Version](./structures.md#branch-key-version)
    //# The version string MUST start with `branch:version:`.
    //# The remaining string encoded as UTF8 bytes MUST be the Branch Key version.
    encryptedHierarchicalKey.type.version, 
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
    //# - [Encryption Context](./structures.md#encryption-context-3) MUST be constructed by
    //# [Custom Encryption Context From Authenticated Encryption Context](#custom-encryption-context-from-authenticated-encryption-context)
    constructCustomEncryptionContext(encryptedHierarchicalKey.encryptionContext));
}
/**
 * This is a utility function that constructs a custom encryption context from
 * an authenticated encryption context
 * @param authenticatedEncryptionContext
 * @returns custom encryption context
 */
function constructCustomEncryptionContext(authenticatedEncryptionContext) {
    const customEncryptionContext = {};
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#custom-encryption-context-from-authenticated-encryption-context
    //# For every key in the [encryption context](./structures.md#encryption-context-3)
    //# the string `aws-crypto-ec:` + the UTF8 decode of this key
    //# MUST exist as a key in the authenticated encryption context.
    //# Also, the value in the [encryption context](./structures.md#encryption-context-3) for this key
    //# MUST equal the value in the authenticated encryption context
    //# for the constructed key.
    for (const [key, value] of Object.entries(authenticatedEncryptionContext)) {
        if (key.startsWith(CUSTOM_ENCRYPTION_CONTEXT_FIELD_PREFIX)) {
            customEncryptionContext[key] = value;
        }
    }
    return customEncryptionContext;
}
//# sourceMappingURL=data:application/json;base64,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