// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { EncryptedHierarchicalKey, ActiveHierarchicalSymmetricVersion, HierarchicalSymmetricVersion, } from './types';
import { DynamoDBClient } from '@aws-sdk/client-dynamodb';
import { getBranchKeyItem, validateBranchKeyRecord, constructAuthenticatedEncryptionContext, } from './branch_keystore_helpers';
import { BRANCH_KEY_ACTIVE_TYPE, BRANCH_KEY_TYPE_PREFIX, BRANCH_KEY_FIELD, } from './constants';
import { immutableClass, needs, readOnlyProperty, } from '@aws-crypto/material-management';
//= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#operations
//= type=implication
//# The Dynamodb Key Storage Interface MUST implement the [key storage interface](./key-storage.md#interface).
export class DynamoDBKeyStorage {
    constructor({ ddbTableName, logicalKeyStoreName, ddbClient, }) {
        /* Precondition: DDB table name must be a string */
        needs(typeof ddbTableName === 'string', 'DDB table name must be a string');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#table-name
        //# The table name of the DynamoDb table that backs this Keystore.
        needs(ddbTableName, 'DynamoDb table name required');
        needs(typeof logicalKeyStoreName === 'string', 'Logical Key Store name must be a string');
        needs(logicalKeyStoreName, 'Logical Key Store name required');
        /* Precondition: DDB client must be a DynamoDBClient */
        needs(ddbClient instanceof DynamoDBClient, 'DDB client must be a DynamoDBClient');
        readOnlyProperty(this, 'ddbTableName', ddbTableName);
        readOnlyProperty(this, 'ddbClient', ddbClient);
        readOnlyProperty(this, 'logicalKeyStoreName', logicalKeyStoreName);
        // make this instance immutable
        Object.freeze(this);
    }
    async getEncryptedActiveBranchKey(branchKeyId) {
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedactivebranchkey
        //# To get the active version for the branch key id from the keystore
        //# this operation MUST call AWS DDB `GetItem`
        //# using the `branch-key-id` as the Partition Key and `"branch:ACTIVE"` value as the Sort Key.
        // get the ddb response item using the partition & sort keys
        const ddbBranchKeyItem = await getBranchKeyItem(this, branchKeyId, BRANCH_KEY_ACTIVE_TYPE);
        // validate and form the branch key record
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedactivebranchkey
        //# If the record does not contain the defined fields, this operation MUST fail.
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedactivebranchkey
        //# The AWS DDB response MUST contain the fields defined in the [branch keystore record format](#record-format).
        const ddbBranchKeyRecord = validateBranchKeyRecord(ddbBranchKeyItem);
        // construct an encryption context from the record
        const authenticatedEncryptionContext = constructAuthenticatedEncryptionContext(this, ddbBranchKeyRecord);
        const encrypted = new EncryptedHierarchicalKey(authenticatedEncryptionContext, ddbBranchKeyRecord[BRANCH_KEY_FIELD]);
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedactivebranchkey
        //# The returned EncryptedHierarchicalKey MUST have the same identifier as the input.
        needs(encrypted.branchKeyId == branchKeyId, 'Unexpected branch key id.');
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedactivebranchkey
        //# The returned EncryptedHierarchicalKey MUST have a type of ActiveHierarchicalSymmetricVersion.
        needs(encrypted.type instanceof ActiveHierarchicalSymmetricVersion, 'Unexpected type. Not an active record.');
        return encrypted;
    }
    async getEncryptedBranchKeyVersion(branchKeyId, branchKeyVersion) {
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedbranchkeyversion
        //# To get a branch key from the keystore this operation MUST call AWS DDB `GetItem`
        //# using the `branch-key-id` as the Partition Key and "branch:version:" + `branchKeyVersion` value as the Sort Key.
        // get the ddb response item using the partition & sort keys
        const ddbBranchKeyItem = await getBranchKeyItem(this, branchKeyId, BRANCH_KEY_TYPE_PREFIX + branchKeyVersion);
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedbranchkeyversion
        //# If the record does not contain the defined fields, this operation MUST fail.
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedbranchkeyversion
        //# The AWS DDB response MUST contain the fields defined in the [branch keystore record format](#record-format).
        // validate and form the branch key record
        const ddbBranchKeyRecord = validateBranchKeyRecord(ddbBranchKeyItem);
        // construct an encryption context from the record
        const authenticatedEncryptionContext = constructAuthenticatedEncryptionContext(this, ddbBranchKeyRecord);
        const encrypted = new EncryptedHierarchicalKey(authenticatedEncryptionContext, ddbBranchKeyRecord[BRANCH_KEY_FIELD]);
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedbranchkeyversion
        //# The returned EncryptedHierarchicalKey MUST have the same identifier as the input.
        needs(encrypted.branchKeyId == branchKeyId, 'Unexpected branch key id. Expected ${branchKeyId}, found ${encrypted.branchKeyId}');
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedbranchkeyversion
        //# The returned EncryptedHierarchicalKey MUST have the same version as the input.
        needs(encrypted.type.version == branchKeyVersion, 'Unexpected branch key version.');
        //= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedbranchkeyversion
        //# The returned EncryptedHierarchicalKey MUST have a type of HierarchicalSymmetricVersion.
        needs(encrypted.type instanceof HierarchicalSymmetricVersion, 'Unexpected type. Not an version record.');
        return encrypted;
    }
    getKeyStorageInfo() {
        return {
            name: this.ddbTableName,
            logicalName: this.logicalKeyStoreName,
        };
    }
}
immutableClass(DynamoDBKeyStorage);
// This is a limited release for JS only.
// The full Key Store operations are available
// in the AWS Cryptographic Material Providers library
// in various languages (Java, .Net, Python, Rust...)
//= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#writenewencryptedbranchkey
//= type=exception
//# To add the branch keys and a beacon key to the keystore the
//# operation MUST call [Amazon DynamoDB API TransactWriteItems](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TransactWriteItems.html).
//# The call to Amazon DynamoDB TransactWriteItems MUST use the configured Amazon DynamoDB Client to make the call.
//# The operation MUST call Amazon DynamoDB TransactWriteItems with a request constructed as follows:
//= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#writenewencryptedbranchkey
//= type=exception
//# If DDB TransactWriteItems is successful, this operation MUST return a successful response containing no additional data.
//# Otherwise, this operation MUST yield an error.
//= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#writenewencryptedbranchkeyversion
//= type=exception
//# To add the new branch key to the keystore,
//# the operation MUST call [Amazon DynamoDB API TransactWriteItems](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TransactWriteItems.html).
//# The call to Amazon DynamoDB TransactWriteItems MUST use the configured Amazon DynamoDB Client to make the call.
//# The operation MUST call Amazon DynamoDB TransactWriteItems with a request constructed as follows:
//= aws-encryption-sdk-specification/framework/key-store/dynamodb-key-storage.md#getencryptedbeaconkey
//= type=exception
//# To get a branch key from the keystore this operation MUST call AWS DDB `GetItem`
//# using the `branch-key-id` as the Partition Key and "beacon:ACTIVE" value as the Sort Key.
//# The AWS DDB response MUST contain the fields defined in the [branch keystore record format](#record-format).
//# The returned EncryptedHierarchicalKey MUST have the same identifier as the input.
//# The returned EncryptedHierarchicalKey MUST have a type of ActiveHierarchicalSymmetricBeacon.
//# If the record does not contain the defined fields, this operation MUST fail.
//# sourceMappingURL=data:application/json;base64,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