"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.cacheEntryHasExceededLimits = exports.decryptMaterials = exports.getEncryptionMaterials = exports.decorateProperties = void 0;
const material_management_1 = require("@aws-crypto/material-management");
const serialize_1 = require("@aws-crypto/serialize");
function decorateProperties(obj, input) {
    const { cache, backingMaterialsManager, maxAge, maxBytesEncrypted, maxMessagesEncrypted, partition, } = input;
    /* Precondition: A caching material manager needs a cache. */
    (0, material_management_1.needs)(cache, 'You must provide a cache.');
    /* Precondition: A caching material manager needs a way to get material. */
    (0, material_management_1.needs)(backingMaterialsManager, 'You must provide a backing material source.');
    /* Precondition: You *can not* cache something forever. */
    (0, material_management_1.needs)(maxAge > 0, 'You must configure a maxAge');
    /* Precondition: maxBytesEncrypted must be inside bounds.  i.e. positive and not more than the maximum. */
    (0, material_management_1.needs)(!maxBytesEncrypted ||
        (maxBytesEncrypted > 0 &&
            serialize_1.Maximum.BYTES_PER_CACHED_KEY_LIMIT >= maxBytesEncrypted), 'maxBytesEncrypted is outside of bounds.');
    /* Precondition: maxMessagesEncrypted must be inside bounds.  i.e. positive and not more than the maximum. */
    (0, material_management_1.needs)(!maxMessagesEncrypted ||
        (maxMessagesEncrypted > 0 &&
            serialize_1.Maximum.MESSAGES_PER_CACHED_KEY_LIMIT >= maxMessagesEncrypted), 'maxMessagesEncrypted is outside of bounds.');
    /* Precondition: partition must be a string. */
    (0, material_management_1.needs)(partition && typeof partition === 'string', 'partition must be a string.');
    (0, material_management_1.readOnlyProperty)(obj, '_cache', cache);
    (0, material_management_1.readOnlyProperty)(obj, '_backingMaterialsManager', backingMaterialsManager);
    (0, material_management_1.readOnlyProperty)(obj, '_maxAge', maxAge);
    (0, material_management_1.readOnlyProperty)(obj, '_maxBytesEncrypted', maxBytesEncrypted || serialize_1.Maximum.BYTES_PER_CACHED_KEY_LIMIT);
    (0, material_management_1.readOnlyProperty)(obj, '_maxMessagesEncrypted', maxMessagesEncrypted || serialize_1.Maximum.MESSAGES_PER_CACHED_KEY_LIMIT);
    (0, material_management_1.readOnlyProperty)(obj, '_partition', partition);
}
exports.decorateProperties = decorateProperties;
function getEncryptionMaterials({ buildEncryptionMaterialCacheKey, }) {
    return async function getEncryptionMaterials(request) {
        const { suite, encryptionContext, plaintextLength, commitmentPolicy } = request;
        /* Check for early return (Postcondition): If I can not cache the EncryptionMaterial, do not even look. */
        if ((suite && !suite.cacheSafe) ||
            typeof plaintextLength !== 'number' ||
            plaintextLength < 0) {
            const material = await this._backingMaterialsManager.getEncryptionMaterials(request);
            return material;
        }
        const cacheKey = await buildEncryptionMaterialCacheKey(this._partition, {
            suite,
            encryptionContext,
        });
        const entry = this._cache.getEncryptionMaterial(cacheKey, plaintextLength);
        /* Check for early return (Postcondition): If I have a valid EncryptionMaterial, return it. */
        if (entry && !this._cacheEntryHasExceededLimits(entry)) {
            return cloneResponse(entry.response);
        }
        else {
            this._cache.del(cacheKey);
        }
        const material = await this._backingMaterialsManager
            /* Strip any information about the plaintext from the backing request,
             * because the resulting response may be used to encrypt multiple plaintexts.
             */
            .getEncryptionMaterials({ suite, encryptionContext, commitmentPolicy });
        /* Check for early return (Postcondition): If I can not cache the EncryptionMaterial, just return it. */
        if (!material.suite.cacheSafe)
            return material;
        /* It is possible for an entry to exceed limits immediately.
         * The simplest case is to need to encrypt more than then maxBytesEncrypted.
         * In this case, I return the response to encrypt the data,
         * but do not put a know invalid item into the cache.
         */
        const testEntry = {
            response: material,
            now: Date.now(),
            messagesEncrypted: 1,
            bytesEncrypted: plaintextLength,
        };
        if (!this._cacheEntryHasExceededLimits(testEntry)) {
            this._cache.putEncryptionMaterial(cacheKey, material, plaintextLength, this._maxAge);
            return cloneResponse(material);
        }
        else {
            /* Postcondition: If the material has exceeded limits it MUST NOT be cloned.
             * If it is cloned, and the clone is returned,
             * then there exist a copy of the unencrypted data key.
             * It is true that this data would be caught by GC, it is better to just not rely on that.
             */
            return material;
        }
    };
}
exports.getEncryptionMaterials = getEncryptionMaterials;
function decryptMaterials({ buildDecryptionMaterialCacheKey, }) {
    return async function decryptMaterials(request) {
        const { suite } = request;
        /* Check for early return (Postcondition): If I can not cache the DecryptionMaterial, do not even look. */
        if (!suite.cacheSafe) {
            const material = await this._backingMaterialsManager.decryptMaterials(request);
            return material;
        }
        const cacheKey = await buildDecryptionMaterialCacheKey(this._partition, request);
        const entry = this._cache.getDecryptionMaterial(cacheKey);
        /* Check for early return (Postcondition): If I have a valid DecryptionMaterial, return it. */
        if (entry && !this._cacheEntryHasExceededLimits(entry)) {
            return cloneResponse(entry.response);
        }
        else {
            this._cache.del(cacheKey);
        }
        const material = await this._backingMaterialsManager.decryptMaterials(request);
        this._cache.putDecryptionMaterial(cacheKey, material, this._maxAge);
        return cloneResponse(material);
    };
}
exports.decryptMaterials = decryptMaterials;
function cacheEntryHasExceededLimits() {
    return function cacheEntryHasExceededLimits({ now, messagesEncrypted, bytesEncrypted }) {
        const age = Date.now() - now;
        return (age > this._maxAge ||
            messagesEncrypted > this._maxMessagesEncrypted ||
            bytesEncrypted > this._maxBytesEncrypted);
    };
}
exports.cacheEntryHasExceededLimits = cacheEntryHasExceededLimits;
/**
 * I need to clone the underlying material.
 * Because when the Encryption SDK is done with material, it will zero it out.
 * Plucking off the material and cloning just that and then returning the rest of the response
 * can just be handled in one place.
 * @param material EncryptionMaterial|DecryptionMaterial
 * @return EncryptionMaterial|DecryptionMaterial
 */
function cloneResponse(material) {
    return (0, material_management_1.cloneMaterial)(material);
}
//# sourceMappingURL=data:application/json;base64,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