"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.VerificationKey = exports.SignatureKey = void 0;
const ecc_encode_1 = require("./ecc_encode");
const ecc_decode_1 = require("./ecc_decode");
const immutable_class_1 = require("./immutable_class");
const pem_helpers_1 = require("./pem_helpers");
/*
 * This public interface to the SignatureKey object is provided for
 * developers of CMMs and keyrings only. If you are a user of the AWS Encryption
 * SDK and you are not developing your own CMMs and/or keyrings, you do not
 * need to use it and you should not do so.
 */
class SignatureKey {
    constructor(privateKey, compressPoint, suite) {
        const { signatureCurve: namedCurve } = suite;
        /* Precondition: Do not create a SignatureKey for an algorithm suite that does not have an EC named curve. */
        if (!namedCurve)
            throw new Error('Unsupported Algorithm');
        /* This is unfortunately complicated.  Node v11 crypto will accept
         * a PEM formated Buffer to sign.  But the ECDH class will still
         * return Buffers that are not PEM formated, but _only_ the points
         * on the curve.  This means I have to make a choice about
         * formating.  I chose to assume that t Buffer/Uin8Array is
         * _only_ the raw points.
         */
        if (privateKey instanceof Uint8Array) {
            const pem = (0, pem_helpers_1.privateKeyPem)(namedCurve, fromBuffer(privateKey), fromBuffer(compressPoint));
            (0, immutable_class_1.readOnlyProperty)(this, 'privateKey', pem);
        }
        else {
            (0, immutable_class_1.readOnlyProperty)(this, 'privateKey', privateKey);
        }
        (0, immutable_class_1.readOnlyBinaryProperty)(this, 'compressPoint', compressPoint);
        (0, immutable_class_1.readOnlyProperty)(this, 'signatureCurve', namedCurve);
        Object.setPrototypeOf(this, SignatureKey.prototype);
        Object.freeze(this);
    }
    static encodeCompressPoint(publicKeyBytes, suite) {
        const { signatureCurve: namedCurve } = suite;
        /* Precondition: Do not return a compress point for an algorithm suite that does not have an EC named curve. */
        if (!namedCurve)
            throw new Error('Unsupported Algorithm');
        return ecc_encode_1.encodeNamedCurves[namedCurve](publicKeyBytes);
    }
}
exports.SignatureKey = SignatureKey;
(0, immutable_class_1.frozenClass)(SignatureKey);
class VerificationKey {
    publicKey;
    signatureCurve;
    constructor(publicKey, suite) {
        const { signatureCurve: namedCurve } = suite;
        /* Precondition: Do not create a VerificationKey for an algorithm suite that does not have an EC named curve. */
        if (!namedCurve)
            throw new Error('Unsupported Algorithm');
        /* This is unfortunately complicated.  Node v11 crypto will accept
         * a PEM formated Buffer to verify.  But the ECDH class will still
         * return Buffers that are not PEM formated, but _only_ the points
         * on the curve.  This means I have to make a choice about
         * formating.  I chose to assume that the Buffer/Uin8Array is
         * _only_ the raw points.
         */
        if (publicKey instanceof Uint8Array) {
            const pem = (0, pem_helpers_1.publicKeyPem)(namedCurve, fromBuffer(publicKey));
            (0, immutable_class_1.readOnlyProperty)(this, 'publicKey', pem);
        }
        else {
            (0, immutable_class_1.readOnlyProperty)(this, 'publicKey', publicKey);
        }
        (0, immutable_class_1.readOnlyProperty)(this, 'signatureCurve', namedCurve);
        Object.setPrototypeOf(this, VerificationKey.prototype);
        Object.freeze(this);
    }
    static decodeCompressPoint(compressPoint, suite) {
        const { signatureCurve: namedCurve } = suite;
        /* Precondition: Do not decode a public key for an algorithm suite that does not have an EC named curve. */
        if (!namedCurve)
            throw new Error('Unsupported Algorithm');
        return ecc_decode_1.decodeNamedCurves[namedCurve](compressPoint);
    }
}
exports.VerificationKey = VerificationKey;
(0, immutable_class_1.frozenClass)(VerificationKey);
function fromBuffer(uint) {
    const { buffer, byteOffset, byteLength } = uint;
    return Buffer.from(buffer, byteOffset, byteLength);
}
//# sourceMappingURL=data:application/json;base64,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