"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.previewRoute = previewRoute;
var _configSchema = require("@osd/config-schema");
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../common/constants");
var _util = require("../utils/util");
var _determine_mapping = require("../utils/determine_mapping");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function previewRoute(router, config, fileProcessors, dataSourceEnabled) {
  router.post({
    path: '/api/data_importer/_preview',
    options: {
      body: {
        maxBytes: config.maxFileSizeBytes,
        accepts: 'multipart/form-data',
        output: 'stream'
      }
    },
    validate: {
      query: _configSchema.schema.object({
        indexName: _configSchema.schema.string({
          minLength: 1
        }),
        createMode: _configSchema.schema.boolean(),
        fileExtension: _configSchema.schema.string({
          minLength: 1
        }),
        dataSource: _configSchema.schema.maybe(_configSchema.schema.string()),
        delimiter: _configSchema.schema.maybe(_configSchema.schema.string({
          validate(value) {
            if (!_constants.CSV_SUPPORTED_DELIMITERS.includes(value)) {
              return `must be a supported delimiter`;
            }
          }
        })),
        previewCount: _configSchema.schema.number({
          min: 1,
          max: config.filePreviewDocumentsCount
        })
      }),
      body: _configSchema.schema.object({
        file: _configSchema.schema.stream()
      })
    }
  }, async (context, request, response) => {
    const fileExtension = request.query.fileExtension.startsWith('.') ? request.query.fileExtension.slice(1) : request.query.fileExtension;
    const processor = fileProcessors.getFileProcessor(fileExtension);
    if (!processor) {
      return response.badRequest({
        body: `${fileExtension} is not a registered or supported filetype`
      });
    }
    const client = await (0, _util.decideClient)(dataSourceEnabled, context, request.query.dataSource);
    if (!!!client) {
      return response.notFound({
        body: 'Data source is not enabled or does not exist'
      });
    }
    if (!request.query.createMode) {
      try {
        const indexExists = await client.indices.exists({
          index: request.query.indexName
        });
        if (!indexExists.body) {
          return response.notFound({
            body: `Index ${request.query.indexName} does not exist`
          });
        }
      } catch (e) {
        return response.internalError({
          body: `Error checking if index exists: ${e}`
        });
      }
    }
    const file = request.body.file;
    const documents = [];
    try {
      documents.push(...(await processor.parseFile(file, request.query.previewCount, {
        delimiter: request.query.delimiter
      })).slice(0, request.query.previewCount));
    } catch (e) {
      return response.badRequest({
        body: `Error parsing file: ${e}`
      });
    }
    try {
      // Ensure OpenSearch can handle deep objects
      const nestedObjectsLimit = await (0, _util.fetchDepthLimit)(client);
      // Some documents may omit fields so we must merge into one large document
      const predictedMapping = (0, _determine_mapping.determineMapping)(_lodash.default.mergeWith(documents, _util.mergeCustomizerForPreview), nestedObjectsLimit);
      const existingMapping = !request.query.createMode ? (await client.indices.getMapping({
        index: request.query.indexName
      })).body[request.query.indexName].mappings : undefined;
      return response.ok({
        body: {
          predictedMapping,
          documents,
          ...(existingMapping && {
            existingMapping
          })
        }
      });
    } catch (e) {
      return response.internalError({
        body: `Error determining mapping: ${e}`
      });
    }
  });
}