use @pony_os_sockopt_level[I32](option: I32)
use @pony_os_sockopt_option[I32](option: I32)

primitive OSSockOpt
  """
  Convenience functions to fetch the option level and option
  name constants (arguments #2 and #3) for the
  `getsockopt(2)` and `setsockopt(2)` operating system calls.

  The values of the option level and option name constants are
  typically C preprocessor macros, e.g., `#define SOMETHING 42`.
  These macro names are upper case and may contain multiple
  consecutive underscore characters (though this is rare, for
  example, `IP_NAT__XXX`).  The function names in this primitive
  are derived by the C macro name and then:

    * converted to lower case
    * any double underscore (`__`) is converted to a
      single underscore (`_`).

  These constants are _not_ stable between Pony releases.
  Values returned by this function may be held by long-lived variables
  by the calling process: values cannot change while the process runs.
  Programmers must not cache any of these values for purposes of
  sharing them for use by any other Pony program (for example,
  sharing via serialization & deserialization or via direct
  shared memory).

  Many functions may return `-1`, which means that the constant's
  value could not be determined at the Pony runtime library compile
  time.  One cause may be that the option truly isn't available,
  for example, the option level constant `IPPROTO_3PC` is available
  on MacOS 10.x but not on Linux 4.4.  Another cause may be the
  Pony runtime library's compilation did not include the correct
  header file(s) for the target OS platform.

  A third cause of error is due to the regular expression-based
  approach used to harvest desirable constants.  It is not fool-proof.
  The regexp used is too broad and finds some macros that are not
  supposed to be used with `getsockopt(2)` and `setsockopt(2)`.
  Please consult your platform's documentation to verify the names
  of the option level and option name macros.

  The following code fragments are equivalent: set the socket
  receive buffer size for the file descriptor `fd` to `4455`.

  ```c
  /* In C */
  int option_value = 4455;
  setsockopt(fd, SOL_SOCKET, SO_RCVBUF, &option_value, 4);

  /* In Pony */
  var option: I32 = 4455;
  @setsockopt[I32](fd, OSSockOpt.sol_socket(), OSSockOpt.so_rcvbuf(),
    addressof option, I32(4))
  ```
  """

  /* Constants are from
   *   macOS Sierra 10.12.6
   *   Ubuntu Linux Xenial/16.04 LTS + kernel 4.4.0-109-generic
   *   FreeBSD 11.1-RELEASE
   *   Windows Winsock function reference for getsockopt & setsockopt:
   *     https://msdn.microsoft.com/en-us/library/windows/desktop/ms738544(v=vs.85).aspx
   *     https://msdn.microsoft.com/en-us/library/windows/desktop/ms740476(v=vs.85).aspx

   * Harvested by recipe given in socket.c
   */

  /*
   * Levels formatted in Pony by:
   *   egrep '^(IP[A-Z0-6]*PROTO_|NSPROTO_|SOL_)' ~/sum-of-all-constants.txt | egrep -v '\(' | sort -u | egrep -v '^$' | sed 's/__/_/g' | awk 'BEGIN { count=4000; } { printf("  fun %s():I32 => @pony_os_sockopt_level(I32(%d))\n", tolower($1), count++); }'
   */

  // levels
  fun ipproto_3pc():I32 => @pony_os_sockopt_level(I32(4000))
  fun ipproto_adfs():I32 => @pony_os_sockopt_level(I32(4001))
  fun ipproto_ah():I32 => @pony_os_sockopt_level(I32(4002))
  fun ipproto_ahip():I32 => @pony_os_sockopt_level(I32(4003))
  fun ipproto_apes():I32 => @pony_os_sockopt_level(I32(4004))
  fun ipproto_argus():I32 => @pony_os_sockopt_level(I32(4005))
  fun ipproto_ax25():I32 => @pony_os_sockopt_level(I32(4006))
  fun ipproto_beetph():I32 => @pony_os_sockopt_level(I32(4007))
  fun ipproto_bha():I32 => @pony_os_sockopt_level(I32(4008))
  fun ipproto_blt():I32 => @pony_os_sockopt_level(I32(4009))
  fun ipproto_brsatmon():I32 => @pony_os_sockopt_level(I32(4010))
  fun ipproto_carp():I32 => @pony_os_sockopt_level(I32(4011))
  fun ipproto_cftp():I32 => @pony_os_sockopt_level(I32(4012))
  fun ipproto_chaos():I32 => @pony_os_sockopt_level(I32(4013))
  fun ipproto_cmtp():I32 => @pony_os_sockopt_level(I32(4014))
  fun ipproto_comp():I32 => @pony_os_sockopt_level(I32(4015))
  fun ipproto_cphb():I32 => @pony_os_sockopt_level(I32(4016))
  fun ipproto_cpnx():I32 => @pony_os_sockopt_level(I32(4017))
  fun ipproto_dccp():I32 => @pony_os_sockopt_level(I32(4018))
  fun ipproto_ddp():I32 => @pony_os_sockopt_level(I32(4019))
  fun ipproto_dgp():I32 => @pony_os_sockopt_level(I32(4020))
  fun ipproto_divert():I32 => @pony_os_sockopt_level(I32(4021))
  fun ipproto_done():I32 => @pony_os_sockopt_level(I32(4022))
  fun ipproto_dstopts():I32 => @pony_os_sockopt_level(I32(4023))
  fun ipproto_egp():I32 => @pony_os_sockopt_level(I32(4024))
  fun ipproto_emcon():I32 => @pony_os_sockopt_level(I32(4025))
  fun ipproto_encap():I32 => @pony_os_sockopt_level(I32(4026))
  fun ipproto_eon():I32 => @pony_os_sockopt_level(I32(4027))
  fun ipproto_esp():I32 => @pony_os_sockopt_level(I32(4028))
  fun ipproto_etherip():I32 => @pony_os_sockopt_level(I32(4029))
  fun ipproto_fragment():I32 => @pony_os_sockopt_level(I32(4030))
  fun ipproto_ggp():I32 => @pony_os_sockopt_level(I32(4031))
  fun ipproto_gmtp():I32 => @pony_os_sockopt_level(I32(4032))
  fun ipproto_gre():I32 => @pony_os_sockopt_level(I32(4033))
  fun ipproto_hello():I32 => @pony_os_sockopt_level(I32(4034))
  fun ipproto_hip():I32 => @pony_os_sockopt_level(I32(4035))
  fun ipproto_hmp():I32 => @pony_os_sockopt_level(I32(4036))
  fun ipproto_hopopts():I32 => @pony_os_sockopt_level(I32(4037))
  fun ipproto_icmp():I32 => @pony_os_sockopt_level(I32(4038))
  fun ipproto_icmpv6():I32 => @pony_os_sockopt_level(I32(4039))
  fun ipproto_idp():I32 => @pony_os_sockopt_level(I32(4040))
  fun ipproto_idpr():I32 => @pony_os_sockopt_level(I32(4041))
  fun ipproto_idrp():I32 => @pony_os_sockopt_level(I32(4042))
  fun ipproto_igmp():I32 => @pony_os_sockopt_level(I32(4043))
  fun ipproto_igp():I32 => @pony_os_sockopt_level(I32(4044))
  fun ipproto_igrp():I32 => @pony_os_sockopt_level(I32(4045))
  fun ipproto_il():I32 => @pony_os_sockopt_level(I32(4046))
  fun ipproto_inlsp():I32 => @pony_os_sockopt_level(I32(4047))
  fun ipproto_inp():I32 => @pony_os_sockopt_level(I32(4048))
  fun ipproto_ip():I32 => @pony_os_sockopt_level(I32(4049))
  fun ipproto_ipcomp():I32 => @pony_os_sockopt_level(I32(4050))
  fun ipproto_ipcv():I32 => @pony_os_sockopt_level(I32(4051))
  fun ipproto_ipeip():I32 => @pony_os_sockopt_level(I32(4052))
  fun ipproto_ipip():I32 => @pony_os_sockopt_level(I32(4053))
  fun ipproto_ippc():I32 => @pony_os_sockopt_level(I32(4054))
  fun ipproto_ipv4():I32 => @pony_os_sockopt_level(I32(4055))
  fun ipproto_ipv6():I32 => @pony_os_sockopt_level(I32(4056))
  fun ipproto_irtp():I32 => @pony_os_sockopt_level(I32(4057))
  fun ipproto_kryptolan():I32 => @pony_os_sockopt_level(I32(4058))
  fun ipproto_larp():I32 => @pony_os_sockopt_level(I32(4059))
  fun ipproto_leaf1():I32 => @pony_os_sockopt_level(I32(4060))
  fun ipproto_leaf2():I32 => @pony_os_sockopt_level(I32(4061))
  fun ipproto_max():I32 => @pony_os_sockopt_level(I32(4062))
  fun ipproto_maxid():I32 => @pony_os_sockopt_level(I32(4063))
  fun ipproto_meas():I32 => @pony_os_sockopt_level(I32(4064))
  fun ipproto_mh():I32 => @pony_os_sockopt_level(I32(4065))
  fun ipproto_mhrp():I32 => @pony_os_sockopt_level(I32(4066))
  fun ipproto_micp():I32 => @pony_os_sockopt_level(I32(4067))
  fun ipproto_mobile():I32 => @pony_os_sockopt_level(I32(4068))
  fun ipproto_mpls():I32 => @pony_os_sockopt_level(I32(4069))
  fun ipproto_mtp():I32 => @pony_os_sockopt_level(I32(4070))
  fun ipproto_mux():I32 => @pony_os_sockopt_level(I32(4071))
  fun ipproto_nd():I32 => @pony_os_sockopt_level(I32(4072))
  fun ipproto_nhrp():I32 => @pony_os_sockopt_level(I32(4073))
  fun ipproto_none():I32 => @pony_os_sockopt_level(I32(4074))
  fun ipproto_nsp():I32 => @pony_os_sockopt_level(I32(4075))
  fun ipproto_nvpii():I32 => @pony_os_sockopt_level(I32(4076))
  fun ipproto_old_divert():I32 => @pony_os_sockopt_level(I32(4077))
  fun ipproto_ospfigp():I32 => @pony_os_sockopt_level(I32(4078))
  fun ipproto_pfsync():I32 => @pony_os_sockopt_level(I32(4079))
  fun ipproto_pgm():I32 => @pony_os_sockopt_level(I32(4080))
  fun ipproto_pigp():I32 => @pony_os_sockopt_level(I32(4081))
  fun ipproto_pim():I32 => @pony_os_sockopt_level(I32(4082))
  fun ipproto_prm():I32 => @pony_os_sockopt_level(I32(4083))
  fun ipproto_pup():I32 => @pony_os_sockopt_level(I32(4084))
  fun ipproto_pvp():I32 => @pony_os_sockopt_level(I32(4085))
  fun ipproto_raw():I32 => @pony_os_sockopt_level(I32(4086))
  fun ipproto_rccmon():I32 => @pony_os_sockopt_level(I32(4087))
  fun ipproto_rdp():I32 => @pony_os_sockopt_level(I32(4088))
  fun ipproto_reserved_253():I32 => @pony_os_sockopt_level(I32(4089))
  fun ipproto_reserved_254():I32 => @pony_os_sockopt_level(I32(4090))
  fun ipproto_routing():I32 => @pony_os_sockopt_level(I32(4091))
  fun ipproto_rsvp():I32 => @pony_os_sockopt_level(I32(4092))
  fun ipproto_rvd():I32 => @pony_os_sockopt_level(I32(4093))
  fun ipproto_satexpak():I32 => @pony_os_sockopt_level(I32(4094))
  fun ipproto_satmon():I32 => @pony_os_sockopt_level(I32(4095))
  fun ipproto_sccsp():I32 => @pony_os_sockopt_level(I32(4096))
  fun ipproto_sctp():I32 => @pony_os_sockopt_level(I32(4097))
  fun ipproto_sdrp():I32 => @pony_os_sockopt_level(I32(4098))
  fun ipproto_send():I32 => @pony_os_sockopt_level(I32(4099))
  fun ipproto_sep():I32 => @pony_os_sockopt_level(I32(4100))
  fun ipproto_shim6():I32 => @pony_os_sockopt_level(I32(4101))
  fun ipproto_skip():I32 => @pony_os_sockopt_level(I32(4102))
  fun ipproto_spacer():I32 => @pony_os_sockopt_level(I32(4103))
  fun ipproto_srpc():I32 => @pony_os_sockopt_level(I32(4104))
  fun ipproto_st():I32 => @pony_os_sockopt_level(I32(4105))
  fun ipproto_svmtp():I32 => @pony_os_sockopt_level(I32(4106))
  fun ipproto_swipe():I32 => @pony_os_sockopt_level(I32(4107))
  fun ipproto_tcf():I32 => @pony_os_sockopt_level(I32(4108))
  fun ipproto_tcp():I32 => @pony_os_sockopt_level(I32(4109))
  fun ipproto_tlsp():I32 => @pony_os_sockopt_level(I32(4110))
  fun ipproto_tp():I32 => @pony_os_sockopt_level(I32(4111))
  fun ipproto_tpxx():I32 => @pony_os_sockopt_level(I32(4112))
  fun ipproto_trunk1():I32 => @pony_os_sockopt_level(I32(4113))
  fun ipproto_trunk2():I32 => @pony_os_sockopt_level(I32(4114))
  fun ipproto_ttp():I32 => @pony_os_sockopt_level(I32(4115))
  fun ipproto_udp():I32 => @pony_os_sockopt_level(I32(4116))
  fun ipproto_udplite():I32 => @pony_os_sockopt_level(I32(4117))
  fun ipproto_vines():I32 => @pony_os_sockopt_level(I32(4118))
  fun ipproto_visa():I32 => @pony_os_sockopt_level(I32(4119))
  fun ipproto_vmtp():I32 => @pony_os_sockopt_level(I32(4120))
  fun ipproto_wbexpak():I32 => @pony_os_sockopt_level(I32(4121))
  fun ipproto_wbmon():I32 => @pony_os_sockopt_level(I32(4122))
  fun ipproto_wsn():I32 => @pony_os_sockopt_level(I32(4123))
  fun ipproto_xnet():I32 => @pony_os_sockopt_level(I32(4124))
  fun ipproto_xtp():I32 => @pony_os_sockopt_level(I32(4125))
  fun sol_atalk():I32 => @pony_os_sockopt_level(I32(4126))
  fun sol_ax25():I32 => @pony_os_sockopt_level(I32(4127))
  fun sol_hci_raw():I32 => @pony_os_sockopt_level(I32(4128))
  fun sol_ipx():I32 => @pony_os_sockopt_level(I32(4129))
  fun sol_l2cap():I32 => @pony_os_sockopt_level(I32(4130))
  fun sol_local():I32 => @pony_os_sockopt_level(I32(4131))
  fun sol_ndrvproto():I32 => @pony_os_sockopt_level(I32(4132))
  fun sol_netrom():I32 => @pony_os_sockopt_level(I32(4133))
  fun sol_rds():I32 => @pony_os_sockopt_level(I32(4134))
  fun sol_rfcomm():I32 => @pony_os_sockopt_level(I32(4135))
  fun sol_rose():I32 => @pony_os_sockopt_level(I32(4136))
  fun sol_sco():I32 => @pony_os_sockopt_level(I32(4137))
  fun sol_socket():I32 => @pony_os_sockopt_level(I32(4138))
  fun sol_tipc():I32 => @pony_os_sockopt_level(I32(4139))
  fun sol_udp():I32 => @pony_os_sockopt_level(I32(4140))

  /*
   *
   * Options formatted in Pony by:
   *   egrep -v '^(IP[A-Z0-6]*PROTO_|NSPROTO_|SOL_)' ~/sum-of-all-constants.txt | egrep -v '\(' | sort -u | egrep -v '^$' | sed 's/__/_/g' | awk 'BEGIN { count=0; } { printf("  fun %s():I32 => @pony_os_sockopt_option(I32(%d))\n", tolower($1), count++); }'
   */

  // options
  fun af_coip():I32 => @pony_os_sockopt_option(I32(0))
  fun af_inet():I32 => @pony_os_sockopt_option(I32(1))
  fun af_inet6():I32 => @pony_os_sockopt_option(I32(2))
  fun bluetooth_proto_sco():I32 => @pony_os_sockopt_option(I32(3))
  fun dccp_nr_pkt_types():I32 => @pony_os_sockopt_option(I32(4))
  fun dccp_service_list_max_len():I32 => @pony_os_sockopt_option(I32(5))
  fun dccp_single_opt_maxlen():I32 => @pony_os_sockopt_option(I32(6))
  fun dccp_sockopt_available_ccids():I32 => @pony_os_sockopt_option(I32(7))
  fun dccp_sockopt_ccid():I32 => @pony_os_sockopt_option(I32(8))
  fun dccp_sockopt_ccid_rx_info():I32 => @pony_os_sockopt_option(I32(9))
  fun dccp_sockopt_ccid_tx_info():I32 => @pony_os_sockopt_option(I32(10))
  fun dccp_sockopt_change_l():I32 => @pony_os_sockopt_option(I32(11))
  fun dccp_sockopt_change_r():I32 => @pony_os_sockopt_option(I32(12))
  fun dccp_sockopt_get_cur_mps():I32 => @pony_os_sockopt_option(I32(13))
  fun dccp_sockopt_packet_size():I32 => @pony_os_sockopt_option(I32(14))
  fun dccp_sockopt_qpolicy_id():I32 => @pony_os_sockopt_option(I32(15))
  fun dccp_sockopt_qpolicy_txqlen():I32 => @pony_os_sockopt_option(I32(16))
  fun dccp_sockopt_recv_cscov():I32 => @pony_os_sockopt_option(I32(17))
  fun dccp_sockopt_rx_ccid():I32 => @pony_os_sockopt_option(I32(18))
  fun dccp_sockopt_send_cscov():I32 => @pony_os_sockopt_option(I32(19))
  fun dccp_sockopt_server_timewait():I32 => @pony_os_sockopt_option(I32(20))
  fun dccp_sockopt_service():I32 => @pony_os_sockopt_option(I32(21))
  fun dccp_sockopt_tx_ccid():I32 => @pony_os_sockopt_option(I32(22))
  fun dso_acceptmode():I32 => @pony_os_sockopt_option(I32(23))
  fun dso_conaccept():I32 => @pony_os_sockopt_option(I32(24))
  fun dso_conaccess():I32 => @pony_os_sockopt_option(I32(25))
  fun dso_condata():I32 => @pony_os_sockopt_option(I32(26))
  fun dso_conreject():I32 => @pony_os_sockopt_option(I32(27))
  fun dso_cork():I32 => @pony_os_sockopt_option(I32(28))
  fun dso_disdata():I32 => @pony_os_sockopt_option(I32(29))
  fun dso_info():I32 => @pony_os_sockopt_option(I32(30))
  fun dso_linkinfo():I32 => @pony_os_sockopt_option(I32(31))
  fun dso_max():I32 => @pony_os_sockopt_option(I32(32))
  fun dso_maxwindow():I32 => @pony_os_sockopt_option(I32(33))
  fun dso_nodelay():I32 => @pony_os_sockopt_option(I32(34))
  fun dso_seqpacket():I32 => @pony_os_sockopt_option(I32(35))
  fun dso_services():I32 => @pony_os_sockopt_option(I32(36))
  fun dso_stream():I32 => @pony_os_sockopt_option(I32(37))
  fun icmp_address():I32 => @pony_os_sockopt_option(I32(38))
  fun icmp_addressreply():I32 => @pony_os_sockopt_option(I32(39))
  fun icmp_dest_unreach():I32 => @pony_os_sockopt_option(I32(40))
  fun icmp_echo():I32 => @pony_os_sockopt_option(I32(41))
  fun icmp_echoreply():I32 => @pony_os_sockopt_option(I32(42))
  fun icmp_exc_fragtime():I32 => @pony_os_sockopt_option(I32(43))
  fun icmp_exc_ttl():I32 => @pony_os_sockopt_option(I32(44))
  fun icmp_filter():I32 => @pony_os_sockopt_option(I32(45))
  fun icmp_frag_needed():I32 => @pony_os_sockopt_option(I32(46))
  fun icmp_host_ano():I32 => @pony_os_sockopt_option(I32(47))
  fun icmp_host_isolated():I32 => @pony_os_sockopt_option(I32(48))
  fun icmp_host_unknown():I32 => @pony_os_sockopt_option(I32(49))
  fun icmp_host_unreach():I32 => @pony_os_sockopt_option(I32(50))
  fun icmp_host_unr_tos():I32 => @pony_os_sockopt_option(I32(51))
  fun icmp_info_reply():I32 => @pony_os_sockopt_option(I32(52))
  fun icmp_info_request():I32 => @pony_os_sockopt_option(I32(53))
  fun icmp_net_ano():I32 => @pony_os_sockopt_option(I32(54))
  fun icmp_net_unknown():I32 => @pony_os_sockopt_option(I32(55))
  fun icmp_net_unreach():I32 => @pony_os_sockopt_option(I32(56))
  fun icmp_net_unr_tos():I32 => @pony_os_sockopt_option(I32(57))
  fun icmp_parameterprob():I32 => @pony_os_sockopt_option(I32(58))
  fun icmp_pkt_filtered():I32 => @pony_os_sockopt_option(I32(59))
  fun icmp_port_unreach():I32 => @pony_os_sockopt_option(I32(60))
  fun icmp_prec_cutoff():I32 => @pony_os_sockopt_option(I32(61))
  fun icmp_prec_violation():I32 => @pony_os_sockopt_option(I32(62))
  fun icmp_prot_unreach():I32 => @pony_os_sockopt_option(I32(63))
  fun icmp_redirect():I32 => @pony_os_sockopt_option(I32(64))
  fun icmp_redir_host():I32 => @pony_os_sockopt_option(I32(65))
  fun icmp_redir_hosttos():I32 => @pony_os_sockopt_option(I32(66))
  fun icmp_redir_net():I32 => @pony_os_sockopt_option(I32(67))
  fun icmp_redir_nettos():I32 => @pony_os_sockopt_option(I32(68))
  fun icmp_source_quench():I32 => @pony_os_sockopt_option(I32(69))
  fun icmp_sr_failed():I32 => @pony_os_sockopt_option(I32(70))
  fun icmp_timestamp():I32 => @pony_os_sockopt_option(I32(71))
  fun icmp_timestampreply():I32 => @pony_os_sockopt_option(I32(72))
  fun icmp_time_exceeded():I32 => @pony_os_sockopt_option(I32(73))
  fun ipctl_acceptsourceroute():I32 => @pony_os_sockopt_option(I32(74))
  fun ipctl_defmtu():I32 => @pony_os_sockopt_option(I32(75))
  fun ipctl_defttl():I32 => @pony_os_sockopt_option(I32(76))
  fun ipctl_directedbroadcast():I32 => @pony_os_sockopt_option(I32(77))
  fun ipctl_fastforwarding():I32 => @pony_os_sockopt_option(I32(78))
  fun ipctl_forwarding():I32 => @pony_os_sockopt_option(I32(79))
  fun ipctl_gif_ttl():I32 => @pony_os_sockopt_option(I32(80))
  fun ipctl_intrdqdrops():I32 => @pony_os_sockopt_option(I32(81))
  fun ipctl_intrdqmaxlen():I32 => @pony_os_sockopt_option(I32(82))
  fun ipctl_intrqdrops():I32 => @pony_os_sockopt_option(I32(83))
  fun ipctl_intrqmaxlen():I32 => @pony_os_sockopt_option(I32(84))
  fun ipctl_keepfaith():I32 => @pony_os_sockopt_option(I32(85))
  fun ipctl_maxid():I32 => @pony_os_sockopt_option(I32(86))
  fun ipctl_rtexpire():I32 => @pony_os_sockopt_option(I32(87))
  fun ipctl_rtmaxcache():I32 => @pony_os_sockopt_option(I32(88))
  fun ipctl_rtminexpire():I32 => @pony_os_sockopt_option(I32(89))
  fun ipctl_sendredirects():I32 => @pony_os_sockopt_option(I32(90))
  fun ipctl_sourceroute():I32 => @pony_os_sockopt_option(I32(91))
  fun ipctl_stats():I32 => @pony_os_sockopt_option(I32(92))
  fun ipport_ephemeralfirst():I32 => @pony_os_sockopt_option(I32(93))
  fun ipport_ephemerallast():I32 => @pony_os_sockopt_option(I32(94))
  fun ipport_hifirstauto():I32 => @pony_os_sockopt_option(I32(95))
  fun ipport_hilastauto():I32 => @pony_os_sockopt_option(I32(96))
  fun ipport_max():I32 => @pony_os_sockopt_option(I32(97))
  fun ipport_reserved():I32 => @pony_os_sockopt_option(I32(98))
  fun ipport_reservedstart():I32 => @pony_os_sockopt_option(I32(99))
  fun ipport_userreserved():I32 => @pony_os_sockopt_option(I32(100))
  fun ipv6_2292dstopts():I32 => @pony_os_sockopt_option(I32(101))
  fun ipv6_2292hoplimit():I32 => @pony_os_sockopt_option(I32(102))
  fun ipv6_2292hopopts():I32 => @pony_os_sockopt_option(I32(103))
  fun ipv6_2292pktinfo():I32 => @pony_os_sockopt_option(I32(104))
  fun ipv6_2292pktoptions():I32 => @pony_os_sockopt_option(I32(105))
  fun ipv6_2292rthdr():I32 => @pony_os_sockopt_option(I32(106))
  fun ipv6_addrform():I32 => @pony_os_sockopt_option(I32(107))
  fun ipv6_addr_preferences():I32 => @pony_os_sockopt_option(I32(108))
  fun ipv6_add_membership():I32 => @pony_os_sockopt_option(I32(109))
  fun ipv6_authhdr():I32 => @pony_os_sockopt_option(I32(110))
  fun ipv6_autoflowlabel():I32 => @pony_os_sockopt_option(I32(111))
  fun ipv6_checksum():I32 => @pony_os_sockopt_option(I32(112))
  fun ipv6_dontfrag():I32 => @pony_os_sockopt_option(I32(113))
  fun ipv6_drop_membership():I32 => @pony_os_sockopt_option(I32(114))
  fun ipv6_dstopts():I32 => @pony_os_sockopt_option(I32(115))
  fun ipv6_flowinfo():I32 => @pony_os_sockopt_option(I32(116))
  fun ipv6_flowinfo_flowlabel():I32 => @pony_os_sockopt_option(I32(117))
  fun ipv6_flowinfo_priority():I32 => @pony_os_sockopt_option(I32(118))
  fun ipv6_flowinfo_send():I32 => @pony_os_sockopt_option(I32(119))
  fun ipv6_flowlabel_mgr():I32 => @pony_os_sockopt_option(I32(120))
  fun ipv6_fl_a_get():I32 => @pony_os_sockopt_option(I32(121))
  fun ipv6_fl_a_put():I32 => @pony_os_sockopt_option(I32(122))
  fun ipv6_fl_a_renew():I32 => @pony_os_sockopt_option(I32(123))
  fun ipv6_fl_f_create():I32 => @pony_os_sockopt_option(I32(124))
  fun ipv6_fl_f_excl():I32 => @pony_os_sockopt_option(I32(125))
  fun ipv6_fl_f_reflect():I32 => @pony_os_sockopt_option(I32(126))
  fun ipv6_fl_f_remote():I32 => @pony_os_sockopt_option(I32(127))
  fun ipv6_fl_s_any():I32 => @pony_os_sockopt_option(I32(128))
  fun ipv6_fl_s_excl():I32 => @pony_os_sockopt_option(I32(129))
  fun ipv6_fl_s_none():I32 => @pony_os_sockopt_option(I32(130))
  fun ipv6_fl_s_process():I32 => @pony_os_sockopt_option(I32(131))
  fun ipv6_fl_s_user():I32 => @pony_os_sockopt_option(I32(132))
  fun ipv6_hoplimit():I32 => @pony_os_sockopt_option(I32(133))
  fun ipv6_hopopts():I32 => @pony_os_sockopt_option(I32(134))
  fun ipv6_ipsec_policy():I32 => @pony_os_sockopt_option(I32(135))
  fun ipv6_join_anycast():I32 => @pony_os_sockopt_option(I32(136))
  fun ipv6_leave_anycast():I32 => @pony_os_sockopt_option(I32(137))
  fun ipv6_minhopcount():I32 => @pony_os_sockopt_option(I32(138))
  fun ipv6_mtu():I32 => @pony_os_sockopt_option(I32(139))
  fun ipv6_mtu_discover():I32 => @pony_os_sockopt_option(I32(140))
  fun ipv6_multicast_hops():I32 => @pony_os_sockopt_option(I32(141))
  fun ipv6_multicast_if():I32 => @pony_os_sockopt_option(I32(142))
  fun ipv6_multicast_loop():I32 => @pony_os_sockopt_option(I32(143))
  fun ipv6_nexthop():I32 => @pony_os_sockopt_option(I32(144))
  fun ipv6_origdstaddr():I32 => @pony_os_sockopt_option(I32(145))
  fun ipv6_pathmtu():I32 => @pony_os_sockopt_option(I32(146))
  fun ipv6_pktinfo():I32 => @pony_os_sockopt_option(I32(147))
  fun ipv6_pmtudisc_do():I32 => @pony_os_sockopt_option(I32(148))
  fun ipv6_pmtudisc_dont():I32 => @pony_os_sockopt_option(I32(149))
  fun ipv6_pmtudisc_interface():I32 => @pony_os_sockopt_option(I32(150))
  fun ipv6_pmtudisc_omit():I32 => @pony_os_sockopt_option(I32(151))
  fun ipv6_pmtudisc_probe():I32 => @pony_os_sockopt_option(I32(152))
  fun ipv6_pmtudisc_want():I32 => @pony_os_sockopt_option(I32(153))
  fun ipv6_prefer_src_cga():I32 => @pony_os_sockopt_option(I32(154))
  fun ipv6_prefer_src_coa():I32 => @pony_os_sockopt_option(I32(155))
  fun ipv6_prefer_src_home():I32 => @pony_os_sockopt_option(I32(156))
  fun ipv6_prefer_src_noncga():I32 => @pony_os_sockopt_option(I32(157))
  fun ipv6_prefer_src_public():I32 => @pony_os_sockopt_option(I32(158))
  fun ipv6_prefer_src_pubtmp_default():I32 => @pony_os_sockopt_option(I32(159))
  fun ipv6_prefer_src_tmp():I32 => @pony_os_sockopt_option(I32(160))
  fun ipv6_priority_10():I32 => @pony_os_sockopt_option(I32(161))
  fun ipv6_priority_11():I32 => @pony_os_sockopt_option(I32(162))
  fun ipv6_priority_12():I32 => @pony_os_sockopt_option(I32(163))
  fun ipv6_priority_13():I32 => @pony_os_sockopt_option(I32(164))
  fun ipv6_priority_14():I32 => @pony_os_sockopt_option(I32(165))
  fun ipv6_priority_15():I32 => @pony_os_sockopt_option(I32(166))
  fun ipv6_priority_8():I32 => @pony_os_sockopt_option(I32(167))
  fun ipv6_priority_9():I32 => @pony_os_sockopt_option(I32(168))
  fun ipv6_priority_bulk():I32 => @pony_os_sockopt_option(I32(169))
  fun ipv6_priority_control():I32 => @pony_os_sockopt_option(I32(170))
  fun ipv6_priority_filler():I32 => @pony_os_sockopt_option(I32(171))
  fun ipv6_priority_interactive():I32 => @pony_os_sockopt_option(I32(172))
  fun ipv6_priority_reserved1():I32 => @pony_os_sockopt_option(I32(173))
  fun ipv6_priority_reserved2():I32 => @pony_os_sockopt_option(I32(174))
  fun ipv6_priority_unattended():I32 => @pony_os_sockopt_option(I32(175))
  fun ipv6_priority_uncharacterized():I32 => @pony_os_sockopt_option(I32(176))
  fun ipv6_recvdstopts():I32 => @pony_os_sockopt_option(I32(177))
  fun ipv6_recverr():I32 => @pony_os_sockopt_option(I32(178))
  fun ipv6_recvhoplimit():I32 => @pony_os_sockopt_option(I32(179))
  fun ipv6_recvhopopts():I32 => @pony_os_sockopt_option(I32(180))
  fun ipv6_recvorigdstaddr():I32 => @pony_os_sockopt_option(I32(181))
  fun ipv6_recvpathmtu():I32 => @pony_os_sockopt_option(I32(182))
  fun ipv6_recvpktinfo():I32 => @pony_os_sockopt_option(I32(183))
  fun ipv6_recvrthdr():I32 => @pony_os_sockopt_option(I32(184))
  fun ipv6_recvtclass():I32 => @pony_os_sockopt_option(I32(185))
  fun ipv6_router_alert():I32 => @pony_os_sockopt_option(I32(186))
  fun ipv6_rthdr():I32 => @pony_os_sockopt_option(I32(187))
  fun ipv6_rthdrdstopts():I32 => @pony_os_sockopt_option(I32(188))
  fun ipv6_tclass():I32 => @pony_os_sockopt_option(I32(189))
  fun ipv6_tlv_hao():I32 => @pony_os_sockopt_option(I32(190))
  fun ipv6_tlv_jumbo():I32 => @pony_os_sockopt_option(I32(191))
  fun ipv6_tlv_pad1():I32 => @pony_os_sockopt_option(I32(192))
  fun ipv6_tlv_padn():I32 => @pony_os_sockopt_option(I32(193))
  fun ipv6_tlv_routeralert():I32 => @pony_os_sockopt_option(I32(194))
  fun ipv6_transparent():I32 => @pony_os_sockopt_option(I32(195))
  fun ipv6_unicast_hops():I32 => @pony_os_sockopt_option(I32(196))
  fun ipv6_unicast_if():I32 => @pony_os_sockopt_option(I32(197))
  fun ipv6_use_min_mtu():I32 => @pony_os_sockopt_option(I32(198))
  fun ipv6_v6only():I32 => @pony_os_sockopt_option(I32(199))
  fun ipv6_xfrm_policy():I32 => @pony_os_sockopt_option(I32(200))
  fun ipx_address():I32 => @pony_os_sockopt_option(I32(201))
  fun ipx_address_notify():I32 => @pony_os_sockopt_option(I32(202))
  fun ipx_crtitf():I32 => @pony_os_sockopt_option(I32(203))
  fun ipx_dltitf():I32 => @pony_os_sockopt_option(I32(204))
  fun ipx_dstype():I32 => @pony_os_sockopt_option(I32(205))
  fun ipx_extended_address():I32 => @pony_os_sockopt_option(I32(206))
  fun ipx_filterptype():I32 => @pony_os_sockopt_option(I32(207))
  fun ipx_frame_8022():I32 => @pony_os_sockopt_option(I32(208))
  fun ipx_frame_8023():I32 => @pony_os_sockopt_option(I32(209))
  fun ipx_frame_etherii():I32 => @pony_os_sockopt_option(I32(210))
  fun ipx_frame_none():I32 => @pony_os_sockopt_option(I32(211))
  fun ipx_frame_snap():I32 => @pony_os_sockopt_option(I32(212))
  fun ipx_frame_tr_8022():I32 => @pony_os_sockopt_option(I32(213))
  fun ipx_getnetinfo():I32 => @pony_os_sockopt_option(I32(214))
  fun ipx_getnetinfo_norip():I32 => @pony_os_sockopt_option(I32(215))
  fun ipx_immediatespxack():I32 => @pony_os_sockopt_option(I32(216))
  fun ipx_internal():I32 => @pony_os_sockopt_option(I32(217))
  fun ipx_maxsize():I32 => @pony_os_sockopt_option(I32(218))
  fun ipx_max_adapter_num():I32 => @pony_os_sockopt_option(I32(219))
  fun ipx_mtu():I32 => @pony_os_sockopt_option(I32(220))
  fun ipx_node_len():I32 => @pony_os_sockopt_option(I32(221))
  fun ipx_primary():I32 => @pony_os_sockopt_option(I32(222))
  fun ipx_ptype():I32 => @pony_os_sockopt_option(I32(223))
  fun ipx_receive_broadcast():I32 => @pony_os_sockopt_option(I32(224))
  fun ipx_recvhdr():I32 => @pony_os_sockopt_option(I32(225))
  fun ipx_reripnetnumber():I32 => @pony_os_sockopt_option(I32(226))
  fun ipx_route_no_router():I32 => @pony_os_sockopt_option(I32(227))
  fun ipx_rt_8022():I32 => @pony_os_sockopt_option(I32(228))
  fun ipx_rt_bluebook():I32 => @pony_os_sockopt_option(I32(229))
  fun ipx_rt_routed():I32 => @pony_os_sockopt_option(I32(230))
  fun ipx_rt_snap():I32 => @pony_os_sockopt_option(I32(231))
  fun ipx_special_none():I32 => @pony_os_sockopt_option(I32(232))
  fun ipx_spxgetconnectionstatus():I32 => @pony_os_sockopt_option(I32(233))
  fun ipx_stopfilterptype():I32 => @pony_os_sockopt_option(I32(234))
  fun ipx_type():I32 => @pony_os_sockopt_option(I32(235))
  fun ip_add_membership():I32 => @pony_os_sockopt_option(I32(236))
  fun ip_add_source_membership():I32 => @pony_os_sockopt_option(I32(237))
  fun ip_bindany():I32 => @pony_os_sockopt_option(I32(238))
  fun ip_bindmulti():I32 => @pony_os_sockopt_option(I32(239))
  fun ip_bind_address_no_port():I32 => @pony_os_sockopt_option(I32(240))
  fun ip_block_source():I32 => @pony_os_sockopt_option(I32(241))
  fun ip_bound_if():I32 => @pony_os_sockopt_option(I32(242))
  fun ip_checksum():I32 => @pony_os_sockopt_option(I32(243))
  fun ip_default_multicast_loop():I32 => @pony_os_sockopt_option(I32(244))
  fun ip_default_multicast_ttl():I32 => @pony_os_sockopt_option(I32(245))
  fun ip_dontfrag():I32 => @pony_os_sockopt_option(I32(246))
  fun ip_drop_membership():I32 => @pony_os_sockopt_option(I32(247))
  fun ip_drop_source_membership():I32 => @pony_os_sockopt_option(I32(248))
  fun ip_dummynet3():I32 => @pony_os_sockopt_option(I32(249))
  fun ip_dummynet_configure():I32 => @pony_os_sockopt_option(I32(250))
  fun ip_dummynet_del():I32 => @pony_os_sockopt_option(I32(251))
  fun ip_dummynet_flush():I32 => @pony_os_sockopt_option(I32(252))
  fun ip_dummynet_get():I32 => @pony_os_sockopt_option(I32(253))
  fun ip_faith():I32 => @pony_os_sockopt_option(I32(254))
  fun ip_flowid():I32 => @pony_os_sockopt_option(I32(255))
  fun ip_flowtype():I32 => @pony_os_sockopt_option(I32(256))
  fun ip_freebind():I32 => @pony_os_sockopt_option(I32(257))
  fun ip_fw3():I32 => @pony_os_sockopt_option(I32(258))
  fun ip_fw_add():I32 => @pony_os_sockopt_option(I32(259))
  fun ip_fw_del():I32 => @pony_os_sockopt_option(I32(260))
  fun ip_fw_flush():I32 => @pony_os_sockopt_option(I32(261))
  fun ip_fw_get():I32 => @pony_os_sockopt_option(I32(262))
  fun ip_fw_nat_cfg():I32 => @pony_os_sockopt_option(I32(263))
  fun ip_fw_nat_del():I32 => @pony_os_sockopt_option(I32(264))
  fun ip_fw_nat_get_config():I32 => @pony_os_sockopt_option(I32(265))
  fun ip_fw_nat_get_log():I32 => @pony_os_sockopt_option(I32(266))
  fun ip_fw_resetlog():I32 => @pony_os_sockopt_option(I32(267))
  fun ip_fw_table_add():I32 => @pony_os_sockopt_option(I32(268))
  fun ip_fw_table_del():I32 => @pony_os_sockopt_option(I32(269))
  fun ip_fw_table_flush():I32 => @pony_os_sockopt_option(I32(270))
  fun ip_fw_table_getsize():I32 => @pony_os_sockopt_option(I32(271))
  fun ip_fw_table_list():I32 => @pony_os_sockopt_option(I32(272))
  fun ip_fw_zero():I32 => @pony_os_sockopt_option(I32(273))
  fun ip_hdrincl():I32 => @pony_os_sockopt_option(I32(274))
  fun ip_ipsec_policy():I32 => @pony_os_sockopt_option(I32(275))
  fun ip_max_group_src_filter():I32 => @pony_os_sockopt_option(I32(276))
  fun ip_max_memberships():I32 => @pony_os_sockopt_option(I32(277))
  fun ip_max_sock_mute_filter():I32 => @pony_os_sockopt_option(I32(278))
  fun ip_max_sock_src_filter():I32 => @pony_os_sockopt_option(I32(279))
  fun ip_max_source_filter():I32 => @pony_os_sockopt_option(I32(280))
  fun ip_minttl():I32 => @pony_os_sockopt_option(I32(281))
  fun ip_min_memberships():I32 => @pony_os_sockopt_option(I32(282))
  fun ip_msfilter():I32 => @pony_os_sockopt_option(I32(283))
  fun ip_mtu():I32 => @pony_os_sockopt_option(I32(284))
  fun ip_mtu_discover():I32 => @pony_os_sockopt_option(I32(285))
  fun ip_multicast_all():I32 => @pony_os_sockopt_option(I32(286))
  fun ip_multicast_if():I32 => @pony_os_sockopt_option(I32(287))
  fun ip_multicast_ifindex():I32 => @pony_os_sockopt_option(I32(288))
  fun ip_multicast_loop():I32 => @pony_os_sockopt_option(I32(289))
  fun ip_multicast_ttl():I32 => @pony_os_sockopt_option(I32(290))
  fun ip_multicast_vif():I32 => @pony_os_sockopt_option(I32(291))
  fun ip_nat_xxx():I32 => @pony_os_sockopt_option(I32(292))
  fun ip_nodefrag():I32 => @pony_os_sockopt_option(I32(293))
  fun ip_old_fw_add():I32 => @pony_os_sockopt_option(I32(294))
  fun ip_old_fw_del():I32 => @pony_os_sockopt_option(I32(295))
  fun ip_old_fw_flush():I32 => @pony_os_sockopt_option(I32(296))
  fun ip_old_fw_get():I32 => @pony_os_sockopt_option(I32(297))
  fun ip_old_fw_resetlog():I32 => @pony_os_sockopt_option(I32(298))
  fun ip_old_fw_zero():I32 => @pony_os_sockopt_option(I32(299))
  fun ip_onesbcast():I32 => @pony_os_sockopt_option(I32(300))
  fun ip_options():I32 => @pony_os_sockopt_option(I32(301))
  fun ip_origdstaddr():I32 => @pony_os_sockopt_option(I32(302))
  fun ip_passsec():I32 => @pony_os_sockopt_option(I32(303))
  fun ip_pktinfo():I32 => @pony_os_sockopt_option(I32(304))
  fun ip_pktoptions():I32 => @pony_os_sockopt_option(I32(305))
  fun ip_pmtudisc_do():I32 => @pony_os_sockopt_option(I32(306))
  fun ip_pmtudisc_dont():I32 => @pony_os_sockopt_option(I32(307))
  fun ip_pmtudisc_interface():I32 => @pony_os_sockopt_option(I32(308))
  fun ip_pmtudisc_omit():I32 => @pony_os_sockopt_option(I32(309))
  fun ip_pmtudisc_probe():I32 => @pony_os_sockopt_option(I32(310))
  fun ip_pmtudisc_want():I32 => @pony_os_sockopt_option(I32(311))
  fun ip_portrange():I32 => @pony_os_sockopt_option(I32(312))
  fun ip_portrange_default():I32 => @pony_os_sockopt_option(I32(313))
  fun ip_portrange_high():I32 => @pony_os_sockopt_option(I32(314))
  fun ip_portrange_low():I32 => @pony_os_sockopt_option(I32(315))
  fun ip_recvdstaddr():I32 => @pony_os_sockopt_option(I32(316))
  fun ip_recverr():I32 => @pony_os_sockopt_option(I32(317))
  fun ip_recvflowid():I32 => @pony_os_sockopt_option(I32(318))
  fun ip_recvif():I32 => @pony_os_sockopt_option(I32(319))
  fun ip_recvopts():I32 => @pony_os_sockopt_option(I32(320))
  fun ip_recvorigdstaddr():I32 => @pony_os_sockopt_option(I32(321))
  fun ip_recvpktinfo():I32 => @pony_os_sockopt_option(I32(322))
  fun ip_recvretopts():I32 => @pony_os_sockopt_option(I32(323))
  fun ip_recvrssbucketid():I32 => @pony_os_sockopt_option(I32(324))
  fun ip_recvtos():I32 => @pony_os_sockopt_option(I32(325))
  fun ip_recvttl():I32 => @pony_os_sockopt_option(I32(326))
  fun ip_retopts():I32 => @pony_os_sockopt_option(I32(327))
  fun ip_router_alert():I32 => @pony_os_sockopt_option(I32(328))
  fun ip_rssbucketid():I32 => @pony_os_sockopt_option(I32(329))
  fun ip_rss_listen_bucket():I32 => @pony_os_sockopt_option(I32(330))
  fun ip_rsvp_off():I32 => @pony_os_sockopt_option(I32(331))
  fun ip_rsvp_on():I32 => @pony_os_sockopt_option(I32(332))
  fun ip_rsvp_vif_off():I32 => @pony_os_sockopt_option(I32(333))
  fun ip_rsvp_vif_on():I32 => @pony_os_sockopt_option(I32(334))
  fun ip_sendsrcaddr():I32 => @pony_os_sockopt_option(I32(335))
  fun ip_striphdr():I32 => @pony_os_sockopt_option(I32(336))
  fun ip_tos():I32 => @pony_os_sockopt_option(I32(337))
  fun ip_traffic_mgt_background():I32 => @pony_os_sockopt_option(I32(338))
  fun ip_transparent():I32 => @pony_os_sockopt_option(I32(339))
  fun ip_ttl():I32 => @pony_os_sockopt_option(I32(340))
  fun ip_unblock_source():I32 => @pony_os_sockopt_option(I32(341))
  fun ip_unicast_if():I32 => @pony_os_sockopt_option(I32(342))
  fun ip_xfrm_policy():I32 => @pony_os_sockopt_option(I32(343))
  fun local_connwait():I32 => @pony_os_sockopt_option(I32(344))
  fun local_creds():I32 => @pony_os_sockopt_option(I32(345))
  fun local_peercred():I32 => @pony_os_sockopt_option(I32(346))
  fun local_peerepid():I32 => @pony_os_sockopt_option(I32(347))
  fun local_peereuuid():I32 => @pony_os_sockopt_option(I32(348))
  fun local_peerpid():I32 => @pony_os_sockopt_option(I32(349))
  fun local_peeruuid():I32 => @pony_os_sockopt_option(I32(350))
  fun local_vendor():I32 => @pony_os_sockopt_option(I32(351))
  fun max_tcpoptlen():I32 => @pony_os_sockopt_option(I32(352))
  fun mcast_block_source():I32 => @pony_os_sockopt_option(I32(353))
  fun mcast_exclude():I32 => @pony_os_sockopt_option(I32(354))
  fun mcast_include():I32 => @pony_os_sockopt_option(I32(355))
  fun mcast_join_group():I32 => @pony_os_sockopt_option(I32(356))
  fun mcast_join_source_group():I32 => @pony_os_sockopt_option(I32(357))
  fun mcast_leave_group():I32 => @pony_os_sockopt_option(I32(358))
  fun mcast_leave_source_group():I32 => @pony_os_sockopt_option(I32(359))
  fun mcast_msfilter():I32 => @pony_os_sockopt_option(I32(360))
  fun mcast_unblock_source():I32 => @pony_os_sockopt_option(I32(361))
  fun mcast_undefined():I32 => @pony_os_sockopt_option(I32(362))
  fun mrt_add_bw_upcall():I32 => @pony_os_sockopt_option(I32(363))
  fun mrt_add_mfc():I32 => @pony_os_sockopt_option(I32(364))
  fun mrt_add_vif():I32 => @pony_os_sockopt_option(I32(365))
  fun mrt_api_config():I32 => @pony_os_sockopt_option(I32(366))
  fun mrt_api_flags_all():I32 => @pony_os_sockopt_option(I32(367))
  fun mrt_api_support():I32 => @pony_os_sockopt_option(I32(368))
  fun mrt_assert():I32 => @pony_os_sockopt_option(I32(369))
  fun mrt_del_bw_upcall():I32 => @pony_os_sockopt_option(I32(370))
  fun mrt_del_mfc():I32 => @pony_os_sockopt_option(I32(371))
  fun mrt_del_vif():I32 => @pony_os_sockopt_option(I32(372))
  fun mrt_done():I32 => @pony_os_sockopt_option(I32(373))
  fun mrt_init():I32 => @pony_os_sockopt_option(I32(374))
  fun mrt_mfc_bw_upcall():I32 => @pony_os_sockopt_option(I32(375))
  fun mrt_mfc_flags_all():I32 => @pony_os_sockopt_option(I32(376))
  fun mrt_mfc_flags_border_vif():I32 => @pony_os_sockopt_option(I32(377))
  fun mrt_mfc_flags_disable_wrongvif():I32 => @pony_os_sockopt_option(I32(378))
  fun mrt_mfc_rp():I32 => @pony_os_sockopt_option(I32(379))
  fun mrt_pim():I32 => @pony_os_sockopt_option(I32(380))
  fun mrt_version():I32 => @pony_os_sockopt_option(I32(381))
  fun msg_notification():I32 => @pony_os_sockopt_option(I32(382))
  fun msg_socallbck():I32 => @pony_os_sockopt_option(I32(383))
  fun ndrvproto_ndrv():I32 => @pony_os_sockopt_option(I32(384))
  fun ndrv_addmulticast():I32 => @pony_os_sockopt_option(I32(385))
  fun ndrv_deldmxspec():I32 => @pony_os_sockopt_option(I32(386))
  fun ndrv_delmulticast():I32 => @pony_os_sockopt_option(I32(387))
  fun ndrv_demuxtype_ethertype():I32 => @pony_os_sockopt_option(I32(388))
  fun ndrv_demuxtype_sap():I32 => @pony_os_sockopt_option(I32(389))
  fun ndrv_demuxtype_snap():I32 => @pony_os_sockopt_option(I32(390))
  fun ndrv_dmux_max_descr():I32 => @pony_os_sockopt_option(I32(391))
  fun ndrv_protocol_desc_vers():I32 => @pony_os_sockopt_option(I32(392))
  fun ndrv_setdmxspec():I32 => @pony_os_sockopt_option(I32(393))
  fun netlink_add_membership():I32 => @pony_os_sockopt_option(I32(394))
  fun netlink_audit():I32 => @pony_os_sockopt_option(I32(395))
  fun netlink_broadcast_error():I32 => @pony_os_sockopt_option(I32(396))
  fun netlink_cap_ack():I32 => @pony_os_sockopt_option(I32(397))
  fun netlink_connector():I32 => @pony_os_sockopt_option(I32(398))
  fun netlink_crypto():I32 => @pony_os_sockopt_option(I32(399))
  fun netlink_dnrtmsg():I32 => @pony_os_sockopt_option(I32(400))
  fun netlink_drop_membership():I32 => @pony_os_sockopt_option(I32(401))
  fun netlink_ecryptfs():I32 => @pony_os_sockopt_option(I32(402))
  fun netlink_fib_lookup():I32 => @pony_os_sockopt_option(I32(403))
  fun netlink_firewall():I32 => @pony_os_sockopt_option(I32(404))
  fun netlink_generic():I32 => @pony_os_sockopt_option(I32(405))
  fun netlink_inet_diag():I32 => @pony_os_sockopt_option(I32(406))
  fun netlink_ip6_fw():I32 => @pony_os_sockopt_option(I32(407))
  fun netlink_iscsi():I32 => @pony_os_sockopt_option(I32(408))
  fun netlink_kobject_uevent():I32 => @pony_os_sockopt_option(I32(409))
  fun netlink_listen_all_nsid():I32 => @pony_os_sockopt_option(I32(410))
  fun netlink_list_memberships():I32 => @pony_os_sockopt_option(I32(411))
  fun netlink_netfilter():I32 => @pony_os_sockopt_option(I32(412))
  fun netlink_nflog():I32 => @pony_os_sockopt_option(I32(413))
  fun netlink_no_enobufs():I32 => @pony_os_sockopt_option(I32(414))
  fun netlink_pktinfo():I32 => @pony_os_sockopt_option(I32(415))
  fun netlink_rdma():I32 => @pony_os_sockopt_option(I32(416))
  fun netlink_route():I32 => @pony_os_sockopt_option(I32(417))
  fun netlink_rx_ring():I32 => @pony_os_sockopt_option(I32(418))
  fun netlink_scsitransport():I32 => @pony_os_sockopt_option(I32(419))
  fun netlink_selinux():I32 => @pony_os_sockopt_option(I32(420))
  fun netlink_sock_diag():I32 => @pony_os_sockopt_option(I32(421))
  fun netlink_tx_ring():I32 => @pony_os_sockopt_option(I32(422))
  fun netlink_unused():I32 => @pony_os_sockopt_option(I32(423))
  fun netlink_usersock():I32 => @pony_os_sockopt_option(I32(424))
  fun netlink_xfrm():I32 => @pony_os_sockopt_option(I32(425))
  fun netrom_idle():I32 => @pony_os_sockopt_option(I32(426))
  fun netrom_kill():I32 => @pony_os_sockopt_option(I32(427))
  fun netrom_n2():I32 => @pony_os_sockopt_option(I32(428))
  fun netrom_neigh():I32 => @pony_os_sockopt_option(I32(429))
  fun netrom_node():I32 => @pony_os_sockopt_option(I32(430))
  fun netrom_paclen():I32 => @pony_os_sockopt_option(I32(431))
  fun netrom_t1():I32 => @pony_os_sockopt_option(I32(432))
  fun netrom_t2():I32 => @pony_os_sockopt_option(I32(433))
  fun netrom_t4():I32 => @pony_os_sockopt_option(I32(434))
  fun nrdv_multicast_addrs_per_sock():I32 => @pony_os_sockopt_option(I32(435))
  fun pvd_config():I32 => @pony_os_sockopt_option(I32(436))
  fun rds_cancel_sent_to():I32 => @pony_os_sockopt_option(I32(437))
  fun rds_cmsg_atomic_cswp():I32 => @pony_os_sockopt_option(I32(438))
  fun rds_cmsg_atomic_fadd():I32 => @pony_os_sockopt_option(I32(439))
  fun rds_cmsg_cong_update():I32 => @pony_os_sockopt_option(I32(440))
  fun rds_cmsg_masked_atomic_cswp():I32 => @pony_os_sockopt_option(I32(441))
  fun rds_cmsg_masked_atomic_fadd():I32 => @pony_os_sockopt_option(I32(442))
  fun rds_cmsg_rdma_args():I32 => @pony_os_sockopt_option(I32(443))
  fun rds_cmsg_rdma_dest():I32 => @pony_os_sockopt_option(I32(444))
  fun rds_cmsg_rdma_map():I32 => @pony_os_sockopt_option(I32(445))
  fun rds_cmsg_rdma_status():I32 => @pony_os_sockopt_option(I32(446))
  fun rds_cong_monitor():I32 => @pony_os_sockopt_option(I32(447))
  fun rds_cong_monitor_size():I32 => @pony_os_sockopt_option(I32(448))
  fun rds_free_mr():I32 => @pony_os_sockopt_option(I32(449))
  fun rds_get_mr():I32 => @pony_os_sockopt_option(I32(450))
  fun rds_get_mr_for_dest():I32 => @pony_os_sockopt_option(I32(451))
  fun rds_ib_abi_version():I32 => @pony_os_sockopt_option(I32(452))
  fun rds_ib_gid_len():I32 => @pony_os_sockopt_option(I32(453))
  fun rds_info_connections():I32 => @pony_os_sockopt_option(I32(454))
  fun rds_info_connection_flag_connected():I32 => @pony_os_sockopt_option(I32(455))
  fun rds_info_connection_flag_connecting():I32 => @pony_os_sockopt_option(I32(456))
  fun rds_info_connection_flag_sending():I32 => @pony_os_sockopt_option(I32(457))
  fun rds_info_connection_stats():I32 => @pony_os_sockopt_option(I32(458))
  fun rds_info_counters():I32 => @pony_os_sockopt_option(I32(459))
  fun rds_info_first():I32 => @pony_os_sockopt_option(I32(460))
  fun rds_info_ib_connections():I32 => @pony_os_sockopt_option(I32(461))
  fun rds_info_iwarp_connections():I32 => @pony_os_sockopt_option(I32(462))
  fun rds_info_last():I32 => @pony_os_sockopt_option(I32(463))
  fun rds_info_message_flag_ack():I32 => @pony_os_sockopt_option(I32(464))
  fun rds_info_message_flag_fast_ack():I32 => @pony_os_sockopt_option(I32(465))
  fun rds_info_recv_messages():I32 => @pony_os_sockopt_option(I32(466))
  fun rds_info_retrans_messages():I32 => @pony_os_sockopt_option(I32(467))
  fun rds_info_send_messages():I32 => @pony_os_sockopt_option(I32(468))
  fun rds_info_sockets():I32 => @pony_os_sockopt_option(I32(469))
  fun rds_info_tcp_sockets():I32 => @pony_os_sockopt_option(I32(470))
  fun rds_rdma_canceled():I32 => @pony_os_sockopt_option(I32(471))
  fun rds_rdma_dontwait():I32 => @pony_os_sockopt_option(I32(472))
  fun rds_rdma_dropped():I32 => @pony_os_sockopt_option(I32(473))
  fun rds_rdma_fence():I32 => @pony_os_sockopt_option(I32(474))
  fun rds_rdma_invalidate():I32 => @pony_os_sockopt_option(I32(475))
  fun rds_rdma_notify_me():I32 => @pony_os_sockopt_option(I32(476))
  fun rds_rdma_other_error():I32 => @pony_os_sockopt_option(I32(477))
  fun rds_rdma_readwrite():I32 => @pony_os_sockopt_option(I32(478))
  fun rds_rdma_remote_error():I32 => @pony_os_sockopt_option(I32(479))
  fun rds_rdma_silent():I32 => @pony_os_sockopt_option(I32(480))
  fun rds_rdma_success():I32 => @pony_os_sockopt_option(I32(481))
  fun rds_rdma_use_once():I32 => @pony_os_sockopt_option(I32(482))
  fun rds_recverr():I32 => @pony_os_sockopt_option(I32(483))
  fun rds_trans_count():I32 => @pony_os_sockopt_option(I32(484))
  fun rds_trans_ib():I32 => @pony_os_sockopt_option(I32(485))
  fun rds_trans_iwarp():I32 => @pony_os_sockopt_option(I32(486))
  fun rds_trans_none():I32 => @pony_os_sockopt_option(I32(487))
  fun rds_trans_tcp():I32 => @pony_os_sockopt_option(I32(488))
  fun rose_access_barred():I32 => @pony_os_sockopt_option(I32(489))
  fun rose_defer():I32 => @pony_os_sockopt_option(I32(490))
  fun rose_dte_originated():I32 => @pony_os_sockopt_option(I32(491))
  fun rose_holdback():I32 => @pony_os_sockopt_option(I32(492))
  fun rose_idle():I32 => @pony_os_sockopt_option(I32(493))
  fun rose_invalid_facility():I32 => @pony_os_sockopt_option(I32(494))
  fun rose_local_procedure():I32 => @pony_os_sockopt_option(I32(495))
  fun rose_max_digis():I32 => @pony_os_sockopt_option(I32(496))
  fun rose_mtu():I32 => @pony_os_sockopt_option(I32(497))
  fun rose_network_congestion():I32 => @pony_os_sockopt_option(I32(498))
  fun rose_not_obtainable():I32 => @pony_os_sockopt_option(I32(499))
  fun rose_number_busy():I32 => @pony_os_sockopt_option(I32(500))
  fun rose_out_of_order():I32 => @pony_os_sockopt_option(I32(501))
  fun rose_qbitincl():I32 => @pony_os_sockopt_option(I32(502))
  fun rose_remote_procedure():I32 => @pony_os_sockopt_option(I32(503))
  fun rose_ship_absent():I32 => @pony_os_sockopt_option(I32(504))
  fun rose_t1():I32 => @pony_os_sockopt_option(I32(505))
  fun rose_t2():I32 => @pony_os_sockopt_option(I32(506))
  fun rose_t3():I32 => @pony_os_sockopt_option(I32(507))
  fun scm_hci_raw_direction():I32 => @pony_os_sockopt_option(I32(508))
  fun scm_timestamp():I32 => @pony_os_sockopt_option(I32(509))
  fun scm_timestamping():I32 => @pony_os_sockopt_option(I32(510))
  fun scm_timestampns():I32 => @pony_os_sockopt_option(I32(511))
  fun scm_wifi_status():I32 => @pony_os_sockopt_option(I32(512))
  fun sctp_abort_association():I32 => @pony_os_sockopt_option(I32(513))
  fun sctp_adaptation_layer():I32 => @pony_os_sockopt_option(I32(514))
  fun sctp_adaption_layer():I32 => @pony_os_sockopt_option(I32(515))
  fun sctp_add_streams():I32 => @pony_os_sockopt_option(I32(516))
  fun sctp_add_vrf_id():I32 => @pony_os_sockopt_option(I32(517))
  fun sctp_asconf():I32 => @pony_os_sockopt_option(I32(518))
  fun sctp_asconf_ack():I32 => @pony_os_sockopt_option(I32(519))
  fun sctp_asconf_supported():I32 => @pony_os_sockopt_option(I32(520))
  fun sctp_associnfo():I32 => @pony_os_sockopt_option(I32(521))
  fun sctp_authentication():I32 => @pony_os_sockopt_option(I32(522))
  fun sctp_auth_active_key():I32 => @pony_os_sockopt_option(I32(523))
  fun sctp_auth_chunk():I32 => @pony_os_sockopt_option(I32(524))
  fun sctp_auth_deactivate_key():I32 => @pony_os_sockopt_option(I32(525))
  fun sctp_auth_delete_key():I32 => @pony_os_sockopt_option(I32(526))
  fun sctp_auth_key():I32 => @pony_os_sockopt_option(I32(527))
  fun sctp_auth_supported():I32 => @pony_os_sockopt_option(I32(528))
  fun sctp_autoclose():I32 => @pony_os_sockopt_option(I32(529))
  fun sctp_auto_asconf():I32 => @pony_os_sockopt_option(I32(530))
  fun sctp_badcrc():I32 => @pony_os_sockopt_option(I32(531))
  fun sctp_bindx_add_addr():I32 => @pony_os_sockopt_option(I32(532))
  fun sctp_bindx_rem_addr():I32 => @pony_os_sockopt_option(I32(533))
  fun sctp_blk_logging_enable():I32 => @pony_os_sockopt_option(I32(534))
  fun sctp_bound():I32 => @pony_os_sockopt_option(I32(535))
  fun sctp_cause_cookie_in_shutdown():I32 => @pony_os_sockopt_option(I32(536))
  fun sctp_cause_deleting_last_addr():I32 => @pony_os_sockopt_option(I32(537))
  fun sctp_cause_deleting_src_addr():I32 => @pony_os_sockopt_option(I32(538))
  fun sctp_cause_illegal_asconf_ack():I32 => @pony_os_sockopt_option(I32(539))
  fun sctp_cause_invalid_param():I32 => @pony_os_sockopt_option(I32(540))
  fun sctp_cause_invalid_stream():I32 => @pony_os_sockopt_option(I32(541))
  fun sctp_cause_missing_param():I32 => @pony_os_sockopt_option(I32(542))
  fun sctp_cause_nat_colliding_state():I32 => @pony_os_sockopt_option(I32(543))
  fun sctp_cause_nat_missing_state():I32 => @pony_os_sockopt_option(I32(544))
  fun sctp_cause_no_error():I32 => @pony_os_sockopt_option(I32(545))
  fun sctp_cause_no_user_data():I32 => @pony_os_sockopt_option(I32(546))
  fun sctp_cause_out_of_resc():I32 => @pony_os_sockopt_option(I32(547))
  fun sctp_cause_protocol_violation():I32 => @pony_os_sockopt_option(I32(548))
  fun sctp_cause_request_refused():I32 => @pony_os_sockopt_option(I32(549))
  fun sctp_cause_resource_shortage():I32 => @pony_os_sockopt_option(I32(550))
  fun sctp_cause_restart_w_newaddr():I32 => @pony_os_sockopt_option(I32(551))
  fun sctp_cause_stale_cookie():I32 => @pony_os_sockopt_option(I32(552))
  fun sctp_cause_unrecog_chunk():I32 => @pony_os_sockopt_option(I32(553))
  fun sctp_cause_unrecog_param():I32 => @pony_os_sockopt_option(I32(554))
  fun sctp_cause_unresolvable_addr():I32 => @pony_os_sockopt_option(I32(555))
  fun sctp_cause_unsupported_hmacid():I32 => @pony_os_sockopt_option(I32(556))
  fun sctp_cause_user_initiated_abt():I32 => @pony_os_sockopt_option(I32(557))
  fun sctp_cc_hstcp():I32 => @pony_os_sockopt_option(I32(558))
  fun sctp_cc_htcp():I32 => @pony_os_sockopt_option(I32(559))
  fun sctp_cc_option():I32 => @pony_os_sockopt_option(I32(560))
  fun sctp_cc_opt_rtcc_setmode():I32 => @pony_os_sockopt_option(I32(561))
  fun sctp_cc_opt_steady_step():I32 => @pony_os_sockopt_option(I32(562))
  fun sctp_cc_opt_use_dccc_ecn():I32 => @pony_os_sockopt_option(I32(563))
  fun sctp_cc_rfc2581():I32 => @pony_os_sockopt_option(I32(564))
  fun sctp_cc_rtcc():I32 => @pony_os_sockopt_option(I32(565))
  fun sctp_closed():I32 => @pony_os_sockopt_option(I32(566))
  fun sctp_clr_stat_log():I32 => @pony_os_sockopt_option(I32(567))
  fun sctp_cmt_base():I32 => @pony_os_sockopt_option(I32(568))
  fun sctp_cmt_max():I32 => @pony_os_sockopt_option(I32(569))
  fun sctp_cmt_mptcp():I32 => @pony_os_sockopt_option(I32(570))
  fun sctp_cmt_off():I32 => @pony_os_sockopt_option(I32(571))
  fun sctp_cmt_on_off():I32 => @pony_os_sockopt_option(I32(572))
  fun sctp_cmt_rpv1():I32 => @pony_os_sockopt_option(I32(573))
  fun sctp_cmt_rpv2():I32 => @pony_os_sockopt_option(I32(574))
  fun sctp_cmt_use_dac():I32 => @pony_os_sockopt_option(I32(575))
  fun sctp_connect_x():I32 => @pony_os_sockopt_option(I32(576))
  fun sctp_connect_x_complete():I32 => @pony_os_sockopt_option(I32(577))
  fun sctp_connect_x_delayed():I32 => @pony_os_sockopt_option(I32(578))
  fun sctp_context():I32 => @pony_os_sockopt_option(I32(579))
  fun sctp_cookie_ack():I32 => @pony_os_sockopt_option(I32(580))
  fun sctp_cookie_echo():I32 => @pony_os_sockopt_option(I32(581))
  fun sctp_cookie_echoed():I32 => @pony_os_sockopt_option(I32(582))
  fun sctp_cookie_wait():I32 => @pony_os_sockopt_option(I32(583))
  fun sctp_cwnd_logging_enable():I32 => @pony_os_sockopt_option(I32(584))
  fun sctp_cwnd_monitor_enable():I32 => @pony_os_sockopt_option(I32(585))
  fun sctp_cwr_in_same_window():I32 => @pony_os_sockopt_option(I32(586))
  fun sctp_cwr_reduce_override():I32 => @pony_os_sockopt_option(I32(587))
  fun sctp_data():I32 => @pony_os_sockopt_option(I32(588))
  fun sctp_data_first_frag():I32 => @pony_os_sockopt_option(I32(589))
  fun sctp_data_frag_mask():I32 => @pony_os_sockopt_option(I32(590))
  fun sctp_data_last_frag():I32 => @pony_os_sockopt_option(I32(591))
  fun sctp_data_middle_frag():I32 => @pony_os_sockopt_option(I32(592))
  fun sctp_data_not_frag():I32 => @pony_os_sockopt_option(I32(593))
  fun sctp_data_sack_immediately():I32 => @pony_os_sockopt_option(I32(594))
  fun sctp_data_unordered():I32 => @pony_os_sockopt_option(I32(595))
  fun sctp_default_prinfo():I32 => @pony_os_sockopt_option(I32(596))
  fun sctp_default_send_param():I32 => @pony_os_sockopt_option(I32(597))
  fun sctp_default_sndinfo():I32 => @pony_os_sockopt_option(I32(598))
  fun sctp_delayed_sack():I32 => @pony_os_sockopt_option(I32(599))
  fun sctp_del_vrf_id():I32 => @pony_os_sockopt_option(I32(600))
  fun sctp_disable_fragments():I32 => @pony_os_sockopt_option(I32(601))
  fun sctp_ecn_cwr():I32 => @pony_os_sockopt_option(I32(602))
  fun sctp_ecn_echo():I32 => @pony_os_sockopt_option(I32(603))
  fun sctp_ecn_supported():I32 => @pony_os_sockopt_option(I32(604))
  fun sctp_enable_change_assoc_req():I32 => @pony_os_sockopt_option(I32(605))
  fun sctp_enable_reset_assoc_req():I32 => @pony_os_sockopt_option(I32(606))
  fun sctp_enable_reset_stream_req():I32 => @pony_os_sockopt_option(I32(607))
  fun sctp_enable_stream_reset():I32 => @pony_os_sockopt_option(I32(608))
  fun sctp_enable_value_mask():I32 => @pony_os_sockopt_option(I32(609))
  fun sctp_established():I32 => @pony_os_sockopt_option(I32(610))
  fun sctp_event():I32 => @pony_os_sockopt_option(I32(611))
  fun sctp_events():I32 => @pony_os_sockopt_option(I32(612))
  fun sctp_explicit_eor():I32 => @pony_os_sockopt_option(I32(613))
  fun sctp_flight_logging_enable():I32 => @pony_os_sockopt_option(I32(614))
  fun sctp_forward_cum_tsn():I32 => @pony_os_sockopt_option(I32(615))
  fun sctp_fragment_interleave():I32 => @pony_os_sockopt_option(I32(616))
  fun sctp_frag_level_0():I32 => @pony_os_sockopt_option(I32(617))
  fun sctp_frag_level_1():I32 => @pony_os_sockopt_option(I32(618))
  fun sctp_frag_level_2():I32 => @pony_os_sockopt_option(I32(619))
  fun sctp_from_middle_box():I32 => @pony_os_sockopt_option(I32(620))
  fun sctp_fr_logging_enable():I32 => @pony_os_sockopt_option(I32(621))
  fun sctp_get_addr_len():I32 => @pony_os_sockopt_option(I32(622))
  fun sctp_get_asoc_vrf():I32 => @pony_os_sockopt_option(I32(623))
  fun sctp_get_assoc_id_list():I32 => @pony_os_sockopt_option(I32(624))
  fun sctp_get_assoc_number():I32 => @pony_os_sockopt_option(I32(625))
  fun sctp_get_local_addresses():I32 => @pony_os_sockopt_option(I32(626))
  fun sctp_get_local_addr_size():I32 => @pony_os_sockopt_option(I32(627))
  fun sctp_get_nonce_values():I32 => @pony_os_sockopt_option(I32(628))
  fun sctp_get_packet_log():I32 => @pony_os_sockopt_option(I32(629))
  fun sctp_get_peer_addresses():I32 => @pony_os_sockopt_option(I32(630))
  fun sctp_get_peer_addr_info():I32 => @pony_os_sockopt_option(I32(631))
  fun sctp_get_remote_addr_size():I32 => @pony_os_sockopt_option(I32(632))
  fun sctp_get_sndbuf_use():I32 => @pony_os_sockopt_option(I32(633))
  fun sctp_get_stat_log():I32 => @pony_os_sockopt_option(I32(634))
  fun sctp_get_vrf_ids():I32 => @pony_os_sockopt_option(I32(635))
  fun sctp_had_no_tcb():I32 => @pony_os_sockopt_option(I32(636))
  fun sctp_heartbeat_ack():I32 => @pony_os_sockopt_option(I32(637))
  fun sctp_heartbeat_request():I32 => @pony_os_sockopt_option(I32(638))
  fun sctp_hmac_ident():I32 => @pony_os_sockopt_option(I32(639))
  fun sctp_idata():I32 => @pony_os_sockopt_option(I32(640))
  fun sctp_iforward_cum_tsn():I32 => @pony_os_sockopt_option(I32(641))
  fun sctp_initiation():I32 => @pony_os_sockopt_option(I32(642))
  fun sctp_initiation_ack():I32 => @pony_os_sockopt_option(I32(643))
  fun sctp_initmsg():I32 => @pony_os_sockopt_option(I32(644))
  fun sctp_interleaving_supported():I32 => @pony_os_sockopt_option(I32(645))
  fun sctp_i_want_mapped_v4_addr():I32 => @pony_os_sockopt_option(I32(646))
  fun sctp_last_packet_tracing():I32 => @pony_os_sockopt_option(I32(647))
  fun sctp_listen():I32 => @pony_os_sockopt_option(I32(648))
  fun sctp_local_auth_chunks():I32 => @pony_os_sockopt_option(I32(649))
  fun sctp_lock_logging_enable():I32 => @pony_os_sockopt_option(I32(650))
  fun sctp_log_at_send_2_outq():I32 => @pony_os_sockopt_option(I32(651))
  fun sctp_log_at_send_2_sctp():I32 => @pony_os_sockopt_option(I32(652))
  fun sctp_log_maxburst_enable():I32 => @pony_os_sockopt_option(I32(653))
  fun sctp_log_rwnd_enable():I32 => @pony_os_sockopt_option(I32(654))
  fun sctp_log_sack_arrivals_enable():I32 => @pony_os_sockopt_option(I32(655))
  fun sctp_log_try_advance():I32 => @pony_os_sockopt_option(I32(656))
  fun sctp_ltrace_chunk_enable():I32 => @pony_os_sockopt_option(I32(657))
  fun sctp_ltrace_error_enable():I32 => @pony_os_sockopt_option(I32(658))
  fun sctp_map_logging_enable():I32 => @pony_os_sockopt_option(I32(659))
  fun sctp_maxburst():I32 => @pony_os_sockopt_option(I32(660))
  fun sctp_maxseg():I32 => @pony_os_sockopt_option(I32(661))
  fun sctp_max_burst():I32 => @pony_os_sockopt_option(I32(662))
  fun sctp_max_cookie_life():I32 => @pony_os_sockopt_option(I32(663))
  fun sctp_max_cwnd():I32 => @pony_os_sockopt_option(I32(664))
  fun sctp_max_hb_interval():I32 => @pony_os_sockopt_option(I32(665))
  fun sctp_max_sack_delay():I32 => @pony_os_sockopt_option(I32(666))
  fun sctp_mbcnt_logging_enable():I32 => @pony_os_sockopt_option(I32(667))
  fun sctp_mbuf_logging_enable():I32 => @pony_os_sockopt_option(I32(668))
  fun sctp_mobility_base():I32 => @pony_os_sockopt_option(I32(669))
  fun sctp_mobility_fasthandoff():I32 => @pony_os_sockopt_option(I32(670))
  fun sctp_mobility_prim_deleted():I32 => @pony_os_sockopt_option(I32(671))
  fun sctp_nagle_logging_enable():I32 => @pony_os_sockopt_option(I32(672))
  fun sctp_nodelay():I32 => @pony_os_sockopt_option(I32(673))
  fun sctp_nrsack_supported():I32 => @pony_os_sockopt_option(I32(674))
  fun sctp_nr_selective_ack():I32 => @pony_os_sockopt_option(I32(675))
  fun sctp_operation_error():I32 => @pony_os_sockopt_option(I32(676))
  fun sctp_packed():I32 => @pony_os_sockopt_option(I32(677))
  fun sctp_packet_dropped():I32 => @pony_os_sockopt_option(I32(678))
  fun sctp_packet_log_size():I32 => @pony_os_sockopt_option(I32(679))
  fun sctp_packet_truncated():I32 => @pony_os_sockopt_option(I32(680))
  fun sctp_pad_chunk():I32 => @pony_os_sockopt_option(I32(681))
  fun sctp_partial_delivery_point():I32 => @pony_os_sockopt_option(I32(682))
  fun sctp_pcb_copy_flags():I32 => @pony_os_sockopt_option(I32(683))
  fun sctp_pcb_flags_accepting():I32 => @pony_os_sockopt_option(I32(684))
  fun sctp_pcb_flags_adaptationevnt():I32 => @pony_os_sockopt_option(I32(685))
  fun sctp_pcb_flags_assoc_resetevnt():I32 => @pony_os_sockopt_option(I32(686))
  fun sctp_pcb_flags_authevnt():I32 => @pony_os_sockopt_option(I32(687))
  fun sctp_pcb_flags_autoclose():I32 => @pony_os_sockopt_option(I32(688))
  fun sctp_pcb_flags_auto_asconf():I32 => @pony_os_sockopt_option(I32(689))
  fun sctp_pcb_flags_blocking_io():I32 => @pony_os_sockopt_option(I32(690))
  fun sctp_pcb_flags_boundall():I32 => @pony_os_sockopt_option(I32(691))
  fun sctp_pcb_flags_bound_v6():I32 => @pony_os_sockopt_option(I32(692))
  fun sctp_pcb_flags_close_ip():I32 => @pony_os_sockopt_option(I32(693))
  fun sctp_pcb_flags_connected():I32 => @pony_os_sockopt_option(I32(694))
  fun sctp_pcb_flags_donot_heartbeat():I32 => @pony_os_sockopt_option(I32(695))
  fun sctp_pcb_flags_dont_wake():I32 => @pony_os_sockopt_option(I32(696))
  fun sctp_pcb_flags_do_asconf():I32 => @pony_os_sockopt_option(I32(697))
  fun sctp_pcb_flags_do_not_pmtud():I32 => @pony_os_sockopt_option(I32(698))
  fun sctp_pcb_flags_dryevnt():I32 => @pony_os_sockopt_option(I32(699))
  fun sctp_pcb_flags_explicit_eor():I32 => @pony_os_sockopt_option(I32(700))
  fun sctp_pcb_flags_ext_rcvinfo():I32 => @pony_os_sockopt_option(I32(701))
  fun sctp_pcb_flags_frag_interleave():I32 => @pony_os_sockopt_option(I32(702))
  fun sctp_pcb_flags_interleave_strms():I32 => @pony_os_sockopt_option(I32(703))
  fun sctp_pcb_flags_in_tcppool():I32 => @pony_os_sockopt_option(I32(704))
  fun sctp_pcb_flags_multiple_asconfs():I32 => @pony_os_sockopt_option(I32(705))
  fun sctp_pcb_flags_needs_mapped_v4():I32 => @pony_os_sockopt_option(I32(706))
  fun sctp_pcb_flags_nodelay():I32 => @pony_os_sockopt_option(I32(707))
  fun sctp_pcb_flags_no_fragment():I32 => @pony_os_sockopt_option(I32(708))
  fun sctp_pcb_flags_pdapievnt():I32 => @pony_os_sockopt_option(I32(709))
  fun sctp_pcb_flags_portreuse():I32 => @pony_os_sockopt_option(I32(710))
  fun sctp_pcb_flags_recvassocevnt():I32 => @pony_os_sockopt_option(I32(711))
  fun sctp_pcb_flags_recvdataioevnt():I32 => @pony_os_sockopt_option(I32(712))
  fun sctp_pcb_flags_recvnsendfailevnt():I32 => @pony_os_sockopt_option(I32(713))
  fun sctp_pcb_flags_recvnxtinfo():I32 => @pony_os_sockopt_option(I32(714))
  fun sctp_pcb_flags_recvpaddrevnt():I32 => @pony_os_sockopt_option(I32(715))
  fun sctp_pcb_flags_recvpeererr():I32 => @pony_os_sockopt_option(I32(716))
  fun sctp_pcb_flags_recvrcvinfo():I32 => @pony_os_sockopt_option(I32(717))
  fun sctp_pcb_flags_recvsendfailevnt():I32 => @pony_os_sockopt_option(I32(718))
  fun sctp_pcb_flags_recvshutdownevnt():I32 => @pony_os_sockopt_option(I32(719))
  fun sctp_pcb_flags_socket_allgone():I32 => @pony_os_sockopt_option(I32(720))
  fun sctp_pcb_flags_socket_cant_read():I32 => @pony_os_sockopt_option(I32(721))
  fun sctp_pcb_flags_socket_gone():I32 => @pony_os_sockopt_option(I32(722))
  fun sctp_pcb_flags_stream_changeevnt():I32 => @pony_os_sockopt_option(I32(723))
  fun sctp_pcb_flags_stream_resetevnt():I32 => @pony_os_sockopt_option(I32(724))
  fun sctp_pcb_flags_tcptype():I32 => @pony_os_sockopt_option(I32(725))
  fun sctp_pcb_flags_udptype():I32 => @pony_os_sockopt_option(I32(726))
  fun sctp_pcb_flags_unbound():I32 => @pony_os_sockopt_option(I32(727))
  fun sctp_pcb_flags_wakeinput():I32 => @pony_os_sockopt_option(I32(728))
  fun sctp_pcb_flags_wakeoutput():I32 => @pony_os_sockopt_option(I32(729))
  fun sctp_pcb_flags_was_aborted():I32 => @pony_os_sockopt_option(I32(730))
  fun sctp_pcb_flags_was_connected():I32 => @pony_os_sockopt_option(I32(731))
  fun sctp_pcb_flags_zero_copy_active():I32 => @pony_os_sockopt_option(I32(732))
  fun sctp_pcb_status():I32 => @pony_os_sockopt_option(I32(733))
  fun sctp_peeloff():I32 => @pony_os_sockopt_option(I32(734))
  fun sctp_peer_addr_params():I32 => @pony_os_sockopt_option(I32(735))
  fun sctp_peer_addr_thlds():I32 => @pony_os_sockopt_option(I32(736))
  fun sctp_peer_auth_chunks():I32 => @pony_os_sockopt_option(I32(737))
  fun sctp_pktdrop_supported():I32 => @pony_os_sockopt_option(I32(738))
  fun sctp_pluggable_cc():I32 => @pony_os_sockopt_option(I32(739))
  fun sctp_pluggable_ss():I32 => @pony_os_sockopt_option(I32(740))
  fun sctp_primary_addr():I32 => @pony_os_sockopt_option(I32(741))
  fun sctp_pr_assoc_status():I32 => @pony_os_sockopt_option(I32(742))
  fun sctp_pr_stream_status():I32 => @pony_os_sockopt_option(I32(743))
  fun sctp_pr_supported():I32 => @pony_os_sockopt_option(I32(744))
  fun sctp_reconfig_supported():I32 => @pony_os_sockopt_option(I32(745))
  fun sctp_recvnxtinfo():I32 => @pony_os_sockopt_option(I32(746))
  fun sctp_recvrcvinfo():I32 => @pony_os_sockopt_option(I32(747))
  fun sctp_recv_rwnd_logging_enable():I32 => @pony_os_sockopt_option(I32(748))
  fun sctp_remote_udp_encaps_port():I32 => @pony_os_sockopt_option(I32(749))
  fun sctp_reset_assoc():I32 => @pony_os_sockopt_option(I32(750))
  fun sctp_reset_streams():I32 => @pony_os_sockopt_option(I32(751))
  fun sctp_reuse_port():I32 => @pony_os_sockopt_option(I32(752))
  fun sctp_rtoinfo():I32 => @pony_os_sockopt_option(I32(753))
  fun sctp_rttvar_logging_enable():I32 => @pony_os_sockopt_option(I32(754))
  fun sctp_sack_cmt_dac():I32 => @pony_os_sockopt_option(I32(755))
  fun sctp_sack_logging_enable():I32 => @pony_os_sockopt_option(I32(756))
  fun sctp_sack_nonce_sum():I32 => @pony_os_sockopt_option(I32(757))
  fun sctp_sack_rwnd_logging_enable():I32 => @pony_os_sockopt_option(I32(758))
  fun sctp_sat_network_burst_incr():I32 => @pony_os_sockopt_option(I32(759))
  fun sctp_sat_network_min():I32 => @pony_os_sockopt_option(I32(760))
  fun sctp_sb_logging_enable():I32 => @pony_os_sockopt_option(I32(761))
  fun sctp_selective_ack():I32 => @pony_os_sockopt_option(I32(762))
  fun sctp_set_debug_level():I32 => @pony_os_sockopt_option(I32(763))
  fun sctp_set_dynamic_primary():I32 => @pony_os_sockopt_option(I32(764))
  fun sctp_set_initial_dbg_seq():I32 => @pony_os_sockopt_option(I32(765))
  fun sctp_set_peer_primary_addr():I32 => @pony_os_sockopt_option(I32(766))
  fun sctp_shutdown():I32 => @pony_os_sockopt_option(I32(767))
  fun sctp_shutdown_ack():I32 => @pony_os_sockopt_option(I32(768))
  fun sctp_shutdown_ack_sent():I32 => @pony_os_sockopt_option(I32(769))
  fun sctp_shutdown_complete():I32 => @pony_os_sockopt_option(I32(770))
  fun sctp_shutdown_pending():I32 => @pony_os_sockopt_option(I32(771))
  fun sctp_shutdown_received():I32 => @pony_os_sockopt_option(I32(772))
  fun sctp_shutdown_sent():I32 => @pony_os_sockopt_option(I32(773))
  fun sctp_smallest_pmtu():I32 => @pony_os_sockopt_option(I32(774))
  fun sctp_ss_default():I32 => @pony_os_sockopt_option(I32(775))
  fun sctp_ss_fair_bandwith():I32 => @pony_os_sockopt_option(I32(776))
  fun sctp_ss_first_come():I32 => @pony_os_sockopt_option(I32(777))
  fun sctp_ss_priority():I32 => @pony_os_sockopt_option(I32(778))
  fun sctp_ss_round_robin():I32 => @pony_os_sockopt_option(I32(779))
  fun sctp_ss_round_robin_packet():I32 => @pony_os_sockopt_option(I32(780))
  fun sctp_ss_value():I32 => @pony_os_sockopt_option(I32(781))
  fun sctp_status():I32 => @pony_os_sockopt_option(I32(782))
  fun sctp_stream_reset():I32 => @pony_os_sockopt_option(I32(783))
  fun sctp_stream_reset_incoming():I32 => @pony_os_sockopt_option(I32(784))
  fun sctp_stream_reset_outgoing():I32 => @pony_os_sockopt_option(I32(785))
  fun sctp_str_logging_enable():I32 => @pony_os_sockopt_option(I32(786))
  fun sctp_threshold_logging():I32 => @pony_os_sockopt_option(I32(787))
  fun sctp_timeouts():I32 => @pony_os_sockopt_option(I32(788))
  fun sctp_use_ext_rcvinfo():I32 => @pony_os_sockopt_option(I32(789))
  fun sctp_vrf_id():I32 => @pony_os_sockopt_option(I32(790))
  fun sctp_wake_logging_enable():I32 => @pony_os_sockopt_option(I32(791))
  fun sock_cloexec():I32 => @pony_os_sockopt_option(I32(792))
  fun sock_dgram():I32 => @pony_os_sockopt_option(I32(793))
  fun sock_maxaddrlen():I32 => @pony_os_sockopt_option(I32(794))
  fun sock_nonblock():I32 => @pony_os_sockopt_option(I32(795))
  fun sock_raw():I32 => @pony_os_sockopt_option(I32(796))
  fun sock_rdm():I32 => @pony_os_sockopt_option(I32(797))
  fun sock_seqpacket():I32 => @pony_os_sockopt_option(I32(798))
  fun sock_stream():I32 => @pony_os_sockopt_option(I32(799))
  fun somaxconn():I32 => @pony_os_sockopt_option(I32(800))
  fun sonpx_setoptshut():I32 => @pony_os_sockopt_option(I32(801))
  fun so_acceptconn():I32 => @pony_os_sockopt_option(I32(802))
  fun so_acceptfilter():I32 => @pony_os_sockopt_option(I32(803))
  fun so_atmpvc():I32 => @pony_os_sockopt_option(I32(804))
  fun so_atmqos():I32 => @pony_os_sockopt_option(I32(805))
  fun so_atmsap():I32 => @pony_os_sockopt_option(I32(806))
  fun so_attach_bpf():I32 => @pony_os_sockopt_option(I32(807))
  fun so_attach_filter():I32 => @pony_os_sockopt_option(I32(808))
  fun so_bindtodevice():I32 => @pony_os_sockopt_option(I32(809))
  fun so_bintime():I32 => @pony_os_sockopt_option(I32(810))
  fun so_bpf_extensions():I32 => @pony_os_sockopt_option(I32(811))
  fun so_broadcast():I32 => @pony_os_sockopt_option(I32(812))
  fun so_bsdcompat():I32 => @pony_os_sockopt_option(I32(813))
  fun so_bsp_state():I32 => @pony_os_sockopt_option(I32(814))
  fun so_busy_poll():I32 => @pony_os_sockopt_option(I32(815))
  fun so_conaccess():I32 => @pony_os_sockopt_option(I32(816))
  fun so_condata():I32 => @pony_os_sockopt_option(I32(817))
  fun so_conditional_accept():I32 => @pony_os_sockopt_option(I32(818))
  fun so_connect_time():I32 => @pony_os_sockopt_option(I32(819))
  fun so_debug():I32 => @pony_os_sockopt_option(I32(820))
  fun so_detach_bpf():I32 => @pony_os_sockopt_option(I32(821))
  fun so_detach_filter():I32 => @pony_os_sockopt_option(I32(822))
  fun so_domain():I32 => @pony_os_sockopt_option(I32(823))
  fun so_dontlinger():I32 => @pony_os_sockopt_option(I32(824))
  fun so_dontroute():I32 => @pony_os_sockopt_option(I32(825))
  fun so_donttrunc():I32 => @pony_os_sockopt_option(I32(826))
  fun so_error():I32 => @pony_os_sockopt_option(I32(827))
  fun so_exclusiveaddruse():I32 => @pony_os_sockopt_option(I32(828))
  fun so_get_filter():I32 => @pony_os_sockopt_option(I32(829))
  fun so_group_id():I32 => @pony_os_sockopt_option(I32(830))
  fun so_group_priority():I32 => @pony_os_sockopt_option(I32(831))
  fun so_hci_raw_direction():I32 => @pony_os_sockopt_option(I32(832))
  fun so_hci_raw_filter():I32 => @pony_os_sockopt_option(I32(833))
  fun so_incoming_cpu():I32 => @pony_os_sockopt_option(I32(834))
  fun so_keepalive():I32 => @pony_os_sockopt_option(I32(835))
  fun so_l2cap_encrypted():I32 => @pony_os_sockopt_option(I32(836))
  fun so_l2cap_flush():I32 => @pony_os_sockopt_option(I32(837))
  fun so_l2cap_iflow():I32 => @pony_os_sockopt_option(I32(838))
  fun so_l2cap_imtu():I32 => @pony_os_sockopt_option(I32(839))
  fun so_l2cap_oflow():I32 => @pony_os_sockopt_option(I32(840))
  fun so_l2cap_omtu():I32 => @pony_os_sockopt_option(I32(841))
  fun so_label():I32 => @pony_os_sockopt_option(I32(842))
  fun so_linger():I32 => @pony_os_sockopt_option(I32(843))
  fun so_linger_sec():I32 => @pony_os_sockopt_option(I32(844))
  fun so_linkinfo():I32 => @pony_os_sockopt_option(I32(845))
  fun so_listenincqlen():I32 => @pony_os_sockopt_option(I32(846))
  fun so_listenqlen():I32 => @pony_os_sockopt_option(I32(847))
  fun so_listenqlimit():I32 => @pony_os_sockopt_option(I32(848))
  fun so_lock_filter():I32 => @pony_os_sockopt_option(I32(849))
  fun so_mark():I32 => @pony_os_sockopt_option(I32(850))
  fun so_max_msg_size():I32 => @pony_os_sockopt_option(I32(851))
  fun so_max_pacing_rate():I32 => @pony_os_sockopt_option(I32(852))
  fun so_multipoint():I32 => @pony_os_sockopt_option(I32(853))
  fun so_netsvc_marking_level():I32 => @pony_os_sockopt_option(I32(854))
  fun so_net_service_type():I32 => @pony_os_sockopt_option(I32(855))
  fun so_nke():I32 => @pony_os_sockopt_option(I32(856))
  fun so_noaddrerr():I32 => @pony_os_sockopt_option(I32(857))
  fun so_nofcs():I32 => @pony_os_sockopt_option(I32(858))
  fun so_nosigpipe():I32 => @pony_os_sockopt_option(I32(859))
  fun so_notifyconflict():I32 => @pony_os_sockopt_option(I32(860))
  fun so_no_check():I32 => @pony_os_sockopt_option(I32(861))
  fun so_no_ddp():I32 => @pony_os_sockopt_option(I32(862))
  fun so_no_offload():I32 => @pony_os_sockopt_option(I32(863))
  fun so_np_extensions():I32 => @pony_os_sockopt_option(I32(864))
  fun so_nread():I32 => @pony_os_sockopt_option(I32(865))
  fun so_numrcvpkt():I32 => @pony_os_sockopt_option(I32(866))
  fun so_nwrite():I32 => @pony_os_sockopt_option(I32(867))
  fun so_oobinline():I32 => @pony_os_sockopt_option(I32(868))
  fun so_original_dst():I32 => @pony_os_sockopt_option(I32(869))
  fun so_passcred():I32 => @pony_os_sockopt_option(I32(870))
  fun so_passsec():I32 => @pony_os_sockopt_option(I32(871))
  fun so_peek_off():I32 => @pony_os_sockopt_option(I32(872))
  fun so_peercred():I32 => @pony_os_sockopt_option(I32(873))
  fun so_peerlabel():I32 => @pony_os_sockopt_option(I32(874))
  fun so_peername():I32 => @pony_os_sockopt_option(I32(875))
  fun so_peersec():I32 => @pony_os_sockopt_option(I32(876))
  fun so_port_scalability():I32 => @pony_os_sockopt_option(I32(877))
  fun so_priority():I32 => @pony_os_sockopt_option(I32(878))
  fun so_protocol():I32 => @pony_os_sockopt_option(I32(879))
  fun so_protocol_info():I32 => @pony_os_sockopt_option(I32(880))
  fun so_prototype():I32 => @pony_os_sockopt_option(I32(881))
  fun so_proxyusr():I32 => @pony_os_sockopt_option(I32(882))
  fun so_randomport():I32 => @pony_os_sockopt_option(I32(883))
  fun so_rcvbuf():I32 => @pony_os_sockopt_option(I32(884))
  fun so_rcvbufforce():I32 => @pony_os_sockopt_option(I32(885))
  fun so_rcvlowat():I32 => @pony_os_sockopt_option(I32(886))
  fun so_rcvtimeo():I32 => @pony_os_sockopt_option(I32(887))
  fun so_rds_transport():I32 => @pony_os_sockopt_option(I32(888))
  fun so_reuseaddr():I32 => @pony_os_sockopt_option(I32(889))
  fun so_reuseport():I32 => @pony_os_sockopt_option(I32(890))
  fun so_reuseshareuid():I32 => @pony_os_sockopt_option(I32(891))
  fun so_rfcomm_fc_info():I32 => @pony_os_sockopt_option(I32(892))
  fun so_rfcomm_mtu():I32 => @pony_os_sockopt_option(I32(893))
  fun so_rxq_ovfl():I32 => @pony_os_sockopt_option(I32(894))
  fun so_sco_conninfo():I32 => @pony_os_sockopt_option(I32(895))
  fun so_sco_mtu():I32 => @pony_os_sockopt_option(I32(896))
  fun so_security_authentication():I32 => @pony_os_sockopt_option(I32(897))
  fun so_security_encryption_network():I32 => @pony_os_sockopt_option(I32(898))
  fun so_security_encryption_transport():I32 => @pony_os_sockopt_option(I32(899))
  fun so_select_err_queue():I32 => @pony_os_sockopt_option(I32(900))
  fun so_setclp():I32 => @pony_os_sockopt_option(I32(901))
  fun so_setfib():I32 => @pony_os_sockopt_option(I32(902))
  fun so_sndbuf():I32 => @pony_os_sockopt_option(I32(903))
  fun so_sndbufforce():I32 => @pony_os_sockopt_option(I32(904))
  fun so_sndlowat():I32 => @pony_os_sockopt_option(I32(905))
  fun so_sndtimeo():I32 => @pony_os_sockopt_option(I32(906))
  fun so_timestamp():I32 => @pony_os_sockopt_option(I32(907))
  fun so_timestamping():I32 => @pony_os_sockopt_option(I32(908))
  fun so_timestampns():I32 => @pony_os_sockopt_option(I32(909))
  fun so_timestamp_monotonic():I32 => @pony_os_sockopt_option(I32(910))
  fun so_type():I32 => @pony_os_sockopt_option(I32(911))
  fun so_upcallclosewait():I32 => @pony_os_sockopt_option(I32(912))
  fun so_update_accept_context():I32 => @pony_os_sockopt_option(I32(913))
  fun so_useloopback():I32 => @pony_os_sockopt_option(I32(914))
  fun so_user_cookie():I32 => @pony_os_sockopt_option(I32(915))
  fun so_vendor():I32 => @pony_os_sockopt_option(I32(916))
  fun so_vm_sockets_buffer_max_size():I32 => @pony_os_sockopt_option(I32(917))
  fun so_vm_sockets_buffer_min_size():I32 => @pony_os_sockopt_option(I32(918))
  fun so_vm_sockets_buffer_size():I32 => @pony_os_sockopt_option(I32(919))
  fun so_vm_sockets_connect_timeout():I32 => @pony_os_sockopt_option(I32(920))
  fun so_vm_sockets_nonblock_txrx():I32 => @pony_os_sockopt_option(I32(921))
  fun so_vm_sockets_peer_host_vm_id():I32 => @pony_os_sockopt_option(I32(922))
  fun so_vm_sockets_trusted():I32 => @pony_os_sockopt_option(I32(923))
  fun so_wantmore():I32 => @pony_os_sockopt_option(I32(924))
  fun so_wantoobflag():I32 => @pony_os_sockopt_option(I32(925))
  fun so_wifi_status():I32 => @pony_os_sockopt_option(I32(926))
  fun tcp6_mss():I32 => @pony_os_sockopt_option(I32(927))
  fun tcpci_flag_lossrecovery():I32 => @pony_os_sockopt_option(I32(928))
  fun tcpci_flag_reordering_detected():I32 => @pony_os_sockopt_option(I32(929))
  fun tcpci_opt_ecn():I32 => @pony_os_sockopt_option(I32(930))
  fun tcpci_opt_sack():I32 => @pony_os_sockopt_option(I32(931))
  fun tcpci_opt_timestamps():I32 => @pony_os_sockopt_option(I32(932))
  fun tcpci_opt_wscale():I32 => @pony_os_sockopt_option(I32(933))
  fun tcpf_ca_cwr():I32 => @pony_os_sockopt_option(I32(934))
  fun tcpf_ca_disorder():I32 => @pony_os_sockopt_option(I32(935))
  fun tcpf_ca_loss():I32 => @pony_os_sockopt_option(I32(936))
  fun tcpf_ca_open():I32 => @pony_os_sockopt_option(I32(937))
  fun tcpf_ca_recovery():I32 => @pony_os_sockopt_option(I32(938))
  fun tcpi_opt_ecn():I32 => @pony_os_sockopt_option(I32(939))
  fun tcpi_opt_ecn_seen():I32 => @pony_os_sockopt_option(I32(940))
  fun tcpi_opt_sack():I32 => @pony_os_sockopt_option(I32(941))
  fun tcpi_opt_syn_data():I32 => @pony_os_sockopt_option(I32(942))
  fun tcpi_opt_timestamps():I32 => @pony_os_sockopt_option(I32(943))
  fun tcpi_opt_toe():I32 => @pony_os_sockopt_option(I32(944))
  fun tcpi_opt_wscale():I32 => @pony_os_sockopt_option(I32(945))
  fun tcpolen_cc():I32 => @pony_os_sockopt_option(I32(946))
  fun tcpolen_cc_appa():I32 => @pony_os_sockopt_option(I32(947))
  fun tcpolen_eol():I32 => @pony_os_sockopt_option(I32(948))
  fun tcpolen_fastopen_req():I32 => @pony_os_sockopt_option(I32(949))
  fun tcpolen_fast_open_empty():I32 => @pony_os_sockopt_option(I32(950))
  fun tcpolen_fast_open_max():I32 => @pony_os_sockopt_option(I32(951))
  fun tcpolen_fast_open_min():I32 => @pony_os_sockopt_option(I32(952))
  fun tcpolen_maxseg():I32 => @pony_os_sockopt_option(I32(953))
  fun tcpolen_nop():I32 => @pony_os_sockopt_option(I32(954))
  fun tcpolen_pad():I32 => @pony_os_sockopt_option(I32(955))
  fun tcpolen_sack():I32 => @pony_os_sockopt_option(I32(956))
  fun tcpolen_sackhdr():I32 => @pony_os_sockopt_option(I32(957))
  fun tcpolen_sack_permitted():I32 => @pony_os_sockopt_option(I32(958))
  fun tcpolen_signature():I32 => @pony_os_sockopt_option(I32(959))
  fun tcpolen_timestamp():I32 => @pony_os_sockopt_option(I32(960))
  fun tcpolen_tstamp_appa():I32 => @pony_os_sockopt_option(I32(961))
  fun tcpolen_window():I32 => @pony_os_sockopt_option(I32(962))
  fun tcpopt_cc():I32 => @pony_os_sockopt_option(I32(963))
  fun tcpopt_ccecho():I32 => @pony_os_sockopt_option(I32(964))
  fun tcpopt_ccnew():I32 => @pony_os_sockopt_option(I32(965))
  fun tcpopt_eol():I32 => @pony_os_sockopt_option(I32(966))
  fun tcpopt_fastopen():I32 => @pony_os_sockopt_option(I32(967))
  fun tcpopt_fast_open():I32 => @pony_os_sockopt_option(I32(968))
  fun tcpopt_maxseg():I32 => @pony_os_sockopt_option(I32(969))
  fun tcpopt_multipath():I32 => @pony_os_sockopt_option(I32(970))
  fun tcpopt_nop():I32 => @pony_os_sockopt_option(I32(971))
  fun tcpopt_pad():I32 => @pony_os_sockopt_option(I32(972))
  fun tcpopt_sack():I32 => @pony_os_sockopt_option(I32(973))
  fun tcpopt_sack_hdr():I32 => @pony_os_sockopt_option(I32(974))
  fun tcpopt_sack_permitted():I32 => @pony_os_sockopt_option(I32(975))
  fun tcpopt_sack_permit_hdr():I32 => @pony_os_sockopt_option(I32(976))
  fun tcpopt_signature():I32 => @pony_os_sockopt_option(I32(977))
  fun tcpopt_timestamp():I32 => @pony_os_sockopt_option(I32(978))
  fun tcpopt_tstamp_hdr():I32 => @pony_os_sockopt_option(I32(979))
  fun tcpopt_window():I32 => @pony_os_sockopt_option(I32(980))
  fun tcp_ca_name_max():I32 => @pony_os_sockopt_option(I32(981))
  fun tcp_ccalgoopt():I32 => @pony_os_sockopt_option(I32(982))
  fun tcp_cc_info():I32 => @pony_os_sockopt_option(I32(983))
  fun tcp_congestion():I32 => @pony_os_sockopt_option(I32(984))
  fun tcp_connectiontimeout():I32 => @pony_os_sockopt_option(I32(985))
  fun tcp_connection_info():I32 => @pony_os_sockopt_option(I32(986))
  fun tcp_cookie_in_always():I32 => @pony_os_sockopt_option(I32(987))
  fun tcp_cookie_max():I32 => @pony_os_sockopt_option(I32(988))
  fun tcp_cookie_min():I32 => @pony_os_sockopt_option(I32(989))
  fun tcp_cookie_out_never():I32 => @pony_os_sockopt_option(I32(990))
  fun tcp_cookie_pair_size():I32 => @pony_os_sockopt_option(I32(991))
  fun tcp_cookie_transactions():I32 => @pony_os_sockopt_option(I32(992))
  fun tcp_cork():I32 => @pony_os_sockopt_option(I32(993))
  fun tcp_defer_accept():I32 => @pony_os_sockopt_option(I32(994))
  fun tcp_enable_ecn():I32 => @pony_os_sockopt_option(I32(995))
  fun tcp_fastopen():I32 => @pony_os_sockopt_option(I32(996))
  fun tcp_function_blk():I32 => @pony_os_sockopt_option(I32(997))
  fun tcp_function_name_len_max():I32 => @pony_os_sockopt_option(I32(998))
  fun tcp_info():I32 => @pony_os_sockopt_option(I32(999))
  fun tcp_keepalive():I32 => @pony_os_sockopt_option(I32(1000))
  fun tcp_keepcnt():I32 => @pony_os_sockopt_option(I32(1001))
  fun tcp_keepidle():I32 => @pony_os_sockopt_option(I32(1002))
  fun tcp_keepinit():I32 => @pony_os_sockopt_option(I32(1003))
  fun tcp_keepintvl():I32 => @pony_os_sockopt_option(I32(1004))
  fun tcp_linger2():I32 => @pony_os_sockopt_option(I32(1005))
  fun tcp_maxburst():I32 => @pony_os_sockopt_option(I32(1006))
  fun tcp_maxhlen():I32 => @pony_os_sockopt_option(I32(1007))
  fun tcp_maxolen():I32 => @pony_os_sockopt_option(I32(1008))
  fun tcp_maxseg():I32 => @pony_os_sockopt_option(I32(1009))
  fun tcp_maxwin():I32 => @pony_os_sockopt_option(I32(1010))
  fun tcp_max_sack():I32 => @pony_os_sockopt_option(I32(1011))
  fun tcp_max_winshift():I32 => @pony_os_sockopt_option(I32(1012))
  fun tcp_md5sig():I32 => @pony_os_sockopt_option(I32(1013))
  fun tcp_md5sig_maxkeylen():I32 => @pony_os_sockopt_option(I32(1014))
  fun tcp_minmss():I32 => @pony_os_sockopt_option(I32(1015))
  fun tcp_mss():I32 => @pony_os_sockopt_option(I32(1016))
  fun tcp_mss_default():I32 => @pony_os_sockopt_option(I32(1017))
  fun tcp_mss_desired():I32 => @pony_os_sockopt_option(I32(1018))
  fun tcp_nodelay():I32 => @pony_os_sockopt_option(I32(1019))
  fun tcp_noopt():I32 => @pony_os_sockopt_option(I32(1020))
  fun tcp_nopush():I32 => @pony_os_sockopt_option(I32(1021))
  fun tcp_notsent_lowat():I32 => @pony_os_sockopt_option(I32(1022))
  fun tcp_pcap_in():I32 => @pony_os_sockopt_option(I32(1023))
  fun tcp_pcap_out():I32 => @pony_os_sockopt_option(I32(1024))
  fun tcp_queue_seq():I32 => @pony_os_sockopt_option(I32(1025))
  fun tcp_quickack():I32 => @pony_os_sockopt_option(I32(1026))
  fun tcp_repair():I32 => @pony_os_sockopt_option(I32(1027))
  fun tcp_repair_options():I32 => @pony_os_sockopt_option(I32(1028))
  fun tcp_repair_queue():I32 => @pony_os_sockopt_option(I32(1029))
  fun tcp_rxt_conndroptime():I32 => @pony_os_sockopt_option(I32(1030))
  fun tcp_rxt_findrop():I32 => @pony_os_sockopt_option(I32(1031))
  fun tcp_saved_syn():I32 => @pony_os_sockopt_option(I32(1032))
  fun tcp_save_syn():I32 => @pony_os_sockopt_option(I32(1033))
  fun tcp_sendmoreacks():I32 => @pony_os_sockopt_option(I32(1034))
  fun tcp_syncnt():I32 => @pony_os_sockopt_option(I32(1035))
  fun tcp_s_data_in():I32 => @pony_os_sockopt_option(I32(1036))
  fun tcp_s_data_out():I32 => @pony_os_sockopt_option(I32(1037))
  fun tcp_thin_dupack():I32 => @pony_os_sockopt_option(I32(1038))
  fun tcp_thin_linear_timeouts():I32 => @pony_os_sockopt_option(I32(1039))
  fun tcp_timestamp():I32 => @pony_os_sockopt_option(I32(1040))
  fun tcp_user_timeout():I32 => @pony_os_sockopt_option(I32(1041))
  fun tcp_vendor():I32 => @pony_os_sockopt_option(I32(1042))
  fun tcp_window_clamp():I32 => @pony_os_sockopt_option(I32(1043))
  fun tipc_addr_id():I32 => @pony_os_sockopt_option(I32(1044))
  fun tipc_addr_mcast():I32 => @pony_os_sockopt_option(I32(1045))
  fun tipc_addr_name():I32 => @pony_os_sockopt_option(I32(1046))
  fun tipc_addr_nameseq():I32 => @pony_os_sockopt_option(I32(1047))
  fun tipc_cfg_srv():I32 => @pony_os_sockopt_option(I32(1048))
  fun tipc_cluster_scope():I32 => @pony_os_sockopt_option(I32(1049))
  fun tipc_conn_shutdown():I32 => @pony_os_sockopt_option(I32(1050))
  fun tipc_conn_timeout():I32 => @pony_os_sockopt_option(I32(1051))
  fun tipc_critical_importance():I32 => @pony_os_sockopt_option(I32(1052))
  fun tipc_destname():I32 => @pony_os_sockopt_option(I32(1053))
  fun tipc_dest_droppable():I32 => @pony_os_sockopt_option(I32(1054))
  fun tipc_errinfo():I32 => @pony_os_sockopt_option(I32(1055))
  fun tipc_err_no_name():I32 => @pony_os_sockopt_option(I32(1056))
  fun tipc_err_no_node():I32 => @pony_os_sockopt_option(I32(1057))
  fun tipc_err_no_port():I32 => @pony_os_sockopt_option(I32(1058))
  fun tipc_err_overload():I32 => @pony_os_sockopt_option(I32(1059))
  fun tipc_high_importance():I32 => @pony_os_sockopt_option(I32(1060))
  fun tipc_importance():I32 => @pony_os_sockopt_option(I32(1061))
  fun tipc_link_state():I32 => @pony_os_sockopt_option(I32(1062))
  fun tipc_low_importance():I32 => @pony_os_sockopt_option(I32(1063))
  fun tipc_max_bearer_name():I32 => @pony_os_sockopt_option(I32(1064))
  fun tipc_max_if_name():I32 => @pony_os_sockopt_option(I32(1065))
  fun tipc_max_link_name():I32 => @pony_os_sockopt_option(I32(1066))
  fun tipc_max_media_name():I32 => @pony_os_sockopt_option(I32(1067))
  fun tipc_max_user_msg_size():I32 => @pony_os_sockopt_option(I32(1068))
  fun tipc_medium_importance():I32 => @pony_os_sockopt_option(I32(1069))
  fun tipc_node_recvq_depth():I32 => @pony_os_sockopt_option(I32(1070))
  fun tipc_node_scope():I32 => @pony_os_sockopt_option(I32(1071))
  fun tipc_ok():I32 => @pony_os_sockopt_option(I32(1072))
  fun tipc_published():I32 => @pony_os_sockopt_option(I32(1073))
  fun tipc_reserved_types():I32 => @pony_os_sockopt_option(I32(1074))
  fun tipc_retdata():I32 => @pony_os_sockopt_option(I32(1075))
  fun tipc_sock_recvq_depth():I32 => @pony_os_sockopt_option(I32(1076))
  fun tipc_src_droppable():I32 => @pony_os_sockopt_option(I32(1077))
  fun tipc_subscr_timeout():I32 => @pony_os_sockopt_option(I32(1078))
  fun tipc_sub_cancel():I32 => @pony_os_sockopt_option(I32(1079))
  fun tipc_sub_ports():I32 => @pony_os_sockopt_option(I32(1080))
  fun tipc_sub_service():I32 => @pony_os_sockopt_option(I32(1081))
  fun tipc_top_srv():I32 => @pony_os_sockopt_option(I32(1082))
  fun tipc_wait_forever():I32 => @pony_os_sockopt_option(I32(1083))
  fun tipc_withdrawn():I32 => @pony_os_sockopt_option(I32(1084))
  fun tipc_zone_scope():I32 => @pony_os_sockopt_option(I32(1085))
  fun ttcp_client_snd_wnd():I32 => @pony_os_sockopt_option(I32(1086))
  fun udp_cork():I32 => @pony_os_sockopt_option(I32(1087))
  fun udp_encap():I32 => @pony_os_sockopt_option(I32(1088))
  fun udp_encap_espinudp():I32 => @pony_os_sockopt_option(I32(1089))
  fun udp_encap_espinudp_maxfraglen():I32 => @pony_os_sockopt_option(I32(1090))
  fun udp_encap_espinudp_non_ike():I32 => @pony_os_sockopt_option(I32(1091))
  fun udp_encap_espinudp_port():I32 => @pony_os_sockopt_option(I32(1092))
  fun udp_encap_l2tpinudp():I32 => @pony_os_sockopt_option(I32(1093))
  fun udp_nocksum():I32 => @pony_os_sockopt_option(I32(1094))
  fun udp_no_check6_rx():I32 => @pony_os_sockopt_option(I32(1095))
  fun udp_no_check6_tx():I32 => @pony_os_sockopt_option(I32(1096))
  fun udp_vendor():I32 => @pony_os_sockopt_option(I32(1097))
  fun so_rcvtimeo_old():I32 => @pony_os_sockopt_option(I32(1098))
  fun so_rcvtimeo_new():I32 => @pony_os_sockopt_option(I32(1099))
  fun so_sndtimeo_old():I32 => @pony_os_sockopt_option(I32(1100))
  fun so_sndtimeo_new():I32 => @pony_os_sockopt_option(I32(1101))
