/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::common::f_fmla;
use crate::polyeval::f_polyeval5;

/// Computes cotangent
///
/// Max found ULP 0.4999999
#[inline]
pub fn f_cotf(x: f32) -> f32 {
    let x_abs = x.to_bits() & 0x7fff_ffffu32;
    let xd = x as f64;

    // |x| < pi/32
    if x_abs <= 0x3dc9_0fdbu32 {
        // |x| < 0.000244141
        if x_abs < 0x3980_0000u32 {
            if x_abs == 0 {
                return 1. / x;
            }

            // When |x| < 2^-12, the relative error of the approximation cot(x)
            // is:
            let ddx = x as f64;
            let dx = 1. / ddx;
            // taylor order 3
            return f_fmla(ddx, f64::from_bits(0xbfd5555555555555), dx) as f32;
        }

        let xsqr = xd * xd;

        /*
           Generated by Sollya:
           f_cotf = x/tan(x);
           Q = fpminimax(f_cotf, [|0, 2, 4, 6, 8|], [|1, D...|], [0, pi/32]);

           See ./notes/cotf.sollya
        */
        let p = f_polyeval5(
            xsqr,
            f64::from_bits(0x3ff0000000000000),
            f64::from_bits(0xbfd5555555555466),
            f64::from_bits(0xbf96c16c16fb8937),
            f64::from_bits(0xbf6156688756cd43),
            f64::from_bits(0xbf2bce669d7cd742),
        );
        return (p / xd) as f32;
    }

    if x_abs >= 0x7f80_0000u32 {
        return x + f32::NAN;
    }

    // For |x| >= pi/32, we use the definition of cot(x) function:
    // cot(a+b) = (1 - tan(a)tan(b)) / (tan(a) + tan(b))
    // tanf_eval returns:
    // - rs.tan_y = tan(pi/32 * y)          -> tangent of the remainder
    // - rs.tan_k = tan(pi/32 * k)          -> tan of the main angle multiple
    let rs = crate::tangent::evalf::tanf_eval(xd, x_abs);

    // Then computing tan through identities
    // num = tan(k*pi/32) + tan(y*pi/32)
    let num = rs.tan_y + rs.tan_k;
    // den = 1 - tan(k*pi/32) * tan(y*pi/32)
    let den = f_fmla(rs.tan_y, -rs.tan_k, 1.);
    (den / num) as f32
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn cotf_test() {
        assert_eq!(f_cotf(0.0010348097), 966.36084);
        assert_eq!(f_cotf(0.0020286469), 492.93872);
        assert_eq!(f_cotf(-0.0020286469), -492.93872);
        assert_eq!(f_cotf(1.0020286469), 0.63923126);
        assert_eq!(f_cotf(-1.0020286469), -0.63923126);
        assert_eq!(f_cotf(0.0), f32::INFINITY);
        assert_eq!(f_cotf(-0.0), f32::NEG_INFINITY);
        assert!(f_cotf(f32::INFINITY).is_nan());
        assert!(f_cotf(f32::NEG_INFINITY).is_nan());
        assert!(f_cotf(f32::NAN).is_nan());
    }
}
