/*
 * Copyright (C) 2015 Igalia S.L.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebContextMenuItemGlib.h"

#include <wtf/TZoneMallocInlines.h>

#if ENABLE(CONTEXT_MENUS)
#include "APIObject.h"
#include "WebKitContextMenuGAction.h"
#include <gio/gio.h>

#if PLATFORM(GTK) && !USE(GTK4)
#include <gtk/gtk.h>
#endif

namespace WebKit {
using namespace WebCore;

WTF_MAKE_TZONE_ALLOCATED_IMPL(WebContextMenuItemGlib);

WebContextMenuItemGlib::WebContextMenuItemGlib(ContextMenuItemType type, ContextMenuAction action, const String& title, bool enabled, bool checked)
    : WebContextMenuItemData(type, action, String { title }, enabled, checked)
{
    ASSERT(type != ContextMenuItemType::Submenu);
    createActionIfNeeded();
}

WebContextMenuItemGlib::WebContextMenuItemGlib(const WebContextMenuItemData& data)
    : WebContextMenuItemData(data.type() == ContextMenuItemType::Submenu ? ContextMenuItemType::Action : data.type(), data.action(), String { data.title() }, data.enabled(), data.checked())
{
    createActionIfNeeded();
}

WebContextMenuItemGlib::WebContextMenuItemGlib(const WebContextMenuItemGlib& data, Vector<WebContextMenuItemGlib>&& submenu)
    : WebContextMenuItemData(ContextMenuItemType::Action, data.action(), String { data.title() }, data.enabled(), false)
{
    m_gAction = data.gAction();
    m_submenuItems = WTF::move(submenu);
#if PLATFORM(GTK) && !USE(GTK4)
    m_gtkAction = data.gtkAction();
#endif
}

static bool isGActionChecked(GAction* action)
{
    if (!g_action_get_state_type(action))
        return false;

    ASSERT(g_variant_type_equal(g_action_get_state_type(action), G_VARIANT_TYPE_BOOLEAN));
    GRefPtr<GVariant> state = adoptGRef(g_action_get_state(action));
    return g_variant_get_boolean(state.get());
}

WebContextMenuItemGlib::WebContextMenuItemGlib(GAction* action, const String& title, GVariant* target)
    : WebContextMenuItemData(g_action_get_state_type(action) ? ContextMenuItemType::CheckableAction : ContextMenuItemType::Action, ContextMenuItemBaseApplicationTag, String { title }, g_action_get_enabled(action), isGActionChecked(action))
    , m_gAction(action)
    , m_gActionTarget(target)
{
    createActionIfNeeded();
}

#if PLATFORM(GTK) && !USE(GTK4)
ALLOW_DEPRECATED_DECLARATIONS_BEGIN
WebContextMenuItemGlib::WebContextMenuItemGlib(GtkAction* action)
    : WebContextMenuItemData(GTK_IS_TOGGLE_ACTION(action) ? ContextMenuItemType::CheckableAction : ContextMenuItemType::Action, ContextMenuItemBaseApplicationTag, String::fromUTF8(gtk_action_get_label(action)), gtk_action_get_sensitive(action), GTK_IS_TOGGLE_ACTION(action) ? gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)) : false)
{
    m_gtkAction = action;
    createActionIfNeeded();
}
ALLOW_DEPRECATED_DECLARATIONS_END
#endif

WebContextMenuItemGlib::~WebContextMenuItemGlib()
{
}

void WebContextMenuItemGlib::createActionIfNeeded()
{
    if (type() == ContextMenuItemType::Separator || type() == ContextMenuItemType::Submenu)
        return;

    if (!m_gAction) {
        const char* name = nullptr;
#if PLATFORM(GTK) && !USE(GTK4)
ALLOW_DEPRECATED_DECLARATIONS_BEGIN
        if (m_gtkAction) // NOLINT
            name = gtk_action_get_name(m_gtkAction);
ALLOW_DEPRECATED_DECLARATIONS_END
#endif
        m_gAction = adoptGRef(webkitContextMenuGActionNew(name, *this));
    }

#if PLATFORM(GTK) && !USE(GTK4)
ALLOW_DEPRECATED_DECLARATIONS_BEGIN
    // Create the GtkAction for backwards compatibility only.
    if (!m_gtkAction) {
        if (type() == ContextMenuItemType::CheckableAction) {
            m_gtkAction = GTK_ACTION(gtk_toggle_action_new(g_action_get_name(m_gAction.get()), title().utf8().data(), nullptr, nullptr));
            gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(m_gtkAction), checked());
        } else
            m_gtkAction = gtk_action_new(g_action_get_name(m_gAction.get()), title().utf8().data(), 0, nullptr);
        gtk_action_set_sensitive(m_gtkAction, enabled());
    }

    if (WEBKIT_IS_CONTEXT_MENU_GACTION(m_gAction.get()))
        webkitContextMenuGActionSetGtkAction(WEBKIT_CONTEXT_MENU_GACTION(m_gAction.get()), m_gtkAction);
    else {
        g_object_set_data_full(G_OBJECT(m_gAction.get()), "webkit-gtk-action", g_object_ref_sink(m_gtkAction), g_object_unref);
        g_signal_connect_object(m_gAction.get(), "activate", G_CALLBACK(gtk_action_activate), m_gtkAction, G_CONNECT_SWAPPED);
    }
ALLOW_DEPRECATED_DECLARATIONS_END
#endif
}

} // namespace WebKit

#endif // ENABLE(CONTEXT_MENUS)
