/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef XMText_H
#include "XMText.h"
#endif

#ifndef ViewEditor_H
#include "ViewEditor.h"
#endif

#ifndef EditorFactory_H
#include "EditorFactory.h"
#endif

#ifndef SimpleEditor_H
#include "SimpleEditor.h"
#endif

#ifndef IconObject_H
#include "IconObject.h"
#endif

#ifndef IconClass_H
#include "IconClass.h"
#endif

#ifndef IconFactory_H
#include "IconFactory.h"
#endif

#ifndef Log_H
#include "Log.h"
#endif

#ifndef PageView_H
#include "PageView.h"
#endif

#ifndef Dropping_H
#include "Dropping.h"
#endif

#ifndef EditorButton_H
#include "EditorButton.h"
#endif

#ifndef Folder_H
#include "Folder.h"
#endif

static struct {
	string      name;
	double      h;
	double      v;
} paper_sizes [] = {
	{"A4", 21,   29.7, },
	{"A3", 29.7, 42,   },
};


//---------------------------------------------------

ViewEditor::ViewEditor(const IconClass& name,const string& kind):
	XEditor<UIViewEditor_c>(name,kind)
{
  
  // Set up callbacks for the paper widget. Needed in order to handle the view
  // properly.
  
}

ViewEditor::~ViewEditor()
{
}

void ViewEditor::apply()
{
	savePaperSize(request_);
	savePages(current_,request_);
	current_->request(request_);
}

void ViewEditor::reset()
{
	cache_.clear();
	save_.clear();
	request_ = current_->request();
	loadPaperSize(request_);
	PaperSetPages(paper_,loadPages(current_,request_));
	updateDrawers();
	PaperSelectFirst(paper_);
}

void ViewEditor::close()
{
	save_.clear();
}

void ViewEditor::merge(IconObject* o)
{
}

void ViewEditor::replace(IconObject* o)
{
}

void ViewEditor::loadPaperSize(const Request& r)
{
	cout << "paperSize " << endl;
	r.print();

	// Set paper size...
	double v = r("CUSTOM_HEIGHT");
	double h = r("CUSTOM_WIDTH");

	const char* size  = r("LAYOUT_SIZE");
	const char* orien = r("LAYOUT_ORIENTATION");

	for(int i = 0; i < XtNumber(paper_sizes) ; i++)
	{
		if(size && paper_sizes[i].name == size)
		{
			if(orien && strcmp(orien,"PORTRAIT") == 0)
			{
				h = paper_sizes[i].h;
				v = paper_sizes[i].v;
			}
			else
			{
				v = paper_sizes[i].h;
				h = paper_sizes[i].v;
			}
		}
	}

	setPaperSize(h,v);

	cout << "h = " << h << " v = " << v << endl;
}

void ViewEditor::savePaperSize(Request& r)
{
	double h,v;
	getPaperSize(h,v);

	r("LAYOUT_SIZE")   = "CUSTOM";
	r("CUSTOM_WIDTH")  = h;
	r("CUSTOM_HEIGHT") = v;
		
	for(int i = 0; i < XtNumber(paper_sizes) ; i++)
	{
		if(h == paper_sizes[i].h && v == paper_sizes[i].v)
		{
			r("LAYOUT_SIZE") = paper_sizes[i].name.c_str();	
			r("LAYOUT_ORIENTATION") = "PORTRAIT";
			r.unsetParam("CUSTOM_WIDTH");
			r.unsetParam("CUSTOM_HEIGHT");
		}
		if(v == paper_sizes[i].h && h == paper_sizes[i].v)
		{
			r("LAYOUT_SIZE") = paper_sizes[i].name.c_str();	
			r("LAYOUT_ORIENTATION") = "LANDSCAPE";
			r.unsetParam("CUSTOM_WIDTH");
			r.unsetParam("CUSTOM_HEIGHT");
		}
	}
}	


PageView* ViewEditor::pageView(IconObject* o)
{
	map<IconObjectH,PageViewH>::iterator j = cache_.find(o);
	if(j == cache_.end())
		cache_[o] = new PageView(o);
	return cache_[o];
}

PaperRectangle* ViewEditor::loadPages(IconObject* o,const Request& r)
{
	PaperRectangle* pages = 0;

	//MvRequest pageRequest = r("PAGES");
	vector<IconObjectH> sub = o->subObjects("PAGES", r );

	for(vector<IconObjectH>::iterator ii = sub.begin(); ii != sub.end(); ++ii)
	  {
	    MvRequest p = (*ii)->request();
	    PaperRectangle* s = loadRectangle(p);
	    vector<IconObjectH> y = (*ii)->subObjects("VIEW",p);
	    
	    IconObject* z = y.size() ? (IconObject*)y[0]: defaultView();
	    s->data = pageView(z);;
	    s->kids = loadSubPages(*ii,p);
	    
	    s->next = pages;
	    pages = s;
	  }
	
	return pages;
}

PaperRectangle* ViewEditor::loadSubPages(IconObject* o,const Request& r)
{
	PaperRectangle* pages = 0;

	// Old style

	int rows = r("ROWS");
	int cols = r("COLUMNS");

	if(rows * cols > 1)
	{
		double height = 1.0 / rows;
		double width  = 1.0 / cols;
		double x = 0;

		for(int i = 0; i < cols; i++) 
		{
			double y = 0;
			for(int j = 0; j < rows; j++)
			{
				PaperRectangle* s = PaperNewPage(
					paper_,
					y, 
					x, 
					(j == rows - 1) ? 1.0 : y + height, 
					(i == cols - 1) ? 1.0 : x + width
					);

				s->next = pages;
				pages = s;
				y += height;
			}
			x +=  width;
		}
	}

	// New Style

	//MvRequest p = r("SUB_PAGES");
	vector<IconObjectH> sub = o->subObjects("SUB_PAGES", r );
	for(vector<IconObjectH>::iterator ii = sub.begin(); ii != sub.end(); ++ii)
	  {
	    MvRequest p = (*ii)->request();
	    PaperRectangle* s = loadRectangle(p);
	    s->next = pages;
	    pages = s;
	  }

	return pages;
}

PaperRectangle* ViewEditor::loadRectangle(const Request& r)
{
	double top    = r("TOP");
	double left   = r("LEFT");
	double right  = r("RIGHT");
	double bottom = r("BOTTOM");
	return PaperNewPage(paper_,top/100.0,left/100.0,bottom/100.0,right/100.0);
}


void ViewEditor::saveRectangle(Request& r,PaperRectangle* s)
{
	r("TOP")    = s->top * 100 ;
	r("LEFT")   = s->left * 100 ;
	r("BOTTOM") = s->bottom * 100; 
	r("RIGHT")  = s->right * 100 ;
}

Widget ViewEditor::dropSite()
{
	return paper_;
}

void ViewEditor::drop(Dropping& drop)
{
	XEvent ev;
	IconObjectH o = drop.object(*this);
	ev.xbutton.x = drop.x();
	ev.xbutton.y = drop.y();

	cout << "dropped " << o->name() << endl;

	if(o->iconClass().type() != "View") // We need something more OO
		return;
	
	if ( drop.copy() )
	  o = o->clone(current_->embeddedFolder(true));

	PaperRectangle* r = PaperFind(paper_,&ev);
	if(!r) {
		double x1=0,y1=0,x2=0,y2=0;

		PaperLocation(paper_,ev.xbutton.x - 32, ev.xbutton.y - 32, &x1, &y1);
		PaperLocation(paper_,ev.xbutton.x + 32, ev.xbutton.y + 32, &x2, &y2);

		r = PaperAddPage(paper_,y1,x1,y2,x2);
	}

	PageView* page = pageView(o);

	if(!r->selected)  
		r->data = page;
	else
	{
		r = PaperGetPages(paper_);
		while(r)
		{
			if(r->selected)
				r->data = page;
			r = r->next;
		}
	}		
		
	PaperUpdate(paper_);
	updateDrawers();

}

IconObject *ViewEditor::defaultView()
{
	return IconClass::find("GEOVIEW").defaultObject();
}

void ViewEditor::drawCB(Widget w,XtPointer data)
{
	PaperCallbackStruct* cb = (PaperCallbackStruct*)data;
	PageView* p = (PageView*)cb->r1->data;

	if(p == 0)
	{

		bool ispage = false;
		PaperRectangle* r = PaperGetPages(paper_);
		while(r)
		{
			if(r == cb->r1) { ispage = true; break ; }
			r = r->next;
		}

		if(ispage)
			cb->r1->data  =  pageView(defaultView());
	}
	if(p) p->draw(w,cb->graphics);
}

void ViewEditor::dblClickCB(Widget,XtPointer data)
{
	PaperCallbackStruct* cb = (PaperCallbackStruct*)data;
	PageView* p = (PageView*)cb->r1->data;
	if(p) p->edit();
}

void ViewEditor::splitCB(Widget, XtPointer data)
{
	PaperCallbackStruct* cb = (PaperCallbackStruct*)data;
	cb->r2->data = cb->r1->data;
	updateDrawers();
}

void ViewEditor::joinCB(Widget, XtPointer data)
{
	PaperCallbackStruct* cb = (PaperCallbackStruct*)data;
	cb->r3->data = cb->r1->data;
	updateDrawers();
}

void ViewEditor::groupCB(Widget, XtPointer data)
{
	PaperCallbackStruct* cb = (PaperCallbackStruct*)data;
	PageView *p1 = ((PageView*)cb->r1->data);
	PageView *p2 = ((PageView*)cb->r2->data);
	  
	if(cb->r2->data == 0 && cb->r1->data != 0) 
		cb->r2->data = cb->r1->data;
	  
	cb->r1->data = 0;

	updateDrawers();
}

void ViewEditor::ungroupCB(Widget, XtPointer data)
{
	PaperCallbackStruct* cb = (PaperCallbackStruct*)data;
	cb->r2->data = cb->r1->data;
	updateDrawers();
}

void ViewEditor::deleteCB(Widget, XtPointer data)
{
	PaperCallbackStruct* cb = (PaperCallbackStruct*)data;
	updateDrawers();
}

void ViewEditor::changeCB(Widget, XtPointer data)
{
}

//-----------------------------------------------------------

void ViewEditor::update()
{
	PaperUpdate(paper_);
	updateDrawers();
}

void ViewEditor::insetSelection(double t,double l,double b,double r)
{
	PaperInsetSelection(paper_,t,l,b,r);
	updateDrawers();
}

void ViewEditor::setPaperSize(double h,double v)
{
	PaperSetSize(paper_,h,v);
	updateDrawers();
}

void ViewEditor::getPaperSize(double& h,double& v)
{
	PaperGetSize(paper_,&h,&v);
}

void ViewEditor::splitSelection(int h,int v,double mh,double mv)
{
	PaperSplitSelection(paper_,h,v,mv,mv);
	updateDrawers();
}

void ViewEditor::joinSelection()
{
	PaperJoinSelection(paper_);
	updateDrawers();
}

void ViewEditor::groupSelection()
{
	PaperGroupSelection(paper_);
	updateDrawers();
}

void ViewEditor::ungroupSelection()
{
	PaperUngroupSelection(paper_);
	updateDrawers();
}

void ViewEditor::alignTop()
{
	PaperAlignTop(paper_);
	updateDrawers();
}

void ViewEditor::alignLeft()
{
	PaperAlignLeft(paper_);
	updateDrawers();
}

void ViewEditor::alignRight()
{
	PaperAlignRight(paper_);
	updateDrawers();
}

void ViewEditor::alignBottom()
{
	PaperAlignBottom(paper_);
	updateDrawers();
}

void ViewEditor::distributeHorizontally()
{
	PaperDistributeHorizontally(paper_);
	updateDrawers();
}

void ViewEditor::distributeVertically()
{
	PaperDistributeVertically(paper_);
	updateDrawers();
}

string ViewEditor::alternateEditor()
{
  return "SimpleEditor";
}

//-----------------------------------------------------------

static EditorMaker<ViewEditor> editorMaker("ViewEditor");

static EditorButtonMaker<ViewEditor> b1("ALIGN_BOTTOM",&ViewEditor::alignBottom,1);
static EditorButtonMaker<ViewEditor> b2("ALIGN_TOP",  &ViewEditor::alignTop,2);
static EditorButtonMaker<ViewEditor> b3("ALIGN_LEFT", &ViewEditor::alignLeft,3);
static EditorButtonMaker<ViewEditor> b4("ALIGN_RIGHT",&ViewEditor::alignRight,4);

static EditorButtonMaker<ViewEditor> b5("DISTR_VCENTER",&ViewEditor::distributeVertically,5);
static EditorButtonMaker<ViewEditor> b6("DISTR_HCENTER",&ViewEditor::distributeHorizontally,6);


void ViewEditor::savePages(IconObject* o,Request& r)
{
	FolderH pageFolder = o->embeddedFolder("Pages",true);

	// Clean up old stuff
	pageFolder->destroy();
	pageFolder = o->embeddedFolder("Pages",true);


	vector<IconObjectH> pages;

	PaperRectangle* p  = PaperGetPages(paper_);
	while(p)
	{
		Request r("PLOT_PAGE");
		
		saveRectangle(r,p);

		//r("ROWS") = 1;
		//r("COLUMNS") = 1;

		IconObjectH page      = IconFactory::create(pageFolder,r);
		FolderH subPageFolder = page->embeddedFolder("Sub Pages",true);
		pages.push_back(page);

		vector<IconObjectH> subpages;

		PaperRectangle* s = p->kids;
		while(s)
		{
			Request r("PLOT_SUBPAGE");
			saveRectangle(r,s);
			subpages.push_back(IconFactory::create(subPageFolder,r));
			s = s->next;
		}

		page->subObjects("SUB_PAGES",subpages,r);

		if(p->data)
		{
		    IconObject* view = ((PageView*)p->data)->view();
		    if(view) page->subObjects("VIEW",vector<IconObjectH>(1,view),r);
		}		  

		page->request(r);
		p = p->next;
	}

	current_->subObjects("PAGES",pages,r);
	current_->request(r);
}

void ViewEditor::setGrid(double x,double y)
{
	PaperSetGrid(paper_,x,y);
}

void ViewEditor::getGrid(double& x,double& y)
{
	PaperGetGrid(paper_,&x,&y);
}

void ViewEditor::showGrid(bool on)
{
	PaperShowGrid(paper_,on);
}

void ViewEditor::snapGrid(bool on)
{
	PaperSnapGrid(paper_,on);
}

void ViewEditor::deleteSelection()
{
	PaperDeleteSelection(paper_);
}

void ViewEditor::addPage()
{
	PaperRectangle* r =  PaperAddPage(paper_,0,0,0.25,0.25);
}
