% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/round.r
\name{round_date}
\alias{round_date}
\alias{floor_date}
\alias{ceiling_date}
\title{Round, floor and ceiling methods for date-time objects}
\usage{
round_date(
  x,
  unit = "second",
  week_start = getOption("lubridate.week.start", 7)
)

floor_date(
  x,
  unit = "seconds",
  week_start = getOption("lubridate.week.start", 7)
)

ceiling_date(
  x,
  unit = "seconds",
  change_on_boundary = NULL,
  week_start = getOption("lubridate.week.start", 7)
)
}
\arguments{
\item{x}{a vector of date-time objects}

\item{unit}{a string, \code{Period} object or a date-time object. When a singleton string,
it specifies a time unit or a multiple of a unit to be rounded to. Valid base units
are \code{second}, \code{minute}, \code{hour}, \code{day}, \code{week}, \code{month}, \code{bimonth}, \code{quarter},
\code{season}, \code{halfyear} and \code{year}. Arbitrary unique English abbreviations as in the
\code{\link[=period]{period()}} constructor are allowed. Rounding to multiples of units (except weeks)
is supported.

When \code{unit} is a \code{Period} object, it is first converted to a string representation
which might not be in the same units as the constructor. For example \code{weeks(1)} is
converted to "7d 0H 0M 0S". Thus, always check the string representation of the
period before passing to this function.

When \code{unit} is a date-time object rounding is done to the nearest of the
elements in \code{unit}. If range of \code{unit} vector does not cover the range of
\code{x} \code{ceiling_date()} and \code{floor_date()} round to the \code{max(x)} and \code{min(x)}
for elements that fall outside of \code{range(unit)}.}

\item{week_start}{week start day (Default is 7, Sunday. Set \code{lubridate.week.start} to
override). Full or abbreviated names of the days of the week can be in English or
as provided by the current locale.}

\item{change_on_boundary}{if this is \code{NULL} (the default), instants on the
boundary remain unchanged, but \code{Date} objects are rounded up to the next
boundary. If this is \code{TRUE}, instants on the boundary are rounded up to the
next boundary. If this is \code{FALSE}, nothing on the boundary is rounded up at
all. This was the default for \pkg{lubridate} prior to \code{v1.6.0}. See
section \verb{Rounding Up Date Objects} below for more details.}
}
\value{
When \code{unit} is a string, return a Date object if \code{x} is a Date and
\code{unit} is larger or equal than "day", otherwise a POSIXct object. When
\code{unit} is a date-time object, return a date-time object of the same class
and same time zone as \code{unit}.
}
\description{
\code{round_date()} takes a date-time object and time unit, and rounds it to the nearest value
of the specified time unit. For rounding date-times which are exactly halfway
between two consecutive units, the convention is to round up. Note that this
is in line with the behavior of R's \code{\link[base:round.POSIXt]{base::round.POSIXt()}} function
but does not follow the convention of the base \code{\link[base:Round]{base::round()}} function
which "rounds to the even digit", as per IEC 60559.

Rounding to the nearest unit or multiple of a unit is supported. All
meaningful specifications in the English language are supported - secs, min,
mins, 2 minutes, 3 years etc.

Rounding to fractional seconds is also supported. Please note that rounding to
fractions smaller than 1 second can lead to large precision errors due to the
floating point representation of the POSIXct objects. See examples.

\code{floor_date()} takes a date-time object and rounds it down to the nearest
boundary of the specified time unit.

\code{ceiling_date()} takes a date-time object and rounds it up to the nearest
boundary of the specified time unit.
}
\details{
In \pkg{lubridate}, functions that round date-time objects try to
preserve the class of the input object whenever possible. This is done by
first rounding to an instant, and then converting to the original class as per
usual R conventions.
}
\section{Rounding Up Date Objects}{


By default, rounding up \code{Date} objects follows 3 steps:
\enumerate{
\item Convert to an instant representing lower bound of the Date:
\code{2000-01-01} --> \verb{2000-01-01 00:00:00}
\item Round up to the \strong{next} closest rounding unit boundary. For example,
if the rounding unit is \code{month} then next closest boundary of \code{2000-01-01}
is \verb{2000-02-01 00:00:00}.

The motivation for this is that the "partial" \code{2000-01-01} is conceptually
an interval (\verb{2000-01-01 00:00:00} -- \verb{2000-01-02 00:00:00}) and the day
hasn't started clocking yet at the exact boundary \code{00:00:00}. Thus, it
seems wrong to round a day to its lower boundary.

Behavior on the boundary can be changed by setting
\code{change_on_boundary} to \code{TRUE} or \code{FALSE}.
\item If the rounding unit is smaller than a day, return the instant from step 2
(\code{POSIXct}), otherwise convert to and return a \code{Date} object.
}
}

\examples{

## print fractional seconds
options(digits.secs = 6)

x <- ymd_hms("2009-08-03 12:01:59.23")
round_date(x, ".5s")
round_date(x, "sec")
round_date(x, "second")
round_date(x, "minute")
round_date(x, "5 mins")
round_date(x, "hour")
round_date(x, "2 hours")
round_date(x, "day")
round_date(x, "week")
round_date(x, "month")
round_date(x, "bimonth")
round_date(x, "quarter") == round_date(x, "3 months")
round_date(x, "halfyear")
round_date(x, "year")

x <- ymd_hms("2009-08-03 12:01:59.23")
floor_date(x, ".1s")
floor_date(x, "second")
floor_date(x, "minute")
floor_date(x, "hour")
floor_date(x, "day")
floor_date(x, "week")
floor_date(x, "month")
floor_date(x, "bimonth")
floor_date(x, "quarter")
floor_date(x, "season")
floor_date(x, "halfyear")
floor_date(x, "year")

x <- ymd_hms("2009-08-03 12:01:59.23")
ceiling_date(x, ".1 sec") # imprecise representation at 0.1 sec !!!
ceiling_date(x, "second")
ceiling_date(x, "minute")
ceiling_date(x, "5 mins")
ceiling_date(x, "hour")
ceiling_date(x, "day")
ceiling_date(x, "week")
ceiling_date(x, "month")
ceiling_date(x, "bimonth") == ceiling_date(x, "2 months")
ceiling_date(x, "quarter")
ceiling_date(x, "season")
ceiling_date(x, "halfyear")
ceiling_date(x, "year")

## Period unit argument
floor_date(x, days(2))
floor_date(x, years(1))

## As of R 3.4.2 POSIXct printing of fractional numbers is wrong
as.POSIXct("2009-08-03 12:01:59.3") ## -> "2009-08-03 12:01:59.2 CEST"
ceiling_date(x, ".1 sec") ## -> "2009-08-03 12:01:59.2 CEST"

## behaviour of `change_on_boundary`
## As per default behaviour `NULL`, instants on the boundary remain the
## same but dates are rounded up
ceiling_date(ymd_hms("2000-01-01 00:00:00"), "month")
ceiling_date(ymd("2000-01-01"), "month")

## If `TRUE`, both instants and dates on the boundary are rounded up
ceiling_date(ymd_hms("2000-01-01 00:00:00"), "month", change_on_boundary = TRUE)
ceiling_date(ymd("2000-01-01"), "month")

## If `FALSE`, both instants and dates on the boundary remain the same
ceiling_date(ymd_hms("2000-01-01 00:00:00"), "month", change_on_boundary = FALSE)
ceiling_date(ymd("2000-01-01"), "month")

x <- ymd_hms("2000-01-01 00:00:00")
ceiling_date(x, "month")
ceiling_date(x, "month", change_on_boundary = TRUE)

## For Date objects first day of the month is not on the
## "boundary". change_on_boundary applies to instants only.
x <- ymd("2000-01-01")
ceiling_date(x, "month")
ceiling_date(x, "month", change_on_boundary = TRUE)
}
\seealso{
\code{\link[base:Round]{base::round()}}
}
\keyword{chron}
\keyword{manip}
