#ifndef _RHEO_CAD_BEZIER_H
#define _RHEO_CAD_BEZIER_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/cad_element.h"
namespace rheolef { 

//
// Evaluate a Bezier curve/surface using Horner algorithm.
//
// from qmg-2.0.2/src/common/qpatchmath.cpp: get_real_point()
//
// see also : 
//  J.-J. Risler
//  Methodes mathematiques pour la CAO
//  Masson, coll. RMA, vol. 18, 1991
//


// Evaluate a one-variable Bezier curve using Horner algorithm.

template<class PointArray, class Patch, class T, class Size> 
T
eval_bezier_edge (
	const T& param,
	const Patch& cp_idx, // control point indexes,
    	Size degree,
        Size k,		     // active coord index in 0...phys_dim-1
        const PointArray& x)
{
    T u = 1.0 - param;
    T pwrv = 1.0; 
    T result = 0.0;
    Size binom = 1;
    for (Size j = 0; j < degree + 1; ++j) {
      result *= u;
      result += T(binom) * pwrv * x[cp_idx[j]][k];
      pwrv *= param;
      binom *= (degree - j);
      binom /= (j + 1);
    }
    return result;
}

// Compute real coord from param coord using Horner's algorithm.

template<class PointArray, class Patch, class T, class Size> 
T
eval_bezier_triangle (
	const T& param0,
	const T& param1,
	const Patch& cp_idx, // control point indexes,
    	Size degree,
	Size k, 
        const PointArray& x)
{
    T p0 = param0;
    T p1 = param1;
    T p2 = 1.0 - p0 - p1;
    Size outer_binom = 1;
    T pwrv = 1.0;
    T result = 0.0;
    for (Size row = degree; row >= 0; --row) {
      Size base_idx = row * (row + 1) / 2;
      Size inner_binom = 1;
      T pwru = 1.0;
      T row_result = 0.0;
      for (Size col = 0; col <= row; ++col) {
        row_result *= p2;
        row_result += T(inner_binom) * x[cp_idx[base_idx + col]][k] * pwru;
        pwru *= p0;
        inner_binom *= (row - col);
        inner_binom /= (col + 1);
      }
      result += row_result * pwrv * T(outer_binom);
      outer_binom *= row;
      outer_binom /= (degree - row + 1);
      pwrv *= p1;
    }
    return result;
}

// Horner algorithm for evaluating bezier quad (parametric
// to real conversion)

template<class PointArray, class Patch, class T, class Size> 
T
eval_bezier_quadrangle (
	const T& param0,
	const T& param1,
	const Patch& cp_idx, // control point indexes,
    	Size degree1,
    	Size degree2,
        Size k,
        const PointArray& x)
{
    T u = param0;
    T oppu = 1.0 - u;
    T v = param1;
    T oppv = 1.0 - v;

    Size outer_binom = 1;
    T outer_result = 0.0;
    T pwrv = 1.0;
    for (Size d2 = 0; d2 <= degree2; ++d2) {
      // Compute the inner coefficient.
      Size inner_binom = 1;
      T pwru = 1.0;
      T inner_result = 0.0;
      for (Size d1 = 0; d1 <= degree1; ++d1) {
        inner_result *= oppu;
        inner_result += x[cp_idx[d2 * (degree1 + 1) + d1]][k] * T(inner_binom) * pwru;
        pwru *= u;
        inner_binom *= (degree1 - d1);
        inner_binom /= (d1 + 1);
      }
      outer_result *= oppv;
      outer_result += inner_result * T(outer_binom) * pwrv;
      pwrv *= v;
      outer_binom *= (degree2 - d2);
      outer_binom /= (d2 + 1);
    }
    return outer_result;
}
}// namespace rheolef
#endif // _RHEO_CAD_BEZIER_H
