/*
 *@BEGIN LICENSE
 *
 * PSI4: an ab initio quantum chemistry software package
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *@END LICENSE
 */

#include <libmints/vector.h>
#include "FT97_Cfunctional.h"
#include "utility.h"
#include <cmath>

using namespace psi;

namespace psi {

FT97_CFunctional::FT97_CFunctional()
{
    name_ = "FT97_C";
    description_ = "    FT97 Correlation (Involves Ei functions)\n";
    citation_ = "    M. Filatov and W. Theil, Int. J. Quant. Chem., 62, 603-616, 1997\n";
    alpha_ = 1.0;
    omega_ = 0.0;
    lrc_ = false;
    gga_ = true;
    meta_ = false;
    parameters_["c0"] =   3.1090690869654901E-02;
    parameters_["c"] =   6.2035049089939986E-01;
    parameters_["tspi_m13"] =   2.0678349696646670E-01;
    parameters_["a1"] =   1.6221187669999999E+00;
    parameters_["a2"] =   4.8995807600000002E-01;
    parameters_["a3"] =   1.3790219410000000E+00;
    parameters_["a4"] =   4.9462813529999998E+00;
    parameters_["a5"] =   3.6006120589999999E+00;
    parameters_["kaa0"] =   1.2008017740000001E+00;
    parameters_["kaa1"] =  -8.1290434499999997E-01;
    parameters_["kaa2"] =   8.5961444499999995E-01;
    parameters_["raa1"] =   6.5563882299999998E-01;
    parameters_["raa2"] =   1.0893388479999999E+00;
    parameters_["kab0"] =   1.2915510740000000E+00;
    parameters_["kab1"] =  -3.4906417299999998E-01;
    parameters_["rab1"] =   8.3275879999999997E-02;
    parameters_["k1"] =   9.3901599999999996E-01;
    parameters_["k2"] =   1.7331700000000001E+00;
}
FT97_CFunctional::~FT97_CFunctional()
{
}
void FT97_CFunctional::compute_functional(const std::map<std::string,SharedVector>& in, const std::map<std::string,SharedVector>& out, int npoints, int deriv, double alpha)
{
    double c0 = parameters_["c0"];
    double c = parameters_["c"];
    double tspi_m13 = parameters_["tspi_m13"];
    double a1 = parameters_["a1"];
    double a2 = parameters_["a2"];
    double a3 = parameters_["a3"];
    double a4 = parameters_["a4"];
    double a5 = parameters_["a5"];
    double kaa0 = parameters_["kaa0"];
    double kaa1 = parameters_["kaa1"];
    double kaa2 = parameters_["kaa2"];
    double raa1 = parameters_["raa1"];
    double raa2 = parameters_["raa2"];
    double kab0 = parameters_["kab0"];
    double kab1 = parameters_["kab1"];
    double rab1 = parameters_["rab1"];
    double k1 = parameters_["k1"];
    double k2 = parameters_["k2"];

    // Overall scale factor
    double scale = alpha_ * alpha;

    // => Input variables <= //

    double* rho_ap = NULL;
    double* rho_bp = NULL;
    double* gamma_aap = NULL;
    double* gamma_abp = NULL;
    double* gamma_bbp = NULL;
    double* tau_ap = NULL;
    double* tau_bp = NULL;

    if (true) {
        rho_ap = in.find("RHO_A")->second->pointer();
        rho_bp = in.find("RHO_B")->second->pointer();
    }
    if (gga_) {  
        gamma_aap = in.find("GAMMA_AA")->second->pointer();
        gamma_abp = in.find("GAMMA_AB")->second->pointer();
        gamma_bbp = in.find("GAMMA_BB")->second->pointer();
    } 
    if (meta_)  {
        tau_ap = in.find("TAU_A")->second->pointer();
        tau_bp = in.find("TAU_B")->second->pointer();
    }

    // => Outut variables <= //

    double* v = NULL;

    double* v_rho_a = NULL;
    double* v_rho_b = NULL;
    double* v_gamma_aa = NULL;
    double* v_gamma_ab = NULL;
    double* v_gamma_bb = NULL;
    double* v_tau_a = NULL;
    double* v_tau_b = NULL;
     
    double* v_rho_a_rho_a = NULL;
    double* v_rho_a_rho_b = NULL;
    double* v_rho_b_rho_b = NULL;
    double* v_gamma_aa_gamma_aa = NULL;
    double* v_gamma_aa_gamma_ab = NULL;
    double* v_gamma_aa_gamma_bb = NULL;
    double* v_gamma_ab_gamma_ab = NULL;
    double* v_gamma_ab_gamma_bb = NULL;
    double* v_gamma_bb_gamma_bb = NULL;
    double* v_tau_a_tau_a = NULL;
    double* v_tau_a_tau_b = NULL;
    double* v_tau_b_tau_b = NULL;
    double* v_rho_a_gamma_aa = NULL;
    double* v_rho_a_gamma_ab = NULL;
    double* v_rho_a_gamma_bb = NULL;
    double* v_rho_b_gamma_aa = NULL;
    double* v_rho_b_gamma_ab = NULL;
    double* v_rho_b_gamma_bb = NULL;
    double* v_rho_a_tau_a = NULL;
    double* v_rho_a_tau_b = NULL;
    double* v_rho_b_tau_a = NULL;
    double* v_rho_b_tau_b = NULL;
    double* v_gamma_aa_tau_a = NULL;
    double* v_gamma_aa_tau_b = NULL;
    double* v_gamma_ab_tau_a = NULL;
    double* v_gamma_ab_tau_b = NULL;
    double* v_gamma_bb_tau_a = NULL;
    double* v_gamma_bb_tau_b = NULL;

    if (deriv >= 0) {
        v = out.find("V")->second->pointer();
    } 
    if (deriv >= 1) {
        if (true) {
            v_rho_a = out.find("V_RHO_A")->second->pointer();
            v_rho_b = out.find("V_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa = out.find("V_GAMMA_AA")->second->pointer();
            v_gamma_ab = out.find("V_GAMMA_AB")->second->pointer();
            v_gamma_bb = out.find("V_GAMMA_BB")->second->pointer();
        }
        if (meta_) {    
            v_tau_a = out.find("V_TAU_A")->second->pointer();
            v_tau_b = out.find("V_TAU_B")->second->pointer();
        }
    }
    if (deriv >= 2) {
        if (true) {
            v_rho_a_rho_a = out.find("V_RHO_A_RHO_A")->second->pointer();
            v_rho_a_rho_b = out.find("V_RHO_A_RHO_B")->second->pointer();
            v_rho_b_rho_b = out.find("V_RHO_B_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa_gamma_aa = out.find("V_GAMMA_AA_GAMMA_AA")->second->pointer();
            v_gamma_aa_gamma_ab = out.find("V_GAMMA_AA_GAMMA_AB")->second->pointer();
            v_gamma_aa_gamma_bb = out.find("V_GAMMA_AA_GAMMA_BB")->second->pointer();
            v_gamma_ab_gamma_ab = out.find("V_GAMMA_AB_GAMMA_AB")->second->pointer();
            v_gamma_ab_gamma_bb = out.find("V_GAMMA_AB_GAMMA_BB")->second->pointer();
            v_gamma_bb_gamma_bb = out.find("V_GAMMA_BB_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_tau_a_tau_a = out.find("V_TAU_A_TAU_A")->second->pointer();
            v_tau_a_tau_b = out.find("V_TAU_A_TAU_B")->second->pointer();
            v_tau_b_tau_b = out.find("V_TAU_B_TAU_B")->second->pointer();
        }
        if (gga_) {
            v_rho_a_gamma_aa = out.find("V_RHO_A_GAMMA_AA")->second->pointer();
            v_rho_a_gamma_ab = out.find("V_RHO_A_GAMMA_AB")->second->pointer();
            v_rho_a_gamma_bb = out.find("V_RHO_A_GAMMA_BB")->second->pointer();
            v_rho_b_gamma_aa = out.find("V_RHO_B_GAMMA_AA")->second->pointer();
            v_rho_b_gamma_ab = out.find("V_RHO_B_GAMMA_AB")->second->pointer();
            v_rho_b_gamma_bb = out.find("V_RHO_B_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_rho_a_tau_a = out.find("V_RHO_A_TAU_A")->second->pointer();
            v_rho_a_tau_b = out.find("V_RHO_A_TAU_B")->second->pointer();
            v_rho_b_tau_a = out.find("V_RHO_B_TAU_A")->second->pointer();
            v_rho_b_tau_b = out.find("V_RHO_B_TAU_B")->second->pointer();
        }
        if (gga_ && meta_) {
            v_gamma_aa_tau_a = out.find("V_GAMMA_AA_TAU_A")->second->pointer();
            v_gamma_aa_tau_b = out.find("V_GAMMA_AA_TAU_B")->second->pointer();
            v_gamma_ab_tau_a = out.find("V_GAMMA_AB_TAU_A")->second->pointer();
            v_gamma_ab_tau_b = out.find("V_GAMMA_AB_TAU_B")->second->pointer();
            v_gamma_bb_tau_a = out.find("V_GAMMA_BB_TAU_A")->second->pointer();
            v_gamma_bb_tau_b = out.find("V_GAMMA_BB_TAU_B")->second->pointer();
        }
    }

    // => Loop over points <= //

    for (int Q = 0; Q < npoints; Q++) {

        // Input variables 
        double rho_a;
        double rho_b;
        double gamma_aa;
        double gamma_ab;
        double gamma_bb;
        double tau_a;
        double tau_b;

        if (true) {
            rho_a = rho_ap[Q];
            rho_b = rho_bp[Q];
        }        
        if (gga_) {
            gamma_aa = gamma_aap[Q];
            gamma_ab = gamma_abp[Q];
            gamma_bb = gamma_bbp[Q];
        }        
        if (meta_) {
            tau_a = tau_ap[Q];
            tau_b = tau_bp[Q];
        }        

        // Definitions (asymptotics to prevent numerical problems)
        if (rho_a < lsda_cutoff_ && rho_b < lsda_cutoff_) {
            continue;
        } else if (rho_a < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t19597 = tspi_m13*tspi_m13;
                double t19598 = 1.0/pow(rho_b,1.0/3.0);
                double t19599 = a2*a2;
                double t19600 = gamma_bb*gamma_bb;
                double t19601 = 1.0/pow(rho_b,1.6E1/3.0);
                double t19602 = t19597*t19597;
                double t19603 = c*t19598;
                double t19604 = a4*a4;
                double t19605 = 1.0/c;
                double t19606 = 1.0/pow(rho_b,7.0/3.0);
                double t19607 = sqrt(t19603);
                double t19608 = t19600*t19601*t19602*t19604*2.0;
                double t19609 = exp(t19608);
                double t19610 = t19600*t19601*t19602*t19604;
                double t19611 = t19610+1.0;
                double t19612 = 1.0/(t19611*t19611);
                double t19613 = a5*gamma_bb*t19605*t19606*t19597;
                double t19614 = t19613+1.0;
                double t19625 = raa2*t19607;
                double t19615 = exp(-t19625);
                double t19616 = t19615-1.0;
                double t19617 = kaa2*t19616;
                double t19618 = pow(t19603,2.0/5.0);
                double t19626 = raa1*t19618;
                double t19619 = exp(-t19626);
                double t19620 = t19619-1.0;
                double t19621 = kaa1*t19620;
                double t19622 = -kaa0+t19621+t19617;
                double t19623 = 1.0/(t19622*t19622);
                double t19624 = c*c0*t19612*t19614*t19623*t19609*t19598*(2.0/3.0);
                double t19627;
                double t19628;
                if (t19624 > expei_cutoff) {
                    t19627 = 1.0;
                    t19628 = expei(t19624);
                } else {
                    t19627 = exp(t19624);
                    t19628 = Ei(-t19624);
                }
                double t19629 = 1.0/t19611;
                double t19630 = c*c0*t19614*t19623*t19609*t19598*(2.0/3.0);
                double t19631 = sqrt(t19630);
                v[Q] += scale * (c0*rho_b*exp(-(c*c)*1.0/pow(rho_b,2.0/3.0)*1.0/pow(k1*t19607+c*k2*t19598,2.0))*heaviside(c*c0*t19598*(-2.0/3.0)+exp(-t19600*t19601*t19602*t19599)*1.0/sqrt(a3*gamma_bb*t19605*t19606*t19597+1.0)*(kab0-kab1*(exp(-rab1*pow(t19603,4.0/5.0))-1.0))*(t19600*t19601*t19602*t19599+a1*gamma_bb*1.0/pow(rho_b,8.0/3.0)*t19597+1.0)*1.0E5)*(t19627*t19628+((c*c0*t19612*t19614*t19623*t19609*t19627*t19628*t19598*(2.0/3.0)+1.0)*(t19631*t19629*4.0+c*c0*t19612*t19614*t19623*t19609*t19598*(8.0/3.0)+6.0))/(t19631*t19629*6.0+c*c0*t19612*t19614*t19623*t19609*t19598*4.0+3.0))*(1.0/2.0));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t19634 = tspi_m13*tspi_m13;
                double t19635 = 1.0/pow(rho_b,1.0/3.0);
                double t19636 = a2*a2;
                double t19637 = gamma_bb*gamma_bb;
                double t19638 = 1.0/pow(rho_b,1.6E1/3.0);
                double t19639 = t19634*t19634;
                double t19640 = c*t19635;
                double t19641 = a4*a4;
                double t19642 = 1.0/c;
                double t19643 = 1.0/pow(rho_b,7.0/3.0);
                double t19644 = sqrt(t19640);
                double t19645 = t19641*t19637*t19638*t19639*2.0;
                double t19646 = exp(t19645);
                double t19647 = t19641*t19637*t19638*t19639;
                double t19648 = t19647+1.0;
                double t19649 = 1.0/(t19648*t19648);
                double t19650 = a5*gamma_bb*t19642*t19634*t19643;
                double t19651 = t19650+1.0;
                double t19662 = raa2*t19644;
                double t19652 = exp(-t19662);
                double t19653 = t19652-1.0;
                double t19654 = kaa2*t19653;
                double t19655 = pow(t19640,2.0/5.0);
                double t19663 = raa1*t19655;
                double t19656 = exp(-t19663);
                double t19657 = t19656-1.0;
                double t19658 = kaa1*t19657;
                double t19659 = -kaa0+t19654+t19658;
                double t19660 = 1.0/(t19659*t19659);
                double t19661 = c*c0*t19651*t19660*t19635*t19646*t19649*(2.0/3.0);
                double t19664;
                double t19665;
                if (t19661 > expei_cutoff) {
                    t19664 = 1.0;
                    t19665 = expei(t19661);
                } else {
                    t19664 = exp(t19661);
                    t19665 = Ei(-t19661);
                }
                double t19666 = 1.0/t19648;
                double t19667 = c*c0*t19651*t19660*t19635*t19646*(2.0/3.0);
                double t19668 = sqrt(t19667);
                double t19669 = t19636*t19637*t19638*t19639;
                double t19670 = pow(t19640,4.0/5.0);
                double t19720 = rab1*t19670;
                double t19671 = exp(-t19720);
                double t19672 = t19671-1.0;
                double t19721 = kab1*t19672;
                double t19673 = kab0-t19721;
                double t19674 = a3*gamma_bb*t19642*t19634*t19643;
                double t19675 = t19674+1.0;
                double t19676 = 1.0/sqrt(t19675);
                double t19677 = 1.0/pow(rho_b,8.0/3.0);
                double t19678 = a1*gamma_bb*t19634*t19677;
                double t19679 = t19669+t19678+1.0;
                double t19680 = c*c;
                double t19681 = 1.0/pow(rho_b,2.0/3.0);
                double t19682 = k1*t19644;
                double t19683 = c*k2*t19635;
                double t19684 = t19682+t19683;
                double t19685 = 1.0/(t19684*t19684);
                double t19725 = t19680*t19681*t19685;
                double t19686 = exp(-t19725);
                double t19687 = c*c0*t19651*t19660*t19635*t19646*t19664*t19665*t19649*(2.0/3.0);
                double t19688 = t19687+1.0;
                double t19689 = t19666*t19668*6.0;
                double t19690 = c*c0*t19651*t19660*t19635*t19646*t19649*4.0;
                double t19691 = t19690+t19689+3.0;
                double t19692 = 1.0/t19691;
                double t19693 = 1.0/pow(rho_b,4.0/3.0);
                double t19694 = 1.0/pow(rho_b,1.1E1/3.0);
                double t19695 = 1.0/sqrt(t19640);
                double t19696 = c*kaa2*raa2*t19652*t19693*t19695*(1.0/6.0);
                double t19697 = 1.0/pow(t19640,3.0/5.0);
                double t19698 = c*kaa1*raa1*t19656*t19693*t19697*(2.0/1.5E1);
                double t19699 = t19696+t19698;
                double t19700 = 1.0/(t19659*t19659*t19659);
                double t19701 = 1.0/pow(rho_b,2.0E1/3.0);
                double t19702 = 1.0/(t19648*t19648*t19648);
                double t19703 = t19666*t19668*4.0;
                double t19704 = c*c0*t19651*t19660*t19635*t19646*t19649*(8.0/3.0);
                double t19705 = t19703+t19704+6.0;
                double t19706 = c*c0*t19651*t19660*t19646*t19693*t19649*(2.0/9.0);
                double t19707 = a5*c0*gamma_bb*t19660*t19634*t19646*t19649*t19694*(1.4E1/9.0);
                double t19708 = c*c0*t19700*t19651*t19635*t19646*t19649*t19699*(4.0/3.0);
                double t19709 = c*c0*t19701*t19641*t19651*t19660*t19637*t19646*t19639*t19649*(6.4E1/9.0);
                double t19711 = c*c0*t19701*t19702*t19641*t19651*t19660*t19637*t19646*t19639*(6.4E1/9.0);
                double t19710 = -t19711+t19706+t19707+t19708+t19709;
                double t19712 = c*c0*t19651*t19660*t19646*t19693*(2.0/9.0);
                double t19713 = a5*c0*gamma_bb*t19660*t19634*t19646*t19694*(1.4E1/9.0);
                double t19714 = c*c0*t19700*t19651*t19635*t19646*t19699*(4.0/3.0);
                double t19715 = c*c0*t19701*t19641*t19651*t19660*t19637*t19646*t19639*(6.4E1/9.0);
                double t19716 = t19712+t19713+t19714+t19715;
                double t19717 = 1.0/sqrt(t19667);
                double t19718 = 1.0/pow(rho_b,1.9E1/3.0);
                double t19719 = exp(-t19669);
                double t19722 = t19673*t19719*t19676*t19679*1.0E5;
                double t19729 = c*c0*t19635*(2.0/3.0);
                double t19723 = t19722-t19729;
                double t19724 = heaviside(t19723);
                double t19726 = t19664*t19665;
                double t19727 = t19705*t19692*t19688;
                double t19728 = t19726+t19727;
                v_rho_b[Q] += scale * (c0*t19728*t19686*heaviside(c*c0*t19635*(-2.0/3.0)+t19673*t19676*t19679*exp(-t19636*t19637*t19638*t19639)*1.0E5)*(1.0/2.0)-c0*rho_b*t19724*t19686*(t19692*t19688*(t19716*t19717*t19666*2.0-t19641*t19637*t19718*t19639*t19649*t19668*(6.4E1/3.0)+c*c0*t19651*t19660*t19646*t19693*t19649*(8.0/9.0)+a5*c0*gamma_bb*t19660*t19634*t19646*t19649*t19694*(5.6E1/9.0)+c*c0*t19700*t19651*t19635*t19646*t19649*t19699*(1.6E1/3.0)-c*c0*t19701*t19702*t19641*t19651*t19660*t19637*t19646*t19639*(2.56E2/9.0)+c*c0*t19701*t19641*t19651*t19660*t19637*t19646*t19639*t19649*(2.56E2/9.0))+t19705*t19692*(-t19711+t19706+t19707+t19708+t19709+c*c0*t19651*t19660*t19646*t19664*t19665*t19693*t19649*(2.0/9.0)+a5*c0*gamma_bb*t19660*t19634*t19646*t19664*t19665*t19649*t19694*(1.4E1/9.0)+c*c0*t19710*t19651*t19660*t19635*t19646*t19664*t19665*t19649*(2.0/3.0)+c*c0*t19700*t19651*t19635*t19646*t19664*t19665*t19649*t19699*(4.0/3.0)-c*c0*t19701*t19702*t19641*t19651*t19660*t19637*t19646*t19664*t19665*t19639*(6.4E1/9.0)+c*c0*t19701*t19641*t19651*t19660*t19637*t19646*t19664*t19665*t19639*t19649*(6.4E1/9.0))+t19710*t19664*t19665-t19705*1.0/(t19691*t19691)*t19688*(t19716*t19717*t19666*3.0-t19641*t19637*t19718*t19639*t19649*t19668*3.2E1+c*c0*t19651*t19660*t19646*t19693*t19649*(4.0/3.0)+a5*c0*gamma_bb*t19660*t19634*t19646*t19649*t19694*(2.8E1/3.0)+c*c0*t19700*t19651*t19635*t19646*t19649*t19699*8.0-c*c0*t19701*t19702*t19641*t19651*t19660*t19637*t19646*t19639*(1.28E2/3.0)+c*c0*t19701*t19641*t19651*t19660*t19637*t19646*t19639*t19649*(1.28E2/3.0))+(pow(rho_b,1.0/3.0)*t19710*t19642*(t19648*t19648)*(t19659*t19659)*exp(-t19645)*(3.0/2.0))/(c0*t19651))*(1.0/2.0)+c0*rho_b*t19724*t19728*t19686*(1.0/pow(rho_b,5.0/3.0)*t19680*t19685*(2.0/3.0)-t19680*t19681*1.0/(t19684*t19684*t19684)*(c*k2*t19693*(1.0/3.0)+c*k1*t19693*t19695*(1.0/6.0))*2.0)*(1.0/2.0)+c0*rho_b*t19728*t19686*dirac(-t19722+t19729)*(c*c0*t19693*(2.0/9.0)-t19673*t19719*t19676*(a1*gamma_bb*t19634*t19694*(8.0/3.0)+t19636*t19637*t19718*t19639*(1.6E1/3.0))*1.0E5+t19636*t19637*t19673*t19718*t19719*t19639*t19676*t19679*(1.6E6/3.0)+a3*gamma_bb*1.0/pow(rho_b,1.0E1/3.0)*t19642*t19634*t19673*t19719*1.0/pow(t19675,3.0/2.0)*t19679*(3.5E5/3.0)-c*kab1*rab1*1.0/pow(t19640,1.0/5.0)*t19671*t19719*t19693*t19676*t19679*(8.0E4/3.0))*(1.0/2.0));
            }
            
            // v_gamma_bb
            if (deriv >= 1) {
                double t19733 = tspi_m13*tspi_m13;
                double t19734 = 1.0/pow(rho_b,1.0/3.0);
                double t19735 = a2*a2;
                double t19736 = gamma_bb*gamma_bb;
                double t19737 = 1.0/pow(rho_b,1.6E1/3.0);
                double t19738 = t19733*t19733;
                double t19739 = c*t19734;
                double t19740 = a4*a4;
                double t19741 = 1.0/c;
                double t19742 = 1.0/pow(rho_b,7.0/3.0);
                double t19743 = sqrt(t19739);
                double t19744 = t19740*t19736*t19737*t19738*2.0;
                double t19745 = exp(t19744);
                double t19746 = t19740*t19736*t19737*t19738;
                double t19747 = t19746+1.0;
                double t19748 = 1.0/(t19747*t19747);
                double t19749 = a5*gamma_bb*t19741*t19733*t19742;
                double t19750 = t19749+1.0;
                double t19762 = raa2*t19743;
                double t19751 = exp(-t19762);
                double t19752 = t19751-1.0;
                double t19753 = kaa2*t19752;
                double t19754 = pow(t19739,2.0/5.0);
                double t19763 = raa1*t19754;
                double t19755 = exp(-t19763);
                double t19756 = t19755-1.0;
                double t19757 = kaa1*t19756;
                double t19758 = -kaa0+t19753+t19757;
                double t19759 = 1.0/(t19758*t19758);
                double t19760 = c*c0*t19750*t19734*t19745*t19748*t19759*(2.0/3.0);
                double t19761 = 1.0/pow(rho_b,8.0/3.0);
                double t19764 = 1.0/pow(rho_b,1.7E1/3.0);
                double t19765;
                double t19766;
                if (t19760 > expei_cutoff) {
                    t19765 = 1.0;
                    t19766 = expei(t19760);
                } else {
                    t19765 = exp(t19760);
                    t19766 = Ei(-t19760);
                }
                double t19767 = 1.0/t19747;
                double t19768 = c*c0*t19750*t19734*t19745*t19759*(2.0/3.0);
                double t19769 = sqrt(t19768);
                double t19770 = 1.0/(t19747*t19747*t19747);
                double t19771 = t19767*t19769*6.0;
                double t19772 = c*c0*t19750*t19734*t19745*t19748*t19759*4.0;
                double t19773 = t19771+t19772+3.0;
                double t19774 = 1.0/t19773;
                double t19775 = a5*c0*t19733*t19761*t19745*t19748*t19759*(2.0/3.0);
                double t19776 = c*c0*gamma_bb*t19740*t19750*t19745*t19764*t19738*t19748*t19759*(8.0/3.0);
                double t19787 = c*c0*gamma_bb*t19740*t19750*t19770*t19745*t19764*t19738*t19759*(8.0/3.0);
                double t19777 = t19775+t19776-t19787;
                double t19778 = c*c0*t19750*t19734*t19745*t19765*t19748*t19766*t19759*(2.0/3.0);
                double t19779 = t19778+1.0;
                double t19780 = t19767*t19769*4.0;
                double t19781 = c*c0*t19750*t19734*t19745*t19748*t19759*(8.0/3.0);
                double t19782 = t19780+t19781+6.0;
                double t19783 = a5*c0*t19733*t19761*t19745*t19759*(2.0/3.0);
                double t19784 = c*c0*gamma_bb*t19740*t19750*t19745*t19764*t19738*t19759*(8.0/3.0);
                double t19785 = t19783+t19784;
                double t19786 = 1.0/sqrt(t19768);
                double t19788 = t19735*t19736*t19737*t19738;
                double t19789 = pow(t19739,4.0/5.0);
                double t19806 = rab1*t19789;
                double t19790 = exp(-t19806);
                double t19791 = t19790-1.0;
                double t19807 = kab1*t19791;
                double t19792 = kab0-t19807;
                double t19793 = a3*gamma_bb*t19741*t19733*t19742;
                double t19794 = t19793+1.0;
                double t19795 = 1.0/sqrt(t19794);
                double t19796 = a1*gamma_bb*t19733*t19761;
                double t19797 = t19796+t19788+1.0;
                double t19798 = c*c;
                double t19799 = 1.0/pow(rho_b,2.0/3.0);
                double t19800 = k1*t19743;
                double t19801 = c*k2*t19734;
                double t19802 = t19800+t19801;
                double t19803 = 1.0/(t19802*t19802);
                double t19804 = exp(-t19803*t19798*t19799);
                double t19805 = exp(-t19788);
                v_gamma_bb[Q] += scale * (c0*rho_b*t19804*heaviside(c*c0*t19734*(-2.0/3.0)+t19792*t19795*t19797*exp(-t19735*t19736*t19737*t19738)*1.0E5)*(t19774*t19779*(t19767*t19785*t19786*2.0-gamma_bb*t19740*t19737*t19738*t19748*t19769*8.0+a5*c0*t19733*t19761*t19745*t19748*t19759*(8.0/3.0)-c*c0*gamma_bb*t19740*t19750*t19770*t19745*t19764*t19738*t19759*(3.2E1/3.0)+c*c0*gamma_bb*t19740*t19750*t19745*t19764*t19738*t19748*t19759*(3.2E1/3.0))+t19782*t19774*(t19775+t19776+a5*c0*t19733*t19761*t19745*t19765*t19748*t19766*t19759*(2.0/3.0)-c*c0*gamma_bb*t19740*t19750*t19770*t19745*t19764*t19738*t19759*(8.0/3.0)+c*c0*t19750*t19734*t19745*t19765*t19748*t19766*t19759*t19777*(2.0/3.0)-c*c0*gamma_bb*t19740*t19750*t19770*t19745*t19764*t19738*t19765*t19766*t19759*(8.0/3.0)+c*c0*gamma_bb*t19740*t19750*t19745*t19764*t19738*t19765*t19748*t19766*t19759*(8.0/3.0))+t19765*t19766*t19777-1.0/(t19773*t19773)*t19782*t19779*(t19767*t19785*t19786*3.0-gamma_bb*t19740*t19737*t19738*t19748*t19769*1.2E1+a5*c0*t19733*t19761*t19745*t19748*t19759*4.0-c*c0*gamma_bb*t19740*t19750*t19770*t19745*t19764*t19738*t19759*1.6E1+c*c0*gamma_bb*t19740*t19750*t19745*t19764*t19738*t19748*t19759*1.6E1)+(pow(rho_b,1.0/3.0)*t19741*(t19747*t19747)*(t19758*t19758)*t19777*exp(-t19744)*(3.0/2.0))/(c0*t19750))*(1.0/2.0)-c0*rho_b*t19804*dirac(c*c0*t19734*(2.0/3.0)-t19805*t19792*t19795*t19797*1.0E5)*(t19765*t19766+t19782*t19774*t19779)*(t19805*t19792*t19795*(a1*t19733*t19761+gamma_bb*t19735*t19737*t19738*2.0)*-1.0E5+a3*t19741*t19733*t19742*t19805*t19792*1.0/pow(t19794,3.0/2.0)*t19797*5.0E4+gamma_bb*t19805*t19735*t19737*t19738*t19792*t19795*t19797*2.0E5)*(1.0/2.0));
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t19813 = tspi_m13*tspi_m13;
                double t19814 = 1.0/pow(rho_b,1.0/3.0);
                double t19815 = a2*a2;
                double t19816 = gamma_bb*gamma_bb;
                double t19817 = 1.0/pow(rho_b,1.6E1/3.0);
                double t19818 = t19813*t19813;
                double t19819 = c*t19814;
                double t19820 = a4*a4;
                double t19821 = t19820*t19816*t19817*t19818*2.0;
                double t19822 = exp(t19821);
                double t19823 = 1.0/c;
                double t19824 = 1.0/pow(rho_b,7.0/3.0);
                double t19825 = sqrt(t19819);
                double t19826 = t19820*t19816*t19817*t19818;
                double t19827 = t19826+1.0;
                double t19828 = 1.0/(t19827*t19827);
                double t19829 = a5*gamma_bb*t19813*t19823*t19824;
                double t19830 = t19829+1.0;
                double t19841 = raa2*t19825;
                double t19831 = exp(-t19841);
                double t19832 = t19831-1.0;
                double t19833 = kaa2*t19832;
                double t19834 = pow(t19819,2.0/5.0);
                double t19842 = raa1*t19834;
                double t19835 = exp(-t19842);
                double t19836 = t19835-1.0;
                double t19837 = kaa1*t19836;
                double t19838 = -kaa0+t19833+t19837;
                double t19839 = 1.0/(t19838*t19838);
                double t19840 = c*c0*t19830*t19822*t19814*t19828*t19839*(2.0/3.0);
                double t19843 = 1.0/t19827;
                double t19844 = 1.0/pow(rho_b,4.0/3.0);
                double t19845 = c*c0*t19830*t19822*t19814*t19839*(2.0/3.0);
                double t19846 = sqrt(t19845);
                double t19847 = 1.0/pow(rho_b,1.1E1/3.0);
                double t19848 = 1.0/sqrt(t19819);
                double t19849 = c*kaa2*raa2*t19831*t19844*t19848*(1.0/6.0);
                double t19850 = 1.0/pow(t19819,3.0/5.0);
                double t19851 = c*kaa1*raa1*t19850*t19835*t19844*(2.0/1.5E1);
                double t19852 = t19851+t19849;
                double t19853 = 1.0/(t19838*t19838*t19838);
                double t19854 = 1.0/pow(rho_b,2.0E1/3.0);
                double t19855;
                double t19856;
                if (t19840 > expei_cutoff) {
                    t19855 = 1.0;
                    t19856 = expei(t19840);
                } else {
                    t19855 = exp(t19840);
                    t19856 = Ei(-t19840);
                }
                double t19857 = 1.0/(t19827*t19827*t19827);
                double t19858 = t19843*t19846*6.0;
                double t19859 = c*c0*t19830*t19822*t19814*t19828*t19839*4.0;
                double t19860 = t19858+t19859+3.0;
                double t19861 = 1.0/t19860;
                double t19862 = c*c0*t19830*t19822*t19844*t19828*t19839*(2.0/9.0);
                double t19863 = a5*c0*gamma_bb*t19813*t19822*t19828*t19847*t19839*(1.4E1/9.0);
                double t19864 = c*c0*t19830*t19822*t19814*t19852*t19853*t19828*(4.0/3.0);
                double t19865 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19828*t19839*(6.4E1/9.0);
                double t19867 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19839*t19857*(6.4E1/9.0);
                double t19866 = t19862+t19863+t19864+t19865-t19867;
                double t19868 = c*c0*t19830*t19822*t19814*t19828*t19855*t19856*t19839*(2.0/3.0);
                double t19869 = t19868+1.0;
                double t19870 = t19843*t19846*4.0;
                double t19871 = c*c0*t19830*t19822*t19814*t19828*t19839*(8.0/3.0);
                double t19872 = t19870+t19871+6.0;
                double t19873 = c*c0*t19830*t19822*t19844*t19839*(2.0/9.0);
                double t19874 = a5*c0*gamma_bb*t19813*t19822*t19847*t19839*(1.4E1/9.0);
                double t19875 = c*c0*t19830*t19822*t19814*t19852*t19853*(4.0/3.0);
                double t19876 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19839*(6.4E1/9.0);
                double t19877 = t19873+t19874+t19875+t19876;
                double t19878 = 1.0/sqrt(t19845);
                double t19879 = 1.0/pow(rho_b,1.9E1/3.0);
                double t19880 = t19815*t19816*t19817*t19818;
                double t19881 = pow(t19819,4.0/5.0);
                double t19899 = rab1*t19881;
                double t19882 = exp(-t19899);
                double t19883 = t19882-1.0;
                double t19900 = kab1*t19883;
                double t19884 = kab0-t19900;
                double t19885 = a3*gamma_bb*t19813*t19823*t19824;
                double t19886 = t19885+1.0;
                double t19887 = 1.0/sqrt(t19886);
                double t19888 = 1.0/pow(rho_b,8.0/3.0);
                double t19889 = a1*gamma_bb*t19813*t19888;
                double t19890 = t19880+t19889+1.0;
                double t19891 = c*c;
                double t19892 = 1.0/pow(rho_b,2.0/3.0);
                double t19893 = k1*t19825;
                double t19894 = c*k2*t19814;
                double t19895 = t19893+t19894;
                double t19896 = 1.0/(t19895*t19895);
                double t19902 = t19891*t19892*t19896;
                double t19897 = exp(-t19902);
                double t19898 = exp(-t19880);
                double t19901 = c*c0*t19814*(2.0/3.0);
                double t19903 = t19843*t19877*t19878*2.0;
                double t19904 = c*c0*t19830*t19822*t19844*t19828*t19839*(8.0/9.0);
                double t19905 = a5*c0*gamma_bb*t19813*t19822*t19828*t19847*t19839*(5.6E1/9.0);
                double t19906 = c*c0*t19830*t19822*t19814*t19852*t19853*t19828*(1.6E1/3.0);
                double t19907 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19828*t19839*(2.56E2/9.0);
                double t19958 = t19820*t19816*t19818*t19828*t19846*t19879*(6.4E1/3.0);
                double t19959 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19839*t19857*(2.56E2/9.0);
                double t19908 = t19903+t19904+t19905+t19906+t19907-t19958-t19959;
                double t19909 = c*c0*t19830*t19822*t19844*t19828*t19855*t19856*t19839*(2.0/9.0);
                double t19910 = c*c0*t19830*t19822*t19814*t19852*t19853*t19828*t19855*t19856*(4.0/3.0);
                double t19911 = c*c0*t19830*t19822*t19814*t19828*t19855*t19856*t19839*t19866*(2.0/3.0);
                double t19912 = a5*c0*gamma_bb*t19813*t19822*t19828*t19855*t19847*t19856*t19839*(1.4E1/9.0);
                double t19913 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19828*t19855*t19856*t19839*(6.4E1/9.0);
                double t19962 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19855*t19856*t19839*t19857*(6.4E1/9.0);
                double t19914 = t19910+t19911+t19912+t19913+t19862+t19863-t19962+t19864+t19909+t19865-t19867;
                double t19915 = t19820*t19820;
                double t19916 = t19816*t19816;
                double t19917 = t19818*t19818;
                double t19918 = raa2*raa2;
                double t19919 = c*kaa2*t19831*t19824*t19918*(1.0/3.6E1);
                double t19920 = raa1*raa1;
                double t19921 = 1.0/pow(t19819,6.0/5.0);
                double t19922 = kaa1*t19920*t19921*t19835*t19891*t19888*(4.0/2.25E2);
                double t19923 = 1.0/pow(t19819,3.0/2.0);
                double t19924 = kaa2*raa2*t19831*t19923*t19891*t19888*(1.0/3.6E1);
                double t19925 = 1.0/pow(t19819,8.0/5.0);
                double t19926 = kaa1*raa1*t19835*t19925*t19891*t19888*(2.0/7.5E1);
                double t19936 = c*kaa2*raa2*t19831*t19824*t19848*(2.0/9.0);
                double t19937 = c*kaa1*raa1*t19850*t19824*t19835*(8.0/4.5E1);
                double t19927 = t19922+t19924+t19926-t19936+t19919-t19937;
                double t19928 = t19852*t19852;
                double t19929 = 1.0/(t19838*t19838*t19838*t19838);
                double t19930 = 1.0/pow(rho_b,1.4E1/3.0);
                double t19931 = 1.0/pow(rho_b,1.0E1);
                double t19932 = 1.0/pow(rho_b,1.3E1);
                double t19933 = 1.0/pow(rho_b,2.3E1/3.0);
                double t19934 = 1.0/pow(rho_b,1.0E1/3.0);
                double t19935 = 1.0/t19830;
                double t19938 = t19866*t19866;
                double t19939 = 1.0/(t19827*t19827*t19827*t19827);
                double t19940 = c*c0*t19830*t19822*t19824*t19828*t19839*(8.0/2.7E1);
                double t19941 = c*c0*t19830*t19822*t19814*t19828*t19928*t19929*4.0;
                double t19942 = a5*c0*gamma_bb*t19813*t19822*t19930*t19828*t19839*(5.6E1/9.0);
                double t19943 = c*c0*t19830*t19822*t19852*t19844*t19853*t19828*(8.0/9.0);
                double t19944 = a5*c0*gamma_bb*t19813*t19822*t19852*t19853*t19828*t19847*(5.6E1/9.0);
                double t19945 = a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19828*t19839*(8.96E2/2.7E1);
                double t19946 = c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19828*t19839*(2.048E3/2.7E1);
                double t19947 = c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19839*t19939*(1.024E3/9.0);
                double t19948 = c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19828*t19839*(4.48E2/9.0);
                double t19949 = c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19828*(2.56E2/9.0);
                double t19982 = c*c0*t19830*t19822*t19814*t19853*t19828*t19927*(4.0/3.0);
                double t19983 = a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19839*t19857*(8.96E2/2.7E1);
                double t19984 = c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19839*t19857*(4.096E3/2.7E1);
                double t19985 = c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19839*t19857*(4.48E2/9.0);
                double t19986 = c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19857*(2.56E2/9.0);
                double t19950 = t19940+t19941+t19942+t19943+t19944+t19945+t19946-t19982+t19947-t19983+t19948-t19984+t19949-t19985-t19986;
                double t19951 = 1.0/(t19860*t19860);
                double t19952 = t19843*t19877*t19878*3.0;
                double t19953 = c*c0*t19830*t19822*t19844*t19828*t19839*(4.0/3.0);
                double t19954 = a5*c0*gamma_bb*t19813*t19822*t19828*t19847*t19839*(2.8E1/3.0);
                double t19955 = c*c0*t19830*t19822*t19814*t19852*t19853*t19828*8.0;
                double t19956 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19828*t19839*(1.28E2/3.0);
                double t19960 = t19820*t19816*t19818*t19828*t19846*t19879*3.2E1;
                double t19961 = c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19839*t19857*(1.28E2/3.0);
                double t19957 = -t19960+t19952-t19961+t19953+t19954+t19955+t19956;
                double t19963 = c*c0*t19830*t19822*t19824*t19839*(8.0/2.7E1);
                double t19964 = a5*c0*gamma_bb*t19813*t19822*t19930*t19839*(5.6E1/9.0);
                double t19965 = c*c0*t19830*t19822*t19852*t19844*t19853*(8.0/9.0);
                double t19966 = c*c0*t19830*t19822*t19814*t19928*t19929*4.0;
                double t19967 = a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19839*(8.96E2/2.7E1);
                double t19968 = a5*c0*gamma_bb*t19813*t19822*t19852*t19853*t19847*(5.6E1/9.0);
                double t19969 = c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19839*(2.048E3/2.7E1);
                double t19970 = c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19839*(4.48E2/9.0);
                double t19971 = c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*(2.56E2/9.0);
                double t19972 = t19970+t19971+t19963+t19964+t19965+t19966+t19967+t19968+t19969-c*c0*t19830*t19822*t19814*t19853*t19927*(4.0/3.0);
                double t19973 = t19877*t19877;
                double t19974 = 1.0/pow(t19845,3.0/2.0);
                double t19975 = 1.0/pow(rho_b,2.2E1/3.0);
                double t19976 = 1.0/pow(rho_b,3.8E1/3.0);
                double t19977 = 1.0/c0;
                double t19978 = exp(-t19821);
                double t19979 = t19827*t19827;
                double t19980 = t19838*t19838;
                double t19981 = pow(rho_b,1.0/3.0);
                double t19987 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b*rho_b);
                double t19988 = t19890*t19884*t19887*t19898*1.0E5;
                double t19989 = -t19901+t19988;
                double t19990 = heaviside(t19989);
                double t19991 = t19855*t19856;
                double t19992 = t19861*t19872*t19869;
                double t19993 = t19991+t19992;
                double t19994 = t19815*t19816*t19818*t19879*(1.6E1/3.0);
                double t19995 = a1*gamma_bb*t19813*t19847*(8.0/3.0);
                double t19996 = t19994+t19995;
                double t19997 = 1.0/(c*c);
                double t19998 = 1.0/pow(t19819,1.0/5.0);
                double t19999 = 1.0/pow(t19886,3.0/2.0);
                double t20000 = 1.0/pow(rho_b,5.0/3.0);
                double t20001 = t19891*t19896*t20000*(2.0/3.0);
                double t20002 = 1.0/(t19895*t19895*t19895);
                double t20003 = c*k2*t19844*(1.0/3.0);
                double t20004 = c*k1*t19844*t19848*(1.0/6.0);
                double t20005 = t20003+t20004;
                double t20019 = t19891*t19892*t20002*t20005*2.0;
                double t20006 = t20001-t20019;
                double t20007 = t19861*t19908*t19869;
                double t20008 = t19855*t19856*t19866;
                double t20009 = t19914*t19861*t19872;
                double t20010 = t19823*t19935*t19980*t19981*t19866*t19977*t19978*t19979*(3.0/2.0);
                double t20014 = t19951*t19872*t19957*t19869;
                double t20011 = t20010-t20014+t20007+t20008+t20009;
                double t20012 = t19901-t19988;
                double t20013 = dirac(t20012);
                double t20015 = c*c0*t19844*(2.0/9.0);
                double t20016 = t19815*t19816*t19818*t19890*t19884*t19887*t19879*t19898*(1.6E6/3.0);
                double t20017 = a3*gamma_bb*t19813*t19823*t19934*t19890*t19884*t19898*t19999*(3.5E5/3.0);
                double t20020 = t19884*t19887*t19996*t19898*1.0E5;
                double t20021 = c*kab1*rab1*t19844*t19890*t19882*t19887*t19898*t19998*(8.0E4/3.0);
                double t20018 = -t20020-t20021+t20015+t20016+t20017;
                v_rho_b_rho_b[Q] += scale * (-c0*t19897*t20011*heaviside(c*c0*t19814*(-2.0/3.0)+t19890*t19884*t19887*exp(-t19815*t19816*t19817*t19818)*1.0E5)+c0*t19993*t19897*t20018*dirac(t19901-t19890*t19884*t19887*t19898*1.0E5)+c0*t19990*t19993*t19897*t20006+c0*rho_b*t19990*t19897*(t19861*t19869*(-t19843*t19973*t19974+t19843*t19972*t19878*2.0-t19820*t19816*t19818*t19828*t19846*t19975*(1.216E3/9.0)+t19915*t19916*t19917*t19846*t19857*t19976*(2.048E3/9.0)+c*c0*t19830*t19822*t19824*t19828*t19839*(3.2E1/2.7E1)-t19820*t19816*t19818*t19828*t19877*t19878*t19879*(6.4E1/3.0)+a5*c0*gamma_bb*t19813*t19822*t19930*t19828*t19839*(2.24E2/9.0)-c*c0*t19830*t19822*t19814*t19853*t19828*t19927*(1.6E1/3.0)+c*c0*t19830*t19822*t19852*t19844*t19853*t19828*(3.2E1/9.0)+c*c0*t19830*t19822*t19814*t19828*t19928*t19929*1.6E1+a5*c0*gamma_bb*t19813*t19822*t19852*t19853*t19828*t19847*(2.24E2/9.0)+c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19828*t19839*(1.792E3/9.0)-c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19839*t19857*(1.792E3/9.0)+c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19828*t19839*(8.192E3/2.7E1)-c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19839*t19857*(1.6384E4/2.7E1)+c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19839*t19939*(4.096E3/9.0)+a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19828*t19839*(3.584E3/2.7E1)-a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19839*t19857*(3.584E3/2.7E1)+c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19828*(1.024E3/9.0)-c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19857*(1.024E3/9.0))+t19914*t19861*t19908*2.0+t19950*t19855*t19856+t19855*t19856*t19938+t19861*t19872*(t19940+t19941+t19942+t19943+t19944+t19945+t19946+t19938+t19947+t19948+t19949+(t19866*(1.0/3.0))/rho_b+(t19852*t19866*2.0)/t19838+t19820*t19816*t19818*t19866*t19879*(3.2E1/3.0)-t19820*t19816*t19843*t19818*t19866*t19879*(3.2E1/3.0)+a5*gamma_bb*t19813*t19823*t19934*t19935*t19866*(7.0/3.0)-c*c0*t19830*t19822*t19814*t19853*t19828*t19927*(4.0/3.0)+c*c0*t19830*t19822*t19824*t19828*t19855*t19856*t19839*(8.0/2.7E1)+a5*c0*gamma_bb*t19813*t19822*t19930*t19828*t19855*t19856*t19839*(5.6E1/9.0)-c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19839*t19857*(4.48E2/9.0)+c*c0*t19830*t19822*t19814*t19950*t19828*t19855*t19856*t19839*(2.0/3.0)-c*c0*t19830*t19822*t19814*t19853*t19828*t19855*t19927*t19856*(4.0/3.0)-c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19839*t19857*(4.096E3/2.7E1)+c*c0*t19830*t19822*t19852*t19844*t19853*t19828*t19855*t19856*(8.0/9.0)+c*c0*t19830*t19822*t19814*t19828*t19855*t19856*t19928*t19929*4.0+c*c0*t19830*t19822*t19814*t19828*t19855*t19856*t19839*t19938*(2.0/3.0)+c*c0*t19830*t19822*t19844*t19828*t19855*t19856*t19839*t19866*(4.0/9.0)-a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19839*t19857*(8.96E2/2.7E1)+a5*c0*gamma_bb*t19813*t19822*t19852*t19853*t19828*t19855*t19847*t19856*(5.6E1/9.0)+a5*c0*gamma_bb*t19813*t19822*t19828*t19855*t19847*t19856*t19839*t19866*(2.8E1/9.0)-c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19857*(2.56E2/9.0)+c*c0*t19830*t19822*t19814*t19852*t19853*t19828*t19855*t19856*t19866*(8.0/3.0)+c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19828*t19855*t19856*t19839*(4.48E2/9.0)-c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19855*t19856*t19839*t19857*(4.48E2/9.0)+c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19828*t19855*t19856*t19839*(2.048E3/2.7E1)-c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19855*t19856*t19839*t19857*(4.096E3/2.7E1)+c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19855*t19856*t19839*t19939*(1.024E3/9.0)+a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19828*t19855*t19856*t19839*(8.96E2/2.7E1)-a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19855*t19856*t19839*t19857*(8.96E2/2.7E1)+c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19828*t19855*t19856*(2.56E2/9.0)-c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19855*t19856*t19857*(2.56E2/9.0)+c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19828*t19855*t19856*t19839*t19866*(1.28E2/9.0)-c*c0*t19820*t19830*t19822*t19816*t19818*t19854*t19855*t19856*t19839*t19857*t19866*(1.28E2/9.0))-t19951*t19872*t19869*(t19843*t19973*t19974*(-3.0/2.0)+t19843*t19972*t19878*3.0-t19820*t19816*t19818*t19828*t19846*t19975*(6.08E2/3.0)+t19915*t19916*t19917*t19846*t19857*t19976*(1.024E3/3.0)+c*c0*t19830*t19822*t19824*t19828*t19839*(1.6E1/9.0)-t19820*t19816*t19818*t19828*t19877*t19878*t19879*3.2E1+a5*c0*gamma_bb*t19813*t19822*t19930*t19828*t19839*(1.12E2/3.0)-c*c0*t19830*t19822*t19814*t19853*t19828*t19927*8.0+c*c0*t19830*t19822*t19852*t19844*t19853*t19828*(1.6E1/3.0)+c*c0*t19830*t19822*t19814*t19828*t19928*t19929*2.4E1+a5*c0*gamma_bb*t19813*t19822*t19852*t19853*t19828*t19847*(1.12E2/3.0)+c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19828*t19839*(8.96E2/3.0)-c*c0*t19820*t19830*t19822*t19816*t19933*t19818*t19839*t19857*(8.96E2/3.0)+c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19828*t19839*(4.096E3/9.0)-c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19839*t19857*(8.192E3/9.0)+c*c0*t19830*t19822*t19932*t19915*t19916*t19917*t19839*t19939*(2.048E3/3.0)+a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19828*t19839*(1.792E3/9.0)-a5*c0*gamma_bb*t19820*t19813*t19822*t19931*t19816*t19818*t19839*t19857*(1.792E3/9.0)+c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19828*(5.12E2/3.0)-c*c0*t19820*t19830*t19822*t19816*t19852*t19853*t19818*t19854*t19857*(5.12E2/3.0))-t19914*t19951*t19872*t19957*2.0-t19951*t19908*t19957*t19869*2.0+1.0/(t19860*t19860*t19860)*t19872*(t19957*t19957)*t19869*2.0+t19823*t19950*t19935*t19980*t19981*t19977*t19978*t19979*(3.0/2.0)+t19823*t19935*t19980*t19981*t19938*t19977*t19978*t19979*(3.0/2.0)-t19823*t19935*t19980*t19892*t19866*t19977*t19978*t19979*(1.0/2.0)-t19823*t19852*t19935*t19981*t19838*t19866*t19977*t19978*t19979*3.0+t19820*t19823*t19816*t19818*t19827*t19935*t19980*t19866*t19977*t19978*t19987*1.6E1-t19820*t19823*t19816*t19818*t19935*t19980*t19866*t19977*t19978*t19987*t19979*1.6E1-a5*gamma_bb*1.0/(rho_b*rho_b*rho_b)*1.0/(t19830*t19830)*t19813*t19980*t19866*t19977*t19978*t19979*t19997*(7.0/2.0))*(1.0/2.0)-c0*rho_b*t19990*t19897*t20011*t20006-c0*rho_b*t19897*t20011*t20013*t20018+c0*rho_b*t19990*t19993*t19897*(t20006*t20006)*(1.0/2.0)-c0*rho_b*t19993*t19897*t20013*(c*c0*t19824*(8.0/2.7E1)-t19884*t19887*t19898*(a1*gamma_bb*t19813*t19930*(8.8E1/9.0)+t19815*t19816*t19818*t19975*(3.04E2/9.0))*1.0E5+t19815*t19816*t19818*t19890*t19884*t19975*t19887*t19898*(3.04E7/9.0)+t19815*t19816*t19818*t19884*t19887*t19879*t19996*t19898*(3.2E6/3.0)-(t19815*t19815)*t19916*t19890*t19917*t19884*t19976*t19887*t19898*(2.56E7/9.0)-(a3*a3)*t19816*t19818*t19854*t19890*t19884*1.0/pow(t19886,5.0/2.0)*t19898*t19997*(1.225E6/3.0)+kab1*(rab1*rab1)*t19890*1.0/pow(t19819,2.0/5.0)*t19882*t19891*t19887*t19888*t19898*(6.4E4/9.0)+a3*gamma_bb*t19813*t19823*t19934*t19884*t19996*t19898*t19999*(7.0E5/3.0)-c*kab1*rab1*t19824*t19890*t19882*t19887*t19898*t19998*(3.2E5/9.0)-c*kab1*rab1*t19844*t19882*t19887*t19996*t19898*t19998*(1.6E5/3.0)+kab1*rab1*t19921*t19890*t19882*t19891*t19887*t19888*t19898*(1.6E4/9.0)+a3*gamma_bb*1.0/pow(rho_b,1.3E1/3.0)*t19813*t19823*t19890*t19884*t19898*t19999*(3.5E6/9.0)+a3*gamma_bb*kab1*rab1*t19813*t19930*t19890*t19882*t19898*t19998*t19999*(5.6E5/9.0)-a3*gamma_bb*1.0/pow(rho_b,2.9E1/3.0)*t19813*t19823*t19815*t19816*t19818*t19890*t19884*t19898*t19999*(1.12E7/9.0)+c*kab1*rab1*t19815*t19816*t19933*t19818*t19890*t19882*t19887*t19898*t19998*(2.56E6/9.0))*(1.0/2.0)-c0*rho_b*t19990*t19993*t19897*(t19891*t19896*t19888*(1.0E1/9.0)-t19891*t20000*t20002*t20005*(8.0/3.0)+t19891*t19892*1.0/(t19895*t19895*t19895*t19895)*(t20005*t20005)*6.0-t19891*t19892*t20002*(c*k2*t19824*(4.0/9.0)+c*k1*t19824*t19848*(2.0/9.0)-k1*t19923*t19891*t19888*(1.0/3.6E1))*2.0)*(1.0/2.0)-c0*rho_b*t19993*t19897*(t20018*t20018)*dirac(t20012,1.0)*(1.0/2.0)+c0*rho_b*t19993*t19897*t20013*t20006*t20018);
            }
            
            // v_gamma_bb_gamma_bb
            if (deriv >= 2) {
                double t20242 = tspi_m13*tspi_m13;
                double t20243 = 1.0/pow(rho_b,1.0/3.0);
                double t20244 = a2*a2;
                double t20245 = gamma_bb*gamma_bb;
                double t20246 = 1.0/pow(rho_b,1.6E1/3.0);
                double t20247 = t20242*t20242;
                double t20248 = c*t20243;
                double t20249 = a4*a4;
                double t20250 = t20245*t20246*t20247*t20249*2.0;
                double t20251 = exp(t20250);
                double t20252 = 1.0/c;
                double t20253 = 1.0/pow(rho_b,7.0/3.0);
                double t20254 = sqrt(t20248);
                double t20255 = t20245*t20246*t20247*t20249;
                double t20256 = t20255+1.0;
                double t20257 = 1.0/(t20256*t20256);
                double t20258 = a5*gamma_bb*t20242*t20252*t20253;
                double t20259 = t20258+1.0;
                double t20270 = raa2*t20254;
                double t20260 = exp(-t20270);
                double t20261 = t20260-1.0;
                double t20262 = kaa2*t20261;
                double t20263 = pow(t20248,2.0/5.0);
                double t20271 = raa1*t20263;
                double t20264 = exp(-t20271);
                double t20265 = t20264-1.0;
                double t20266 = kaa1*t20265;
                double t20267 = -kaa0+t20262+t20266;
                double t20268 = 1.0/(t20267*t20267);
                double t20269 = c*c0*t20251*t20243*t20257*t20259*t20268*(2.0/3.0);
                double t20272 = 1.0/t20256;
                double t20273 = c*c0*t20251*t20243*t20259*t20268*(2.0/3.0);
                double t20274 = 1.0/pow(rho_b,8.0/3.0);
                double t20275 = 1.0/pow(rho_b,1.7E1/3.0);
                double t20280 = a5*c0*t20242*t20251*t20274*t20268*(2.0/3.0);
                double t20281 = c*c0*gamma_bb*t20251*t20247*t20275*t20249*t20259*t20268*(8.0/3.0);
                double t20276 = t20280+t20281;
                double t20277 = sqrt(t20273);
                double t20278 = t20249*t20249;
                double t20279 = t20247*t20247;
                double t20282 = 1.0/sqrt(t20273);
                double t20283 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b);
                double t20284 = 1.0/(t20256*t20256*t20256);
                double t20285 = 1.0/pow(rho_b,1.1E1);
                double t20286 = t20272*t20277*6.0;
                double t20287 = c*c0*t20251*t20243*t20257*t20259*t20268*4.0;
                double t20288 = t20286+t20287+3.0;
                double t20289 = 1.0/t20288;
                double t20291 = a5*c0*t20242*t20251*t20274*t20257*t20268*(2.0/3.0);
                double t20292 = c*c0*gamma_bb*t20251*t20247*t20257*t20275*t20249*t20259*t20268*(8.0/3.0);
                double t20293 = c*c0*gamma_bb*t20251*t20247*t20275*t20284*t20249*t20259*t20268*(8.0/3.0);
                double t20290 = t20291+t20292-t20293;
                double t20294;
                double t20295;
                if (t20269 > expei_cutoff) {
                    t20294 = 1.0;
                    t20295 = expei(t20269);
                } else {
                    t20294 = exp(t20269);
                    t20295 = Ei(-t20269);
                }
                double t20296 = t20290*t20290;
                double t20297 = 1.0/(t20256*t20256*t20256*t20256);
                double t20298 = a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20257*t20249*t20268*(1.6E1/3.0);
                double t20299 = c*c0*t20251*t20247*t20257*t20275*t20249*t20259*t20268*(8.0/3.0);
                double t20300 = c*c0*t20251*t20245*t20257*t20285*t20259*t20268*t20278*t20279*(3.2E1/3.0);
                double t20301 = c*c0*t20251*t20245*t20285*t20259*t20268*t20278*t20279*t20297*1.6E1;
                double t20336 = a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20284*t20249*t20268*(1.6E1/3.0);
                double t20337 = c*c0*t20251*t20247*t20275*t20284*t20249*t20259*t20268*(8.0/3.0);
                double t20338 = c*c0*t20251*t20245*t20284*t20285*t20259*t20268*t20278*t20279*(6.4E1/3.0);
                double t20302 = t20300+t20301-t20336-t20337-t20338+t20298+t20299;
                double t20303 = t20272*t20277*4.0;
                double t20304 = c*c0*t20251*t20243*t20257*t20259*t20268*(8.0/3.0);
                double t20305 = t20303+t20304+6.0;
                double t20306 = a5*c0*t20242*t20251*t20274*t20257*t20294*t20268*t20295*(2.0/3.0);
                double t20307 = c*c0*t20251*t20243*t20290*t20257*t20294*t20259*t20268*t20295*(2.0/3.0);
                double t20308 = c*c0*gamma_bb*t20251*t20247*t20257*t20275*t20249*t20294*t20259*t20268*t20295*(8.0/3.0);
                double t20373 = c*c0*gamma_bb*t20251*t20247*t20275*t20284*t20249*t20294*t20259*t20268*t20295*(8.0/3.0);
                double t20309 = t20306+t20307+t20308+t20291+t20292-t20373-t20293;
                double t20310 = c*c0*t20251*t20243*t20257*t20294*t20259*t20268*t20295*(2.0/3.0);
                double t20311 = t20310+1.0;
                double t20312 = t20272*t20282*t20276*3.0;
                double t20313 = a5*c0*t20242*t20251*t20274*t20257*t20268*4.0;
                double t20314 = c*c0*gamma_bb*t20251*t20247*t20257*t20275*t20249*t20259*t20268*1.6E1;
                double t20317 = gamma_bb*t20246*t20247*t20257*t20249*t20277*1.2E1;
                double t20318 = c*c0*gamma_bb*t20251*t20247*t20275*t20284*t20249*t20259*t20268*1.6E1;
                double t20315 = t20312+t20313+t20314-t20317-t20318;
                double t20316 = 1.0/(t20288*t20288);
                double t20319 = t20272*t20282*t20276*2.0;
                double t20320 = a5*c0*t20242*t20251*t20274*t20257*t20268*(8.0/3.0);
                double t20321 = c*c0*gamma_bb*t20251*t20247*t20257*t20275*t20249*t20259*t20268*(3.2E1/3.0);
                double t20371 = gamma_bb*t20246*t20247*t20257*t20249*t20277*8.0;
                double t20372 = c*c0*gamma_bb*t20251*t20247*t20275*t20284*t20249*t20259*t20268*(3.2E1/3.0);
                double t20322 = t20320+t20321-t20371-t20372+t20319;
                double t20323 = a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20249*t20268*(1.6E1/3.0);
                double t20324 = c*c0*t20251*t20247*t20275*t20249*t20259*t20268*(8.0/3.0);
                double t20325 = c*c0*t20251*t20245*t20285*t20259*t20268*t20278*t20279*(3.2E1/3.0);
                double t20326 = t20323+t20324+t20325;
                double t20327 = t20276*t20276;
                double t20328 = 1.0/pow(t20273,3.0/2.0);
                double t20329 = 1.0/pow(rho_b,3.2E1/3.0);
                double t20330 = 1.0/t20259;
                double t20331 = 1.0/c0;
                double t20332 = pow(rho_b,1.0/3.0);
                double t20333 = exp(-t20250);
                double t20334 = t20256*t20256;
                double t20335 = t20267*t20267;
                double t20339 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b);
                double t20340 = c*c;
                double t20341 = 1.0/pow(rho_b,2.0/3.0);
                double t20342 = k1*t20254;
                double t20343 = c*k2*t20243;
                double t20344 = t20342+t20343;
                double t20345 = 1.0/(t20344*t20344);
                double t20363 = t20340*t20341*t20345;
                double t20346 = exp(-t20363);
                double t20347 = t20244*t20245*t20246*t20247;
                double t20348 = pow(t20248,4.0/5.0);
                double t20358 = rab1*t20348;
                double t20349 = exp(-t20358);
                double t20350 = t20349-1.0;
                double t20359 = kab1*t20350;
                double t20351 = kab0-t20359;
                double t20352 = a3*gamma_bb*t20242*t20252*t20253;
                double t20353 = t20352+1.0;
                double t20354 = 1.0/sqrt(t20353);
                double t20355 = a1*gamma_bb*t20242*t20274;
                double t20356 = t20355+t20347+1.0;
                double t20357 = exp(-t20347);
                double t20364 = a1*t20242*t20274;
                double t20365 = gamma_bb*t20244*t20246*t20247*2.0;
                double t20366 = t20364+t20365;
                double t20367 = t20351*t20354*t20357*t20366*1.0E5;
                double t20368 = 1.0/pow(t20353,3.0/2.0);
                double t20369 = a3*t20242*t20252*t20351*t20253*t20356*t20357*t20368*5.0E4;
                double t20370 = gamma_bb*t20351*t20244*t20246*t20354*t20247*t20356*t20357*2.0E5;
                double t20360 = t20370-t20367+t20369;
                double t20361 = c*c0*t20243*(2.0/3.0);
                double t20374 = t20351*t20354*t20356*t20357*1.0E5;
                double t20362 = t20361-t20374;
                double t20375 = dirac(t20362);
                double t20376 = t20294*t20295;
                double t20377 = t20311*t20305*t20289;
                double t20378 = t20376+t20377;
                double t20379 = 1.0/(c*c);
                v_gamma_bb_gamma_bb[Q] += scale * (c0*rho_b*t20346*heaviside(c*c0*t20243*(-2.0/3.0)+t20351*t20354*t20356*exp(-t20244*t20245*t20246*t20247)*1.0E5)*(t20305*t20289*(t20300+t20301+t20296+t20298+t20299+gamma_bb*t20290*t20246*t20247*t20249*4.0+a5*t20330*t20242*t20252*t20253*t20290-gamma_bb*t20272*t20290*t20246*t20247*t20249*4.0-c*c0*t20251*t20247*t20275*t20284*t20249*t20259*t20268*(8.0/3.0)-a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20284*t20249*t20268*(1.6E1/3.0)+a5*c0*t20242*t20251*t20290*t20274*t20257*t20294*t20268*t20295*(4.0/3.0)+c*c0*t20302*t20251*t20243*t20257*t20294*t20259*t20268*t20295*(2.0/3.0)+c*c0*t20251*t20243*t20257*t20294*t20259*t20268*t20295*t20296*(2.0/3.0)-c*c0*t20251*t20245*t20284*t20285*t20259*t20268*t20278*t20279*(6.4E1/3.0)+c*c0*t20251*t20247*t20257*t20275*t20249*t20294*t20259*t20268*t20295*(8.0/3.0)-c*c0*t20251*t20247*t20275*t20284*t20249*t20294*t20259*t20268*t20295*(8.0/3.0)+a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20257*t20249*t20294*t20268*t20295*(1.6E1/3.0)-a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20284*t20249*t20294*t20268*t20295*(1.6E1/3.0)+c*c0*t20251*t20245*t20257*t20285*t20294*t20259*t20268*t20295*t20278*t20279*(3.2E1/3.0)-c*c0*t20251*t20245*t20284*t20285*t20294*t20259*t20268*t20295*t20278*t20279*(6.4E1/3.0)+c*c0*t20251*t20245*t20285*t20294*t20259*t20268*t20295*t20278*t20279*t20297*1.6E1+c*c0*gamma_bb*t20251*t20290*t20247*t20257*t20275*t20249*t20294*t20259*t20268*t20295*(1.6E1/3.0)-c*c0*gamma_bb*t20251*t20290*t20247*t20275*t20284*t20249*t20294*t20259*t20268*t20295*(1.6E1/3.0))+t20302*t20294*t20295+t20322*t20309*t20289*2.0+t20294*t20295*t20296+t20311*t20289*(t20272*t20326*t20282*2.0-t20272*t20327*t20328-t20246*t20247*t20257*t20249*t20277*8.0+t20245*t20284*t20329*t20277*t20278*t20279*3.2E1-gamma_bb*t20246*t20282*t20247*t20257*t20249*t20276*8.0+c*c0*t20251*t20247*t20257*t20275*t20249*t20259*t20268*(3.2E1/3.0)-c*c0*t20251*t20247*t20275*t20284*t20249*t20259*t20268*(3.2E1/3.0)+a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20257*t20249*t20268*(6.4E1/3.0)-a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20284*t20249*t20268*(6.4E1/3.0)+c*c0*t20251*t20245*t20257*t20285*t20259*t20268*t20278*t20279*(1.28E2/3.0)-c*c0*t20251*t20245*t20284*t20285*t20259*t20268*t20278*t20279*(2.56E2/3.0)+c*c0*t20251*t20245*t20285*t20259*t20268*t20278*t20279*t20297*6.4E1)-t20311*t20322*t20315*t20316*2.0-t20305*t20315*t20316*t20309*2.0-t20311*t20305*t20316*(t20272*t20326*t20282*3.0-t20272*t20327*t20328*(3.0/2.0)-t20246*t20247*t20257*t20249*t20277*1.2E1+t20245*t20284*t20329*t20277*t20278*t20279*4.8E1-gamma_bb*t20246*t20282*t20247*t20257*t20249*t20276*1.2E1+c*c0*t20251*t20247*t20257*t20275*t20249*t20259*t20268*1.6E1-c*c0*t20251*t20247*t20275*t20284*t20249*t20259*t20268*1.6E1+a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20257*t20249*t20268*3.2E1-a5*c0*gamma_bb*t20242*t20251*t20247*t20283*t20284*t20249*t20268*3.2E1+c*c0*t20251*t20245*t20257*t20285*t20259*t20268*t20278*t20279*6.4E1-c*c0*t20251*t20245*t20284*t20285*t20259*t20268*t20278*t20279*1.28E2+c*c0*t20251*t20245*t20285*t20259*t20268*t20278*t20279*t20297*9.6E1)+t20311*t20305*(t20315*t20315)*1.0/(t20288*t20288*t20288)*2.0+t20302*t20330*t20331*t20332*t20252*t20333*t20334*t20335*(3.0/2.0)+t20330*t20331*t20332*t20252*t20333*t20334*t20335*t20296*(3.0/2.0)-a5*1.0/(rho_b*rho_b)*t20331*t20242*t20333*t20334*t20290*t20335*1.0/(t20259*t20259)*t20379*(3.0/2.0)-gamma_bb*t20330*t20331*t20252*t20333*t20334*t20290*t20335*t20247*t20249*t20339*6.0+gamma_bb*t20330*t20331*t20252*t20333*t20290*t20335*t20247*t20256*t20249*t20339*6.0)*(1.0/2.0)-c0*rho_b*t20360*t20346*t20375*(t20311*t20322*t20289+t20305*t20309*t20289+t20290*t20294*t20295-t20311*t20305*t20315*t20316+t20330*t20331*t20332*t20252*t20333*t20334*t20290*t20335*(3.0/2.0))+c0*rho_b*t20346*t20375*t20378*(t20351*t20244*t20246*t20354*t20247*t20357*2.0E5-t20351*t20244*t20246*t20354*t20247*t20356*t20357*2.0E5+(a3*a3)*1.0/pow(rho_b,1.4E1/3.0)*t20351*1.0/pow(t20353,5.0/2.0)*t20247*t20356*t20357*t20379*7.5E4+t20351*(t20244*t20244)*t20245*t20354*t20329*t20356*t20357*t20279*4.0E5-a3*t20242*t20252*t20351*t20253*t20357*t20366*t20368*1.0E5-gamma_bb*t20351*t20244*t20246*t20354*t20247*t20357*t20366*4.0E5+a3*gamma_bb*1.0/pow(rho_b,2.3E1/3.0)*t20242*t20252*t20351*t20244*t20247*t20356*t20357*t20368*2.0E5)*(1.0/2.0)-c0*rho_b*(t20360*t20360)*t20346*t20378*dirac(t20362,1.0)*(1.0/2.0));
            }
            
            // v_rho_b_gamma_bb
            if (deriv >= 2) {
                double t20028 = tspi_m13*tspi_m13;
                double t20029 = 1.0/pow(rho_b,1.0/3.0);
                double t20030 = a2*a2;
                double t20031 = gamma_bb*gamma_bb;
                double t20032 = 1.0/pow(rho_b,1.6E1/3.0);
                double t20033 = t20028*t20028;
                double t20034 = c*t20029;
                double t20035 = a4*a4;
                double t20036 = 1.0/c;
                double t20037 = 1.0/pow(rho_b,7.0/3.0);
                double t20038 = sqrt(t20034);
                double t20039 = t20031*t20032*t20033*t20035*2.0;
                double t20040 = exp(t20039);
                double t20041 = t20031*t20032*t20033*t20035;
                double t20042 = t20041+1.0;
                double t20043 = 1.0/(t20042*t20042);
                double t20044 = a5*gamma_bb*t20036*t20028*t20037;
                double t20045 = t20044+1.0;
                double t20057 = raa2*t20038;
                double t20046 = exp(-t20057);
                double t20047 = t20046-1.0;
                double t20048 = kaa2*t20047;
                double t20049 = pow(t20034,2.0/5.0);
                double t20058 = raa1*t20049;
                double t20050 = exp(-t20058);
                double t20051 = t20050-1.0;
                double t20052 = kaa1*t20051;
                double t20053 = -kaa0+t20052+t20048;
                double t20054 = 1.0/(t20053*t20053);
                double t20055 = c*c0*t20040*t20043*t20045*t20054*t20029*(2.0/3.0);
                double t20056 = 1.0/pow(rho_b,8.0/3.0);
                double t20059 = 1.0/pow(rho_b,1.7E1/3.0);
                double t20060;
                double t20061;
                if (t20055 > expei_cutoff) {
                    t20060 = 1.0;
                    t20061 = expei(t20055);
                } else {
                    t20060 = exp(t20055);
                    t20061 = Ei(-t20055);
                }
                double t20062 = 1.0/t20042;
                double t20063 = c*c0*t20040*t20045*t20054*t20029*(2.0/3.0);
                double t20064 = sqrt(t20063);
                double t20065 = 1.0/(t20042*t20042*t20042);
                double t20066 = t20062*t20064*6.0;
                double t20067 = c*c0*t20040*t20043*t20045*t20054*t20029*4.0;
                double t20068 = t20066+t20067+3.0;
                double t20069 = 1.0/t20068;
                double t20070 = a5*c0*t20040*t20043*t20054*t20028*t20056*(2.0/3.0);
                double t20071 = c*c0*gamma_bb*t20040*t20033*t20043*t20035*t20045*t20054*t20059*(8.0/3.0);
                double t20082 = c*c0*gamma_bb*t20040*t20033*t20035*t20045*t20054*t20065*t20059*(8.0/3.0);
                double t20072 = t20070+t20071-t20082;
                double t20073 = c*c0*t20040*t20060*t20043*t20061*t20045*t20054*t20029*(2.0/3.0);
                double t20074 = t20073+1.0;
                double t20075 = t20062*t20064*4.0;
                double t20076 = c*c0*t20040*t20043*t20045*t20054*t20029*(8.0/3.0);
                double t20077 = t20075+t20076+6.0;
                double t20078 = a5*c0*t20040*t20054*t20028*t20056*(2.0/3.0);
                double t20079 = c*c0*gamma_bb*t20040*t20033*t20035*t20045*t20054*t20059*(8.0/3.0);
                double t20080 = t20078+t20079;
                double t20081 = 1.0/sqrt(t20063);
                double t20083 = t20030*t20031*t20032*t20033;
                double t20084 = pow(t20034,4.0/5.0);
                double t20192 = rab1*t20084;
                double t20085 = exp(-t20192);
                double t20086 = t20085-1.0;
                double t20193 = kab1*t20086;
                double t20087 = kab0-t20193;
                double t20088 = a3*gamma_bb*t20036*t20028*t20037;
                double t20089 = t20088+1.0;
                double t20090 = 1.0/sqrt(t20089);
                double t20091 = a1*gamma_bb*t20028*t20056;
                double t20092 = t20091+t20083+1.0;
                double t20093 = c*c;
                double t20094 = 1.0/pow(rho_b,2.0/3.0);
                double t20095 = k1*t20038;
                double t20096 = c*k2*t20029;
                double t20097 = t20095+t20096;
                double t20098 = 1.0/(t20097*t20097);
                double t20195 = t20093*t20094*t20098;
                double t20099 = exp(-t20195);
                double t20100 = 1.0/pow(rho_b,4.0/3.0);
                double t20101 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b);
                double t20102 = 1.0/pow(rho_b,2.0E1/3.0);
                double t20103 = t20035*t20035;
                double t20104 = 1.0/pow(rho_b,1.2E1);
                double t20105 = t20033*t20033;
                double t20106 = 1.0/sqrt(t20034);
                double t20107 = c*kaa2*raa2*t20100*t20106*t20046*(1.0/6.0);
                double t20108 = 1.0/pow(t20034,3.0/5.0);
                double t20109 = c*kaa1*raa1*t20100*t20050*t20108*(2.0/1.5E1);
                double t20110 = t20107+t20109;
                double t20111 = 1.0/(t20053*t20053*t20053);
                double t20112 = t20062*t20080*t20081*2.0;
                double t20113 = a5*c0*t20040*t20043*t20054*t20028*t20056*(8.0/3.0);
                double t20114 = c*c0*gamma_bb*t20040*t20033*t20043*t20035*t20045*t20054*t20059*(3.2E1/3.0);
                double t20156 = gamma_bb*t20032*t20033*t20043*t20035*t20064*8.0;
                double t20157 = c*c0*gamma_bb*t20040*t20033*t20035*t20045*t20054*t20065*t20059*(3.2E1/3.0);
                double t20115 = t20112+t20113+t20114-t20156-t20157;
                double t20116 = 1.0/pow(rho_b,1.1E1/3.0);
                double t20117 = c*c0*t20100*t20040*t20043*t20045*t20054*(2.0/9.0);
                double t20118 = a5*c0*gamma_bb*t20040*t20043*t20116*t20054*t20028*(1.4E1/9.0);
                double t20119 = c*c0*t20110*t20111*t20040*t20043*t20045*t20029*(4.0/3.0);
                double t20120 = c*c0*t20102*t20031*t20040*t20033*t20043*t20035*t20045*t20054*(6.4E1/9.0);
                double t20121 = a5*c0*t20040*t20060*t20043*t20061*t20054*t20028*t20056*(2.0/3.0);
                double t20122 = c*c0*t20040*t20060*t20043*t20061*t20045*t20054*t20072*t20029*(2.0/3.0);
                double t20123 = c*c0*gamma_bb*t20040*t20033*t20060*t20043*t20061*t20035*t20045*t20054*t20059*(8.0/3.0);
                double t20136 = c*c0*t20102*t20031*t20040*t20033*t20035*t20045*t20054*t20065*(6.4E1/9.0);
                double t20124 = t20120+t20117+t20118-t20136+t20119;
                double t20125 = 1.0/pow(rho_b,1.9E1/3.0);
                double t20126 = 1.0/t20045;
                double t20127 = a5*c0*t20040*t20043*t20116*t20054*t20028*(1.6E1/9.0);
                double t20128 = a5*c0*t20110*t20111*t20040*t20043*t20028*t20056*(4.0/3.0);
                double t20129 = a5*c0*t20101*t20031*t20040*t20033*t20043*t20035*t20054*t20028*(4.0E1/3.0);
                double t20130 = c*c0*gamma_bb*t20102*t20040*t20033*t20043*t20035*t20045*t20054*(1.36E2/9.0);
                double t20131 = c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20043*t20045*t20054*(2.56E2/9.0);
                double t20132 = 1.0/(t20042*t20042*t20042*t20042);
                double t20133 = c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20132*t20045*t20054*(1.28E2/3.0);
                double t20134 = c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20043*t20035*t20045*t20059*(1.6E1/3.0);
                double t20137 = c*c0*gamma_bb*t20102*t20040*t20033*t20035*t20045*t20054*t20065*(1.36E2/9.0);
                double t20138 = c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20045*t20054*t20065*(5.12E2/9.0);
                double t20139 = c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20035*t20045*t20065*t20059*(1.6E1/3.0);
                double t20188 = a5*c0*t20101*t20031*t20040*t20033*t20035*t20054*t20028*t20065*(4.0E1/3.0);
                double t20135 = t20130+t20131+t20133+t20134+t20127+t20128-t20137+t20129-t20138-t20139-t20188;
                double t20140 = c*c0*t20100*t20040*t20045*t20054*(2.0/9.0);
                double t20141 = a5*c0*gamma_bb*t20040*t20116*t20054*t20028*(1.4E1/9.0);
                double t20142 = c*c0*t20110*t20111*t20040*t20045*t20029*(4.0/3.0);
                double t20143 = c*c0*t20102*t20031*t20040*t20033*t20035*t20045*t20054*(6.4E1/9.0);
                double t20144 = t20140+t20141+t20142+t20143;
                double t20145 = 1.0/(t20068*t20068);
                double t20211 = c*c0*gamma_bb*t20040*t20033*t20060*t20061*t20035*t20045*t20054*t20065*t20059*(8.0/3.0);
                double t20146 = t20121-t20211+t20122+t20123+t20070+t20071-t20082;
                double t20147 = a5*c0*t20040*t20116*t20054*t20028*(1.6E1/9.0);
                double t20148 = a5*c0*t20110*t20111*t20040*t20028*t20056*(4.0/3.0);
                double t20149 = a5*c0*t20101*t20031*t20040*t20033*t20035*t20054*t20028*(4.0E1/3.0);
                double t20150 = c*c0*gamma_bb*t20102*t20040*t20033*t20035*t20045*t20054*(1.36E2/9.0);
                double t20151 = c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20045*t20054*(2.56E2/9.0);
                double t20152 = c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20035*t20045*t20059*(1.6E1/3.0);
                double t20153 = t20150+t20151+t20152+t20147+t20148+t20149;
                double t20154 = 1.0/pow(t20063,3.0/2.0);
                double t20155 = 1.0/pow(rho_b,3.5E1/3.0);
                double t20158 = t20062*t20081*t20144*3.0;
                double t20159 = c*c0*t20100*t20040*t20043*t20045*t20054*(4.0/3.0);
                double t20160 = a5*c0*gamma_bb*t20040*t20043*t20116*t20054*t20028*(2.8E1/3.0);
                double t20161 = c*c0*t20110*t20111*t20040*t20043*t20045*t20029*8.0;
                double t20162 = c*c0*t20102*t20031*t20040*t20033*t20043*t20035*t20045*t20054*(1.28E2/3.0);
                double t20181 = t20031*t20033*t20043*t20035*t20125*t20064*3.2E1;
                double t20182 = c*c0*t20102*t20031*t20040*t20033*t20035*t20045*t20054*t20065*(1.28E2/3.0);
                double t20163 = t20160+t20161+t20162-t20181-t20182+t20158+t20159;
                double t20164 = t20062*t20080*t20081*3.0;
                double t20165 = a5*c0*t20040*t20043*t20054*t20028*t20056*4.0;
                double t20166 = c*c0*gamma_bb*t20040*t20033*t20043*t20035*t20045*t20054*t20059*1.6E1;
                double t20174 = gamma_bb*t20032*t20033*t20043*t20035*t20064*1.2E1;
                double t20175 = c*c0*gamma_bb*t20040*t20033*t20035*t20045*t20054*t20065*t20059*1.6E1;
                double t20167 = t20164+t20165-t20174+t20166-t20175;
                double t20168 = t20062*t20081*t20144*2.0;
                double t20169 = c*c0*t20100*t20040*t20043*t20045*t20054*(8.0/9.0);
                double t20170 = a5*c0*gamma_bb*t20040*t20043*t20116*t20054*t20028*(5.6E1/9.0);
                double t20171 = c*c0*t20110*t20111*t20040*t20043*t20045*t20029*(1.6E1/3.0);
                double t20172 = c*c0*t20102*t20031*t20040*t20033*t20043*t20035*t20045*t20054*(2.56E2/9.0);
                double t20206 = t20031*t20033*t20043*t20035*t20125*t20064*(6.4E1/3.0);
                double t20207 = c*c0*t20102*t20031*t20040*t20033*t20035*t20045*t20054*t20065*(2.56E2/9.0);
                double t20173 = t20170-t20206+t20171-t20207+t20172+t20168+t20169;
                double t20176 = c*c0*t20100*t20040*t20060*t20043*t20061*t20045*t20054*(2.0/9.0);
                double t20177 = c*c0*t20110*t20111*t20040*t20060*t20043*t20061*t20045*t20029*(4.0/3.0);
                double t20178 = c*c0*t20040*t20060*t20043*t20061*t20124*t20045*t20054*t20029*(2.0/3.0);
                double t20179 = a5*c0*gamma_bb*t20040*t20060*t20043*t20061*t20116*t20054*t20028*(1.4E1/9.0);
                double t20180 = c*c0*t20102*t20031*t20040*t20033*t20060*t20043*t20061*t20035*t20045*t20054*(6.4E1/9.0);
                double t20183 = 1.0/c0;
                double t20184 = pow(rho_b,1.0/3.0);
                double t20185 = exp(-t20039);
                double t20186 = t20042*t20042;
                double t20187 = t20053*t20053;
                double t20189 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b);
                double t20208 = c*c0*t20102*t20031*t20040*t20033*t20060*t20061*t20035*t20045*t20054*t20065*(6.4E1/9.0);
                double t20190 = t20120+t20117+t20180+t20118-t20136-t20208+t20119+t20176+t20177+t20178+t20179;
                double t20191 = exp(-t20083);
                double t20194 = t20090*t20092*t20191*t20087*1.0E5;
                double t20196 = c*c0*t20029*(2.0/3.0);
                double t20197 = -t20194+t20196;
                double t20198 = dirac(t20197);
                double t20199 = a1*t20028*t20056;
                double t20200 = gamma_bb*t20030*t20032*t20033*2.0;
                double t20201 = t20200+t20199;
                double t20202 = 1.0/pow(t20089,3.0/2.0);
                double t20203 = a3*t20202*t20036*t20028*t20037*t20092*t20191*t20087*5.0E4;
                double t20204 = gamma_bb*t20030*t20032*t20033*t20090*t20092*t20191*t20087*2.0E5;
                double t20231 = t20201*t20090*t20191*t20087*1.0E5;
                double t20205 = t20203+t20204-t20231;
                double t20209 = t20060*t20061*t20072;
                double t20210 = t20115*t20074*t20069;
                double t20212 = t20036*t20072*t20126*t20183*t20184*t20185*t20186*t20187*(3.0/2.0);
                double t20213 = t20060*t20061;
                double t20214 = t20074*t20077*t20069;
                double t20215 = t20213+t20214;
                double t20216 = 1.0/pow(rho_b,1.0E1/3.0);
                double t20217 = t20030*t20031*t20033*t20125*(1.6E1/3.0);
                double t20218 = a1*gamma_bb*t20116*t20028*(8.0/3.0);
                double t20219 = t20217+t20218;
                double t20220 = 1.0/(c*c);
                double t20221 = 1.0/pow(t20034,1.0/5.0);
                double t20222 = t20146*t20077*t20069;
                double t20223 = t20210+t20212+t20222+t20209-t20145*t20074*t20077*t20167;
                double t20224 = 1.0/pow(rho_b,5.0/3.0);
                double t20225 = t20224*t20093*t20098*(2.0/3.0);
                double t20226 = 1.0/(t20097*t20097*t20097);
                double t20227 = c*k2*t20100*(1.0/3.0);
                double t20228 = c*k1*t20100*t20106*(1.0/6.0);
                double t20229 = t20227+t20228;
                double t20230 = t20225-t20226*t20093*t20094*t20229*2.0;
                double t20232 = c*c0*t20100*(2.0/9.0);
                double t20233 = t20030*t20031*t20033*t20125*t20090*t20092*t20191*t20087*(1.6E6/3.0);
                double t20234 = a3*gamma_bb*t20202*t20036*t20216*t20028*t20092*t20191*t20087*(3.5E5/3.0);
                double t20235 = t20232+t20233+t20234-t20090*t20191*t20219*t20087*1.0E5-c*kab1*rab1*t20100*t20221*t20090*t20092*t20191*t20085*(8.0E4/3.0);
                v_rho_b_gamma_bb[Q] += scale * (c0*t20099*heaviside(c*c0*t20029*(-2.0/3.0)+t20090*t20092*t20087*exp(-t20030*t20031*t20032*t20033)*1.0E5)*(t20210+t20212+t20209+t20077*t20069*(t20121+t20122+t20123+t20070+t20071-c*c0*gamma_bb*t20040*t20033*t20035*t20045*t20054*t20065*t20059*(8.0/3.0)-c*c0*gamma_bb*t20040*t20033*t20060*t20061*t20035*t20045*t20054*t20065*t20059*(8.0/3.0))-t20145*t20074*t20077*t20167)*(1.0/2.0)-c0*t20205*t20215*t20099*t20198*(1.0/2.0)-c0*rho_b*t20099*heaviside(t20194-c*c0*t20029*(2.0/3.0))*(t20074*t20069*(t20062*t20081*t20153*2.0-t20062*t20080*t20144*t20154-gamma_bb*t20033*t20043*t20035*t20125*t20064*(1.28E2/3.0)+a5*c0*t20040*t20043*t20116*t20054*t20028*(6.4E1/9.0)-gamma_bb*t20032*t20033*t20043*t20035*t20081*t20144*4.0+gamma_bb*t20031*t20103*t20105*t20064*t20065*t20155*(2.56E2/3.0)-t20031*t20033*t20043*t20035*t20080*t20125*t20081*(3.2E1/3.0)+a5*c0*t20110*t20111*t20040*t20043*t20028*t20056*(1.6E1/3.0)+c*c0*gamma_bb*t20102*t20040*t20033*t20043*t20035*t20045*t20054*(5.44E2/9.0)-c*c0*gamma_bb*t20102*t20040*t20033*t20035*t20045*t20054*t20065*(5.44E2/9.0)+a5*c0*t20101*t20031*t20040*t20033*t20043*t20035*t20054*t20028*(1.6E2/3.0)-a5*c0*t20101*t20031*t20040*t20033*t20035*t20054*t20028*t20065*(1.6E2/3.0)+c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20132*t20045*t20054*(5.12E2/3.0)+c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20043*t20045*t20054*(1.024E3/9.0)-c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20045*t20054*t20065*(2.048E3/9.0)+c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20043*t20035*t20045*t20059*(6.4E1/3.0)-c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20035*t20045*t20065*t20059*(6.4E1/3.0))+t20060*t20061*t20135+t20115*t20190*t20069+t20146*t20173*t20069+t20077*t20069*(t20130+t20131+t20133+t20134+t20127+t20128-t20137+t20129-t20138-t20139-t20188+t20124*t20072+(t20072*(1.0/3.0))/rho_b+(t20110*t20072*2.0)/t20053+t20031*t20033*t20035*t20125*t20072*(3.2E1/3.0)-t20031*t20033*t20035*t20062*t20125*t20072*(3.2E1/3.0)+a5*gamma_bb*t20036*t20072*t20126*t20216*t20028*(7.0/3.0)+a5*c0*t20040*t20060*t20043*t20061*t20116*t20054*t20028*(1.6E1/9.0)+a5*c0*t20110*t20111*t20040*t20060*t20043*t20061*t20028*t20056*(4.0/3.0)+a5*c0*t20040*t20060*t20043*t20061*t20124*t20054*t20028*t20056*(2.0/3.0)+c*c0*t20100*t20040*t20060*t20043*t20061*t20045*t20054*t20072*(2.0/9.0)+c*c0*t20040*t20060*t20043*t20061*t20045*t20054*t20135*t20029*(2.0/3.0)+a5*c0*gamma_bb*t20040*t20060*t20043*t20061*t20116*t20054*t20072*t20028*(1.4E1/9.0)+c*c0*t20110*t20111*t20040*t20060*t20043*t20061*t20045*t20072*t20029*(4.0/3.0)+c*c0*t20040*t20060*t20043*t20061*t20124*t20045*t20054*t20072*t20029*(2.0/3.0)+c*c0*gamma_bb*t20102*t20040*t20033*t20060*t20043*t20061*t20035*t20045*t20054*(1.36E2/9.0)-c*c0*gamma_bb*t20102*t20040*t20033*t20060*t20061*t20035*t20045*t20054*t20065*(1.36E2/9.0)+a5*c0*t20101*t20031*t20040*t20033*t20060*t20043*t20061*t20035*t20054*t20028*(4.0E1/3.0)-a5*c0*t20101*t20031*t20040*t20033*t20060*t20061*t20035*t20054*t20028*t20065*(4.0E1/3.0)+c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20060*t20105*t20132*t20061*t20045*t20054*(1.28E2/3.0)+c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20060*t20105*t20043*t20061*t20045*t20054*(2.56E2/9.0)-c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20060*t20105*t20061*t20045*t20054*t20065*(5.12E2/9.0)+c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20060*t20043*t20061*t20035*t20045*t20059*(1.6E1/3.0)-c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20060*t20061*t20035*t20045*t20065*t20059*(1.6E1/3.0)+c*c0*gamma_bb*t20040*t20033*t20060*t20043*t20061*t20124*t20035*t20045*t20054*t20059*(8.0/3.0)-c*c0*gamma_bb*t20040*t20033*t20060*t20061*t20124*t20035*t20045*t20054*t20065*t20059*(8.0/3.0)+c*c0*t20102*t20031*t20040*t20033*t20060*t20043*t20061*t20035*t20045*t20054*t20072*(6.4E1/9.0)-c*c0*t20102*t20031*t20040*t20033*t20060*t20061*t20035*t20045*t20054*t20072*t20065*(6.4E1/9.0))+t20060*t20061*t20124*t20072-t20115*t20145*t20163*t20074-t20145*t20163*t20146*t20077-t20145*t20074*t20173*t20167-t20145*t20190*t20077*t20167-t20145*t20074*t20077*(t20062*t20081*t20153*3.0-t20062*t20080*t20144*t20154*(3.0/2.0)-gamma_bb*t20033*t20043*t20035*t20125*t20064*6.4E1+a5*c0*t20040*t20043*t20116*t20054*t20028*(3.2E1/3.0)-gamma_bb*t20032*t20033*t20043*t20035*t20081*t20144*6.0+gamma_bb*t20031*t20103*t20105*t20064*t20065*t20155*1.28E2-t20031*t20033*t20043*t20035*t20080*t20125*t20081*1.6E1+a5*c0*t20110*t20111*t20040*t20043*t20028*t20056*8.0+c*c0*gamma_bb*t20102*t20040*t20033*t20043*t20035*t20045*t20054*(2.72E2/3.0)-c*c0*gamma_bb*t20102*t20040*t20033*t20035*t20045*t20054*t20065*(2.72E2/3.0)+a5*c0*t20101*t20031*t20040*t20033*t20043*t20035*t20054*t20028*8.0E1-a5*c0*t20101*t20031*t20040*t20033*t20035*t20054*t20028*t20065*8.0E1+c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20132*t20045*t20054*2.56E2+c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20043*t20045*t20054*(5.12E2/3.0)-c*c0*gamma_bb*t20031*t20040*t20103*t20104*t20105*t20045*t20054*t20065*(1.024E3/3.0)+c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20043*t20035*t20045*t20059*3.2E1-c*c0*gamma_bb*t20110*t20111*t20040*t20033*t20035*t20045*t20065*t20059*3.2E1)+t20163*t20074*1.0/(t20068*t20068*t20068)*t20077*t20167*2.0+t20036*t20126*t20135*t20183*t20184*t20185*t20186*t20187*(3.0/2.0)+t20124*t20036*t20072*t20126*t20183*t20184*t20185*t20186*t20187*(3.0/2.0)-a5*1.0/(rho_b*rho_b)*t20220*t20124*1.0/(t20045*t20045)*t20028*t20183*t20185*t20186*t20187*(3.0/2.0)+gamma_bb*t20033*t20042*t20124*t20035*t20036*t20126*t20183*t20185*t20187*t20189*6.0-gamma_bb*t20033*t20124*t20035*t20036*t20126*t20183*t20185*t20186*t20187*t20189*6.0)*(1.0/2.0)+c0*rho_b*t20223*t20235*t20099*t20198*(1.0/2.0)+c0*rho_b*t20215*t20099*t20198*(t20090*t20191*t20087*(a1*t20116*t20028*(8.0/3.0)+gamma_bb*t20030*t20033*t20125*(3.2E1/3.0))*-1.0E5+t20030*t20201*t20031*t20033*t20125*t20090*t20191*t20087*(1.6E6/3.0)+a3*t20202*t20036*t20216*t20028*t20092*t20191*t20087*(3.5E5/3.0)+a3*t20202*t20036*t20028*t20037*t20191*t20219*t20087*5.0E4+gamma_bb*t20030*t20032*t20033*t20090*t20191*t20219*t20087*2.0E5+gamma_bb*t20030*t20033*t20125*t20090*t20092*t20191*t20087*(3.2E6/3.0)-(a3*a3)*gamma_bb*t20220*t20033*t20092*t20191*t20059*t20087*1.0/pow(t20089,5.0/2.0)*1.75E5+a3*gamma_bb*t20201*t20202*t20036*t20216*t20028*t20191*t20087*(3.5E5/3.0)-c*kab1*rab1*t20100*t20201*t20221*t20090*t20191*t20085*(8.0E4/3.0)-gamma_bb*(t20030*t20030)*t20031*t20105*t20090*t20092*t20155*t20191*t20087*(3.2E6/3.0)+a3*kab1*rab1*t20202*t20221*t20116*t20028*t20092*t20191*t20085*(4.0E4/3.0)-a3*1.0/pow(rho_b,2.6E1/3.0)*t20030*t20031*t20202*t20033*t20036*t20028*t20092*t20191*t20087*5.0E5+c*gamma_bb*kab1*rab1*t20030*t20102*t20221*t20033*t20090*t20092*t20191*t20085*(1.6E5/3.0))*(1.0/2.0)+c0*rho_b*t20205*t20099*t20198*(t20060*t20061*t20124+t20074*t20173*t20069+t20190*t20077*t20069-t20145*t20163*t20074*t20077+t20124*t20036*t20126*t20183*t20184*t20185*t20186*t20187*(3.0/2.0))*(1.0/2.0)+c0*rho_b*t20230*t20223*t20099*heaviside(t20194-t20196)*(1.0/2.0)-c0*rho_b*t20230*t20205*t20215*t20099*t20198*(1.0/2.0)+c0*rho_b*t20205*t20215*t20235*t20099*dirac(t20197,1.0)*(1.0/2.0));
            }
            
        } else if (rho_b < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t20394 = tspi_m13*tspi_m13;
                double t20395 = 1.0/pow(rho_a,1.0/3.0);
                double t20396 = a2*a2;
                double t20397 = gamma_aa*gamma_aa;
                double t20398 = 1.0/pow(rho_a,1.6E1/3.0);
                double t20399 = t20394*t20394;
                double t20400 = c*t20395;
                double t20401 = a4*a4;
                double t20402 = 1.0/c;
                double t20403 = 1.0/pow(rho_a,7.0/3.0);
                double t20404 = sqrt(t20400);
                double t20405 = t20401*t20397*t20398*t20399*2.0;
                double t20406 = exp(t20405);
                double t20407 = t20401*t20397*t20398*t20399;
                double t20408 = t20407+1.0;
                double t20409 = 1.0/(t20408*t20408);
                double t20410 = a5*gamma_aa*t20402*t20403*t20394;
                double t20411 = t20410+1.0;
                double t20422 = raa2*t20404;
                double t20412 = exp(-t20422);
                double t20413 = t20412-1.0;
                double t20414 = kaa2*t20413;
                double t20415 = pow(t20400,2.0/5.0);
                double t20423 = raa1*t20415;
                double t20416 = exp(-t20423);
                double t20417 = t20416-1.0;
                double t20418 = kaa1*t20417;
                double t20419 = -kaa0+t20414+t20418;
                double t20420 = 1.0/(t20419*t20419);
                double t20421 = c*c0*t20411*t20420*t20406*t20409*t20395*(2.0/3.0);
                double t20424;
                double t20425;
                if (t20421 > expei_cutoff) {
                    t20424 = 1.0;
                    t20425 = expei(t20421);
                } else {
                    t20424 = exp(t20421);
                    t20425 = Ei(-t20421);
                }
                double t20426 = 1.0/t20408;
                double t20427 = c*c0*t20411*t20420*t20406*t20395*(2.0/3.0);
                double t20428 = sqrt(t20427);
                v[Q] += scale * (c0*rho_a*exp(-(c*c)*1.0/pow(rho_a,2.0/3.0)*1.0/pow(k1*t20404+c*k2*t20395,2.0))*heaviside(c*c0*t20395*(-2.0/3.0)+exp(-t20396*t20397*t20398*t20399)*1.0/sqrt(a3*gamma_aa*t20402*t20403*t20394+1.0)*(kab0-kab1*(exp(-rab1*pow(t20400,4.0/5.0))-1.0))*(t20396*t20397*t20398*t20399+a1*gamma_aa*1.0/pow(rho_a,8.0/3.0)*t20394+1.0)*1.0E5)*(t20424*t20425+((c*c0*t20411*t20420*t20406*t20424*t20425*t20409*t20395*(2.0/3.0)+1.0)*(t20426*t20428*4.0+c*c0*t20411*t20420*t20406*t20409*t20395*(8.0/3.0)+6.0))/(t20426*t20428*6.0+c*c0*t20411*t20420*t20406*t20409*t20395*4.0+3.0))*(1.0/2.0));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t20430 = tspi_m13*tspi_m13;
                double t20431 = 1.0/pow(rho_a,1.0/3.0);
                double t20432 = a2*a2;
                double t20433 = gamma_aa*gamma_aa;
                double t20434 = 1.0/pow(rho_a,1.6E1/3.0);
                double t20435 = t20430*t20430;
                double t20436 = c*t20431;
                double t20437 = a4*a4;
                double t20438 = 1.0/c;
                double t20439 = 1.0/pow(rho_a,7.0/3.0);
                double t20440 = sqrt(t20436);
                double t20441 = t20433*t20434*t20435*t20437*2.0;
                double t20442 = exp(t20441);
                double t20443 = t20433*t20434*t20435*t20437;
                double t20444 = t20443+1.0;
                double t20445 = 1.0/(t20444*t20444);
                double t20446 = a5*gamma_aa*t20430*t20438*t20439;
                double t20447 = t20446+1.0;
                double t20458 = raa2*t20440;
                double t20448 = exp(-t20458);
                double t20449 = t20448-1.0;
                double t20450 = kaa2*t20449;
                double t20451 = pow(t20436,2.0/5.0);
                double t20459 = raa1*t20451;
                double t20452 = exp(-t20459);
                double t20453 = t20452-1.0;
                double t20454 = kaa1*t20453;
                double t20455 = -kaa0+t20450+t20454;
                double t20456 = 1.0/(t20455*t20455);
                double t20457 = c*c0*t20431*t20442*t20445*t20447*t20456*(2.0/3.0);
                double t20460;
                double t20461;
                if (t20457 > expei_cutoff) {
                    t20460 = 1.0;
                    t20461 = expei(t20457);
                } else {
                    t20460 = exp(t20457);
                    t20461 = Ei(-t20457);
                }
                double t20462 = 1.0/t20444;
                double t20463 = c*c0*t20431*t20442*t20447*t20456*(2.0/3.0);
                double t20464 = sqrt(t20463);
                double t20465 = t20432*t20433*t20434*t20435;
                double t20466 = pow(t20436,4.0/5.0);
                double t20516 = rab1*t20466;
                double t20467 = exp(-t20516);
                double t20468 = t20467-1.0;
                double t20517 = kab1*t20468;
                double t20469 = kab0-t20517;
                double t20470 = a3*gamma_aa*t20430*t20438*t20439;
                double t20471 = t20470+1.0;
                double t20472 = 1.0/sqrt(t20471);
                double t20473 = 1.0/pow(rho_a,8.0/3.0);
                double t20474 = a1*gamma_aa*t20430*t20473;
                double t20475 = t20465+t20474+1.0;
                double t20476 = c*c;
                double t20477 = 1.0/pow(rho_a,2.0/3.0);
                double t20478 = k1*t20440;
                double t20479 = c*k2*t20431;
                double t20480 = t20478+t20479;
                double t20481 = 1.0/(t20480*t20480);
                double t20521 = t20481*t20476*t20477;
                double t20482 = exp(-t20521);
                double t20483 = c*c0*t20431*t20442*t20460*t20461*t20445*t20447*t20456*(2.0/3.0);
                double t20484 = t20483+1.0;
                double t20485 = t20462*t20464*6.0;
                double t20486 = c*c0*t20431*t20442*t20445*t20447*t20456*4.0;
                double t20487 = t20485+t20486+3.0;
                double t20488 = 1.0/t20487;
                double t20489 = 1.0/pow(rho_a,4.0/3.0);
                double t20490 = 1.0/pow(rho_a,1.1E1/3.0);
                double t20491 = 1.0/sqrt(t20436);
                double t20492 = c*kaa2*raa2*t20491*t20448*t20489*(1.0/6.0);
                double t20493 = 1.0/pow(t20436,3.0/5.0);
                double t20494 = c*kaa1*raa1*t20452*t20493*t20489*(2.0/1.5E1);
                double t20495 = t20492+t20494;
                double t20496 = 1.0/(t20455*t20455*t20455);
                double t20497 = 1.0/pow(rho_a,2.0E1/3.0);
                double t20498 = 1.0/(t20444*t20444*t20444);
                double t20499 = t20462*t20464*4.0;
                double t20500 = c*c0*t20431*t20442*t20445*t20447*t20456*(8.0/3.0);
                double t20501 = t20500+t20499+6.0;
                double t20502 = c*c0*t20442*t20445*t20447*t20456*t20489*(2.0/9.0);
                double t20503 = a5*c0*gamma_aa*t20430*t20442*t20445*t20490*t20456*(1.4E1/9.0);
                double t20504 = c*c0*t20431*t20442*t20445*t20447*t20495*t20496*(4.0/3.0);
                double t20505 = c*c0*t20433*t20442*t20435*t20445*t20437*t20447*t20456*t20497*(6.4E1/9.0);
                double t20507 = c*c0*t20433*t20442*t20435*t20437*t20447*t20456*t20497*t20498*(6.4E1/9.0);
                double t20506 = t20502+t20503+t20504+t20505-t20507;
                double t20508 = c*c0*t20442*t20447*t20456*t20489*(2.0/9.0);
                double t20509 = a5*c0*gamma_aa*t20430*t20442*t20490*t20456*(1.4E1/9.0);
                double t20510 = c*c0*t20431*t20442*t20447*t20495*t20496*(4.0/3.0);
                double t20511 = c*c0*t20433*t20442*t20435*t20437*t20447*t20456*t20497*(6.4E1/9.0);
                double t20512 = t20510+t20511+t20508+t20509;
                double t20513 = 1.0/sqrt(t20463);
                double t20514 = 1.0/pow(rho_a,1.9E1/3.0);
                double t20515 = exp(-t20465);
                double t20518 = t20515*t20472*t20475*t20469*1.0E5;
                double t20525 = c*c0*t20431*(2.0/3.0);
                double t20519 = -t20525+t20518;
                double t20520 = heaviside(t20519);
                double t20522 = t20460*t20461;
                double t20523 = t20501*t20484*t20488;
                double t20524 = t20522+t20523;
                v_rho_a[Q] += scale * (c0*t20524*t20482*heaviside(c*c0*t20431*(-2.0/3.0)+t20472*t20475*t20469*exp(-t20432*t20433*t20434*t20435)*1.0E5)*(1.0/2.0)-c0*rho_a*t20520*t20482*(t20484*t20488*(t20512*t20513*t20462*2.0-t20433*t20514*t20435*t20445*t20437*t20464*(6.4E1/3.0)+c*c0*t20442*t20445*t20447*t20456*t20489*(8.0/9.0)+a5*c0*gamma_aa*t20430*t20442*t20445*t20490*t20456*(5.6E1/9.0)+c*c0*t20431*t20442*t20445*t20447*t20495*t20496*(1.6E1/3.0)+c*c0*t20433*t20442*t20435*t20445*t20437*t20447*t20456*t20497*(2.56E2/9.0)-c*c0*t20433*t20442*t20435*t20437*t20447*t20456*t20497*t20498*(2.56E2/9.0))+t20501*t20488*(t20502+t20503+t20504+t20505-t20507+c*c0*t20442*t20460*t20461*t20445*t20447*t20456*t20489*(2.0/9.0)+a5*c0*gamma_aa*t20430*t20442*t20460*t20461*t20445*t20490*t20456*(1.4E1/9.0)+c*c0*t20431*t20442*t20460*t20461*t20506*t20445*t20447*t20456*(2.0/3.0)+c*c0*t20431*t20442*t20460*t20461*t20445*t20447*t20495*t20496*(4.0/3.0)+c*c0*t20433*t20442*t20460*t20461*t20435*t20445*t20437*t20447*t20456*t20497*(6.4E1/9.0)-c*c0*t20433*t20442*t20460*t20461*t20435*t20437*t20447*t20456*t20497*t20498*(6.4E1/9.0))+t20460*t20461*t20506-t20501*t20484*1.0/(t20487*t20487)*(t20512*t20513*t20462*3.0-t20433*t20514*t20435*t20445*t20437*t20464*3.2E1+c*c0*t20442*t20445*t20447*t20456*t20489*(4.0/3.0)+a5*c0*gamma_aa*t20430*t20442*t20445*t20490*t20456*(2.8E1/3.0)+c*c0*t20431*t20442*t20445*t20447*t20495*t20496*8.0+c*c0*t20433*t20442*t20435*t20445*t20437*t20447*t20456*t20497*(1.28E2/3.0)-c*c0*t20433*t20442*t20435*t20437*t20447*t20456*t20497*t20498*(1.28E2/3.0))+(pow(rho_a,1.0/3.0)*t20506*(t20444*t20444)*(t20455*t20455)*t20438*exp(-t20441)*(3.0/2.0))/(c0*t20447))*(1.0/2.0)+c0*rho_a*t20520*t20524*t20482*(1.0/pow(rho_a,5.0/3.0)*t20481*t20476*(2.0/3.0)-1.0/(t20480*t20480*t20480)*t20476*t20477*(c*k2*t20489*(1.0/3.0)+c*k1*t20491*t20489*(1.0/6.0))*2.0)*(1.0/2.0)+c0*rho_a*t20524*t20482*dirac(t20525-t20518)*(c*c0*t20489*(2.0/9.0)-t20515*t20472*t20469*(a1*gamma_aa*t20430*t20490*(8.0/3.0)+t20432*t20433*t20514*t20435*(1.6E1/3.0))*1.0E5+t20432*t20433*t20514*t20515*t20435*t20472*t20475*t20469*(1.6E6/3.0)+a3*gamma_aa*1.0/pow(rho_a,1.0E1/3.0)*t20430*t20515*1.0/pow(t20471,3.0/2.0)*t20438*t20475*t20469*(3.5E5/3.0)-c*kab1*rab1*t20515*1.0/pow(t20436,1.0/5.0)*t20472*t20475*t20467*t20489*(8.0E4/3.0))*(1.0/2.0));
            }
            
            // v_gamma_aa
            if (deriv >= 1) {
                double t20528 = tspi_m13*tspi_m13;
                double t20529 = 1.0/pow(rho_a,1.0/3.0);
                double t20530 = a2*a2;
                double t20531 = gamma_aa*gamma_aa;
                double t20532 = 1.0/pow(rho_a,1.6E1/3.0);
                double t20533 = t20528*t20528;
                double t20534 = c*t20529;
                double t20535 = a4*a4;
                double t20536 = 1.0/c;
                double t20537 = 1.0/pow(rho_a,7.0/3.0);
                double t20538 = sqrt(t20534);
                double t20539 = t20531*t20532*t20533*t20535*2.0;
                double t20540 = exp(t20539);
                double t20541 = t20531*t20532*t20533*t20535;
                double t20542 = t20541+1.0;
                double t20543 = 1.0/(t20542*t20542);
                double t20544 = a5*gamma_aa*t20536*t20528*t20537;
                double t20545 = t20544+1.0;
                double t20557 = raa2*t20538;
                double t20546 = exp(-t20557);
                double t20547 = t20546-1.0;
                double t20548 = kaa2*t20547;
                double t20549 = pow(t20534,2.0/5.0);
                double t20558 = raa1*t20549;
                double t20550 = exp(-t20558);
                double t20551 = t20550-1.0;
                double t20552 = kaa1*t20551;
                double t20553 = -kaa0+t20552+t20548;
                double t20554 = 1.0/(t20553*t20553);
                double t20555 = c*c0*t20540*t20543*t20545*t20554*t20529*(2.0/3.0);
                double t20556 = 1.0/pow(rho_a,8.0/3.0);
                double t20559 = 1.0/pow(rho_a,1.7E1/3.0);
                double t20560;
                double t20561;
                if (t20555 > expei_cutoff) {
                    t20560 = 1.0;
                    t20561 = expei(t20555);
                } else {
                    t20560 = exp(t20555);
                    t20561 = Ei(-t20555);
                }
                double t20562 = 1.0/t20542;
                double t20563 = c*c0*t20540*t20545*t20554*t20529*(2.0/3.0);
                double t20564 = sqrt(t20563);
                double t20565 = 1.0/(t20542*t20542*t20542);
                double t20566 = t20562*t20564*6.0;
                double t20567 = c*c0*t20540*t20543*t20545*t20554*t20529*4.0;
                double t20568 = t20566+t20567+3.0;
                double t20569 = 1.0/t20568;
                double t20570 = a5*c0*t20540*t20543*t20554*t20528*t20556*(2.0/3.0);
                double t20571 = c*c0*gamma_aa*t20540*t20533*t20543*t20535*t20545*t20554*t20559*(8.0/3.0);
                double t20582 = c*c0*gamma_aa*t20540*t20533*t20535*t20545*t20554*t20565*t20559*(8.0/3.0);
                double t20572 = t20570+t20571-t20582;
                double t20573 = c*c0*t20540*t20560*t20543*t20561*t20545*t20554*t20529*(2.0/3.0);
                double t20574 = t20573+1.0;
                double t20575 = t20562*t20564*4.0;
                double t20576 = c*c0*t20540*t20543*t20545*t20554*t20529*(8.0/3.0);
                double t20577 = t20575+t20576+6.0;
                double t20578 = a5*c0*t20540*t20554*t20528*t20556*(2.0/3.0);
                double t20579 = c*c0*gamma_aa*t20540*t20533*t20535*t20545*t20554*t20559*(8.0/3.0);
                double t20580 = t20578+t20579;
                double t20581 = 1.0/sqrt(t20563);
                double t20583 = t20530*t20531*t20532*t20533;
                double t20584 = pow(t20534,4.0/5.0);
                double t20601 = rab1*t20584;
                double t20585 = exp(-t20601);
                double t20586 = t20585-1.0;
                double t20602 = kab1*t20586;
                double t20587 = kab0-t20602;
                double t20588 = a3*gamma_aa*t20536*t20528*t20537;
                double t20589 = t20588+1.0;
                double t20590 = 1.0/sqrt(t20589);
                double t20591 = a1*gamma_aa*t20528*t20556;
                double t20592 = t20591+t20583+1.0;
                double t20593 = c*c;
                double t20594 = 1.0/pow(rho_a,2.0/3.0);
                double t20595 = k1*t20538;
                double t20596 = c*k2*t20529;
                double t20597 = t20595+t20596;
                double t20598 = 1.0/(t20597*t20597);
                double t20599 = exp(-t20593*t20594*t20598);
                double t20600 = exp(-t20583);
                v_gamma_aa[Q] += scale * (c0*rho_a*t20599*heaviside(c*c0*t20529*(-2.0/3.0)+t20590*t20592*t20587*exp(-t20530*t20531*t20532*t20533)*1.0E5)*(t20574*t20569*(t20562*t20580*t20581*2.0-gamma_aa*t20532*t20533*t20543*t20535*t20564*8.0+a5*c0*t20540*t20543*t20554*t20528*t20556*(8.0/3.0)+c*c0*gamma_aa*t20540*t20533*t20543*t20535*t20545*t20554*t20559*(3.2E1/3.0)-c*c0*gamma_aa*t20540*t20533*t20535*t20545*t20554*t20565*t20559*(3.2E1/3.0))+t20577*t20569*(t20570+t20571+a5*c0*t20540*t20560*t20543*t20561*t20554*t20528*t20556*(2.0/3.0)-c*c0*gamma_aa*t20540*t20533*t20535*t20545*t20554*t20565*t20559*(8.0/3.0)+c*c0*t20540*t20560*t20543*t20561*t20545*t20554*t20572*t20529*(2.0/3.0)+c*c0*gamma_aa*t20540*t20533*t20560*t20543*t20561*t20535*t20545*t20554*t20559*(8.0/3.0)-c*c0*gamma_aa*t20540*t20533*t20560*t20561*t20535*t20545*t20554*t20565*t20559*(8.0/3.0))+t20560*t20561*t20572-t20574*1.0/(t20568*t20568)*t20577*(t20562*t20580*t20581*3.0-gamma_aa*t20532*t20533*t20543*t20535*t20564*1.2E1+a5*c0*t20540*t20543*t20554*t20528*t20556*4.0+c*c0*gamma_aa*t20540*t20533*t20543*t20535*t20545*t20554*t20559*1.6E1-c*c0*gamma_aa*t20540*t20533*t20535*t20545*t20554*t20565*t20559*1.6E1)+(pow(rho_a,1.0/3.0)*(t20542*t20542)*(t20553*t20553)*t20536*t20572*exp(-t20539)*(3.0/2.0))/(c0*t20545))*(1.0/2.0)-c0*rho_a*t20599*dirac(c*c0*t20529*(2.0/3.0)-t20600*t20590*t20592*t20587*1.0E5)*(t20560*t20561+t20574*t20577*t20569)*(t20600*t20590*t20587*(a1*t20528*t20556+gamma_aa*t20530*t20532*t20533*2.0)*-1.0E5+a3*t20600*t20536*t20528*t20537*t20592*t20587*1.0/pow(t20589,3.0/2.0)*5.0E4+gamma_aa*t20600*t20530*t20532*t20533*t20590*t20592*t20587*2.0E5)*(1.0/2.0));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t20608 = tspi_m13*tspi_m13;
                double t20609 = 1.0/pow(rho_a,1.0/3.0);
                double t20610 = a2*a2;
                double t20611 = gamma_aa*gamma_aa;
                double t20612 = 1.0/pow(rho_a,1.6E1/3.0);
                double t20613 = t20608*t20608;
                double t20614 = c*t20609;
                double t20615 = a4*a4;
                double t20616 = t20611*t20612*t20613*t20615*2.0;
                double t20617 = exp(t20616);
                double t20618 = 1.0/c;
                double t20619 = 1.0/pow(rho_a,7.0/3.0);
                double t20620 = sqrt(t20614);
                double t20621 = t20611*t20612*t20613*t20615;
                double t20622 = t20621+1.0;
                double t20623 = 1.0/(t20622*t20622);
                double t20624 = a5*gamma_aa*t20608*t20618*t20619;
                double t20625 = t20624+1.0;
                double t20636 = raa2*t20620;
                double t20626 = exp(-t20636);
                double t20627 = t20626-1.0;
                double t20628 = kaa2*t20627;
                double t20629 = pow(t20614,2.0/5.0);
                double t20637 = raa1*t20629;
                double t20630 = exp(-t20637);
                double t20631 = t20630-1.0;
                double t20632 = kaa1*t20631;
                double t20633 = -kaa0+t20632+t20628;
                double t20634 = 1.0/(t20633*t20633);
                double t20635 = c*c0*t20623*t20625*t20634*t20617*t20609*(2.0/3.0);
                double t20638 = 1.0/t20622;
                double t20639 = 1.0/pow(rho_a,4.0/3.0);
                double t20640 = c*c0*t20625*t20634*t20617*t20609*(2.0/3.0);
                double t20641 = sqrt(t20640);
                double t20642 = 1.0/pow(rho_a,1.1E1/3.0);
                double t20643 = 1.0/sqrt(t20614);
                double t20644 = c*kaa2*raa2*t20643*t20626*t20639*(1.0/6.0);
                double t20645 = 1.0/pow(t20614,3.0/5.0);
                double t20646 = c*kaa1*raa1*t20630*t20645*t20639*(2.0/1.5E1);
                double t20647 = t20644+t20646;
                double t20648 = 1.0/(t20633*t20633*t20633);
                double t20649 = 1.0/pow(rho_a,2.0E1/3.0);
                double t20650;
                double t20651;
                if (t20635 > expei_cutoff) {
                    t20650 = 1.0;
                    t20651 = expei(t20635);
                } else {
                    t20650 = exp(t20635);
                    t20651 = Ei(-t20635);
                }
                double t20652 = 1.0/(t20622*t20622*t20622);
                double t20653 = t20641*t20638*6.0;
                double t20654 = c*c0*t20623*t20625*t20634*t20617*t20609*4.0;
                double t20655 = t20653+t20654+3.0;
                double t20656 = 1.0/t20655;
                double t20657 = c*c0*t20623*t20625*t20634*t20617*t20639*(2.0/9.0);
                double t20658 = a5*c0*gamma_aa*t20623*t20642*t20634*t20608*t20617*(1.4E1/9.0);
                double t20659 = c*c0*t20623*t20625*t20617*t20609*t20647*t20648*(4.0/3.0);
                double t20660 = c*c0*t20611*t20613*t20623*t20615*t20625*t20634*t20617*t20649*(6.4E1/9.0);
                double t20662 = c*c0*t20611*t20613*t20615*t20625*t20634*t20652*t20617*t20649*(6.4E1/9.0);
                double t20661 = t20660-t20662+t20657+t20658+t20659;
                double t20663 = c*c0*t20623*t20650*t20651*t20625*t20634*t20617*t20609*(2.0/3.0);
                double t20664 = t20663+1.0;
                double t20665 = t20641*t20638*4.0;
                double t20666 = c*c0*t20623*t20625*t20634*t20617*t20609*(8.0/3.0);
                double t20667 = t20665+t20666+6.0;
                double t20668 = c*c0*t20625*t20634*t20617*t20639*(2.0/9.0);
                double t20669 = a5*c0*gamma_aa*t20642*t20634*t20608*t20617*(1.4E1/9.0);
                double t20670 = c*c0*t20625*t20617*t20609*t20647*t20648*(4.0/3.0);
                double t20671 = c*c0*t20611*t20613*t20615*t20625*t20634*t20617*t20649*(6.4E1/9.0);
                double t20672 = t20670+t20671+t20668+t20669;
                double t20673 = 1.0/sqrt(t20640);
                double t20674 = 1.0/pow(rho_a,1.9E1/3.0);
                double t20675 = t20610*t20611*t20612*t20613;
                double t20676 = pow(t20614,4.0/5.0);
                double t20694 = rab1*t20676;
                double t20677 = exp(-t20694);
                double t20678 = t20677-1.0;
                double t20695 = kab1*t20678;
                double t20679 = kab0-t20695;
                double t20680 = a3*gamma_aa*t20608*t20618*t20619;
                double t20681 = t20680+1.0;
                double t20682 = 1.0/sqrt(t20681);
                double t20683 = 1.0/pow(rho_a,8.0/3.0);
                double t20684 = a1*gamma_aa*t20608*t20683;
                double t20685 = t20675+t20684+1.0;
                double t20686 = c*c;
                double t20687 = 1.0/pow(rho_a,2.0/3.0);
                double t20688 = k1*t20620;
                double t20689 = c*k2*t20609;
                double t20690 = t20688+t20689;
                double t20691 = 1.0/(t20690*t20690);
                double t20697 = t20691*t20686*t20687;
                double t20692 = exp(-t20697);
                double t20693 = exp(-t20675);
                double t20696 = c*c0*t20609*(2.0/3.0);
                double t20698 = t20672*t20673*t20638*2.0;
                double t20699 = c*c0*t20623*t20625*t20634*t20617*t20639*(8.0/9.0);
                double t20700 = a5*c0*gamma_aa*t20623*t20642*t20634*t20608*t20617*(5.6E1/9.0);
                double t20701 = c*c0*t20623*t20625*t20617*t20609*t20647*t20648*(1.6E1/3.0);
                double t20702 = c*c0*t20611*t20613*t20623*t20615*t20625*t20634*t20617*t20649*(2.56E2/9.0);
                double t20753 = t20611*t20613*t20623*t20641*t20615*t20674*(6.4E1/3.0);
                double t20754 = c*c0*t20611*t20613*t20615*t20625*t20634*t20652*t20617*t20649*(2.56E2/9.0);
                double t20703 = t20700+t20701+t20702-t20753-t20754+t20698+t20699;
                double t20704 = c*c0*t20623*t20650*t20651*t20625*t20634*t20617*t20639*(2.0/9.0);
                double t20705 = c*c0*t20623*t20650*t20651*t20625*t20617*t20609*t20647*t20648*(4.0/3.0);
                double t20706 = c*c0*t20623*t20650*t20651*t20625*t20634*t20661*t20617*t20609*(2.0/3.0);
                double t20707 = a5*c0*gamma_aa*t20623*t20650*t20642*t20651*t20634*t20608*t20617*(1.4E1/9.0);
                double t20708 = c*c0*t20611*t20613*t20623*t20650*t20615*t20651*t20625*t20634*t20617*t20649*(6.4E1/9.0);
                double t20757 = c*c0*t20611*t20613*t20650*t20615*t20651*t20625*t20634*t20652*t20617*t20649*(6.4E1/9.0);
                double t20709 = t20704+t20660+t20705+t20706-t20662+t20707+t20708+t20657+t20658-t20757+t20659;
                double t20710 = t20615*t20615;
                double t20711 = t20611*t20611;
                double t20712 = t20613*t20613;
                double t20713 = raa2*raa2;
                double t20714 = c*kaa2*t20713*t20626*t20619*(1.0/3.6E1);
                double t20715 = raa1*raa1;
                double t20716 = 1.0/pow(t20614,6.0/5.0);
                double t20717 = kaa1*t20630*t20715*t20716*t20683*t20686*(4.0/2.25E2);
                double t20718 = 1.0/pow(t20614,3.0/2.0);
                double t20719 = kaa2*raa2*t20626*t20718*t20683*t20686*(1.0/3.6E1);
                double t20720 = 1.0/pow(t20614,8.0/5.0);
                double t20721 = kaa1*raa1*t20630*t20720*t20683*t20686*(2.0/7.5E1);
                double t20731 = c*kaa2*raa2*t20643*t20626*t20619*(2.0/9.0);
                double t20732 = c*kaa1*raa1*t20630*t20645*t20619*(8.0/4.5E1);
                double t20722 = t20721-t20731+t20714-t20732+t20717+t20719;
                double t20723 = t20647*t20647;
                double t20724 = 1.0/(t20633*t20633*t20633*t20633);
                double t20725 = 1.0/pow(rho_a,1.4E1/3.0);
                double t20726 = 1.0/pow(rho_a,1.0E1);
                double t20727 = 1.0/pow(rho_a,1.3E1);
                double t20728 = 1.0/pow(rho_a,2.3E1/3.0);
                double t20729 = 1.0/pow(rho_a,1.0E1/3.0);
                double t20730 = 1.0/t20625;
                double t20733 = t20661*t20661;
                double t20734 = 1.0/(t20622*t20622*t20622*t20622);
                double t20735 = c*c0*t20623*t20625*t20634*t20617*t20619*(8.0/2.7E1);
                double t20736 = c*c0*t20623*t20723*t20625*t20724*t20617*t20609*4.0;
                double t20737 = a5*c0*gamma_aa*t20623*t20634*t20608*t20617*t20725*(5.6E1/9.0);
                double t20738 = c*c0*t20623*t20625*t20617*t20647*t20639*t20648*(8.0/9.0);
                double t20739 = a5*c0*gamma_aa*t20623*t20642*t20608*t20617*t20647*t20648*(5.6E1/9.0);
                double t20740 = a5*c0*gamma_aa*t20611*t20613*t20623*t20615*t20634*t20608*t20617*t20726*(8.96E2/2.7E1);
                double t20741 = c*c0*t20710*t20711*t20712*t20623*t20625*t20634*t20617*t20727*(2.048E3/2.7E1);
                double t20742 = c*c0*t20710*t20711*t20712*t20625*t20634*t20617*t20734*t20727*(1.024E3/9.0);
                double t20743 = c*c0*t20611*t20613*t20623*t20615*t20625*t20634*t20617*t20728*(4.48E2/9.0);
                double t20744 = c*c0*t20611*t20613*t20623*t20615*t20625*t20617*t20647*t20648*t20649*(2.56E2/9.0);
                double t20777 = c*c0*t20623*t20722*t20625*t20617*t20609*t20648*(4.0/3.0);
                double t20778 = a5*c0*gamma_aa*t20611*t20613*t20615*t20634*t20652*t20608*t20617*t20726*(8.96E2/2.7E1);
                double t20779 = c*c0*t20710*t20711*t20712*t20625*t20634*t20652*t20617*t20727*(4.096E3/2.7E1);
                double t20780 = c*c0*t20611*t20613*t20615*t20625*t20634*t20652*t20617*t20728*(4.48E2/9.0);
                double t20781 = c*c0*t20611*t20613*t20615*t20625*t20652*t20617*t20647*t20648*t20649*(2.56E2/9.0);
                double t20745 = t20740+t20741+t20742+t20743+t20735+t20744-t20780+t20736-t20781+t20737+t20738+t20739-t20777-t20778-t20779;
                double t20746 = 1.0/(t20655*t20655);
                double t20747 = t20672*t20673*t20638*3.0;
                double t20748 = c*c0*t20623*t20625*t20634*t20617*t20639*(4.0/3.0);
                double t20749 = a5*c0*gamma_aa*t20623*t20642*t20634*t20608*t20617*(2.8E1/3.0);
                double t20750 = c*c0*t20623*t20625*t20617*t20609*t20647*t20648*8.0;
                double t20751 = c*c0*t20611*t20613*t20623*t20615*t20625*t20634*t20617*t20649*(1.28E2/3.0);
                double t20755 = t20611*t20613*t20623*t20641*t20615*t20674*3.2E1;
                double t20756 = c*c0*t20611*t20613*t20615*t20625*t20634*t20652*t20617*t20649*(1.28E2/3.0);
                double t20752 = t20750+t20751-t20755+t20747-t20756+t20748+t20749;
                double t20758 = c*c0*t20625*t20634*t20617*t20619*(8.0/2.7E1);
                double t20759 = a5*c0*gamma_aa*t20634*t20608*t20617*t20725*(5.6E1/9.0);
                double t20760 = c*c0*t20625*t20617*t20647*t20639*t20648*(8.0/9.0);
                double t20761 = c*c0*t20723*t20625*t20724*t20617*t20609*4.0;
                double t20762 = a5*c0*gamma_aa*t20611*t20613*t20615*t20634*t20608*t20617*t20726*(8.96E2/2.7E1);
                double t20763 = a5*c0*gamma_aa*t20642*t20608*t20617*t20647*t20648*(5.6E1/9.0);
                double t20764 = c*c0*t20710*t20711*t20712*t20625*t20634*t20617*t20727*(2.048E3/2.7E1);
                double t20765 = c*c0*t20611*t20613*t20615*t20625*t20634*t20617*t20728*(4.48E2/9.0);
                double t20766 = c*c0*t20611*t20613*t20615*t20625*t20617*t20647*t20648*t20649*(2.56E2/9.0);
                double t20767 = t20760+t20761+t20762+t20763+t20764+t20765+t20766+t20758+t20759-c*c0*t20722*t20625*t20617*t20609*t20648*(4.0/3.0);
                double t20768 = t20672*t20672;
                double t20769 = 1.0/pow(t20640,3.0/2.0);
                double t20770 = 1.0/pow(rho_a,2.2E1/3.0);
                double t20771 = 1.0/pow(rho_a,3.8E1/3.0);
                double t20772 = 1.0/c0;
                double t20773 = exp(-t20616);
                double t20774 = t20622*t20622;
                double t20775 = t20633*t20633;
                double t20776 = pow(rho_a,1.0/3.0);
                double t20782 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a*rho_a);
                double t20783 = t20682*t20693*t20685*t20679*1.0E5;
                double t20784 = t20783-t20696;
                double t20785 = heaviside(t20784);
                double t20786 = t20650*t20651;
                double t20787 = t20664*t20656*t20667;
                double t20788 = t20786+t20787;
                double t20789 = t20610*t20611*t20613*t20674*(1.6E1/3.0);
                double t20790 = a1*gamma_aa*t20642*t20608*(8.0/3.0);
                double t20791 = t20790+t20789;
                double t20792 = 1.0/(c*c);
                double t20793 = 1.0/pow(t20614,1.0/5.0);
                double t20794 = 1.0/pow(t20681,3.0/2.0);
                double t20795 = 1.0/pow(rho_a,5.0/3.0);
                double t20796 = t20691*t20686*t20795*(2.0/3.0);
                double t20797 = 1.0/(t20690*t20690*t20690);
                double t20798 = c*k2*t20639*(1.0/3.0);
                double t20799 = c*k1*t20643*t20639*(1.0/6.0);
                double t20800 = t20798+t20799;
                double t20814 = t20800*t20686*t20687*t20797*2.0;
                double t20801 = -t20814+t20796;
                double t20802 = t20703*t20664*t20656;
                double t20803 = t20650*t20651*t20661;
                double t20804 = t20709*t20656*t20667;
                double t20805 = t20730*t20661*t20618*t20772*t20773*t20774*t20775*t20776*(3.0/2.0);
                double t20809 = t20752*t20664*t20746*t20667;
                double t20806 = t20802+t20803+t20804+t20805-t20809;
                double t20807 = -t20783+t20696;
                double t20808 = dirac(t20807);
                double t20810 = c*c0*t20639*(2.0/9.0);
                double t20811 = t20610*t20611*t20613*t20682*t20674*t20693*t20685*t20679*(1.6E6/3.0);
                double t20812 = a3*gamma_aa*t20608*t20618*t20693*t20729*t20685*t20794*t20679*(3.5E5/3.0);
                double t20815 = t20682*t20791*t20693*t20679*1.0E5;
                double t20816 = c*kab1*rab1*t20682*t20639*t20693*t20685*t20793*t20677*(8.0E4/3.0);
                double t20813 = t20810+t20811+t20812-t20815-t20816;
                v_rho_a_rho_a[Q] += scale * (-c0*t20806*t20692*heaviside(c*c0*t20609*(-2.0/3.0)+t20682*t20685*t20679*exp(-t20610*t20611*t20612*t20613)*1.0E5)+c0*t20813*t20692*t20788*dirac(t20696-t20682*t20693*t20685*t20679*1.0E5)+c0*t20801*t20692*t20785*t20788+c0*rho_a*t20692*t20785*(t20664*t20656*(t20673*t20638*t20767*2.0-t20638*t20768*t20769-t20611*t20613*t20623*t20641*t20615*t20770*(1.216E3/9.0)+t20710*t20711*t20712*t20641*t20652*t20771*(2.048E3/9.0)+c*c0*t20623*t20625*t20634*t20617*t20619*(3.2E1/2.7E1)-t20611*t20613*t20623*t20615*t20672*t20673*t20674*(6.4E1/3.0)+a5*c0*gamma_aa*t20623*t20634*t20608*t20617*t20725*(2.24E2/9.0)+c*c0*t20623*t20723*t20625*t20724*t20617*t20609*1.6E1-c*c0*t20623*t20722*t20625*t20617*t20609*t20648*(1.6E1/3.0)+c*c0*t20623*t20625*t20617*t20647*t20639*t20648*(3.2E1/9.0)+a5*c0*gamma_aa*t20623*t20642*t20608*t20617*t20647*t20648*(2.24E2/9.0)+c*c0*t20710*t20711*t20712*t20623*t20625*t20634*t20617*t20727*(8.192E3/2.7E1)-c*c0*t20710*t20711*t20712*t20625*t20634*t20652*t20617*t20727*(1.6384E4/2.7E1)+c*c0*t20710*t20711*t20712*t20625*t20634*t20617*t20734*t20727*(4.096E3/9.0)+c*c0*t20611*t20613*t20623*t20615*t20625*t20634*t20617*t20728*(1.792E3/9.0)-c*c0*t20611*t20613*t20615*t20625*t20634*t20652*t20617*t20728*(1.792E3/9.0)+a5*c0*gamma_aa*t20611*t20613*t20623*t20615*t20634*t20608*t20617*t20726*(3.584E3/2.7E1)-a5*c0*gamma_aa*t20611*t20613*t20615*t20634*t20652*t20608*t20617*t20726*(3.584E3/2.7E1)+c*c0*t20611*t20613*t20623*t20615*t20625*t20617*t20647*t20648*t20649*(1.024E3/9.0)-c*c0*t20611*t20613*t20615*t20625*t20652*t20617*t20647*t20648*t20649*(1.024E3/9.0))+t20650*t20651*t20733+t20650*t20651*t20745+t20703*t20709*t20656*2.0+t20656*t20667*(t20740+t20741+t20733+t20742+t20743+t20735+t20744+t20736+t20737+t20738+t20739+(t20661*(1.0/3.0))/rho_a+(t20661*t20647*2.0)/t20633+t20611*t20613*t20615*t20661*t20674*(3.2E1/3.0)-t20611*t20613*t20615*t20661*t20638*t20674*(3.2E1/3.0)+a5*gamma_aa*t20730*t20661*t20608*t20618*t20729*(7.0/3.0)-c*c0*t20623*t20722*t20625*t20617*t20609*t20648*(4.0/3.0)+c*c0*t20623*t20650*t20651*t20625*t20634*t20617*t20619*(8.0/2.7E1)+a5*c0*gamma_aa*t20623*t20650*t20651*t20634*t20608*t20617*t20725*(5.6E1/9.0)-c*c0*t20710*t20711*t20712*t20625*t20634*t20652*t20617*t20727*(4.096E3/2.7E1)-c*c0*t20611*t20613*t20615*t20625*t20634*t20652*t20617*t20728*(4.48E2/9.0)+c*c0*t20623*t20650*t20651*t20723*t20625*t20724*t20617*t20609*4.0+c*c0*t20623*t20650*t20651*t20625*t20634*t20733*t20617*t20609*(2.0/3.0)-c*c0*t20623*t20650*t20722*t20651*t20625*t20617*t20609*t20648*(4.0/3.0)+c*c0*t20623*t20650*t20651*t20625*t20634*t20661*t20617*t20639*(4.0/9.0)+c*c0*t20623*t20650*t20651*t20625*t20634*t20617*t20609*t20745*(2.0/3.0)+c*c0*t20623*t20650*t20651*t20625*t20617*t20647*t20639*t20648*(8.0/9.0)-a5*c0*gamma_aa*t20611*t20613*t20615*t20634*t20652*t20608*t20617*t20726*(8.96E2/2.7E1)+a5*c0*gamma_aa*t20623*t20650*t20642*t20651*t20634*t20661*t20608*t20617*(2.8E1/9.0)+a5*c0*gamma_aa*t20623*t20650*t20642*t20651*t20608*t20617*t20647*t20648*(5.6E1/9.0)-c*c0*t20611*t20613*t20615*t20625*t20652*t20617*t20647*t20648*t20649*(2.56E2/9.0)+c*c0*t20623*t20650*t20651*t20625*t20661*t20617*t20609*t20647*t20648*(8.0/3.0)+c*c0*t20710*t20711*t20712*t20623*t20650*t20651*t20625*t20634*t20617*t20727*(2.048E3/2.7E1)-c*c0*t20710*t20711*t20712*t20650*t20651*t20625*t20634*t20652*t20617*t20727*(4.096E3/2.7E1)+c*c0*t20710*t20711*t20712*t20650*t20651*t20625*t20634*t20617*t20734*t20727*(1.024E3/9.0)+c*c0*t20611*t20613*t20623*t20650*t20615*t20651*t20625*t20634*t20617*t20728*(4.48E2/9.0)-c*c0*t20611*t20613*t20650*t20615*t20651*t20625*t20634*t20652*t20617*t20728*(4.48E2/9.0)+a5*c0*gamma_aa*t20611*t20613*t20623*t20650*t20615*t20651*t20634*t20608*t20617*t20726*(8.96E2/2.7E1)-a5*c0*gamma_aa*t20611*t20613*t20650*t20615*t20651*t20634*t20652*t20608*t20617*t20726*(8.96E2/2.7E1)+c*c0*t20611*t20613*t20623*t20650*t20615*t20651*t20625*t20634*t20661*t20617*t20649*(1.28E2/9.0)-c*c0*t20611*t20613*t20650*t20615*t20651*t20625*t20634*t20652*t20661*t20617*t20649*(1.28E2/9.0)+c*c0*t20611*t20613*t20623*t20650*t20615*t20651*t20625*t20617*t20647*t20648*t20649*(2.56E2/9.0)-c*c0*t20611*t20613*t20650*t20615*t20651*t20625*t20652*t20617*t20647*t20648*t20649*(2.56E2/9.0))-t20664*t20746*t20667*(t20673*t20638*t20767*3.0-t20638*t20768*t20769*(3.0/2.0)-t20611*t20613*t20623*t20641*t20615*t20770*(6.08E2/3.0)+t20710*t20711*t20712*t20641*t20652*t20771*(1.024E3/3.0)+c*c0*t20623*t20625*t20634*t20617*t20619*(1.6E1/9.0)-t20611*t20613*t20623*t20615*t20672*t20673*t20674*3.2E1+a5*c0*gamma_aa*t20623*t20634*t20608*t20617*t20725*(1.12E2/3.0)+c*c0*t20623*t20723*t20625*t20724*t20617*t20609*2.4E1-c*c0*t20623*t20722*t20625*t20617*t20609*t20648*8.0+c*c0*t20623*t20625*t20617*t20647*t20639*t20648*(1.6E1/3.0)+a5*c0*gamma_aa*t20623*t20642*t20608*t20617*t20647*t20648*(1.12E2/3.0)+c*c0*t20710*t20711*t20712*t20623*t20625*t20634*t20617*t20727*(4.096E3/9.0)-c*c0*t20710*t20711*t20712*t20625*t20634*t20652*t20617*t20727*(8.192E3/9.0)+c*c0*t20710*t20711*t20712*t20625*t20634*t20617*t20734*t20727*(2.048E3/3.0)+c*c0*t20611*t20613*t20623*t20615*t20625*t20634*t20617*t20728*(8.96E2/3.0)-c*c0*t20611*t20613*t20615*t20625*t20634*t20652*t20617*t20728*(8.96E2/3.0)+a5*c0*gamma_aa*t20611*t20613*t20623*t20615*t20634*t20608*t20617*t20726*(1.792E3/9.0)-a5*c0*gamma_aa*t20611*t20613*t20615*t20634*t20652*t20608*t20617*t20726*(1.792E3/9.0)+c*c0*t20611*t20613*t20623*t20615*t20625*t20617*t20647*t20648*t20649*(5.12E2/3.0)-c*c0*t20611*t20613*t20615*t20625*t20652*t20617*t20647*t20648*t20649*(5.12E2/3.0))-t20703*t20752*t20664*t20746*2.0-t20752*t20709*t20746*t20667*2.0+(t20752*t20752)*1.0/(t20655*t20655*t20655)*t20664*t20667*2.0+t20730*t20733*t20618*t20772*t20773*t20774*t20775*t20776*(3.0/2.0)-t20730*t20661*t20618*t20772*t20773*t20774*t20775*t20687*(1.0/2.0)+t20730*t20618*t20745*t20772*t20773*t20774*t20775*t20776*(3.0/2.0)-t20730*t20633*t20661*t20618*t20772*t20647*t20773*t20774*t20776*3.0+t20611*t20613*t20622*t20730*t20615*t20661*t20618*t20772*t20773*t20782*t20775*1.6E1-t20611*t20613*t20730*t20615*t20661*t20618*t20772*t20773*t20782*t20774*t20775*1.6E1-a5*gamma_aa*1.0/(rho_a*rho_a*rho_a)*1.0/(t20625*t20625)*t20661*t20608*t20772*t20773*t20774*t20792*t20775*(7.0/2.0))*(1.0/2.0)-c0*rho_a*t20813*t20806*t20808*t20692-c0*rho_a*t20801*t20806*t20692*t20785+c0*rho_a*(t20801*t20801)*t20692*t20785*t20788*(1.0/2.0)-c0*rho_a*t20808*t20692*t20788*(c*c0*t20619*(8.0/2.7E1)-t20682*t20693*t20679*(a1*gamma_aa*t20608*t20725*(8.8E1/9.0)+t20610*t20611*t20613*t20770*(3.04E2/9.0))*1.0E5+t20610*t20611*t20613*t20770*t20682*t20693*t20685*t20679*(3.04E7/9.0)+t20610*t20611*t20613*t20682*t20674*t20791*t20693*t20679*(3.2E6/3.0)-(t20610*t20610)*t20711*t20712*t20771*t20682*t20693*t20685*t20679*(2.56E7/9.0)-(a3*a3)*t20611*t20613*1.0/pow(t20681,5.0/2.0)*t20693*t20792*t20649*t20685*t20679*(1.225E6/3.0)+kab1*(rab1*rab1)*1.0/pow(t20614,2.0/5.0)*t20682*t20683*t20693*t20685*t20677*t20686*(6.4E4/9.0)+a3*gamma_aa*t20608*t20618*t20791*t20693*t20729*t20794*t20679*(7.0E5/3.0)-c*kab1*rab1*t20619*t20682*t20693*t20685*t20793*t20677*(3.2E5/9.0)-c*kab1*rab1*t20682*t20791*t20639*t20693*t20793*t20677*(1.6E5/3.0)+kab1*rab1*t20716*t20682*t20683*t20693*t20685*t20677*t20686*(1.6E4/9.0)+a3*gamma_aa*1.0/pow(rho_a,1.3E1/3.0)*t20608*t20618*t20693*t20685*t20794*t20679*(3.5E6/9.0)+a3*gamma_aa*kab1*rab1*t20608*t20725*t20693*t20685*t20793*t20677*t20794*(5.6E5/9.0)-a3*gamma_aa*1.0/pow(rho_a,2.9E1/3.0)*t20610*t20611*t20613*t20608*t20618*t20693*t20685*t20794*t20679*(1.12E7/9.0)+c*kab1*rab1*t20610*t20611*t20613*t20682*t20728*t20693*t20685*t20793*t20677*(2.56E6/9.0))*(1.0/2.0)-c0*rho_a*t20692*t20785*t20788*(t20691*t20683*t20686*(1.0E1/9.0)-t20800*t20686*t20795*t20797*(8.0/3.0)+(t20800*t20800)*1.0/(t20690*t20690*t20690*t20690)*t20686*t20687*6.0-t20686*t20687*t20797*(c*k2*t20619*(4.0/9.0)+c*k1*t20643*t20619*(2.0/9.0)-k1*t20718*t20683*t20686*(1.0/3.6E1))*2.0)*(1.0/2.0)-c0*rho_a*(t20813*t20813)*t20692*t20788*dirac(t20807,1.0)*(1.0/2.0)+c0*rho_a*t20801*t20813*t20808*t20692*t20788);
            }
            
            // v_gamma_aa_gamma_aa
            if (deriv >= 2) {
                double t21034 = tspi_m13*tspi_m13;
                double t21035 = 1.0/pow(rho_a,1.0/3.0);
                double t21036 = a2*a2;
                double t21037 = gamma_aa*gamma_aa;
                double t21038 = 1.0/pow(rho_a,1.6E1/3.0);
                double t21039 = t21034*t21034;
                double t21040 = c*t21035;
                double t21041 = a4*a4;
                double t21042 = t21041*t21037*t21038*t21039*2.0;
                double t21043 = exp(t21042);
                double t21044 = 1.0/c;
                double t21045 = 1.0/pow(rho_a,7.0/3.0);
                double t21046 = sqrt(t21040);
                double t21047 = t21041*t21037*t21038*t21039;
                double t21048 = t21047+1.0;
                double t21049 = 1.0/(t21048*t21048);
                double t21050 = a5*gamma_aa*t21034*t21044*t21045;
                double t21051 = t21050+1.0;
                double t21062 = raa2*t21046;
                double t21052 = exp(-t21062);
                double t21053 = t21052-1.0;
                double t21054 = kaa2*t21053;
                double t21055 = pow(t21040,2.0/5.0);
                double t21063 = raa1*t21055;
                double t21056 = exp(-t21063);
                double t21057 = t21056-1.0;
                double t21058 = kaa1*t21057;
                double t21059 = -kaa0+t21054+t21058;
                double t21060 = 1.0/(t21059*t21059);
                double t21061 = c*c0*t21051*t21060*t21043*t21035*t21049*(2.0/3.0);
                double t21064 = 1.0/t21048;
                double t21065 = c*c0*t21051*t21060*t21043*t21035*(2.0/3.0);
                double t21066 = 1.0/pow(rho_a,8.0/3.0);
                double t21067 = 1.0/pow(rho_a,1.7E1/3.0);
                double t21072 = a5*c0*t21060*t21034*t21043*t21066*(2.0/3.0);
                double t21073 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21067*(8.0/3.0);
                double t21068 = t21072+t21073;
                double t21069 = sqrt(t21065);
                double t21070 = t21041*t21041;
                double t21071 = t21039*t21039;
                double t21074 = 1.0/sqrt(t21065);
                double t21075 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a);
                double t21076 = 1.0/(t21048*t21048*t21048);
                double t21077 = 1.0/pow(rho_a,1.1E1);
                double t21078 = t21064*t21069*6.0;
                double t21079 = c*c0*t21051*t21060*t21043*t21035*t21049*4.0;
                double t21080 = t21078+t21079+3.0;
                double t21081 = 1.0/t21080;
                double t21083 = a5*c0*t21060*t21034*t21043*t21066*t21049*(2.0/3.0);
                double t21084 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21049*t21067*(8.0/3.0);
                double t21085 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21067*t21076*(8.0/3.0);
                double t21082 = t21083+t21084-t21085;
                double t21086;
                double t21087;
                if (t21061 > expei_cutoff) {
                    t21086 = 1.0;
                    t21087 = expei(t21061);
                } else {
                    t21086 = exp(t21061);
                    t21087 = Ei(-t21061);
                }
                double t21088 = t21082*t21082;
                double t21089 = 1.0/(t21048*t21048*t21048*t21048);
                double t21090 = a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21049*(1.6E1/3.0);
                double t21091 = c*c0*t21041*t21051*t21060*t21043*t21039*t21049*t21067*(8.0/3.0);
                double t21092 = c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21049*t21077*(3.2E1/3.0);
                double t21093 = c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21077*t21089*1.6E1;
                double t21128 = a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21076*(1.6E1/3.0);
                double t21129 = c*c0*t21041*t21051*t21060*t21043*t21039*t21067*t21076*(8.0/3.0);
                double t21130 = c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21076*t21077*(6.4E1/3.0);
                double t21094 = -t21130+t21090+t21091+t21092-t21128+t21093-t21129;
                double t21095 = t21064*t21069*4.0;
                double t21096 = c*c0*t21051*t21060*t21043*t21035*t21049*(8.0/3.0);
                double t21097 = t21095+t21096+6.0;
                double t21098 = a5*c0*t21060*t21034*t21043*t21066*t21049*t21086*t21087*(2.0/3.0);
                double t21099 = c*c0*t21051*t21060*t21043*t21035*t21082*t21049*t21086*t21087*(2.0/3.0);
                double t21100 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21049*t21067*t21086*t21087*(8.0/3.0);
                double t21165 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21067*t21076*t21086*t21087*(8.0/3.0);
                double t21101 = t21100+t21083+t21084-t21165-t21085+t21098+t21099;
                double t21102 = c*c0*t21051*t21060*t21043*t21035*t21049*t21086*t21087*(2.0/3.0);
                double t21103 = t21102+1.0;
                double t21104 = t21064*t21074*t21068*3.0;
                double t21105 = a5*c0*t21060*t21034*t21043*t21066*t21049*4.0;
                double t21106 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21049*t21067*1.6E1;
                double t21109 = gamma_aa*t21041*t21038*t21039*t21049*t21069*1.2E1;
                double t21110 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21067*t21076*1.6E1;
                double t21107 = -t21110+t21104+t21105+t21106-t21109;
                double t21108 = 1.0/(t21080*t21080);
                double t21111 = t21064*t21074*t21068*2.0;
                double t21112 = a5*c0*t21060*t21034*t21043*t21066*t21049*(8.0/3.0);
                double t21113 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21049*t21067*(3.2E1/3.0);
                double t21163 = gamma_aa*t21041*t21038*t21039*t21049*t21069*8.0;
                double t21164 = c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21039*t21067*t21076*(3.2E1/3.0);
                double t21114 = t21111+t21112+t21113-t21163-t21164;
                double t21115 = a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*(1.6E1/3.0);
                double t21116 = c*c0*t21041*t21051*t21060*t21043*t21039*t21067*(8.0/3.0);
                double t21117 = c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21077*(3.2E1/3.0);
                double t21118 = t21115+t21116+t21117;
                double t21119 = t21068*t21068;
                double t21120 = 1.0/pow(t21065,3.0/2.0);
                double t21121 = 1.0/pow(rho_a,3.2E1/3.0);
                double t21122 = 1.0/t21051;
                double t21123 = 1.0/c0;
                double t21124 = pow(rho_a,1.0/3.0);
                double t21125 = exp(-t21042);
                double t21126 = t21048*t21048;
                double t21127 = t21059*t21059;
                double t21131 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a);
                double t21132 = c*c;
                double t21133 = 1.0/pow(rho_a,2.0/3.0);
                double t21134 = k1*t21046;
                double t21135 = c*k2*t21035;
                double t21136 = t21134+t21135;
                double t21137 = 1.0/(t21136*t21136);
                double t21155 = t21132*t21133*t21137;
                double t21138 = exp(-t21155);
                double t21139 = t21036*t21037*t21038*t21039;
                double t21140 = pow(t21040,4.0/5.0);
                double t21150 = rab1*t21140;
                double t21141 = exp(-t21150);
                double t21142 = t21141-1.0;
                double t21151 = kab1*t21142;
                double t21143 = kab0-t21151;
                double t21144 = a3*gamma_aa*t21034*t21044*t21045;
                double t21145 = t21144+1.0;
                double t21146 = 1.0/sqrt(t21145);
                double t21147 = a1*gamma_aa*t21034*t21066;
                double t21148 = t21147+t21139+1.0;
                double t21149 = exp(-t21139);
                double t21156 = a1*t21034*t21066;
                double t21157 = gamma_aa*t21036*t21038*t21039*2.0;
                double t21158 = t21156+t21157;
                double t21159 = t21143*t21146*t21149*t21158*1.0E5;
                double t21160 = 1.0/pow(t21145,3.0/2.0);
                double t21161 = a3*t21034*t21160*t21044*t21143*t21045*t21148*t21149*5.0E4;
                double t21162 = gamma_aa*t21143*t21036*t21038*t21146*t21039*t21148*t21149*2.0E5;
                double t21152 = t21161+t21162-t21159;
                double t21153 = c*c0*t21035*(2.0/3.0);
                double t21166 = t21143*t21146*t21148*t21149*1.0E5;
                double t21154 = t21153-t21166;
                double t21167 = dirac(t21154);
                double t21168 = t21086*t21087;
                double t21169 = t21103*t21081*t21097;
                double t21170 = t21168+t21169;
                double t21171 = 1.0/(c*c);
                v_gamma_aa_gamma_aa[Q] += scale * (c0*rho_a*t21138*heaviside(c*c0*t21035*(-2.0/3.0)+t21143*t21146*t21148*exp(-t21036*t21037*t21038*t21039)*1.0E5)*(t21081*t21097*(t21090+t21091+t21092+t21093+t21088+gamma_aa*t21041*t21082*t21038*t21039*4.0+a5*t21122*t21034*t21044*t21045*t21082-gamma_aa*t21041*t21064*t21082*t21038*t21039*4.0-c*c0*t21041*t21051*t21060*t21043*t21039*t21067*t21076*(8.0/3.0)-a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21076*(1.6E1/3.0)+a5*c0*t21060*t21034*t21043*t21082*t21066*t21049*t21086*t21087*(4.0/3.0)-c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21076*t21077*(6.4E1/3.0)+c*c0*t21051*t21060*t21043*t21035*t21049*t21094*t21086*t21087*(2.0/3.0)+c*c0*t21051*t21060*t21043*t21035*t21049*t21086*t21087*t21088*(2.0/3.0)+c*c0*t21041*t21051*t21060*t21043*t21039*t21049*t21067*t21086*t21087*(8.0/3.0)-c*c0*t21041*t21051*t21060*t21043*t21039*t21067*t21076*t21086*t21087*(8.0/3.0)+a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21049*t21086*t21087*(1.6E1/3.0)-a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21076*t21086*t21087*(1.6E1/3.0)+c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21049*t21077*t21086*t21087*(3.2E1/3.0)-c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21076*t21077*t21086*t21087*(6.4E1/3.0)+c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21077*t21086*t21087*t21089*1.6E1+c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21082*t21039*t21049*t21067*t21086*t21087*(1.6E1/3.0)-c*c0*gamma_aa*t21041*t21051*t21060*t21043*t21082*t21039*t21067*t21076*t21086*t21087*(1.6E1/3.0))+t21101*t21114*t21081*2.0+t21094*t21086*t21087+t21086*t21087*t21088-t21103*t21081*(t21120*t21064*t21119-t21064*t21118*t21074*2.0+t21041*t21038*t21039*t21049*t21069*8.0-t21121*t21070*t21071*t21037*t21076*t21069*3.2E1+gamma_aa*t21041*t21038*t21074*t21039*t21049*t21068*8.0-c*c0*t21041*t21051*t21060*t21043*t21039*t21049*t21067*(3.2E1/3.0)+c*c0*t21041*t21051*t21060*t21043*t21039*t21067*t21076*(3.2E1/3.0)-a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21049*(6.4E1/3.0)+a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21076*(6.4E1/3.0)-c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21049*t21077*(1.28E2/3.0)+c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21076*t21077*(2.56E2/3.0)-c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21077*t21089*6.4E1)-t21103*t21114*t21107*t21108*2.0-t21101*t21107*t21108*t21097*2.0+t21103*t21108*t21097*(t21120*t21064*t21119*(3.0/2.0)-t21064*t21118*t21074*3.0+t21041*t21038*t21039*t21049*t21069*1.2E1-t21121*t21070*t21071*t21037*t21076*t21069*4.8E1+gamma_aa*t21041*t21038*t21074*t21039*t21049*t21068*1.2E1-c*c0*t21041*t21051*t21060*t21043*t21039*t21049*t21067*1.6E1+c*c0*t21041*t21051*t21060*t21043*t21039*t21067*t21076*1.6E1-a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21049*3.2E1+a5*c0*gamma_aa*t21041*t21060*t21034*t21043*t21039*t21075*t21076*3.2E1-c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21049*t21077*6.4E1+c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21076*t21077*1.28E2-c*c0*t21051*t21060*t21043*t21070*t21071*t21037*t21077*t21089*9.6E1)+t21103*1.0/(t21080*t21080*t21080)*(t21107*t21107)*t21097*2.0+t21122*t21123*t21124*t21044*t21125*t21126*t21127*t21094*(3.0/2.0)+t21122*t21123*t21124*t21044*t21125*t21126*t21127*t21088*(3.0/2.0)-a5*1.0/(rho_a*rho_a)*1.0/(t21051*t21051)*t21123*t21034*t21125*t21126*t21171*t21082*t21127*(3.0/2.0)-gamma_aa*t21041*t21122*t21131*t21123*t21044*t21125*t21126*t21082*t21127*t21039*6.0+gamma_aa*t21041*t21122*t21131*t21123*t21044*t21125*t21082*t21127*t21039*t21048*6.0)*(1.0/2.0)-c0*rho_a*t21152*t21138*t21167*(t21103*t21114*t21081+t21101*t21081*t21097+t21082*t21086*t21087-t21103*t21107*t21108*t21097+t21122*t21123*t21124*t21044*t21125*t21126*t21082*t21127*(3.0/2.0))+c0*rho_a*t21170*t21138*t21167*(t21143*t21036*t21038*t21146*t21039*t21149*2.0E5-t21143*t21036*t21038*t21146*t21039*t21148*t21149*2.0E5+(a3*a3)*1.0/pow(rho_a,1.4E1/3.0)*t21143*t21171*1.0/pow(t21145,5.0/2.0)*t21039*t21148*t21149*7.5E4+t21121*t21071*t21143*(t21036*t21036)*t21037*t21146*t21148*t21149*4.0E5-a3*t21034*t21160*t21044*t21143*t21045*t21149*t21158*1.0E5-gamma_aa*t21143*t21036*t21038*t21146*t21039*t21149*t21158*4.0E5+a3*gamma_aa*1.0/pow(rho_a,2.3E1/3.0)*t21034*t21160*t21044*t21143*t21036*t21039*t21148*t21149*2.0E5)*(1.0/2.0)-c0*rho_a*(t21152*t21152)*t21170*t21138*dirac(t21154,1.0)*(1.0/2.0));
            }
            
            // v_rho_a_gamma_aa
            if (deriv >= 2) {
                double t20820 = tspi_m13*tspi_m13;
                double t20821 = 1.0/pow(rho_a,1.0/3.0);
                double t20822 = a2*a2;
                double t20823 = gamma_aa*gamma_aa;
                double t20824 = 1.0/pow(rho_a,1.6E1/3.0);
                double t20825 = t20820*t20820;
                double t20826 = c*t20821;
                double t20827 = a4*a4;
                double t20828 = 1.0/c;
                double t20829 = 1.0/pow(rho_a,7.0/3.0);
                double t20830 = sqrt(t20826);
                double t20831 = t20823*t20824*t20825*t20827*2.0;
                double t20832 = exp(t20831);
                double t20833 = t20823*t20824*t20825*t20827;
                double t20834 = t20833+1.0;
                double t20835 = 1.0/(t20834*t20834);
                double t20836 = a5*gamma_aa*t20820*t20828*t20829;
                double t20837 = t20836+1.0;
                double t20849 = raa2*t20830;
                double t20838 = exp(-t20849);
                double t20839 = t20838-1.0;
                double t20840 = kaa2*t20839;
                double t20841 = pow(t20826,2.0/5.0);
                double t20850 = raa1*t20841;
                double t20842 = exp(-t20850);
                double t20843 = t20842-1.0;
                double t20844 = kaa1*t20843;
                double t20845 = -kaa0+t20840+t20844;
                double t20846 = 1.0/(t20845*t20845);
                double t20847 = c*c0*t20821*t20832*t20835*t20837*t20846*(2.0/3.0);
                double t20848 = 1.0/pow(rho_a,8.0/3.0);
                double t20851 = 1.0/pow(rho_a,1.7E1/3.0);
                double t20852;
                double t20853;
                if (t20847 > expei_cutoff) {
                    t20852 = 1.0;
                    t20853 = expei(t20847);
                } else {
                    t20852 = exp(t20847);
                    t20853 = Ei(-t20847);
                }
                double t20854 = 1.0/t20834;
                double t20855 = c*c0*t20821*t20832*t20837*t20846*(2.0/3.0);
                double t20856 = sqrt(t20855);
                double t20857 = 1.0/(t20834*t20834*t20834);
                double t20858 = t20854*t20856*6.0;
                double t20859 = c*c0*t20821*t20832*t20835*t20837*t20846*4.0;
                double t20860 = t20858+t20859+3.0;
                double t20861 = 1.0/t20860;
                double t20862 = a5*c0*t20820*t20832*t20835*t20846*t20848*(2.0/3.0);
                double t20863 = c*c0*gamma_aa*t20832*t20851*t20825*t20835*t20827*t20837*t20846*(8.0/3.0);
                double t20874 = c*c0*gamma_aa*t20832*t20851*t20825*t20827*t20837*t20846*t20857*(8.0/3.0);
                double t20864 = t20862+t20863-t20874;
                double t20865 = c*c0*t20821*t20832*t20852*t20835*t20853*t20837*t20846*(2.0/3.0);
                double t20866 = t20865+1.0;
                double t20867 = t20854*t20856*4.0;
                double t20868 = c*c0*t20821*t20832*t20835*t20837*t20846*(8.0/3.0);
                double t20869 = t20867+t20868+6.0;
                double t20870 = a5*c0*t20820*t20832*t20846*t20848*(2.0/3.0);
                double t20871 = c*c0*gamma_aa*t20832*t20851*t20825*t20827*t20837*t20846*(8.0/3.0);
                double t20872 = t20870+t20871;
                double t20873 = 1.0/sqrt(t20855);
                double t20875 = t20822*t20823*t20824*t20825;
                double t20876 = pow(t20826,4.0/5.0);
                double t20984 = rab1*t20876;
                double t20877 = exp(-t20984);
                double t20878 = t20877-1.0;
                double t20985 = kab1*t20878;
                double t20879 = kab0-t20985;
                double t20880 = a3*gamma_aa*t20820*t20828*t20829;
                double t20881 = t20880+1.0;
                double t20882 = 1.0/sqrt(t20881);
                double t20883 = a1*gamma_aa*t20820*t20848;
                double t20884 = t20883+t20875+1.0;
                double t20885 = c*c;
                double t20886 = 1.0/pow(rho_a,2.0/3.0);
                double t20887 = k1*t20830;
                double t20888 = c*k2*t20821;
                double t20889 = t20887+t20888;
                double t20890 = 1.0/(t20889*t20889);
                double t20987 = t20890*t20885*t20886;
                double t20891 = exp(-t20987);
                double t20892 = 1.0/pow(rho_a,4.0/3.0);
                double t20893 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a);
                double t20894 = 1.0/pow(rho_a,2.0E1/3.0);
                double t20895 = t20827*t20827;
                double t20896 = 1.0/pow(rho_a,1.2E1);
                double t20897 = t20825*t20825;
                double t20898 = 1.0/sqrt(t20826);
                double t20899 = c*kaa2*raa2*t20838*t20892*t20898*(1.0/6.0);
                double t20900 = 1.0/pow(t20826,3.0/5.0);
                double t20901 = c*kaa1*raa1*t20900*t20842*t20892*(2.0/1.5E1);
                double t20902 = t20901+t20899;
                double t20903 = 1.0/(t20845*t20845*t20845);
                double t20904 = t20854*t20872*t20873*2.0;
                double t20905 = a5*c0*t20820*t20832*t20835*t20846*t20848*(8.0/3.0);
                double t20906 = c*c0*gamma_aa*t20832*t20851*t20825*t20835*t20827*t20837*t20846*(3.2E1/3.0);
                double t20948 = gamma_aa*t20824*t20825*t20835*t20827*t20856*8.0;
                double t20949 = c*c0*gamma_aa*t20832*t20851*t20825*t20827*t20837*t20846*t20857*(3.2E1/3.0);
                double t20907 = t20904+t20905+t20906-t20948-t20949;
                double t20908 = 1.0/pow(rho_a,1.1E1/3.0);
                double t20909 = c*c0*t20832*t20835*t20837*t20846*t20892*(2.0/9.0);
                double t20910 = a5*c0*gamma_aa*t20820*t20832*t20835*t20908*t20846*(1.4E1/9.0);
                double t20911 = c*c0*t20821*t20902*t20903*t20832*t20835*t20837*(4.0/3.0);
                double t20912 = c*c0*t20823*t20832*t20825*t20835*t20827*t20837*t20846*t20894*(6.4E1/9.0);
                double t20913 = a5*c0*t20820*t20832*t20852*t20835*t20853*t20846*t20848*(2.0/3.0);
                double t20914 = c*c0*t20821*t20832*t20852*t20835*t20853*t20837*t20846*t20864*(2.0/3.0);
                double t20915 = c*c0*gamma_aa*t20832*t20851*t20825*t20852*t20835*t20853*t20827*t20837*t20846*(8.0/3.0);
                double t20928 = c*c0*t20823*t20832*t20825*t20827*t20837*t20846*t20857*t20894*(6.4E1/9.0);
                double t20916 = t20910+t20911+t20912+t20909-t20928;
                double t20917 = 1.0/pow(rho_a,1.9E1/3.0);
                double t20918 = 1.0/t20837;
                double t20919 = a5*c0*t20820*t20832*t20835*t20908*t20846*(1.6E1/9.0);
                double t20920 = a5*c0*t20820*t20902*t20903*t20832*t20835*t20848*(4.0/3.0);
                double t20921 = a5*c0*t20820*t20823*t20832*t20825*t20835*t20827*t20846*t20893*(4.0E1/3.0);
                double t20922 = c*c0*gamma_aa*t20832*t20825*t20835*t20827*t20837*t20846*t20894*(1.36E2/9.0);
                double t20923 = c*c0*gamma_aa*t20823*t20832*t20835*t20837*t20846*t20895*t20896*t20897*(2.56E2/9.0);
                double t20924 = 1.0/(t20834*t20834*t20834*t20834);
                double t20925 = c*c0*gamma_aa*t20823*t20832*t20924*t20837*t20846*t20895*t20896*t20897*(1.28E2/3.0);
                double t20926 = c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20835*t20827*t20837*(1.6E1/3.0);
                double t20929 = c*c0*gamma_aa*t20832*t20825*t20827*t20837*t20846*t20857*t20894*(1.36E2/9.0);
                double t20930 = c*c0*gamma_aa*t20823*t20832*t20837*t20846*t20857*t20895*t20896*t20897*(5.12E2/9.0);
                double t20931 = c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20827*t20837*t20857*(1.6E1/3.0);
                double t20980 = a5*c0*t20820*t20823*t20832*t20825*t20827*t20846*t20857*t20893*(4.0E1/3.0);
                double t20927 = t20920+t20921-t20930+t20922-t20931+t20923+t20925+t20926-t20980+t20919-t20929;
                double t20932 = c*c0*t20832*t20837*t20846*t20892*(2.0/9.0);
                double t20933 = a5*c0*gamma_aa*t20820*t20832*t20908*t20846*(1.4E1/9.0);
                double t20934 = c*c0*t20821*t20902*t20903*t20832*t20837*(4.0/3.0);
                double t20935 = c*c0*t20823*t20832*t20825*t20827*t20837*t20846*t20894*(6.4E1/9.0);
                double t20936 = t20932+t20933+t20934+t20935;
                double t20937 = 1.0/(t20860*t20860);
                double t21003 = c*c0*gamma_aa*t20832*t20851*t20825*t20852*t20853*t20827*t20837*t20846*t20857*(8.0/3.0);
                double t20938 = t20913+t20914+t20915+t20862+t20863-t20874-t21003;
                double t20939 = a5*c0*t20820*t20832*t20908*t20846*(1.6E1/9.0);
                double t20940 = a5*c0*t20820*t20902*t20903*t20832*t20848*(4.0/3.0);
                double t20941 = a5*c0*t20820*t20823*t20832*t20825*t20827*t20846*t20893*(4.0E1/3.0);
                double t20942 = c*c0*gamma_aa*t20832*t20825*t20827*t20837*t20846*t20894*(1.36E2/9.0);
                double t20943 = c*c0*gamma_aa*t20823*t20832*t20837*t20846*t20895*t20896*t20897*(2.56E2/9.0);
                double t20944 = c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20827*t20837*(1.6E1/3.0);
                double t20945 = t20940+t20941+t20942+t20943+t20944+t20939;
                double t20946 = 1.0/pow(t20855,3.0/2.0);
                double t20947 = 1.0/pow(rho_a,3.5E1/3.0);
                double t20950 = t20854*t20873*t20936*3.0;
                double t20951 = c*c0*t20832*t20835*t20837*t20846*t20892*(4.0/3.0);
                double t20952 = a5*c0*gamma_aa*t20820*t20832*t20835*t20908*t20846*(2.8E1/3.0);
                double t20953 = c*c0*t20821*t20902*t20903*t20832*t20835*t20837*8.0;
                double t20954 = c*c0*t20823*t20832*t20825*t20835*t20827*t20837*t20846*t20894*(1.28E2/3.0);
                double t20973 = t20823*t20825*t20835*t20827*t20917*t20856*3.2E1;
                double t20974 = c*c0*t20823*t20832*t20825*t20827*t20837*t20846*t20857*t20894*(1.28E2/3.0);
                double t20955 = t20950+t20951+t20952+t20953+t20954-t20973-t20974;
                double t20956 = t20854*t20872*t20873*3.0;
                double t20957 = a5*c0*t20820*t20832*t20835*t20846*t20848*4.0;
                double t20958 = c*c0*gamma_aa*t20832*t20851*t20825*t20835*t20827*t20837*t20846*1.6E1;
                double t20966 = gamma_aa*t20824*t20825*t20835*t20827*t20856*1.2E1;
                double t20967 = c*c0*gamma_aa*t20832*t20851*t20825*t20827*t20837*t20846*t20857*1.6E1;
                double t20959 = t20956+t20957-t20966+t20958-t20967;
                double t20960 = t20854*t20873*t20936*2.0;
                double t20961 = c*c0*t20832*t20835*t20837*t20846*t20892*(8.0/9.0);
                double t20962 = a5*c0*gamma_aa*t20820*t20832*t20835*t20908*t20846*(5.6E1/9.0);
                double t20963 = c*c0*t20821*t20902*t20903*t20832*t20835*t20837*(1.6E1/3.0);
                double t20964 = c*c0*t20823*t20832*t20825*t20835*t20827*t20837*t20846*t20894*(2.56E2/9.0);
                double t20998 = t20823*t20825*t20835*t20827*t20917*t20856*(6.4E1/3.0);
                double t20999 = c*c0*t20823*t20832*t20825*t20827*t20837*t20846*t20857*t20894*(2.56E2/9.0);
                double t20965 = t20960+t20961+t20962+t20963+t20964-t20998-t20999;
                double t20968 = c*c0*t20832*t20852*t20835*t20853*t20837*t20846*t20892*(2.0/9.0);
                double t20969 = c*c0*t20821*t20902*t20903*t20832*t20852*t20835*t20853*t20837*(4.0/3.0);
                double t20970 = c*c0*t20821*t20832*t20852*t20835*t20853*t20916*t20837*t20846*(2.0/3.0);
                double t20971 = a5*c0*gamma_aa*t20820*t20832*t20852*t20835*t20853*t20908*t20846*(1.4E1/9.0);
                double t20972 = c*c0*t20823*t20832*t20825*t20852*t20835*t20853*t20827*t20837*t20846*t20894*(6.4E1/9.0);
                double t20975 = 1.0/c0;
                double t20976 = pow(rho_a,1.0/3.0);
                double t20977 = exp(-t20831);
                double t20978 = t20834*t20834;
                double t20979 = t20845*t20845;
                double t20981 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a);
                double t21000 = c*c0*t20823*t20832*t20825*t20852*t20853*t20827*t20837*t20846*t20857*t20894*(6.4E1/9.0);
                double t20982 = t20910+t20911+t20912+t20970+t20971+t20909+t20972-t20928+t20968+t20969-t21000;
                double t20983 = exp(-t20875);
                double t20986 = t20882*t20884*t20983*t20879*1.0E5;
                double t20988 = c*c0*t20821*(2.0/3.0);
                double t20989 = -t20986+t20988;
                double t20990 = dirac(t20989);
                double t20991 = a1*t20820*t20848;
                double t20992 = gamma_aa*t20822*t20824*t20825*2.0;
                double t20993 = t20991+t20992;
                double t20994 = 1.0/pow(t20881,3.0/2.0);
                double t20995 = a3*t20820*t20828*t20829*t20884*t20983*t20994*t20879*5.0E4;
                double t20996 = gamma_aa*t20822*t20824*t20825*t20882*t20884*t20983*t20879*2.0E5;
                double t21023 = t20882*t20983*t20993*t20879*1.0E5;
                double t20997 = t20995+t20996-t21023;
                double t21001 = t20852*t20853*t20864;
                double t21002 = t20861*t20907*t20866;
                double t21004 = t20828*t20864*t20918*t20975*t20976*t20977*t20978*t20979*(3.0/2.0);
                double t21005 = t20852*t20853;
                double t21006 = t20861*t20866*t20869;
                double t21007 = t21005+t21006;
                double t21008 = 1.0/pow(rho_a,1.0E1/3.0);
                double t21009 = t20822*t20823*t20825*t20917*(1.6E1/3.0);
                double t21010 = a1*gamma_aa*t20820*t20908*(8.0/3.0);
                double t21011 = t21010+t21009;
                double t21012 = 1.0/(c*c);
                double t21013 = 1.0/pow(t20826,1.0/5.0);
                double t21014 = t20861*t20938*t20869;
                double t21015 = t21001+t21002+t21004+t21014-t20937*t20866*t20869*t20959;
                double t21016 = 1.0/pow(rho_a,5.0/3.0);
                double t21017 = t20890*t20885*t21016*(2.0/3.0);
                double t21018 = 1.0/(t20889*t20889*t20889);
                double t21019 = c*k2*t20892*(1.0/3.0);
                double t21020 = c*k1*t20892*t20898*(1.0/6.0);
                double t21021 = t21020+t21019;
                double t21022 = t21017-t20885*t20886*t21021*t21018*2.0;
                double t21024 = c*c0*t20892*(2.0/9.0);
                double t21025 = t20822*t20823*t20825*t20917*t20882*t20884*t20983*t20879*(1.6E6/3.0);
                double t21026 = a3*gamma_aa*t20820*t20828*t20884*t20983*t20994*t20879*t21008*(3.5E5/3.0);
                double t21027 = t21024+t21025+t21026-t20882*t20983*t20879*t21011*1.0E5-c*kab1*rab1*t20882*t20892*t20884*t20983*t20877*t21013*(8.0E4/3.0);
                v_rho_a_gamma_aa[Q] += scale * (c0*t20891*heaviside(c*c0*t20821*(-2.0/3.0)+t20882*t20884*t20879*exp(-t20822*t20823*t20824*t20825)*1.0E5)*(t21001+t21002+t21004+t20861*t20869*(t20913+t20914+t20915+t20862+t20863-c*c0*gamma_aa*t20832*t20851*t20825*t20827*t20837*t20846*t20857*(8.0/3.0)-c*c0*gamma_aa*t20832*t20851*t20825*t20852*t20853*t20827*t20837*t20846*t20857*(8.0/3.0))-t20937*t20866*t20869*t20959)*(1.0/2.0)-c0*t20891*t20990*t20997*t21007*(1.0/2.0)-c0*rho_a*t20891*heaviside(t20986-c*c0*t20821*(2.0/3.0))*(t20861*t20866*(t20854*t20873*t20945*2.0-t20854*t20872*t20936*t20946-gamma_aa*t20825*t20835*t20827*t20917*t20856*(1.28E2/3.0)+a5*c0*t20820*t20832*t20835*t20908*t20846*(6.4E1/9.0)-gamma_aa*t20824*t20825*t20835*t20827*t20873*t20936*4.0+gamma_aa*t20823*t20856*t20857*t20947*t20895*t20897*(2.56E2/3.0)-t20823*t20825*t20835*t20827*t20872*t20917*t20873*(3.2E1/3.0)+a5*c0*t20820*t20902*t20903*t20832*t20835*t20848*(1.6E1/3.0)+c*c0*gamma_aa*t20832*t20825*t20835*t20827*t20837*t20846*t20894*(5.44E2/9.0)-c*c0*gamma_aa*t20832*t20825*t20827*t20837*t20846*t20857*t20894*(5.44E2/9.0)+a5*c0*t20820*t20823*t20832*t20825*t20835*t20827*t20846*t20893*(1.6E2/3.0)-a5*c0*t20820*t20823*t20832*t20825*t20827*t20846*t20857*t20893*(1.6E2/3.0)+c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20835*t20827*t20837*(6.4E1/3.0)-c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20827*t20837*t20857*(6.4E1/3.0)+c*c0*gamma_aa*t20823*t20832*t20924*t20837*t20846*t20895*t20896*t20897*(5.12E2/3.0)+c*c0*gamma_aa*t20823*t20832*t20835*t20837*t20846*t20895*t20896*t20897*(1.024E3/9.0)-c*c0*gamma_aa*t20823*t20832*t20837*t20846*t20857*t20895*t20896*t20897*(2.048E3/9.0))+t20852*t20853*t20927+t20861*t20907*t20982+t20861*t20938*t20965+t20861*t20869*(t20920+t20921-t20930+t20922-t20931+t20923+t20925+t20926-t20980+t20919-t20929+t20916*t20864+(t20864*(1.0/3.0))/rho_a+(t20902*t20864*2.0)/t20845+t20823*t20825*t20827*t20917*t20864*(3.2E1/3.0)-t20823*t20825*t20827*t20854*t20917*t20864*(3.2E1/3.0)+a5*gamma_aa*t20820*t20828*t20864*t20918*t21008*(7.0/3.0)+a5*c0*t20820*t20832*t20852*t20835*t20853*t20908*t20846*(1.6E1/9.0)+a5*c0*t20820*t20902*t20903*t20832*t20852*t20835*t20853*t20848*(4.0/3.0)+a5*c0*t20820*t20832*t20852*t20835*t20853*t20916*t20846*t20848*(2.0/3.0)+c*c0*t20821*t20832*t20852*t20835*t20853*t20837*t20846*t20927*(2.0/3.0)+c*c0*t20832*t20852*t20835*t20853*t20837*t20846*t20864*t20892*(2.0/9.0)+a5*c0*gamma_aa*t20820*t20832*t20852*t20835*t20853*t20908*t20846*t20864*(1.4E1/9.0)+c*c0*t20821*t20902*t20903*t20832*t20852*t20835*t20853*t20837*t20864*(4.0/3.0)+c*c0*t20821*t20832*t20852*t20835*t20853*t20916*t20837*t20846*t20864*(2.0/3.0)+c*c0*gamma_aa*t20832*t20825*t20852*t20835*t20853*t20827*t20837*t20846*t20894*(1.36E2/9.0)-c*c0*gamma_aa*t20832*t20825*t20852*t20853*t20827*t20837*t20846*t20857*t20894*(1.36E2/9.0)+a5*c0*t20820*t20823*t20832*t20825*t20852*t20835*t20853*t20827*t20846*t20893*(4.0E1/3.0)-a5*c0*t20820*t20823*t20832*t20825*t20852*t20853*t20827*t20846*t20857*t20893*(4.0E1/3.0)+c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20852*t20835*t20853*t20827*t20837*(1.6E1/3.0)-c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20852*t20853*t20827*t20837*t20857*(1.6E1/3.0)+c*c0*gamma_aa*t20832*t20851*t20825*t20852*t20835*t20853*t20916*t20827*t20837*t20846*(8.0/3.0)-c*c0*gamma_aa*t20832*t20851*t20825*t20852*t20853*t20916*t20827*t20837*t20846*t20857*(8.0/3.0)+c*c0*gamma_aa*t20823*t20832*t20852*t20924*t20853*t20837*t20846*t20895*t20896*t20897*(1.28E2/3.0)+c*c0*gamma_aa*t20823*t20832*t20852*t20835*t20853*t20837*t20846*t20895*t20896*t20897*(2.56E2/9.0)-c*c0*gamma_aa*t20823*t20832*t20852*t20853*t20837*t20846*t20857*t20895*t20896*t20897*(5.12E2/9.0)+c*c0*t20823*t20832*t20825*t20852*t20835*t20853*t20827*t20837*t20846*t20864*t20894*(6.4E1/9.0)-c*c0*t20823*t20832*t20825*t20852*t20853*t20827*t20837*t20846*t20864*t20857*t20894*(6.4E1/9.0))+t20852*t20853*t20916*t20864-t20907*t20937*t20955*t20866-t20937*t20955*t20938*t20869-t20937*t20866*t20965*t20959-t20937*t20982*t20869*t20959-t20937*t20866*t20869*(t20854*t20873*t20945*3.0-t20854*t20872*t20936*t20946*(3.0/2.0)-gamma_aa*t20825*t20835*t20827*t20917*t20856*6.4E1+a5*c0*t20820*t20832*t20835*t20908*t20846*(3.2E1/3.0)-gamma_aa*t20824*t20825*t20835*t20827*t20873*t20936*6.0+gamma_aa*t20823*t20856*t20857*t20947*t20895*t20897*1.28E2-t20823*t20825*t20835*t20827*t20872*t20917*t20873*1.6E1+a5*c0*t20820*t20902*t20903*t20832*t20835*t20848*8.0+c*c0*gamma_aa*t20832*t20825*t20835*t20827*t20837*t20846*t20894*(2.72E2/3.0)-c*c0*gamma_aa*t20832*t20825*t20827*t20837*t20846*t20857*t20894*(2.72E2/3.0)+a5*c0*t20820*t20823*t20832*t20825*t20835*t20827*t20846*t20893*8.0E1-a5*c0*t20820*t20823*t20832*t20825*t20827*t20846*t20857*t20893*8.0E1+c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20835*t20827*t20837*3.2E1-c*c0*gamma_aa*t20902*t20903*t20832*t20851*t20825*t20827*t20837*t20857*3.2E1+c*c0*gamma_aa*t20823*t20832*t20924*t20837*t20846*t20895*t20896*t20897*2.56E2+c*c0*gamma_aa*t20823*t20832*t20835*t20837*t20846*t20895*t20896*t20897*(5.12E2/3.0)-c*c0*gamma_aa*t20823*t20832*t20837*t20846*t20857*t20895*t20896*t20897*(1.024E3/3.0))+1.0/(t20860*t20860*t20860)*t20955*t20866*t20869*t20959*2.0+t20828*t20918*t20927*t20975*t20976*t20977*t20978*t20979*(3.0/2.0)+t20916*t20828*t20864*t20918*t20975*t20976*t20977*t20978*t20979*(3.0/2.0)-a5*1.0/(rho_a*rho_a)*t20820*t20916*1.0/(t20837*t20837)*t20975*t20977*t20978*t20979*t21012*(3.0/2.0)+gamma_aa*t20825*t20834*t20916*t20827*t20828*t20918*t20981*t20975*t20977*t20979*6.0-gamma_aa*t20825*t20916*t20827*t20828*t20918*t20981*t20975*t20977*t20978*t20979*6.0)*(1.0/2.0)+c0*rho_a*t20891*t20990*t21015*t21027*(1.0/2.0)+c0*rho_a*t20891*t20990*t21007*(t20882*t20983*t20879*(a1*t20820*t20908*(8.0/3.0)+gamma_aa*t20822*t20825*t20917*(3.2E1/3.0))*-1.0E5+t20822*t20823*t20825*t20917*t20882*t20983*t20993*t20879*(1.6E6/3.0)+a3*t20820*t20828*t20829*t20983*t20994*t20879*t21011*5.0E4+a3*t20820*t20828*t20884*t20983*t20994*t20879*t21008*(3.5E5/3.0)+gamma_aa*t20822*t20825*t20917*t20882*t20884*t20983*t20879*(3.2E6/3.0)+gamma_aa*t20822*t20824*t20825*t20882*t20983*t20879*t21011*2.0E5-(a3*a3)*gamma_aa*t20851*t20825*1.0/pow(t20881,5.0/2.0)*t20884*t20983*t20879*t21012*1.75E5+a3*gamma_aa*t20820*t20828*t20983*t20993*t20994*t20879*t21008*(3.5E5/3.0)-c*kab1*rab1*t20882*t20892*t20983*t20993*t20877*t21013*(8.0E4/3.0)-gamma_aa*(t20822*t20822)*t20823*t20882*t20884*t20947*t20983*t20879*t20897*(3.2E6/3.0)+a3*kab1*rab1*t20820*t20908*t20884*t20983*t20877*t20994*t21013*(4.0E4/3.0)-a3*1.0/pow(rho_a,2.6E1/3.0)*t20820*t20822*t20823*t20825*t20828*t20884*t20983*t20994*t20879*5.0E5+c*gamma_aa*kab1*rab1*t20822*t20825*t20882*t20884*t20983*t20894*t20877*t21013*(1.6E5/3.0))*(1.0/2.0)+c0*rho_a*t20891*t20990*t20997*(t20852*t20853*t20916+t20861*t20866*t20965+t20861*t20982*t20869-t20937*t20955*t20866*t20869+t20916*t20828*t20918*t20975*t20976*t20977*t20978*t20979*(3.0/2.0))*(1.0/2.0)+c0*rho_a*t20891*t21022*t21015*heaviside(t20986-t20988)*(1.0/2.0)-c0*rho_a*t20891*t20990*t20997*t21022*t21007*(1.0/2.0)+c0*rho_a*t20891*t20997*t21007*t21027*dirac(t20989,1.0)*(1.0/2.0));
            }
            
        } else {
            // v
            if (deriv >= 0) {
                double t17003 = tspi_m13*tspi_m13;
                double t17004 = 1.0/pow(rho_a,1.0/3.0);
                double t17005 = a2*a2;
                double t17006 = gamma_aa*gamma_aa;
                double t17007 = 1.0/pow(rho_a,1.6E1/3.0);
                double t17008 = t17003*t17003;
                double t17009 = c*t17004;
                double t17010 = pow(t17009,4.0/5.0);
                double t17024 = rab1*t17010;
                double t17011 = exp(-t17024);
                double t17012 = t17011-1.0;
                double t17025 = kab1*t17012;
                double t17013 = kab0-t17025;
                double t17014 = 1.0/c;
                double t17015 = 1.0/pow(rho_a,7.0/3.0);
                double t17016 = a3*gamma_aa*t17003*t17014*t17015;
                double t17017 = t17016+1.0;
                double t17018 = t17005*t17006*t17007*t17008;
                double t17019 = 1.0/pow(rho_a,8.0/3.0);
                double t17020 = a1*gamma_aa*t17003*t17019;
                double t17021 = t17020+t17018+1.0;
                double t17022 = t17005*t17006*t17007*t17008*2.0;
                double t17023 = exp(t17022);
                double t17026 = 1.0/(t17013*t17013);
                double t17027 = 1.0/(t17021*t17021);
                double t17028 = c*c0*t17004*t17023*t17017*t17026*t17027*(2.0/3.0);
                double t17029;
                double t17030;
                if (t17028 > expei_cutoff) {
                    t17029 = 1.0;
                    t17030 = expei(t17028);
                } else {
                    t17029 = exp(t17028);
                    t17030 = Ei(-t17028);
                }
                double t17031 = sqrt(t17028);
                double t17032 = 1.0/pow(rho_b,1.0/3.0);
                double t17033 = gamma_bb*gamma_bb;
                double t17034 = 1.0/pow(rho_b,1.6E1/3.0);
                double t17035 = c*t17032;
                double t17036 = a4*a4;
                double t17037 = 1.0/pow(rho_b,7.0/3.0);
                double t17038 = sqrt(t17035);
                double t17039 = t17033*t17034*t17008*t17036*2.0;
                double t17040 = exp(t17039);
                double t17041 = t17033*t17034*t17008*t17036;
                double t17042 = t17041+1.0;
                double t17043 = 1.0/(t17042*t17042);
                double t17044 = a5*gamma_bb*t17003*t17014*t17037;
                double t17045 = t17044+1.0;
                double t17056 = raa2*t17038;
                double t17046 = exp(-t17056);
                double t17047 = t17046-1.0;
                double t17048 = kaa2*t17047;
                double t17049 = pow(t17035,2.0/5.0);
                double t17057 = raa1*t17049;
                double t17050 = exp(-t17057);
                double t17051 = t17050-1.0;
                double t17052 = kaa1*t17051;
                double t17053 = -kaa0+t17052+t17048;
                double t17054 = 1.0/(t17053*t17053);
                double t17055 = c*c0*t17040*t17032*t17043*t17045*t17054*(2.0/3.0);
                double t17058;
                double t17059;
                if (t17055 > expei_cutoff) {
                    t17058 = 1.0;
                    t17059 = expei(t17055);
                } else {
                    t17058 = exp(t17055);
                    t17059 = Ei(-t17055);
                }
                double t17060 = 1.0/t17042;
                double t17061 = c*c0*t17040*t17032*t17045*t17054*(2.0/3.0);
                double t17062 = sqrt(t17061);
                double t17063 = t17005*t17033*t17034*t17008;
                double t17064 = pow(t17035,4.0/5.0);
                double t17074 = rab1*t17064;
                double t17065 = exp(-t17074);
                double t17066 = t17065-1.0;
                double t17075 = kab1*t17066;
                double t17067 = kab0-t17075;
                double t17068 = a3*gamma_bb*t17003*t17014*t17037;
                double t17069 = t17068+1.0;
                double t17070 = 1.0/sqrt(t17069);
                double t17071 = 1.0/pow(rho_b,8.0/3.0);
                double t17072 = a1*gamma_bb*t17003*t17071;
                double t17073 = t17063+t17072+1.0;
                double t17076 = t17005*t17033*t17034*t17008*2.0;
                double t17077 = exp(t17076);
                double t17078 = 1.0/(t17067*t17067);
                double t17079 = 1.0/(t17073*t17073);
                double t17080 = c*c0*t17032*t17077*t17069*t17078*t17079*(2.0/3.0);
                double t17081;
                double t17082;
                if (t17080 > expei_cutoff) {
                    t17081 = 1.0;
                    t17082 = expei(t17080);
                } else {
                    t17081 = exp(t17080);
                    t17082 = Ei(-t17080);
                }
                double t17083 = sqrt(t17080);
                double t17084 = 1.0/sqrt(t17017);
                double t17085 = c*c;
                double t17086 = sqrt(t17009);
                double t17087 = t17006*t17007*t17008*t17036*2.0;
                double t17088 = exp(t17087);
                double t17089 = t17006*t17007*t17008*t17036;
                double t17090 = t17089+1.0;
                double t17091 = 1.0/(t17090*t17090);
                double t17092 = a5*gamma_aa*t17003*t17014*t17015;
                double t17093 = t17092+1.0;
                double t17104 = raa2*t17086;
                double t17094 = exp(-t17104);
                double t17095 = t17094-1.0;
                double t17096 = kaa2*t17095;
                double t17097 = pow(t17009,2.0/5.0);
                double t17105 = raa1*t17097;
                double t17098 = exp(-t17105);
                double t17099 = t17098-1.0;
                double t17100 = kaa1*t17099;
                double t17101 = -kaa0+t17100+t17096;
                double t17102 = 1.0/(t17101*t17101);
                double t17103 = c*c0*t17102*t17004*t17091*t17093*t17088*(2.0/3.0);
                double t17106;
                double t17107;
                if (t17103 > expei_cutoff) {
                    t17106 = 1.0;
                    t17107 = expei(t17103);
                } else {
                    t17106 = exp(t17103);
                    t17107 = Ei(-t17103);
                }
                double t17108 = 1.0/t17090;
                double t17109 = c*c0*t17102*t17004*t17093*t17088*(2.0/3.0);
                double t17110 = sqrt(t17109);
                v[Q] += scale * (rho_b*(c0*heaviside(c*c0*t17004*(-2.0/3.0)+t17021*t17013*t17084*exp(-t17005*t17006*t17007*t17008)*1.0E5)*(t17030*t17029+((c*c0*t17030*t17004*t17023*t17017*t17026*t17027*t17029*(2.0/3.0)+1.0)*(t17031*4.0+c*c0*t17004*t17023*t17017*t17026*t17027*(8.0/3.0)+6.0))/(t17031*6.0+c*c0*t17004*t17023*t17017*t17026*t17027*4.0+3.0))*(1.0/2.0)+c0*exp(-1.0/pow(rho_b,2.0/3.0)*t17085*1.0/pow(k1*t17038+c*k2*t17032,2.0))*heaviside(c*c0*t17032*(-2.0/3.0)+t17070*t17073*t17067*exp(-t17005*t17033*t17034*t17008)*1.0E5)*(t17058*t17059+((c*c0*t17040*t17032*t17043*t17045*t17054*t17058*t17059*(2.0/3.0)+1.0)*(t17060*t17062*4.0+c*c0*t17040*t17032*t17043*t17045*t17054*(8.0/3.0)+6.0))/(t17060*t17062*6.0+c*c0*t17040*t17032*t17043*t17045*t17054*4.0+3.0))*(1.0/2.0))+rho_a*(c0*heaviside(c*c0*t17032*(-2.0/3.0)+t17070*t17073*t17067*exp(-t17063)*1.0E5)*(t17081*t17082+((c*c0*t17032*t17081*t17082*t17077*t17069*t17078*t17079*(2.0/3.0)+1.0)*(t17083*4.0+c*c0*t17032*t17077*t17069*t17078*t17079*(8.0/3.0)+6.0))/(t17083*6.0+c*c0*t17032*t17077*t17069*t17078*t17079*4.0+3.0))*(1.0/2.0)+c0*heaviside(c*c0*t17004*(-2.0/3.0)+t17021*t17013*t17084*exp(-t17018)*1.0E5)*exp(-1.0/pow(rho_a,2.0/3.0)*t17085*1.0/pow(k1*t17086+c*k2*t17004,2.0))*(t17106*t17107+((c*c0*t17102*t17004*t17106*t17107*t17091*t17093*t17088*(2.0/3.0)+1.0)*(t17110*t17108*4.0+c*c0*t17102*t17004*t17091*t17093*t17088*(8.0/3.0)+6.0))/(t17110*t17108*6.0+c*c0*t17102*t17004*t17091*t17093*t17088*4.0+3.0))*(1.0/2.0)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t17112 = tspi_m13*tspi_m13;
                double t17113 = 1.0/pow(rho_a,1.0/3.0);
                double t17114 = a2*a2;
                double t17115 = gamma_aa*gamma_aa;
                double t17116 = 1.0/pow(rho_a,1.6E1/3.0);
                double t17117 = t17112*t17112;
                double t17118 = c*t17113;
                double t17119 = a4*a4;
                double t17120 = t17115*t17116*t17117*t17119*2.0;
                double t17121 = exp(t17120);
                double t17122 = 1.0/c;
                double t17123 = 1.0/pow(rho_a,7.0/3.0);
                double t17124 = sqrt(t17118);
                double t17125 = t17115*t17116*t17117*t17119;
                double t17126 = t17125+1.0;
                double t17127 = 1.0/(t17126*t17126);
                double t17128 = a5*gamma_aa*t17112*t17122*t17123;
                double t17129 = t17128+1.0;
                double t17140 = raa2*t17124;
                double t17130 = exp(-t17140);
                double t17131 = t17130-1.0;
                double t17132 = kaa2*t17131;
                double t17133 = pow(t17118,2.0/5.0);
                double t17141 = raa1*t17133;
                double t17134 = exp(-t17141);
                double t17135 = t17134-1.0;
                double t17136 = kaa1*t17135;
                double t17137 = -kaa0+t17132+t17136;
                double t17138 = 1.0/(t17137*t17137);
                double t17139 = c*c0*t17121*t17113*t17127*t17129*t17138*(2.0/3.0);
                double t17142 = 1.0/t17126;
                double t17143 = 1.0/pow(rho_a,4.0/3.0);
                double t17144 = c*c0*t17121*t17113*t17129*t17138*(2.0/3.0);
                double t17145 = sqrt(t17144);
                double t17146 = 1.0/pow(rho_a,1.1E1/3.0);
                double t17147 = 1.0/sqrt(t17118);
                double t17148 = c*kaa2*raa2*t17130*t17143*t17147*(1.0/6.0);
                double t17149 = 1.0/pow(t17118,3.0/5.0);
                double t17150 = c*kaa1*raa1*t17134*t17143*t17149*(2.0/1.5E1);
                double t17151 = t17150+t17148;
                double t17152 = 1.0/(t17137*t17137*t17137);
                double t17153 = 1.0/pow(rho_a,2.0E1/3.0);
                double t17154;
                double t17155;
                if (t17139 > expei_cutoff) {
                    t17154 = 1.0;
                    t17155 = expei(t17139);
                } else {
                    t17154 = exp(t17139);
                    t17155 = Ei(-t17139);
                }
                double t17156 = 1.0/(t17126*t17126*t17126);
                double t17157 = t17142*t17145*6.0;
                double t17158 = c*c0*t17121*t17113*t17127*t17129*t17138*4.0;
                double t17159 = t17157+t17158+3.0;
                double t17160 = 1.0/t17159;
                double t17161 = c*c0*t17121*t17143*t17127*t17129*t17138*(2.0/9.0);
                double t17162 = a5*c0*gamma_aa*t17112*t17121*t17127*t17146*t17138*(1.4E1/9.0);
                double t17163 = c*c0*t17121*t17113*t17151*t17152*t17127*t17129*(4.0/3.0);
                double t17164 = c*c0*t17121*t17115*t17117*t17153*t17127*t17119*t17129*t17138*(6.4E1/9.0);
                double t17166 = c*c0*t17121*t17115*t17117*t17153*t17119*t17129*t17138*t17156*(6.4E1/9.0);
                double t17165 = t17161+t17162+t17163+t17164-t17166;
                double t17167 = c*c0*t17121*t17113*t17127*t17154*t17155*t17129*t17138*(2.0/3.0);
                double t17168 = t17167+1.0;
                double t17169 = t17142*t17145*4.0;
                double t17170 = c*c0*t17121*t17113*t17127*t17129*t17138*(8.0/3.0);
                double t17171 = t17170+t17169+6.0;
                double t17172 = c*c0*t17121*t17143*t17129*t17138*(2.0/9.0);
                double t17173 = a5*c0*gamma_aa*t17112*t17121*t17146*t17138*(1.4E1/9.0);
                double t17174 = c*c0*t17121*t17113*t17151*t17152*t17129*(4.0/3.0);
                double t17175 = c*c0*t17121*t17115*t17117*t17153*t17119*t17129*t17138*(6.4E1/9.0);
                double t17176 = t17172+t17173+t17174+t17175;
                double t17177 = 1.0/sqrt(t17144);
                double t17178 = 1.0/pow(rho_a,1.9E1/3.0);
                double t17179 = t17114*t17115*t17116*t17117;
                double t17180 = pow(t17118,4.0/5.0);
                double t17198 = rab1*t17180;
                double t17181 = exp(-t17198);
                double t17182 = t17181-1.0;
                double t17199 = kab1*t17182;
                double t17183 = kab0-t17199;
                double t17184 = a3*gamma_aa*t17112*t17122*t17123;
                double t17185 = t17184+1.0;
                double t17186 = 1.0/sqrt(t17185);
                double t17187 = 1.0/pow(rho_a,8.0/3.0);
                double t17188 = a1*gamma_aa*t17112*t17187;
                double t17189 = t17179+t17188+1.0;
                double t17190 = c*c;
                double t17191 = 1.0/pow(rho_a,2.0/3.0);
                double t17192 = k1*t17124;
                double t17193 = c*k2*t17113;
                double t17194 = t17192+t17193;
                double t17195 = 1.0/(t17194*t17194);
                double t17201 = t17190*t17191*t17195;
                double t17196 = exp(-t17201);
                double t17197 = exp(-t17179);
                double t17200 = c*c0*t17113*(2.0/3.0);
                double t17202 = t17154*t17155;
                double t17203 = t17160*t17171*t17168;
                double t17204 = t17202+t17203;
                double t17205 = t17183*t17186*t17197*t17189*1.0E5;
                double t17206 = -t17200+t17205;
                double t17207 = heaviside(t17206);
                double t17208 = t17114*t17115*t17116*t17117*2.0;
                double t17209 = exp(t17208);
                double t17210 = 1.0/(t17183*t17183);
                double t17211 = 1.0/(t17189*t17189);
                double t17212 = c*c0*t17210*t17211*t17113*t17209*t17185*(2.0/3.0);
                double t17213 = sqrt(t17212);
                double t17214 = t17114*t17115*t17117*t17178*(1.6E1/3.0);
                double t17215 = a1*gamma_aa*t17112*t17146*(8.0/3.0);
                double t17216 = t17214+t17215;
                double t17217;
                double t17218;
                if (t17212 > expei_cutoff) {
                    t17217 = 1.0;
                    t17218 = expei(t17212);
                } else {
                    t17217 = exp(t17212);
                    t17218 = Ei(-t17212);
                }
                double t17219 = c*c0*t17210*t17211*t17143*t17209*t17185*(2.0/9.0);
                double t17220 = 1.0/(t17189*t17189*t17189);
                double t17221 = a3*c0*gamma_aa*t17210*t17112*t17211*t17146*t17209*(1.4E1/9.0);
                double t17222 = 1.0/pow(rho_a,5.0/3.0);
                double t17223 = 1.0/pow(t17118,1.0/5.0);
                double t17224 = c*c0*t17210*t17211*t17114*t17115*t17117*t17153*t17209*t17185*(6.4E1/9.0);
                double t17225 = 1.0/(t17183*t17183*t17183);
                double t17231 = c*c0*t17210*t17220*t17113*t17216*t17209*t17185*(4.0/3.0);
                double t17232 = c0*kab1*rab1*t17211*t17222*t17223*t17225*t17181*t17190*t17209*t17185*(1.6E1/4.5E1);
                double t17226 = t17221-t17231-t17232+t17224+t17219;
                double t17227 = t17213*6.0;
                double t17228 = c*c0*t17210*t17211*t17113*t17209*t17185*4.0;
                double t17229 = t17227+t17228+3.0;
                double t17230 = 1.0/t17229;
                double t17233 = c*c0*t17210*t17211*t17113*t17217*t17209*t17218*t17185*(2.0/3.0);
                double t17234 = t17233+1.0;
                double t17235 = t17213*4.0;
                double t17236 = c*c0*t17210*t17211*t17113*t17209*t17185*(8.0/3.0);
                double t17237 = t17235+t17236+6.0;
                double t17238 = 1.0/sqrt(t17212);
                double t17239 = 1.0/c0;
                double t17240 = pow(rho_a,1.0/3.0);
                double t17241 = c*c0*t17143*(2.0/9.0);
                double t17242 = t17114*t17115*t17117*t17183*t17186*t17178*t17197*t17189*(1.6E6/3.0);
                double t17243 = 1.0/pow(rho_a,1.0E1/3.0);
                double t17244 = 1.0/pow(t17185,3.0/2.0);
                double t17245 = a3*gamma_aa*t17112*t17122*t17243*t17244*t17183*t17197*t17189*(3.5E5/3.0);
                double t17246 = t17241+t17242+t17245-t17216*t17183*t17186*t17197*1.0E5-c*kab1*rab1*t17223*t17143*t17181*t17186*t17197*t17189*(8.0E4/3.0);
                double t17247 = 1.0/pow(rho_b,1.0/3.0);
                double t17248 = gamma_bb*gamma_bb;
                double t17249 = 1.0/pow(rho_b,1.6E1/3.0);
                double t17250 = c*t17247;
                double t17251 = pow(t17250,4.0/5.0);
                double t17264 = rab1*t17251;
                double t17252 = exp(-t17264);
                double t17253 = t17252-1.0;
                double t17265 = kab1*t17253;
                double t17254 = kab0-t17265;
                double t17255 = 1.0/pow(rho_b,7.0/3.0);
                double t17256 = a3*gamma_bb*t17112*t17122*t17255;
                double t17257 = t17256+1.0;
                double t17258 = t17114*t17117*t17248*t17249;
                double t17259 = 1.0/pow(rho_b,8.0/3.0);
                double t17260 = a1*gamma_bb*t17112*t17259;
                double t17261 = t17260+t17258+1.0;
                double t17262 = t17114*t17117*t17248*t17249*2.0;
                double t17263 = exp(t17262);
                double t17266 = 1.0/(t17254*t17254);
                double t17267 = 1.0/(t17261*t17261);
                double t17268 = c*c0*t17263*t17247*t17257*t17266*t17267*(2.0/3.0);
                double t17269;
                double t17270;
                if (t17268 > expei_cutoff) {
                    t17269 = 1.0;
                    t17270 = expei(t17268);
                } else {
                    t17269 = exp(t17268);
                    t17270 = Ei(-t17268);
                }
                double t17271 = sqrt(t17268);
                v_rho_a[Q] += scale * (-rho_b*(c0*t17207*(t17230*t17234*(t17226*t17238*2.0+c*c0*t17210*t17211*t17143*t17209*t17185*(8.0/9.0)+a3*c0*gamma_aa*t17210*t17112*t17211*t17146*t17209*(5.6E1/9.0)-c*c0*t17210*t17220*t17113*t17216*t17209*t17185*(1.6E1/3.0)+c*c0*t17210*t17211*t17114*t17115*t17117*t17153*t17209*t17185*(2.56E2/9.0)-c0*kab1*rab1*t17211*t17222*t17223*t17225*t17181*t17190*t17209*t17185*(6.4E1/4.5E1))+t17230*t17237*(t17221-t17231-t17232+t17224+t17219+c*c0*t17210*t17211*t17143*t17217*t17209*t17218*t17185*(2.0/9.0)+a3*c0*gamma_aa*t17210*t17112*t17211*t17217*t17146*t17209*t17218*(1.4E1/9.0)-c*c0*t17210*t17220*t17113*t17216*t17217*t17209*t17218*t17185*(4.0/3.0)+c*c0*t17210*t17211*t17113*t17217*t17226*t17209*t17218*t17185*(2.0/3.0)+c*c0*t17210*t17211*t17114*t17115*t17117*t17153*t17217*t17209*t17218*t17185*(6.4E1/9.0)-c0*kab1*rab1*t17211*t17222*t17223*t17225*t17181*t17190*t17217*t17209*t17218*t17185*(1.6E1/4.5E1))+t17217*t17226*t17218-t17234*t17237*1.0/(t17229*t17229)*(t17226*t17238*3.0+c*c0*t17210*t17211*t17143*t17209*t17185*(4.0/3.0)+a3*c0*gamma_aa*t17210*t17112*t17211*t17146*t17209*(2.8E1/3.0)-c*c0*t17210*t17220*t17113*t17216*t17209*t17185*8.0+c*c0*t17210*t17211*t17114*t17115*t17117*t17153*t17209*t17185*(1.28E2/3.0)-c0*kab1*rab1*t17211*t17222*t17223*t17225*t17181*t17190*t17209*t17185*(3.2E1/1.5E1))+(t17122*t17240*t17226*(t17183*t17183)*t17239*(t17189*t17189)*exp(-t17208)*(3.0/2.0))/t17185)*(1.0/2.0)-c0*t17246*dirac(t17200-t17205)*(t17217*t17218+t17230*t17234*t17237)*(1.0/2.0))+rho_a*(c0*t17196*heaviside(c*c0*t17113*(-2.0/3.0)+t17183*t17186*t17189*exp(-t17114*t17115*t17116*t17117)*1.0E5)*(t17160*t17168*(t17142*t17176*t17177*2.0-t17115*t17117*t17127*t17145*t17119*t17178*(6.4E1/3.0)+c*c0*t17121*t17143*t17127*t17129*t17138*(8.0/9.0)+a5*c0*gamma_aa*t17112*t17121*t17127*t17146*t17138*(5.6E1/9.0)+c*c0*t17121*t17113*t17151*t17152*t17127*t17129*(1.6E1/3.0)+c*c0*t17121*t17115*t17117*t17153*t17127*t17119*t17129*t17138*(2.56E2/9.0)-c*c0*t17121*t17115*t17117*t17153*t17119*t17129*t17138*t17156*(2.56E2/9.0))+t17160*t17171*(t17161+t17162+t17163+t17164-t17166+c*c0*t17121*t17143*t17127*t17154*t17155*t17129*t17138*(2.0/9.0)+a5*c0*gamma_aa*t17112*t17121*t17127*t17154*t17146*t17155*t17138*(1.4E1/9.0)+c*c0*t17121*t17113*t17151*t17152*t17127*t17154*t17155*t17129*(4.0/3.0)+c*c0*t17121*t17113*t17127*t17154*t17155*t17129*t17138*t17165*(2.0/3.0)+c*c0*t17121*t17115*t17117*t17153*t17127*t17154*t17119*t17155*t17129*t17138*(6.4E1/9.0)-c*c0*t17121*t17115*t17117*t17153*t17154*t17119*t17155*t17129*t17138*t17156*(6.4E1/9.0))+t17154*t17155*t17165-t17171*1.0/(t17159*t17159)*t17168*(t17142*t17176*t17177*3.0-t17115*t17117*t17127*t17145*t17119*t17178*3.2E1+c*c0*t17121*t17143*t17127*t17129*t17138*(4.0/3.0)+a5*c0*gamma_aa*t17112*t17121*t17127*t17146*t17138*(2.8E1/3.0)+c*c0*t17121*t17113*t17151*t17152*t17127*t17129*8.0+c*c0*t17121*t17115*t17117*t17153*t17127*t17119*t17129*t17138*(1.28E2/3.0)-c*c0*t17121*t17115*t17117*t17153*t17119*t17129*t17138*t17156*(1.28E2/3.0))+(t17122*t17240*(t17126*t17126)*(t17137*t17137)*t17165*t17239*exp(-t17120)*(3.0/2.0))/t17129)*(-1.0/2.0)+c0*t17204*t17207*t17196*(t17222*t17190*t17195*(2.0/3.0)-t17190*t17191*1.0/(t17194*t17194*t17194)*(c*k2*t17143*(1.0/3.0)+c*k1*t17143*t17147*(1.0/6.0))*2.0)*(1.0/2.0)+c0*t17204*t17246*t17196*dirac(t17200-t17183*t17186*t17197*t17189*1.0E5)*(1.0/2.0))+c0*heaviside(c*c0*t17247*(-2.0/3.0)+t17261*t17254*1.0/sqrt(t17257)*exp(-t17114*t17117*t17248*t17249)*1.0E5)*(t17270*t17269+((c*c0*t17270*t17263*t17247*t17257*t17266*t17267*t17269*(2.0/3.0)+1.0)*(t17271*4.0+c*c0*t17263*t17247*t17257*t17266*t17267*(8.0/3.0)+6.0))/(t17271*6.0+c*c0*t17263*t17247*t17257*t17266*t17267*4.0+3.0))*(1.0/2.0)+c0*t17204*t17207*t17196*(1.0/2.0));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t17273 = tspi_m13*tspi_m13;
                double t17274 = 1.0/pow(rho_b,1.0/3.0);
                double t17275 = a2*a2;
                double t17276 = gamma_bb*gamma_bb;
                double t17277 = 1.0/pow(rho_b,1.6E1/3.0);
                double t17278 = t17273*t17273;
                double t17279 = c*t17274;
                double t17280 = a4*a4;
                double t17281 = t17280*t17276*t17277*t17278*2.0;
                double t17282 = exp(t17281);
                double t17283 = 1.0/c;
                double t17284 = 1.0/pow(rho_b,7.0/3.0);
                double t17285 = sqrt(t17279);
                double t17286 = t17280*t17276*t17277*t17278;
                double t17287 = t17286+1.0;
                double t17288 = 1.0/(t17287*t17287);
                double t17289 = a5*gamma_bb*t17273*t17283*t17284;
                double t17290 = t17289+1.0;
                double t17301 = raa2*t17285;
                double t17291 = exp(-t17301);
                double t17292 = t17291-1.0;
                double t17293 = kaa2*t17292;
                double t17294 = pow(t17279,2.0/5.0);
                double t17302 = raa1*t17294;
                double t17295 = exp(-t17302);
                double t17296 = t17295-1.0;
                double t17297 = kaa1*t17296;
                double t17298 = -kaa0+t17293+t17297;
                double t17299 = 1.0/(t17298*t17298);
                double t17300 = c*c0*t17290*t17282*t17274*t17288*t17299*(2.0/3.0);
                double t17303 = 1.0/t17287;
                double t17304 = 1.0/pow(rho_b,4.0/3.0);
                double t17305 = c*c0*t17290*t17282*t17274*t17299*(2.0/3.0);
                double t17306 = sqrt(t17305);
                double t17307 = 1.0/pow(rho_b,1.1E1/3.0);
                double t17308 = 1.0/sqrt(t17279);
                double t17309 = c*kaa2*raa2*t17304*t17308*t17291*(1.0/6.0);
                double t17310 = 1.0/pow(t17279,3.0/5.0);
                double t17311 = c*kaa1*raa1*t17310*t17304*t17295*(2.0/1.5E1);
                double t17312 = t17311+t17309;
                double t17313 = 1.0/(t17298*t17298*t17298);
                double t17314 = 1.0/pow(rho_b,2.0E1/3.0);
                double t17315;
                double t17316;
                if (t17300 > expei_cutoff) {
                    t17315 = 1.0;
                    t17316 = expei(t17300);
                } else {
                    t17315 = exp(t17300);
                    t17316 = Ei(-t17300);
                }
                double t17317 = 1.0/(t17287*t17287*t17287);
                double t17318 = t17303*t17306*6.0;
                double t17319 = c*c0*t17290*t17282*t17274*t17288*t17299*4.0;
                double t17320 = t17318+t17319+3.0;
                double t17321 = 1.0/t17320;
                double t17322 = c*c0*t17304*t17290*t17282*t17288*t17299*(2.0/9.0);
                double t17323 = a5*c0*gamma_bb*t17307*t17273*t17282*t17288*t17299*(1.4E1/9.0);
                double t17324 = c*c0*t17312*t17313*t17290*t17282*t17274*t17288*(4.0/3.0);
                double t17325 = c*c0*t17314*t17280*t17290*t17282*t17276*t17278*t17288*t17299*(6.4E1/9.0);
                double t17327 = c*c0*t17314*t17280*t17290*t17317*t17282*t17276*t17278*t17299*(6.4E1/9.0);
                double t17326 = t17322+t17323+t17324+t17325-t17327;
                double t17328 = c*c0*t17315*t17316*t17290*t17282*t17274*t17288*t17299*(2.0/3.0);
                double t17329 = t17328+1.0;
                double t17330 = t17303*t17306*4.0;
                double t17331 = c*c0*t17290*t17282*t17274*t17288*t17299*(8.0/3.0);
                double t17332 = t17330+t17331+6.0;
                double t17333 = c*c0*t17304*t17290*t17282*t17299*(2.0/9.0);
                double t17334 = a5*c0*gamma_bb*t17307*t17273*t17282*t17299*(1.4E1/9.0);
                double t17335 = c*c0*t17312*t17313*t17290*t17282*t17274*(4.0/3.0);
                double t17336 = c*c0*t17314*t17280*t17290*t17282*t17276*t17278*t17299*(6.4E1/9.0);
                double t17337 = t17333+t17334+t17335+t17336;
                double t17338 = 1.0/sqrt(t17305);
                double t17339 = 1.0/pow(rho_b,1.9E1/3.0);
                double t17340 = t17275*t17276*t17277*t17278;
                double t17341 = pow(t17279,4.0/5.0);
                double t17359 = rab1*t17341;
                double t17342 = exp(-t17359);
                double t17343 = t17342-1.0;
                double t17360 = kab1*t17343;
                double t17344 = kab0-t17360;
                double t17345 = a3*gamma_bb*t17273*t17283*t17284;
                double t17346 = t17345+1.0;
                double t17347 = 1.0/sqrt(t17346);
                double t17348 = 1.0/pow(rho_b,8.0/3.0);
                double t17349 = a1*gamma_bb*t17273*t17348;
                double t17350 = t17340+t17349+1.0;
                double t17351 = c*c;
                double t17352 = 1.0/pow(rho_b,2.0/3.0);
                double t17353 = k1*t17285;
                double t17354 = c*k2*t17274;
                double t17355 = t17353+t17354;
                double t17356 = 1.0/(t17355*t17355);
                double t17362 = t17351*t17352*t17356;
                double t17357 = exp(-t17362);
                double t17358 = exp(-t17340);
                double t17361 = c*c0*t17274*(2.0/3.0);
                double t17363 = t17315*t17316;
                double t17364 = t17321*t17332*t17329;
                double t17365 = t17363+t17364;
                double t17366 = t17350*t17344*t17347*t17358*1.0E5;
                double t17367 = -t17361+t17366;
                double t17368 = heaviside(t17367);
                double t17369 = t17275*t17276*t17277*t17278*2.0;
                double t17370 = exp(t17369);
                double t17371 = 1.0/(t17344*t17344);
                double t17372 = 1.0/(t17350*t17350);
                double t17373 = c*c0*t17370*t17371*t17372*t17274*t17346*(2.0/3.0);
                double t17374 = sqrt(t17373);
                double t17375 = t17275*t17276*t17339*t17278*(1.6E1/3.0);
                double t17376 = a1*gamma_bb*t17307*t17273*(8.0/3.0);
                double t17377 = t17375+t17376;
                double t17378;
                double t17379;
                if (t17373 > expei_cutoff) {
                    t17378 = 1.0;
                    t17379 = expei(t17373);
                } else {
                    t17378 = exp(t17373);
                    t17379 = Ei(-t17373);
                }
                double t17380 = c*c0*t17304*t17370*t17371*t17372*t17346*(2.0/9.0);
                double t17381 = 1.0/(t17350*t17350*t17350);
                double t17382 = a3*c0*gamma_bb*t17307*t17370*t17371*t17273*t17372*(1.4E1/9.0);
                double t17383 = 1.0/pow(rho_b,5.0/3.0);
                double t17384 = 1.0/pow(t17279,1.0/5.0);
                double t17385 = c*c0*t17314*t17370*t17371*t17372*t17346*t17275*t17276*t17278*(6.4E1/9.0);
                double t17386 = 1.0/(t17344*t17344*t17344);
                double t17392 = c*c0*t17370*t17371*t17381*t17274*t17346*t17377*(4.0/3.0);
                double t17393 = c0*kab1*rab1*t17342*t17351*t17370*t17372*t17346*t17383*t17384*t17386*(1.6E1/4.5E1);
                double t17387 = t17380+t17382-t17392-t17393+t17385;
                double t17388 = t17374*6.0;
                double t17389 = c*c0*t17370*t17371*t17372*t17274*t17346*4.0;
                double t17390 = t17388+t17389+3.0;
                double t17391 = 1.0/t17390;
                double t17394 = c*c0*t17370*t17371*t17372*t17274*t17346*t17378*t17379*(2.0/3.0);
                double t17395 = t17394+1.0;
                double t17396 = t17374*4.0;
                double t17397 = c*c0*t17370*t17371*t17372*t17274*t17346*(8.0/3.0);
                double t17398 = t17396+t17397+6.0;
                double t17399 = 1.0/sqrt(t17373);
                double t17400 = 1.0/c0;
                double t17401 = pow(rho_b,1.0/3.0);
                double t17402 = c*c0*t17304*(2.0/9.0);
                double t17403 = t17350*t17344*t17275*t17347*t17276*t17339*t17358*t17278*(1.6E6/3.0);
                double t17404 = 1.0/pow(rho_b,1.0E1/3.0);
                double t17405 = 1.0/pow(t17346,3.0/2.0);
                double t17406 = a3*gamma_bb*t17350*t17404*t17405*t17344*t17273*t17283*t17358*(3.5E5/3.0);
                double t17407 = t17402+t17403+t17406-t17344*t17347*t17358*t17377*1.0E5-c*kab1*rab1*t17304*t17350*t17342*t17347*t17384*t17358*(8.0E4/3.0);
                double t17408 = 1.0/pow(rho_a,1.0/3.0);
                double t17409 = gamma_aa*gamma_aa;
                double t17410 = 1.0/pow(rho_a,1.6E1/3.0);
                double t17411 = c*t17408;
                double t17412 = pow(t17411,4.0/5.0);
                double t17425 = rab1*t17412;
                double t17413 = exp(-t17425);
                double t17414 = t17413-1.0;
                double t17426 = kab1*t17414;
                double t17415 = kab0-t17426;
                double t17416 = 1.0/pow(rho_a,7.0/3.0);
                double t17417 = a3*gamma_aa*t17416*t17273*t17283;
                double t17418 = t17417+1.0;
                double t17419 = t17410*t17409*t17275*t17278;
                double t17420 = 1.0/pow(rho_a,8.0/3.0);
                double t17421 = a1*gamma_aa*t17420*t17273;
                double t17422 = t17421+t17419+1.0;
                double t17423 = t17410*t17409*t17275*t17278*2.0;
                double t17424 = exp(t17423);
                double t17427 = 1.0/(t17415*t17415);
                double t17428 = 1.0/(t17422*t17422);
                double t17429 = c*c0*t17424*t17408*t17418*t17427*t17428*(2.0/3.0);
                double t17430;
                double t17431;
                if (t17429 > expei_cutoff) {
                    t17430 = 1.0;
                    t17431 = expei(t17429);
                } else {
                    t17430 = exp(t17429);
                    t17431 = Ei(-t17429);
                }
                double t17432 = sqrt(t17429);
                v_rho_b[Q] += scale * (-rho_a*(c0*t17368*(t17391*t17395*(t17387*t17399*2.0+c*c0*t17304*t17370*t17371*t17372*t17346*(8.0/9.0)+a3*c0*gamma_bb*t17307*t17370*t17371*t17273*t17372*(5.6E1/9.0)-c*c0*t17370*t17371*t17381*t17274*t17346*t17377*(1.6E1/3.0)+c*c0*t17314*t17370*t17371*t17372*t17346*t17275*t17276*t17278*(2.56E2/9.0)-c0*kab1*rab1*t17342*t17351*t17370*t17372*t17346*t17383*t17384*t17386*(6.4E1/4.5E1))+t17391*t17398*(t17380+t17382-t17392-t17393+t17385+c*c0*t17304*t17370*t17371*t17372*t17346*t17378*t17379*(2.0/9.0)+a3*c0*gamma_bb*t17307*t17370*t17371*t17273*t17372*t17378*t17379*(1.4E1/9.0)-c*c0*t17370*t17371*t17381*t17274*t17346*t17377*t17378*t17379*(4.0/3.0)+c*c0*t17370*t17371*t17372*t17274*t17346*t17378*t17387*t17379*(2.0/3.0)+c*c0*t17314*t17370*t17371*t17372*t17346*t17275*t17276*t17278*t17378*t17379*(6.4E1/9.0)-c0*kab1*rab1*t17342*t17351*t17370*t17372*t17346*t17383*t17384*t17386*t17378*t17379*(1.6E1/4.5E1))+t17378*t17387*t17379-1.0/(t17390*t17390)*t17395*t17398*(t17387*t17399*3.0+c*c0*t17304*t17370*t17371*t17372*t17346*(4.0/3.0)+a3*c0*gamma_bb*t17307*t17370*t17371*t17273*t17372*(2.8E1/3.0)-c*c0*t17370*t17371*t17381*t17274*t17346*t17377*8.0+c*c0*t17314*t17370*t17371*t17372*t17346*t17275*t17276*t17278*(1.28E2/3.0)-c0*kab1*rab1*t17342*t17351*t17370*t17372*t17346*t17383*t17384*t17386*(3.2E1/1.5E1))+(t17400*t17401*(t17350*t17350)*(t17344*t17344)*t17283*t17387*exp(-t17369)*(3.0/2.0))/t17346)*(1.0/2.0)-c0*t17407*dirac(t17361-t17366)*(t17378*t17379+t17391*t17395*t17398)*(1.0/2.0))+rho_b*(c0*t17357*heaviside(c*c0*t17274*(-2.0/3.0)+t17350*t17344*t17347*exp(-t17275*t17276*t17277*t17278)*1.0E5)*(t17321*t17329*(t17303*t17337*t17338*2.0-t17306*t17280*t17276*t17339*t17278*t17288*(6.4E1/3.0)+c*c0*t17304*t17290*t17282*t17288*t17299*(8.0/9.0)+a5*c0*gamma_bb*t17307*t17273*t17282*t17288*t17299*(5.6E1/9.0)+c*c0*t17312*t17313*t17290*t17282*t17274*t17288*(1.6E1/3.0)-c*c0*t17314*t17280*t17290*t17317*t17282*t17276*t17278*t17299*(2.56E2/9.0)+c*c0*t17314*t17280*t17290*t17282*t17276*t17278*t17288*t17299*(2.56E2/9.0))+t17321*t17332*(t17322+t17323+t17324+t17325-t17327+c*c0*t17304*t17315*t17316*t17290*t17282*t17288*t17299*(2.0/9.0)+a5*c0*gamma_bb*t17315*t17307*t17316*t17273*t17282*t17288*t17299*(1.4E1/9.0)+c*c0*t17312*t17313*t17315*t17316*t17290*t17282*t17274*t17288*(4.0/3.0)+c*c0*t17315*t17316*t17290*t17326*t17282*t17274*t17288*t17299*(2.0/3.0)-c*c0*t17314*t17315*t17280*t17316*t17290*t17317*t17282*t17276*t17278*t17299*(6.4E1/9.0)+c*c0*t17314*t17315*t17280*t17316*t17290*t17282*t17276*t17278*t17288*t17299*(6.4E1/9.0))+t17315*t17316*t17326-1.0/(t17320*t17320)*t17332*t17329*(t17303*t17337*t17338*3.0-t17306*t17280*t17276*t17339*t17278*t17288*3.2E1+c*c0*t17304*t17290*t17282*t17288*t17299*(4.0/3.0)+a5*c0*gamma_bb*t17307*t17273*t17282*t17288*t17299*(2.8E1/3.0)+c*c0*t17312*t17313*t17290*t17282*t17274*t17288*8.0-c*c0*t17314*t17280*t17290*t17317*t17282*t17276*t17278*t17299*(1.28E2/3.0)+c*c0*t17314*t17280*t17290*t17282*t17276*t17278*t17288*t17299*(1.28E2/3.0))+(t17400*t17401*t17326*t17283*(t17287*t17287)*(t17298*t17298)*exp(-t17281)*(3.0/2.0))/t17290)*(-1.0/2.0)+c0*t17365*t17357*t17368*(t17351*t17356*t17383*(2.0/3.0)-t17351*t17352*1.0/(t17355*t17355*t17355)*(c*k2*t17304*(1.0/3.0)+c*k1*t17304*t17308*(1.0/6.0))*2.0)*(1.0/2.0)+c0*t17407*t17365*t17357*dirac(t17361-t17350*t17344*t17347*t17358*1.0E5)*(1.0/2.0))+c0*heaviside(c*c0*t17408*(-2.0/3.0)+t17422*t17415*1.0/sqrt(t17418)*exp(-t17410*t17409*t17275*t17278)*1.0E5)*(t17430*t17431+((c*c0*t17430*t17431*t17424*t17408*t17418*t17427*t17428*(2.0/3.0)+1.0)*(t17432*4.0+c*c0*t17424*t17408*t17418*t17427*t17428*(8.0/3.0)+6.0))/(t17432*6.0+c*c0*t17424*t17408*t17418*t17427*t17428*4.0+3.0))*(1.0/2.0)+c0*t17365*t17357*t17368*(1.0/2.0));
            }
            
            // v_gamma_aa
            if (deriv >= 1) {
                double t17434 = tspi_m13*tspi_m13;
                double t17435 = 1.0/pow(rho_a,1.0/3.0);
                double t17436 = a2*a2;
                double t17437 = gamma_aa*gamma_aa;
                double t17438 = 1.0/pow(rho_a,1.6E1/3.0);
                double t17439 = t17434*t17434;
                double t17440 = t17436*t17437*t17438*t17439*2.0;
                double t17441 = exp(t17440);
                double t17442 = c*t17435;
                double t17443 = pow(t17442,4.0/5.0);
                double t17455 = rab1*t17443;
                double t17444 = exp(-t17455);
                double t17445 = t17444-1.0;
                double t17456 = kab1*t17445;
                double t17446 = kab0-t17456;
                double t17447 = 1.0/c;
                double t17448 = 1.0/pow(rho_a,7.0/3.0);
                double t17449 = a3*gamma_aa*t17434*t17447*t17448;
                double t17450 = t17449+1.0;
                double t17451 = t17436*t17437*t17438*t17439;
                double t17452 = 1.0/pow(rho_a,8.0/3.0);
                double t17453 = a1*gamma_aa*t17434*t17452;
                double t17454 = t17451+t17453+1.0;
                double t17457 = 1.0/(t17446*t17446);
                double t17458 = 1.0/(t17454*t17454);
                double t17459 = c*c0*t17441*t17450*t17435*t17457*t17458*(2.0/3.0);
                double t17460 = a1*t17434*t17452;
                double t17461 = gamma_aa*t17436*t17438*t17439*2.0;
                double t17462 = t17460+t17461;
                double t17463 = 1.0/(t17454*t17454*t17454);
                double t17464 = 1.0/pow(rho_a,1.7E1/3.0);
                double t17465;
                double t17466;
                if (t17459 > expei_cutoff) {
                    t17465 = 1.0;
                    t17466 = expei(t17459);
                } else {
                    t17465 = exp(t17459);
                    t17466 = Ei(-t17459);
                }
                double t17467 = a3*c0*t17441*t17434*t17452*t17457*t17458*(2.0/3.0);
                double t17468 = c*c0*gamma_aa*t17441*t17450*t17436*t17464*t17439*t17457*t17458*(8.0/3.0);
                double t17475 = c*c0*t17441*t17450*t17435*t17462*t17463*t17457*(4.0/3.0);
                double t17469 = -t17475+t17467+t17468;
                double t17470 = sqrt(t17459);
                double t17471 = t17470*6.0;
                double t17472 = c*c0*t17441*t17450*t17435*t17457*t17458*4.0;
                double t17473 = t17471+t17472+3.0;
                double t17474 = 1.0/t17473;
                double t17476 = c*c0*t17441*t17450*t17435*t17465*t17457*t17466*t17458*(2.0/3.0);
                double t17477 = t17476+1.0;
                double t17478 = t17470*4.0;
                double t17479 = c*c0*t17441*t17450*t17435*t17457*t17458*(8.0/3.0);
                double t17480 = t17478+t17479+6.0;
                double t17481 = 1.0/sqrt(t17459);
                double t17482 = 1.0/sqrt(t17450);
                double t17483 = exp(-t17451);
                double t17484 = c*c0*t17435*(2.0/3.0);
                double t17485 = a4*a4;
                double t17486 = sqrt(t17442);
                double t17487 = t17437*t17438*t17439*t17485*2.0;
                double t17488 = exp(t17487);
                double t17489 = t17437*t17438*t17439*t17485;
                double t17490 = t17489+1.0;
                double t17491 = 1.0/(t17490*t17490);
                double t17492 = a5*gamma_aa*t17434*t17447*t17448;
                double t17493 = t17492+1.0;
                double t17504 = raa2*t17486;
                double t17494 = exp(-t17504);
                double t17495 = t17494-1.0;
                double t17496 = kaa2*t17495;
                double t17497 = pow(t17442,2.0/5.0);
                double t17505 = raa1*t17497;
                double t17498 = exp(-t17505);
                double t17499 = t17498-1.0;
                double t17500 = kaa1*t17499;
                double t17501 = -kaa0+t17500+t17496;
                double t17502 = 1.0/(t17501*t17501);
                double t17503 = c*c0*t17502*t17435*t17491*t17493*t17488*(2.0/3.0);
                double t17506;
                double t17507;
                if (t17503 > expei_cutoff) {
                    t17506 = 1.0;
                    t17507 = expei(t17503);
                } else {
                    t17506 = exp(t17503);
                    t17507 = Ei(-t17503);
                }
                double t17508 = 1.0/t17490;
                double t17509 = c*c0*t17502*t17435*t17493*t17488*(2.0/3.0);
                double t17510 = sqrt(t17509);
                double t17511 = 1.0/(t17490*t17490*t17490);
                double t17512 = t17510*t17508*6.0;
                double t17513 = c*c0*t17502*t17435*t17491*t17493*t17488*4.0;
                double t17514 = t17512+t17513+3.0;
                double t17515 = 1.0/t17514;
                double t17516 = a5*c0*t17502*t17434*t17452*t17491*t17488*(2.0/3.0);
                double t17517 = c*c0*gamma_aa*t17502*t17464*t17491*t17439*t17493*t17485*t17488*(8.0/3.0);
                double t17530 = c*c0*gamma_aa*t17502*t17511*t17464*t17439*t17493*t17485*t17488*(8.0/3.0);
                double t17518 = -t17530+t17516+t17517;
                double t17519 = c*c0*t17502*t17506*t17435*t17507*t17491*t17493*t17488*(2.0/3.0);
                double t17520 = t17519+1.0;
                double t17521 = t17510*t17508*4.0;
                double t17522 = c*c0*t17502*t17435*t17491*t17493*t17488*(8.0/3.0);
                double t17523 = t17521+t17522+6.0;
                double t17524 = a5*c0*t17502*t17434*t17452*t17488*(2.0/3.0);
                double t17525 = c*c0*gamma_aa*t17502*t17464*t17439*t17493*t17485*t17488*(8.0/3.0);
                double t17526 = t17524+t17525;
                double t17527 = 1.0/sqrt(t17509);
                double t17528 = 1.0/c0;
                double t17529 = pow(rho_a,1.0/3.0);
                double t17531 = t17454*t17446*t17482*t17483*1.0E5;
                double t17532 = c*c;
                double t17533 = 1.0/pow(rho_a,2.0/3.0);
                double t17534 = k1*t17486;
                double t17535 = c*k2*t17435;
                double t17536 = t17534+t17535;
                double t17537 = 1.0/(t17536*t17536);
                double t17538 = exp(-t17532*t17533*t17537);
                double t17539 = 1.0/pow(t17450,3.0/2.0);
                double t17540 = a3*t17434*t17454*t17446*t17447*t17483*t17448*t17539*5.0E4;
                double t17541 = gamma_aa*t17436*t17454*t17446*t17482*t17438*t17483*t17439*2.0E5;
                double t17542 = t17540+t17541-t17462*t17446*t17482*t17483*1.0E5;
                v_gamma_aa[Q] += scale * (rho_b*(c0*heaviside(c*c0*t17435*(-2.0/3.0)+t17454*t17446*t17482*exp(-t17436*t17437*t17438*t17439)*1.0E5)*(t17465*t17466*t17469+t17480*t17474*(-t17475+t17467+t17468+a3*c0*t17441*t17434*t17452*t17465*t17457*t17466*t17458*(2.0/3.0)-c*c0*t17441*t17450*t17435*t17462*t17463*t17465*t17457*t17466*(4.0/3.0)+c*c0*t17441*t17450*t17435*t17465*t17457*t17466*t17458*t17469*(2.0/3.0)+c*c0*gamma_aa*t17441*t17450*t17436*t17464*t17465*t17439*t17457*t17466*t17458*(8.0/3.0))+t17474*t17477*(t17481*t17469*2.0+a3*c0*t17441*t17434*t17452*t17457*t17458*(8.0/3.0)-c*c0*t17441*t17450*t17435*t17462*t17463*t17457*(1.6E1/3.0)+c*c0*gamma_aa*t17441*t17450*t17436*t17464*t17439*t17457*t17458*(3.2E1/3.0))-t17480*1.0/(t17473*t17473)*t17477*(t17481*t17469*3.0+a3*c0*t17441*t17434*t17452*t17457*t17458*4.0-c*c0*t17441*t17450*t17435*t17462*t17463*t17457*8.0+c*c0*gamma_aa*t17441*t17450*t17436*t17464*t17439*t17457*t17458*1.6E1)+((t17454*t17454)*(t17446*t17446)*t17447*t17528*t17529*t17469*exp(-t17440)*(3.0/2.0))/t17450)*(1.0/2.0)-c0*t17542*dirac(t17484-t17454*t17446*t17482*t17483*1.0E5)*(t17465*t17466+t17480*t17474*t17477)*(1.0/2.0))+rho_a*(c0*t17538*heaviside(t17531-t17484)*(t17520*t17515*(t17508*t17526*t17527*2.0-gamma_aa*t17510*t17491*t17438*t17439*t17485*8.0+a5*c0*t17502*t17434*t17452*t17491*t17488*(8.0/3.0)-c*c0*gamma_aa*t17502*t17511*t17464*t17439*t17493*t17485*t17488*(3.2E1/3.0)+c*c0*gamma_aa*t17502*t17464*t17491*t17439*t17493*t17485*t17488*(3.2E1/3.0))+t17523*t17515*(t17516+t17517+a5*c0*t17502*t17434*t17452*t17506*t17507*t17491*t17488*(2.0/3.0)-c*c0*gamma_aa*t17502*t17511*t17464*t17439*t17493*t17485*t17488*(8.0/3.0)+c*c0*t17502*t17506*t17435*t17507*t17491*t17518*t17493*t17488*(2.0/3.0)-c*c0*gamma_aa*t17502*t17511*t17506*t17507*t17464*t17439*t17493*t17485*t17488*(8.0/3.0)+c*c0*gamma_aa*t17502*t17506*t17507*t17464*t17491*t17439*t17493*t17485*t17488*(8.0/3.0))+t17506*t17507*t17518-t17520*1.0/(t17514*t17514)*t17523*(t17508*t17526*t17527*3.0-gamma_aa*t17510*t17491*t17438*t17439*t17485*1.2E1+a5*c0*t17502*t17434*t17452*t17491*t17488*4.0-c*c0*gamma_aa*t17502*t17511*t17464*t17439*t17493*t17485*t17488*1.6E1+c*c0*gamma_aa*t17502*t17464*t17491*t17439*t17493*t17485*t17488*1.6E1)+((t17501*t17501)*(t17490*t17490)*t17518*t17447*t17528*t17529*exp(-t17487)*(3.0/2.0))/t17493)*(1.0/2.0)-c0*t17542*t17538*dirac(-t17531+t17484)*(t17506*t17507+t17520*t17523*t17515)*(1.0/2.0)));
            }
            
            // v_gamma_bb
            if (deriv >= 1) {
                double t17545 = tspi_m13*tspi_m13;
                double t17546 = 1.0/pow(rho_b,1.0/3.0);
                double t17547 = a2*a2;
                double t17548 = gamma_bb*gamma_bb;
                double t17549 = 1.0/pow(rho_b,1.6E1/3.0);
                double t17550 = t17545*t17545;
                double t17551 = t17550*t17547*t17548*t17549*2.0;
                double t17552 = exp(t17551);
                double t17553 = c*t17546;
                double t17554 = pow(t17553,4.0/5.0);
                double t17566 = rab1*t17554;
                double t17555 = exp(-t17566);
                double t17556 = t17555-1.0;
                double t17567 = kab1*t17556;
                double t17557 = kab0-t17567;
                double t17558 = 1.0/c;
                double t17559 = 1.0/pow(rho_b,7.0/3.0);
                double t17560 = a3*gamma_bb*t17545*t17558*t17559;
                double t17561 = t17560+1.0;
                double t17562 = t17550*t17547*t17548*t17549;
                double t17563 = 1.0/pow(rho_b,8.0/3.0);
                double t17564 = a1*gamma_bb*t17545*t17563;
                double t17565 = t17562+t17564+1.0;
                double t17568 = 1.0/(t17557*t17557);
                double t17569 = 1.0/(t17565*t17565);
                double t17570 = c*c0*t17552*t17561*t17546*t17568*t17569*(2.0/3.0);
                double t17571 = a1*t17545*t17563;
                double t17572 = gamma_bb*t17550*t17547*t17549*2.0;
                double t17573 = t17571+t17572;
                double t17574 = 1.0/(t17565*t17565*t17565);
                double t17575 = 1.0/pow(rho_b,1.7E1/3.0);
                double t17576;
                double t17577;
                if (t17570 > expei_cutoff) {
                    t17576 = 1.0;
                    t17577 = expei(t17570);
                } else {
                    t17576 = exp(t17570);
                    t17577 = Ei(-t17570);
                }
                double t17578 = a3*c0*t17552*t17545*t17563*t17568*t17569*(2.0/3.0);
                double t17579 = c*c0*gamma_bb*t17550*t17552*t17561*t17547*t17575*t17568*t17569*(8.0/3.0);
                double t17586 = c*c0*t17552*t17561*t17546*t17573*t17574*t17568*(4.0/3.0);
                double t17580 = -t17586+t17578+t17579;
                double t17581 = sqrt(t17570);
                double t17582 = t17581*6.0;
                double t17583 = c*c0*t17552*t17561*t17546*t17568*t17569*4.0;
                double t17584 = t17582+t17583+3.0;
                double t17585 = 1.0/t17584;
                double t17587 = c*c0*t17552*t17561*t17546*t17576*t17568*t17577*t17569*(2.0/3.0);
                double t17588 = t17587+1.0;
                double t17589 = t17581*4.0;
                double t17590 = c*c0*t17552*t17561*t17546*t17568*t17569*(8.0/3.0);
                double t17591 = t17590+t17589+6.0;
                double t17592 = 1.0/sqrt(t17570);
                double t17593 = 1.0/sqrt(t17561);
                double t17594 = exp(-t17562);
                double t17595 = c*c0*t17546*(2.0/3.0);
                double t17596 = a4*a4;
                double t17597 = sqrt(t17553);
                double t17598 = t17550*t17548*t17549*t17596*2.0;
                double t17599 = exp(t17598);
                double t17600 = t17550*t17548*t17549*t17596;
                double t17601 = t17600+1.0;
                double t17602 = 1.0/(t17601*t17601);
                double t17603 = a5*gamma_bb*t17545*t17558*t17559;
                double t17604 = t17603+1.0;
                double t17615 = raa2*t17597;
                double t17605 = exp(-t17615);
                double t17606 = t17605-1.0;
                double t17607 = kaa2*t17606;
                double t17608 = pow(t17553,2.0/5.0);
                double t17616 = raa1*t17608;
                double t17609 = exp(-t17616);
                double t17610 = t17609-1.0;
                double t17611 = kaa1*t17610;
                double t17612 = -kaa0+t17611+t17607;
                double t17613 = 1.0/(t17612*t17612);
                double t17614 = c*c0*t17602*t17604*t17613*t17546*t17599*(2.0/3.0);
                double t17617;
                double t17618;
                if (t17614 > expei_cutoff) {
                    t17617 = 1.0;
                    t17618 = expei(t17614);
                } else {
                    t17617 = exp(t17614);
                    t17618 = Ei(-t17614);
                }
                double t17619 = 1.0/t17601;
                double t17620 = c*c0*t17604*t17613*t17546*t17599*(2.0/3.0);
                double t17621 = sqrt(t17620);
                double t17622 = 1.0/(t17601*t17601*t17601);
                double t17623 = t17621*t17619*6.0;
                double t17624 = c*c0*t17602*t17604*t17613*t17546*t17599*4.0;
                double t17625 = t17623+t17624+3.0;
                double t17626 = 1.0/t17625;
                double t17627 = a5*c0*t17602*t17613*t17545*t17563*t17599*(2.0/3.0);
                double t17628 = c*c0*gamma_bb*t17602*t17550*t17604*t17613*t17575*t17596*t17599*(8.0/3.0);
                double t17641 = c*c0*gamma_bb*t17550*t17604*t17613*t17622*t17575*t17596*t17599*(8.0/3.0);
                double t17629 = -t17641+t17627+t17628;
                double t17630 = c*c0*t17602*t17604*t17613*t17617*t17546*t17618*t17599*(2.0/3.0);
                double t17631 = t17630+1.0;
                double t17632 = t17621*t17619*4.0;
                double t17633 = c*c0*t17602*t17604*t17613*t17546*t17599*(8.0/3.0);
                double t17634 = t17632+t17633+6.0;
                double t17635 = a5*c0*t17613*t17545*t17563*t17599*(2.0/3.0);
                double t17636 = c*c0*gamma_bb*t17550*t17604*t17613*t17575*t17596*t17599*(8.0/3.0);
                double t17637 = t17635+t17636;
                double t17638 = 1.0/sqrt(t17620);
                double t17639 = 1.0/c0;
                double t17640 = pow(rho_b,1.0/3.0);
                double t17642 = t17565*t17557*t17593*t17594*1.0E5;
                double t17643 = c*c;
                double t17644 = 1.0/pow(rho_b,2.0/3.0);
                double t17645 = k1*t17597;
                double t17646 = c*k2*t17546;
                double t17647 = t17645+t17646;
                double t17648 = 1.0/(t17647*t17647);
                double t17649 = exp(-t17643*t17644*t17648);
                double t17650 = 1.0/pow(t17561,3.0/2.0);
                double t17651 = a3*t17650*t17545*t17565*t17557*t17558*t17594*t17559*5.0E4;
                double t17652 = gamma_bb*t17550*t17547*t17565*t17557*t17593*t17549*t17594*2.0E5;
                double t17653 = t17651+t17652-t17573*t17557*t17593*t17594*1.0E5;
                v_gamma_bb[Q] += scale * (rho_a*(c0*heaviside(c*c0*t17546*(-2.0/3.0)+t17565*t17557*t17593*exp(-t17550*t17547*t17548*t17549)*1.0E5)*(t17580*t17576*t17577+t17591*t17585*(-t17586+t17578+t17579+a3*c0*t17552*t17545*t17563*t17576*t17568*t17577*t17569*(2.0/3.0)-c*c0*t17552*t17561*t17546*t17573*t17574*t17576*t17568*t17577*(4.0/3.0)+c*c0*t17552*t17561*t17580*t17546*t17576*t17568*t17577*t17569*(2.0/3.0)+c*c0*gamma_bb*t17550*t17552*t17561*t17547*t17575*t17576*t17568*t17577*t17569*(8.0/3.0))+t17585*t17588*(t17580*t17592*2.0+a3*c0*t17552*t17545*t17563*t17568*t17569*(8.0/3.0)-c*c0*t17552*t17561*t17546*t17573*t17574*t17568*(1.6E1/3.0)+c*c0*gamma_bb*t17550*t17552*t17561*t17547*t17575*t17568*t17569*(3.2E1/3.0))-t17591*1.0/(t17584*t17584)*t17588*(t17580*t17592*3.0+a3*c0*t17552*t17545*t17563*t17568*t17569*4.0-c*c0*t17552*t17561*t17546*t17573*t17574*t17568*8.0+c*c0*gamma_bb*t17550*t17552*t17561*t17547*t17575*t17568*t17569*1.6E1)+(t17640*t17580*(t17565*t17565)*(t17557*t17557)*t17558*t17639*exp(-t17551)*(3.0/2.0))/t17561)*(1.0/2.0)-c0*t17653*dirac(t17595-t17565*t17557*t17593*t17594*1.0E5)*(t17576*t17577+t17591*t17585*t17588)*(1.0/2.0))+rho_b*(c0*t17649*heaviside(t17642-t17595)*(t17631*t17626*(t17619*t17637*t17638*2.0-gamma_bb*t17602*t17621*t17550*t17549*t17596*8.0+a5*c0*t17602*t17613*t17545*t17563*t17599*(8.0/3.0)+c*c0*gamma_bb*t17602*t17550*t17604*t17613*t17575*t17596*t17599*(3.2E1/3.0)-c*c0*gamma_bb*t17550*t17604*t17613*t17622*t17575*t17596*t17599*(3.2E1/3.0))+t17634*t17626*(t17627+t17628+a5*c0*t17602*t17613*t17545*t17563*t17617*t17618*t17599*(2.0/3.0)-c*c0*gamma_bb*t17550*t17604*t17613*t17622*t17575*t17596*t17599*(8.0/3.0)+c*c0*t17602*t17604*t17613*t17617*t17546*t17618*t17629*t17599*(2.0/3.0)+c*c0*gamma_bb*t17602*t17550*t17604*t17613*t17617*t17618*t17575*t17596*t17599*(8.0/3.0)-c*c0*gamma_bb*t17550*t17604*t17613*t17622*t17617*t17618*t17575*t17596*t17599*(8.0/3.0))+t17617*t17618*t17629-t17631*1.0/(t17625*t17625)*t17634*(t17619*t17637*t17638*3.0-gamma_bb*t17602*t17621*t17550*t17549*t17596*1.2E1+a5*c0*t17602*t17613*t17545*t17563*t17599*4.0+c*c0*gamma_bb*t17602*t17550*t17604*t17613*t17575*t17596*t17599*1.6E1-c*c0*gamma_bb*t17550*t17604*t17613*t17622*t17575*t17596*t17599*1.6E1)+((t17601*t17601)*(t17612*t17612)*t17640*t17629*t17558*t17639*exp(-t17598)*(3.0/2.0))/t17604)*(1.0/2.0)-c0*t17653*t17649*dirac(-t17642+t17595)*(t17617*t17618+t17631*t17634*t17626)*(1.0/2.0)));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t17657 = tspi_m13*tspi_m13;
                double t17658 = 1.0/pow(rho_a,1.0/3.0);
                double t17659 = a2*a2;
                double t17660 = gamma_aa*gamma_aa;
                double t17661 = 1.0/pow(rho_a,1.6E1/3.0);
                double t17662 = t17657*t17657;
                double t17663 = c*t17658;
                double t17664 = a4*a4;
                double t17665 = 1.0/c;
                double t17666 = 1.0/pow(rho_a,7.0/3.0);
                double t17667 = sqrt(t17663);
                double t17668 = t17660*t17661*t17662*t17664*2.0;
                double t17669 = exp(t17668);
                double t17670 = t17660*t17661*t17662*t17664;
                double t17671 = t17670+1.0;
                double t17672 = a5*gamma_aa*t17665*t17657*t17666;
                double t17673 = t17672+1.0;
                double t17684 = raa2*t17667;
                double t17674 = exp(-t17684);
                double t17675 = t17674-1.0;
                double t17676 = kaa2*t17675;
                double t17677 = pow(t17663,2.0/5.0);
                double t17685 = raa1*t17677;
                double t17678 = exp(-t17685);
                double t17679 = t17678-1.0;
                double t17680 = kaa1*t17679;
                double t17681 = -kaa0+t17680+t17676;
                double t17682 = 1.0/(t17681*t17681);
                double t17683 = 1.0/t17671;
                double t17686 = 1.0/pow(rho_a,4.0/3.0);
                double t17687 = c*c0*t17673*t17682*t17658*t17669*(2.0/3.0);
                double t17688 = 1.0/(t17671*t17671);
                double t17689 = sqrt(t17687);
                double t17690 = 1.0/pow(rho_a,1.1E1/3.0);
                double t17691 = 1.0/sqrt(t17663);
                double t17692 = c*kaa2*raa2*t17691*t17674*t17686*(1.0/6.0);
                double t17693 = 1.0/pow(t17663,3.0/5.0);
                double t17694 = c*kaa1*raa1*t17693*t17686*t17678*(2.0/1.5E1);
                double t17695 = t17692+t17694;
                double t17696 = 1.0/(t17681*t17681*t17681);
                double t17697 = 1.0/pow(rho_a,2.0E1/3.0);
                double t17698 = c*c0*t17673*t17682*t17658*t17669*t17688*(2.0/3.0);
                double t17699;
                double t17700;
                if (t17698 > expei_cutoff) {
                    t17699 = 1.0;
                    t17700 = expei(t17698);
                } else {
                    t17699 = exp(t17698);
                    t17700 = Ei(-t17698);
                }
                double t17701 = c*c0*t17673*t17682*t17686*t17669*t17688*(2.0/9.0);
                double t17702 = a5*c0*gamma_aa*t17690*t17682*t17657*t17669*t17688*(1.4E1/9.0);
                double t17703 = c*c0*t17673*t17658*t17695*t17669*t17696*t17688*(4.0/3.0);
                double t17704 = 1.0/(t17671*t17671*t17671);
                double t17705 = c*c0*t17660*t17662*t17664*t17673*t17682*t17669*t17688*t17697*(6.4E1/9.0);
                double t17706 = t17683*t17689*6.0;
                double t17707 = c*c0*t17673*t17682*t17658*t17669*t17688*4.0;
                double t17708 = t17706+t17707+3.0;
                double t17709 = 1.0/t17708;
                double t17710 = 1.0/sqrt(t17687);
                double t17711 = c*c;
                double t17712 = 1.0/pow(rho_a,8.0/3.0);
                double t17713 = c*c0*t17673*t17682*t17686*t17669*(2.0/9.0);
                double t17714 = a5*c0*gamma_aa*t17690*t17682*t17657*t17669*(1.4E1/9.0);
                double t17715 = c*c0*t17673*t17658*t17695*t17669*t17696*(4.0/3.0);
                double t17716 = c*c0*t17660*t17662*t17664*t17673*t17682*t17669*t17697*(6.4E1/9.0);
                double t17717 = t17713+t17714+t17715+t17716;
                double t17718 = t17664*t17664;
                double t17719 = t17660*t17660;
                double t17720 = t17662*t17662;
                double t17721 = 1.0/pow(rho_a,1.9E1/3.0);
                double t17722 = raa2*raa2;
                double t17723 = c*kaa2*t17722*t17674*t17666*(1.0/3.6E1);
                double t17724 = raa1*raa1;
                double t17725 = 1.0/pow(t17663,6.0/5.0);
                double t17726 = kaa1*t17711*t17712*t17724*t17725*t17678*(4.0/2.25E2);
                double t17727 = 1.0/pow(t17663,3.0/2.0);
                double t17728 = kaa2*raa2*t17711*t17712*t17727*t17674*(1.0/3.6E1);
                double t17729 = 1.0/pow(t17663,8.0/5.0);
                double t17730 = kaa1*raa1*t17711*t17712*t17729*t17678*(2.0/7.5E1);
                double t17740 = c*kaa2*raa2*t17691*t17674*t17666*(2.0/9.0);
                double t17741 = c*kaa1*raa1*t17666*t17693*t17678*(8.0/4.5E1);
                double t17731 = t17730-t17740+t17723-t17741+t17726+t17728;
                double t17732 = t17695*t17695;
                double t17733 = 1.0/(t17681*t17681*t17681*t17681);
                double t17734 = 1.0/pow(rho_a,1.4E1/3.0);
                double t17735 = 1.0/pow(rho_a,1.0E1);
                double t17736 = 1.0/pow(rho_a,1.3E1);
                double t17737 = 1.0/pow(rho_a,2.3E1/3.0);
                double t17739 = c*c0*t17704*t17660*t17662*t17664*t17673*t17682*t17669*t17697*(6.4E1/9.0);
                double t17738 = t17701+t17702+t17703+t17705-t17739;
                double t17742 = t17738*t17738;
                double t17743 = 1.0/(t17671*t17671*t17671*t17671);
                double t17744 = c*c0*t17673*t17682*t17666*t17669*t17688*(8.0/2.7E1);
                double t17745 = c*c0*t17732*t17733*t17673*t17658*t17669*t17688*4.0;
                double t17746 = a5*c0*gamma_aa*t17734*t17682*t17657*t17669*t17688*(5.6E1/9.0);
                double t17747 = c*c0*t17673*t17686*t17695*t17669*t17696*t17688*(8.0/9.0);
                double t17748 = a5*c0*gamma_aa*t17690*t17657*t17695*t17669*t17696*t17688*(5.6E1/9.0);
                double t17749 = a5*c0*gamma_aa*t17660*t17662*t17735*t17664*t17682*t17657*t17669*t17688*(8.96E2/2.7E1);
                double t17750 = c*c0*t17720*t17673*t17682*t17718*t17736*t17719*t17669*t17688*(2.048E3/2.7E1);
                double t17751 = c*c0*t17720*t17743*t17673*t17682*t17718*t17736*t17719*t17669*(1.024E3/9.0);
                double t17752 = c*c0*t17660*t17662*t17664*t17673*t17682*t17737*t17669*t17688*(4.48E2/9.0);
                double t17753 = c*c0*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17688*t17697*(2.56E2/9.0);
                double t17801 = c*c0*t17731*t17673*t17658*t17669*t17696*t17688*(4.0/3.0);
                double t17802 = a5*c0*gamma_aa*t17704*t17660*t17662*t17735*t17664*t17682*t17657*t17669*(8.96E2/2.7E1);
                double t17803 = c*c0*t17720*t17704*t17673*t17682*t17718*t17736*t17719*t17669*(4.096E3/2.7E1);
                double t17804 = c*c0*t17704*t17660*t17662*t17664*t17673*t17682*t17737*t17669*(4.48E2/9.0);
                double t17805 = c*c0*t17704*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17697*(2.56E2/9.0);
                double t17754 = -t17801-t17802-t17803+t17750-t17804+t17751-t17805+t17752+t17744+t17753+t17745+t17746+t17747+t17748+t17749;
                double t17755 = c*c0*t17700*t17673*t17682*t17658*t17669*t17688*t17699*(2.0/3.0);
                double t17756 = t17755+1.0;
                double t17757 = t17710*t17717*t17683*2.0;
                double t17758 = c*c0*t17673*t17682*t17686*t17669*t17688*(8.0/9.0);
                double t17759 = a5*c0*gamma_aa*t17690*t17682*t17657*t17669*t17688*(5.6E1/9.0);
                double t17760 = c*c0*t17673*t17658*t17695*t17669*t17696*t17688*(1.6E1/3.0);
                double t17761 = c*c0*t17660*t17662*t17664*t17673*t17682*t17669*t17688*t17697*(2.56E2/9.0);
                double t17823 = t17721*t17660*t17662*t17664*t17688*t17689*(6.4E1/3.0);
                double t17824 = c*c0*t17704*t17660*t17662*t17664*t17673*t17682*t17669*t17697*(2.56E2/9.0);
                double t17762 = t17760-t17823+t17761-t17824+t17757+t17758+t17759;
                double t17763 = 1.0/(t17708*t17708);
                double t17764 = t17683*t17689*4.0;
                double t17765 = c*c0*t17673*t17682*t17658*t17669*t17688*(8.0/3.0);
                double t17766 = t17764+t17765+6.0;
                double t17767 = t17710*t17717*t17683*3.0;
                double t17768 = c*c0*t17673*t17682*t17686*t17669*t17688*(4.0/3.0);
                double t17769 = a5*c0*gamma_aa*t17690*t17682*t17657*t17669*t17688*(2.8E1/3.0);
                double t17770 = c*c0*t17673*t17658*t17695*t17669*t17696*t17688*8.0;
                double t17771 = c*c0*t17660*t17662*t17664*t17673*t17682*t17669*t17688*t17697*(1.28E2/3.0);
                double t17792 = t17721*t17660*t17662*t17664*t17688*t17689*3.2E1;
                double t17793 = c*c0*t17704*t17660*t17662*t17664*t17673*t17682*t17669*t17697*(1.28E2/3.0);
                double t17772 = t17770+t17771-t17792-t17793+t17767+t17768+t17769;
                double t17773 = c*c0*t17700*t17673*t17682*t17686*t17669*t17688*t17699*(2.0/9.0);
                double t17774 = c*c0*t17700*t17673*t17658*t17695*t17669*t17696*t17688*t17699*(4.0/3.0);
                double t17775 = c*c0*t17700*t17673*t17682*t17738*t17658*t17669*t17688*t17699*(2.0/3.0);
                double t17776 = a5*c0*gamma_aa*t17700*t17690*t17682*t17657*t17669*t17688*t17699*(1.4E1/9.0);
                double t17777 = c*c0*t17700*t17660*t17662*t17664*t17673*t17682*t17669*t17688*t17697*t17699*(6.4E1/9.0);
                double t17778 = c*c0*t17673*t17682*t17666*t17669*(8.0/2.7E1);
                double t17779 = a5*c0*gamma_aa*t17734*t17682*t17657*t17669*(5.6E1/9.0);
                double t17780 = c*c0*t17673*t17686*t17695*t17669*t17696*(8.0/9.0);
                double t17781 = c*c0*t17732*t17733*t17673*t17658*t17669*4.0;
                double t17782 = a5*c0*gamma_aa*t17660*t17662*t17735*t17664*t17682*t17657*t17669*(8.96E2/2.7E1);
                double t17783 = a5*c0*gamma_aa*t17690*t17657*t17695*t17669*t17696*(5.6E1/9.0);
                double t17784 = c*c0*t17720*t17673*t17682*t17718*t17736*t17719*t17669*(2.048E3/2.7E1);
                double t17785 = c*c0*t17660*t17662*t17664*t17673*t17682*t17737*t17669*(4.48E2/9.0);
                double t17786 = c*c0*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17697*(2.56E2/9.0);
                double t17787 = t17780+t17781+t17782+t17783+t17784+t17785+t17786+t17778+t17779-c*c0*t17731*t17673*t17658*t17669*t17696*(4.0/3.0);
                double t17788 = t17717*t17717;
                double t17789 = 1.0/pow(t17687,3.0/2.0);
                double t17790 = 1.0/pow(rho_a,2.2E1/3.0);
                double t17791 = 1.0/pow(rho_a,3.8E1/3.0);
                double t17794 = 1.0/pow(rho_a,2.0/3.0);
                double t17795 = 1.0/t17673;
                double t17796 = 1.0/c0;
                double t17797 = exp(-t17668);
                double t17798 = t17671*t17671;
                double t17799 = t17681*t17681;
                double t17800 = pow(rho_a,1.0/3.0);
                double t17806 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a*rho_a);
                double t17825 = c*c0*t17700*t17704*t17660*t17662*t17664*t17673*t17682*t17669*t17697*t17699*(6.4E1/9.0);
                double t17807 = t17701+t17702+t17703+t17705-t17825+t17773+t17774-t17739+t17775+t17776+t17777;
                double t17808 = t17660*t17661*t17662*t17659;
                double t17809 = pow(t17663,4.0/5.0);
                double t17827 = rab1*t17809;
                double t17810 = exp(-t17827);
                double t17811 = t17810-1.0;
                double t17828 = kab1*t17811;
                double t17812 = kab0-t17828;
                double t17813 = a3*gamma_aa*t17665*t17657*t17666;
                double t17814 = t17813+1.0;
                double t17815 = 1.0/sqrt(t17814);
                double t17816 = a1*gamma_aa*t17712*t17657;
                double t17817 = t17816+t17808+1.0;
                double t17818 = k1*t17667;
                double t17819 = c*k2*t17658;
                double t17820 = t17818+t17819;
                double t17821 = 1.0/(t17820*t17820);
                double t17830 = t17711*t17821*t17794;
                double t17822 = exp(-t17830);
                double t17826 = exp(-t17808);
                double t17829 = t17812*t17815*t17817*t17826*1.0E5;
                double t17831 = t17762*t17709*t17756;
                double t17832 = t17700*t17738*t17699;
                double t17833 = t17807*t17709*t17766;
                double t17834 = t17800*t17665*t17738*t17795*t17796*t17797*t17798*t17799*(3.0/2.0);
                double t17972 = t17763*t17772*t17756*t17766;
                double t17835 = t17831+t17832+t17833+t17834-t17972;
                double t17836 = 1.0/pow(rho_a,1.0E1/3.0);
                double t17837 = c*c0*t17658*(2.0/3.0);
                double t17838 = 1.0/(t17820*t17820*t17820);
                double t17839 = c*k2*t17686*(1.0/3.0);
                double t17840 = c*k1*t17691*t17686*(1.0/6.0);
                double t17841 = t17840+t17839;
                double t17842 = 1.0/pow(rho_a,5.0/3.0);
                double t17843 = -t17837+t17829;
                double t17844 = heaviside(t17843);
                double t17845 = t17700*t17699;
                double t17846 = t17709*t17756*t17766;
                double t17847 = t17845+t17846;
                double t17848 = t17711*t17821*t17842*(2.0/3.0);
                double t17862 = t17711*t17841*t17838*t17794*2.0;
                double t17849 = -t17862+t17848;
                double t17850 = t17837-t17829;
                double t17851 = c*c0*t17686*(2.0/9.0);
                double t17852 = t17721*t17660*t17662*t17659*(1.6E1/3.0);
                double t17853 = a1*gamma_aa*t17690*t17657*(8.0/3.0);
                double t17854 = t17852+t17853;
                double t17855 = t17721*t17812*t17660*t17662*t17815*t17817*t17826*t17659*(1.6E6/3.0);
                double t17856 = 1.0/pow(t17663,1.0/5.0);
                double t17857 = 1.0/pow(t17814,3.0/2.0);
                double t17858 = a3*gamma_aa*t17812*t17817*t17826*t17665*t17836*t17657*t17857*(3.5E5/3.0);
                double t17863 = t17812*t17815*t17826*t17854*1.0E5;
                double t17864 = c*kab1*rab1*t17810*t17815*t17817*t17826*t17856*t17686*(8.0E4/3.0);
                double t17859 = t17851-t17863+t17855-t17864+t17858;
                double t17860 = dirac(t17850);
                double t17861 = 1.0/(c*c);
                double t17865 = t17660*t17661*t17662*t17659*2.0;
                double t17866 = exp(t17865);
                double t17867 = 1.0/(t17812*t17812);
                double t17868 = 1.0/(t17817*t17817);
                double t17869 = c*c0*t17814*t17658*t17866*t17867*t17868*(2.0/3.0);
                double t17870 = sqrt(t17869);
                double t17871 = 1.0/rho_a;
                double t17872 = c*c0*t17814*t17686*t17866*t17867*t17868*(2.0/9.0);
                double t17873 = 1.0/(t17817*t17817*t17817);
                double t17874 = a3*c0*gamma_aa*t17690*t17657*t17866*t17867*t17868*(1.4E1/9.0);
                double t17875 = c*c0*t17660*t17814*t17662*t17659*t17866*t17867*t17697*t17868*(6.4E1/9.0);
                double t17876 = 1.0/(t17812*t17812*t17812);
                double t17878 = c*c0*t17814*t17854*t17873*t17658*t17866*t17867*(4.0/3.0);
                double t17879 = c0*kab1*rab1*t17711*t17810*t17814*t17842*t17856*t17866*t17876*t17868*(1.6E1/4.5E1);
                double t17877 = t17872+t17874+t17875-t17878-t17879;
                double t17880 = t17660*t17662*t17790*t17659*(3.04E2/9.0);
                double t17881 = a1*gamma_aa*t17734*t17657*(8.8E1/9.0);
                double t17882 = t17880+t17881;
                double t17883;
                double t17884;
                if (t17869 > expei_cutoff) {
                    t17883 = 1.0;
                    t17884 = expei(t17869);
                } else {
                    t17883 = exp(t17869);
                    t17884 = Ei(-t17869);
                }
                double t17885 = c*c0*t17814*t17666*t17866*t17867*t17868*(8.0/2.7E1);
                double t17886 = t17854*t17854;
                double t17887 = 1.0/(t17817*t17817*t17817*t17817);
                double t17888 = c*c0*t17814*t17658*t17866*t17867*t17886*t17887*4.0;
                double t17889 = a3*c0*gamma_aa*t17734*t17657*t17866*t17867*t17868*(5.6E1/9.0);
                double t17890 = a3*c0*gamma_aa*t17660*t17662*t17735*t17657*t17659*t17866*t17867*t17868*(8.96E2/2.7E1);
                double t17891 = t17659*t17659;
                double t17892 = rab1*rab1;
                double t17893 = 1.0/(rho_a*rho_a*rho_a);
                double t17894 = 1.0/pow(t17663,2.0/5.0);
                double t17895 = t17877*t17877;
                double t17896 = c*c0*t17720*t17814*t17736*t17719*t17891*t17866*t17867*t17868*(2.048E3/2.7E1);
                double t17897 = c*c0*t17660*t17814*t17662*t17737*t17659*t17866*t17867*t17868*(4.48E2/9.0);
                double t17898 = c*c0*kab1*t17711*t17810*t17814*t17892*t17866*t17893*t17876*t17894*t17868*(6.4E1/6.75E2);
                double t17899 = kab1*kab1;
                double t17906 = rab1*t17809*2.0;
                double t17900 = exp(-t17906);
                double t17901 = 1.0/(t17812*t17812*t17812*t17812);
                double t17902 = c*c0*t17711*t17900*t17901*t17814*t17892*t17866*t17893*t17894*t17868*t17899*(6.4E1/2.25E2);
                double t17903 = c*c0*kab1*rab1*t17711*t17810*t17814*t17725*t17866*t17893*t17876*t17868*(1.6E1/6.75E2);
                double t17904 = c0*kab1*rab1*t17711*t17810*t17814*t17842*t17854*t17873*t17856*t17866*t17876*(6.4E1/4.5E1);
                double t17905 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a);
                double t17907 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a);
                double t17908 = t17870*6.0;
                double t17909 = c*c0*t17814*t17658*t17866*t17867*t17868*4.0;
                double t17910 = t17908+t17909+3.0;
                double t17911 = 1.0/t17910;
                double t17913 = c*c0*t17814*t17854*t17873*t17686*t17866*t17867*(8.0/9.0);
                double t17914 = c*c0*t17814*t17873*t17882*t17658*t17866*t17867*(4.0/3.0);
                double t17915 = a3*c0*gamma_aa*t17690*t17854*t17657*t17873*t17866*t17867*(5.6E1/9.0);
                double t17916 = c*c0*t17660*t17814*t17662*t17854*t17873*t17659*t17866*t17867*t17697*(2.56E2/9.0);
                double t17917 = c0*kab1*rab1*t17711*t17810*t17712*t17814*t17856*t17866*t17876*t17868*(3.2E1/4.5E1);
                double t17918 = a3*c*c0*gamma_aa*kab1*rab1*t17810*t17905*t17657*t17856*t17866*t17876*t17868*(2.24E2/1.35E2);
                double t17919 = c0*kab1*rab1*t17711*t17810*t17660*t17814*t17662*t17907*t17856*t17659*t17866*t17876*t17868*(1.024E3/1.35E2);
                double t17912 = t17902+t17903+t17904-t17913-t17914-t17915-t17916+t17890-t17917-t17918-t17919+t17885+t17896+t17888+t17897+t17889+t17898;
                double t17920 = 1.0/sqrt(t17869);
                double t17921 = t17870*4.0;
                double t17922 = c*c0*t17814*t17658*t17866*t17867*t17868*(8.0/3.0);
                double t17923 = t17921+t17922+6.0;
                double t17924 = c*c0*t17814*t17883*t17686*t17866*t17884*t17867*t17868*(2.0/9.0);
                double t17925 = a3*c0*gamma_aa*t17690*t17657*t17883*t17866*t17884*t17867*t17868*(1.4E1/9.0);
                double t17926 = c*c0*t17814*t17658*t17883*t17866*t17884*t17867*t17868*t17877*(2.0/3.0);
                double t17927 = c*c0*t17660*t17814*t17662*t17883*t17659*t17866*t17884*t17867*t17697*t17868*(6.4E1/9.0);
                double t17949 = c*c0*t17814*t17854*t17873*t17658*t17883*t17866*t17884*t17867*(4.0/3.0);
                double t17950 = c0*kab1*rab1*t17711*t17810*t17814*t17842*t17856*t17883*t17866*t17884*t17876*t17868*(1.6E1/4.5E1);
                double t17928 = -t17950+t17924+t17925+t17872+t17926+t17927+t17874+t17875-t17949-t17878-t17879;
                double t17929 = c*c0*t17814*t17658*t17883*t17866*t17884*t17867*t17868*(2.0/3.0);
                double t17930 = t17929+1.0;
                double t17931 = 1.0/(t17910*t17910);
                double t17932 = 1.0/pow(t17869,3.0/2.0);
                double t17933 = t17920*t17877*3.0;
                double t17934 = c*c0*t17814*t17686*t17866*t17867*t17868*(4.0/3.0);
                double t17935 = a3*c0*gamma_aa*t17690*t17657*t17866*t17867*t17868*(2.8E1/3.0);
                double t17936 = c*c0*t17660*t17814*t17662*t17659*t17866*t17867*t17697*t17868*(1.28E2/3.0);
                double t17938 = c*c0*t17814*t17854*t17873*t17658*t17866*t17867*8.0;
                double t17939 = c0*kab1*rab1*t17711*t17810*t17814*t17842*t17856*t17866*t17876*t17868*(3.2E1/1.5E1);
                double t17937 = t17933+t17934+t17935+t17936-t17938-t17939;
                double t17940 = t17920*t17877*2.0;
                double t17941 = c*c0*t17814*t17686*t17866*t17867*t17868*(8.0/9.0);
                double t17942 = a3*c0*gamma_aa*t17690*t17657*t17866*t17867*t17868*(5.6E1/9.0);
                double t17943 = c*c0*t17660*t17814*t17662*t17659*t17866*t17867*t17697*t17868*(2.56E2/9.0);
                double t17951 = c*c0*t17814*t17854*t17873*t17658*t17866*t17867*(1.6E1/3.0);
                double t17952 = c0*kab1*rab1*t17711*t17810*t17814*t17842*t17856*t17866*t17876*t17868*(6.4E1/4.5E1);
                double t17944 = t17940+t17941+t17942-t17951+t17943-t17952;
                double t17945 = 1.0/t17814;
                double t17946 = exp(-t17865);
                double t17947 = t17812*t17812;
                double t17948 = t17817*t17817;
                double t17953 = dirac(t17850,1.0);
                double t17954 = t17859*t17859;
                double t17955 = t17883*t17884;
                double t17956 = t17911*t17930*t17923;
                double t17957 = t17955+t17956;
                double t17958 = c*c0*t17666*(8.0/2.7E1);
                double t17959 = t17812*t17660*t17662*t17815*t17790*t17817*t17826*t17659*(3.04E7/9.0);
                double t17960 = t17721*t17812*t17660*t17662*t17815*t17826*t17854*t17659*(3.2E6/3.0);
                double t17961 = kab1*t17711*t17810*t17712*t17815*t17817*t17826*t17892*t17894*(6.4E4/9.0);
                double t17962 = a3*a3;
                double t17963 = 1.0/pow(t17814,5.0/2.0);
                double t17964 = 1.0/pow(rho_a,1.3E1/3.0);
                double t17965 = a3*gamma_aa*t17812*t17817*t17826*t17665*t17657*t17964*t17857*(3.5E6/9.0);
                double t17966 = a3*gamma_aa*t17812*t17826*t17665*t17836*t17854*t17657*t17857*(7.0E5/3.0);
                double t17967 = kab1*rab1*t17711*t17810*t17712*t17725*t17815*t17817*t17826*(1.6E4/9.0);
                double t17968 = 1.0/pow(rho_a,2.9E1/3.0);
                double t17969 = a3*gamma_aa*kab1*rab1*t17810*t17734*t17817*t17826*t17657*t17856*t17857*(5.6E5/9.0);
                double t17970 = c*kab1*rab1*t17810*t17660*t17662*t17815*t17817*t17826*t17737*t17856*t17659*(2.56E6/9.0);
                double t17971 = t17960+t17961+t17970+t17965+t17966+t17958+t17967+t17959+t17969-t17812*t17815*t17826*t17882*1.0E5-t17720*t17812*t17815*t17817*t17826*t17719*t17791*t17891*(2.56E7/9.0)-c*kab1*rab1*t17810*t17815*t17817*t17826*t17666*t17856*(3.2E5/9.0)-c*kab1*rab1*t17810*t17815*t17826*t17854*t17856*t17686*(1.6E5/3.0)-t17812*t17660*t17662*t17861*t17817*t17826*t17962*t17963*t17697*(1.225E6/3.0)-a3*gamma_aa*t17812*t17660*t17662*t17817*t17826*t17665*t17657*t17659*t17857*t17968*(1.12E7/9.0);
                v_rho_a_rho_a[Q] += scale * (-rho_b*(c0*t17844*(t17911*t17923*(t17902+t17903+t17904-t17913-t17914-t17915-t17916+t17890-t17917-t17918-t17919+t17885+t17895+t17896+t17888+t17897+t17889+t17898+t17871*t17877*(1.0/3.0)-(t17854*t17877*2.0)/t17817+t17721*t17660*t17662*t17659*t17877*(3.2E1/3.0)+a3*gamma_aa*t17665*t17836*t17657*t17945*t17877*(7.0/3.0)-(c*kab1*rab1*t17810*t17856*t17686*t17877*(8.0/1.5E1))/t17812+c*c0*t17814*t17666*t17883*t17866*t17884*t17867*t17868*(8.0/2.7E1)+a3*c0*gamma_aa*t17734*t17657*t17883*t17866*t17884*t17867*t17868*(5.6E1/9.0)+c*c0*t17912*t17814*t17658*t17883*t17866*t17884*t17867*t17868*(2.0/3.0)-c*c0*t17814*t17854*t17873*t17883*t17686*t17866*t17884*t17867*(8.0/9.0)-c*c0*t17814*t17873*t17882*t17658*t17883*t17866*t17884*t17867*(4.0/3.0)+c*c0*t17814*t17658*t17883*t17866*t17884*t17867*t17868*t17895*(2.0/3.0)+c*c0*t17814*t17658*t17883*t17866*t17884*t17867*t17886*t17887*4.0+c*c0*t17814*t17883*t17686*t17866*t17884*t17867*t17868*t17877*(4.0/9.0)-a3*c0*gamma_aa*t17690*t17854*t17657*t17873*t17883*t17866*t17884*t17867*(5.6E1/9.0)+a3*c0*gamma_aa*t17690*t17657*t17883*t17866*t17884*t17867*t17868*t17877*(2.8E1/9.0)-c*c0*t17814*t17854*t17873*t17658*t17883*t17866*t17884*t17867*t17877*(8.0/3.0)+c*c0*t17720*t17814*t17736*t17719*t17891*t17883*t17866*t17884*t17867*t17868*(2.048E3/2.7E1)+c*c0*t17660*t17814*t17662*t17737*t17883*t17659*t17866*t17884*t17867*t17868*(4.48E2/9.0)+a3*c0*gamma_aa*t17660*t17662*t17735*t17657*t17883*t17659*t17866*t17884*t17867*t17868*(8.96E2/2.7E1)-c*c0*t17660*t17814*t17662*t17854*t17873*t17883*t17659*t17866*t17884*t17867*t17697*(2.56E2/9.0)+c*c0*t17660*t17814*t17662*t17883*t17659*t17866*t17884*t17867*t17697*t17868*t17877*(1.28E2/9.0)-c0*kab1*rab1*t17711*t17810*t17712*t17814*t17856*t17883*t17866*t17884*t17876*t17868*(3.2E1/4.5E1)+c*c0*kab1*rab1*t17711*t17810*t17814*t17725*t17883*t17866*t17884*t17893*t17876*t17868*(1.6E1/6.75E2)+c*c0*kab1*t17711*t17810*t17814*t17883*t17892*t17866*t17884*t17893*t17876*t17894*t17868*(6.4E1/6.75E2)+c*c0*t17711*t17900*t17901*t17814*t17883*t17892*t17866*t17884*t17893*t17894*t17868*t17899*(6.4E1/2.25E2)+c0*kab1*rab1*t17711*t17810*t17814*t17842*t17854*t17873*t17856*t17883*t17866*t17884*t17876*(6.4E1/4.5E1)-c0*kab1*rab1*t17711*t17810*t17814*t17842*t17856*t17883*t17866*t17884*t17876*t17868*t17877*(3.2E1/4.5E1)-a3*c*c0*gamma_aa*kab1*rab1*t17810*t17905*t17657*t17856*t17883*t17866*t17884*t17876*t17868*(2.24E2/1.35E2)-c0*kab1*rab1*t17711*t17810*t17660*t17814*t17662*t17907*t17856*t17883*t17659*t17866*t17884*t17876*t17868*(1.024E3/1.35E2))+t17911*t17944*t17928*2.0+t17912*t17883*t17884+t17883*t17884*t17895+t17911*t17930*(t17920*t17912*2.0-t17932*t17895+c*c0*t17814*t17666*t17866*t17867*t17868*(3.2E1/2.7E1)+a3*c0*gamma_aa*t17734*t17657*t17866*t17867*t17868*(2.24E2/9.0)-c*c0*t17814*t17854*t17873*t17686*t17866*t17867*(3.2E1/9.0)-c*c0*t17814*t17873*t17882*t17658*t17866*t17867*(1.6E1/3.0)+c*c0*t17814*t17658*t17866*t17867*t17886*t17887*1.6E1-a3*c0*gamma_aa*t17690*t17854*t17657*t17873*t17866*t17867*(2.24E2/9.0)+c*c0*t17720*t17814*t17736*t17719*t17891*t17866*t17867*t17868*(8.192E3/2.7E1)+c*c0*t17660*t17814*t17662*t17737*t17659*t17866*t17867*t17868*(1.792E3/9.0)+a3*c0*gamma_aa*t17660*t17662*t17735*t17657*t17659*t17866*t17867*t17868*(3.584E3/2.7E1)-c*c0*t17660*t17814*t17662*t17854*t17873*t17659*t17866*t17867*t17697*(1.024E3/9.0)-c0*kab1*rab1*t17711*t17810*t17712*t17814*t17856*t17866*t17876*t17868*(1.28E2/4.5E1)+c*c0*kab1*rab1*t17711*t17810*t17814*t17725*t17866*t17893*t17876*t17868*(6.4E1/6.75E2)+c*c0*kab1*t17711*t17810*t17814*t17892*t17866*t17893*t17876*t17894*t17868*(2.56E2/6.75E2)+c*c0*t17711*t17900*t17901*t17814*t17892*t17866*t17893*t17894*t17868*t17899*(2.56E2/2.25E2)+c0*kab1*rab1*t17711*t17810*t17814*t17842*t17854*t17873*t17856*t17866*t17876*(2.56E2/4.5E1)-a3*c*c0*gamma_aa*kab1*rab1*t17810*t17905*t17657*t17856*t17866*t17876*t17868*(8.96E2/1.35E2)-c0*kab1*rab1*t17711*t17810*t17660*t17814*t17662*t17907*t17856*t17659*t17866*t17876*t17868*(4.096E3/1.35E2))-t17930*t17931*t17944*t17937*2.0-t17931*t17923*t17928*t17937*2.0-t17930*t17931*t17923*(t17920*t17912*3.0-t17932*t17895*(3.0/2.0)+c*c0*t17814*t17666*t17866*t17867*t17868*(1.6E1/9.0)+a3*c0*gamma_aa*t17734*t17657*t17866*t17867*t17868*(1.12E2/3.0)-c*c0*t17814*t17854*t17873*t17686*t17866*t17867*(1.6E1/3.0)-c*c0*t17814*t17873*t17882*t17658*t17866*t17867*8.0+c*c0*t17814*t17658*t17866*t17867*t17886*t17887*2.4E1-a3*c0*gamma_aa*t17690*t17854*t17657*t17873*t17866*t17867*(1.12E2/3.0)+c*c0*t17720*t17814*t17736*t17719*t17891*t17866*t17867*t17868*(4.096E3/9.0)+c*c0*t17660*t17814*t17662*t17737*t17659*t17866*t17867*t17868*(8.96E2/3.0)+a3*c0*gamma_aa*t17660*t17662*t17735*t17657*t17659*t17866*t17867*t17868*(1.792E3/9.0)-c*c0*t17660*t17814*t17662*t17854*t17873*t17659*t17866*t17867*t17697*(5.12E2/3.0)-c0*kab1*rab1*t17711*t17810*t17712*t17814*t17856*t17866*t17876*t17868*(6.4E1/1.5E1)+c*c0*kab1*rab1*t17711*t17810*t17814*t17725*t17866*t17893*t17876*t17868*(3.2E1/2.25E2)+c*c0*kab1*t17711*t17810*t17814*t17892*t17866*t17893*t17876*t17894*t17868*(1.28E2/2.25E2)+c*c0*t17711*t17900*t17901*t17814*t17892*t17866*t17893*t17894*t17868*t17899*(1.28E2/7.5E1)+c0*kab1*rab1*t17711*t17810*t17814*t17842*t17854*t17873*t17856*t17866*t17876*(1.28E2/1.5E1)-a3*c*c0*gamma_aa*kab1*rab1*t17810*t17905*t17657*t17856*t17866*t17876*t17868*(4.48E2/4.5E1)-c0*kab1*rab1*t17711*t17810*t17660*t17814*t17662*t17907*t17856*t17659*t17866*t17876*t17868*(2.048E3/4.5E1))+1.0/(t17910*t17910*t17910)*t17930*t17923*(t17937*t17937)*2.0+t17800*t17912*t17665*t17945*t17946*t17947*t17948*t17796*(3.0/2.0)+t17800*t17665*t17945*t17946*t17947*t17948*t17796*t17895*(3.0/2.0)-t17665*t17945*t17946*t17794*t17947*t17948*t17796*t17877*(1.0/2.0)+t17800*t17817*t17665*t17854*t17945*t17946*t17947*t17796*t17877*3.0+kab1*rab1*t17810*t17812*t17871*t17945*t17856*t17946*t17948*t17796*t17877*(4.0/5.0)-t17660*t17662*t17806*t17665*t17945*t17946*t17659*t17947*t17948*t17796*t17877*1.6E1-a3*gamma_aa*1.0/(t17814*t17814)*t17861*t17657*t17946*t17893*t17947*t17948*t17796*t17877*(7.0/2.0))*(-1.0/2.0)+c0*t17860*t17971*t17957*(1.0/2.0)+c0*t17953*t17954*t17957*(1.0/2.0)+c0*t17860*t17859*(t17911*t17930*t17944+t17911*t17923*t17928+t17883*t17884*t17877-t17930*t17931*t17923*t17937+t17800*t17665*t17945*t17946*t17947*t17948*t17796*t17877*(3.0/2.0)))-rho_a*(c0*t17822*t17844*(t17709*t17756*(t17710*t17683*t17787*2.0-t17683*t17788*t17789+t17720*t17704*t17718*t17719*t17791*t17689*(2.048E3/9.0)-t17660*t17662*t17664*t17790*t17688*t17689*(1.216E3/9.0)+c*c0*t17673*t17682*t17666*t17669*t17688*(3.2E1/2.7E1)-t17710*t17721*t17660*t17662*t17717*t17664*t17688*(6.4E1/3.0)+a5*c0*gamma_aa*t17734*t17682*t17657*t17669*t17688*(2.24E2/9.0)+c*c0*t17732*t17733*t17673*t17658*t17669*t17688*1.6E1-c*c0*t17731*t17673*t17658*t17669*t17696*t17688*(1.6E1/3.0)+c*c0*t17673*t17686*t17695*t17669*t17696*t17688*(3.2E1/9.0)+a5*c0*gamma_aa*t17690*t17657*t17695*t17669*t17696*t17688*(2.24E2/9.0)-c*c0*t17720*t17704*t17673*t17682*t17718*t17736*t17719*t17669*(1.6384E4/2.7E1)-c*c0*t17704*t17660*t17662*t17664*t17673*t17682*t17737*t17669*(1.792E3/9.0)+c*c0*t17720*t17743*t17673*t17682*t17718*t17736*t17719*t17669*(4.096E3/9.0)+c*c0*t17720*t17673*t17682*t17718*t17736*t17719*t17669*t17688*(8.192E3/2.7E1)+c*c0*t17660*t17662*t17664*t17673*t17682*t17737*t17669*t17688*(1.792E3/9.0)-a5*c0*gamma_aa*t17704*t17660*t17662*t17735*t17664*t17682*t17657*t17669*(3.584E3/2.7E1)+a5*c0*gamma_aa*t17660*t17662*t17735*t17664*t17682*t17657*t17669*t17688*(3.584E3/2.7E1)-c*c0*t17704*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17697*(1.024E3/9.0)+c*c0*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17688*t17697*(1.024E3/9.0))+t17700*t17742*t17699+t17762*t17807*t17709*2.0+t17700*t17754*t17699+t17709*t17766*(-t17801-t17802-t17803+t17750-t17804+t17742+t17751-t17805+t17752+t17744+t17753+t17745+t17746+t17747+t17748+t17749+t17871*t17738*(1.0/3.0)+(t17738*t17695*2.0)/t17681+t17721*t17660*t17662*t17664*t17738*(3.2E1/3.0)-t17721*t17660*t17662*t17664*t17683*t17738*(3.2E1/3.0)+a5*gamma_aa*t17665*t17836*t17657*t17738*t17795*(7.0/3.0)+c*c0*t17700*t17673*t17682*t17666*t17669*t17688*t17699*(8.0/2.7E1)+a5*c0*gamma_aa*t17700*t17734*t17682*t17657*t17669*t17688*t17699*(5.6E1/9.0)+c*c0*t17700*t17732*t17733*t17673*t17658*t17669*t17688*t17699*4.0+c*c0*t17700*t17742*t17673*t17682*t17658*t17669*t17688*t17699*(2.0/3.0)-c*c0*t17700*t17731*t17673*t17658*t17669*t17696*t17688*t17699*(4.0/3.0)+c*c0*t17700*t17673*t17682*t17754*t17658*t17669*t17688*t17699*(2.0/3.0)+c*c0*t17700*t17673*t17682*t17738*t17686*t17669*t17688*t17699*(4.0/9.0)+c*c0*t17700*t17673*t17686*t17695*t17669*t17696*t17688*t17699*(8.0/9.0)+a5*c0*gamma_aa*t17700*t17690*t17682*t17657*t17738*t17669*t17688*t17699*(2.8E1/9.0)+a5*c0*gamma_aa*t17700*t17690*t17657*t17695*t17669*t17696*t17688*t17699*(5.6E1/9.0)+c*c0*t17700*t17673*t17738*t17658*t17695*t17669*t17696*t17688*t17699*(8.0/3.0)-c*c0*t17700*t17720*t17704*t17673*t17682*t17718*t17736*t17719*t17669*t17699*(4.096E3/2.7E1)-c*c0*t17700*t17704*t17660*t17662*t17664*t17673*t17682*t17737*t17669*t17699*(4.48E2/9.0)+c*c0*t17700*t17720*t17743*t17673*t17682*t17718*t17736*t17719*t17669*t17699*(1.024E3/9.0)+c*c0*t17700*t17720*t17673*t17682*t17718*t17736*t17719*t17669*t17688*t17699*(2.048E3/2.7E1)+c*c0*t17700*t17660*t17662*t17664*t17673*t17682*t17737*t17669*t17688*t17699*(4.48E2/9.0)-a5*c0*gamma_aa*t17700*t17704*t17660*t17662*t17735*t17664*t17682*t17657*t17669*t17699*(8.96E2/2.7E1)+a5*c0*gamma_aa*t17700*t17660*t17662*t17735*t17664*t17682*t17657*t17669*t17688*t17699*(8.96E2/2.7E1)-c*c0*t17700*t17704*t17660*t17662*t17664*t17673*t17682*t17738*t17669*t17697*t17699*(1.28E2/9.0)-c*c0*t17700*t17704*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17697*t17699*(2.56E2/9.0)+c*c0*t17700*t17660*t17662*t17664*t17673*t17682*t17738*t17669*t17688*t17697*t17699*(1.28E2/9.0)+c*c0*t17700*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17688*t17697*t17699*(2.56E2/9.0))-t17763*t17756*t17766*(t17710*t17683*t17787*3.0-t17683*t17788*t17789*(3.0/2.0)+t17720*t17704*t17718*t17719*t17791*t17689*(1.024E3/3.0)-t17660*t17662*t17664*t17790*t17688*t17689*(6.08E2/3.0)+c*c0*t17673*t17682*t17666*t17669*t17688*(1.6E1/9.0)-t17710*t17721*t17660*t17662*t17717*t17664*t17688*3.2E1+a5*c0*gamma_aa*t17734*t17682*t17657*t17669*t17688*(1.12E2/3.0)+c*c0*t17732*t17733*t17673*t17658*t17669*t17688*2.4E1-c*c0*t17731*t17673*t17658*t17669*t17696*t17688*8.0+c*c0*t17673*t17686*t17695*t17669*t17696*t17688*(1.6E1/3.0)+a5*c0*gamma_aa*t17690*t17657*t17695*t17669*t17696*t17688*(1.12E2/3.0)-c*c0*t17720*t17704*t17673*t17682*t17718*t17736*t17719*t17669*(8.192E3/9.0)-c*c0*t17704*t17660*t17662*t17664*t17673*t17682*t17737*t17669*(8.96E2/3.0)+c*c0*t17720*t17743*t17673*t17682*t17718*t17736*t17719*t17669*(2.048E3/3.0)+c*c0*t17720*t17673*t17682*t17718*t17736*t17719*t17669*t17688*(4.096E3/9.0)+c*c0*t17660*t17662*t17664*t17673*t17682*t17737*t17669*t17688*(8.96E2/3.0)-a5*c0*gamma_aa*t17704*t17660*t17662*t17735*t17664*t17682*t17657*t17669*(1.792E3/9.0)+a5*c0*gamma_aa*t17660*t17662*t17735*t17664*t17682*t17657*t17669*t17688*(1.792E3/9.0)-c*c0*t17704*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17697*(5.12E2/3.0)+c*c0*t17660*t17662*t17664*t17673*t17695*t17669*t17696*t17688*t17697*(5.12E2/3.0))-t17762*t17763*t17772*t17756*2.0-t17807*t17763*t17772*t17766*2.0+1.0/(t17708*t17708*t17708)*(t17772*t17772)*t17756*t17766*2.0+t17800*t17742*t17665*t17795*t17796*t17797*t17798*t17799*(3.0/2.0)+t17800*t17754*t17665*t17795*t17796*t17797*t17798*t17799*(3.0/2.0)-t17665*t17738*t17794*t17795*t17796*t17797*t17798*t17799*(1.0/2.0)-t17800*t17681*t17665*t17738*t17695*t17795*t17796*t17797*t17798*3.0+t17660*t17662*t17671*t17806*t17664*t17665*t17738*t17795*t17796*t17797*t17799*1.6E1-t17660*t17662*t17806*t17664*t17665*t17738*t17795*t17796*t17797*t17798*t17799*1.6E1-a5*gamma_aa*t17861*1.0/(t17673*t17673)*t17657*t17738*t17893*t17796*t17797*t17798*t17799*(7.0/2.0))*(-1.0/2.0)+c0*t17822*t17844*t17847*(t17711*t17712*t17821*(1.0E1/9.0)-t17711*t17841*t17842*t17838*(8.0/3.0)+t17711*1.0/(t17820*t17820*t17820*t17820)*(t17841*t17841)*t17794*6.0-t17711*t17838*t17794*(c*k2*t17666*(4.0/9.0)+c*k1*t17691*t17666*(2.0/9.0)-k1*t17711*t17712*t17727*(1.0/3.6E1))*2.0)*(1.0/2.0)+c0*t17822*t17860*t17971*t17847*(1.0/2.0)+c0*t17822*t17860*t17835*t17859+c0*t17822*t17835*t17844*t17849+c0*t17822*t17953*t17954*t17847*(1.0/2.0)-c0*t17822*t17844*t17847*(t17849*t17849)*(1.0/2.0)-c0*t17822*t17860*t17847*t17849*t17859)-c0*t17822*t17835*t17844+c0*t17822*t17860*t17847*t17859+c0*t17822*t17844*t17847*t17849);
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t17974 = tspi_m13*tspi_m13;
                double t17975 = 1.0/pow(rho_a,1.0/3.0);
                double t17976 = a2*a2;
                double t17977 = gamma_aa*gamma_aa;
                double t17978 = 1.0/pow(rho_a,1.6E1/3.0);
                double t17979 = t17974*t17974;
                double t17980 = c*t17975;
                double t17981 = pow(t17980,4.0/5.0);
                double t17995 = rab1*t17981;
                double t17982 = exp(-t17995);
                double t17983 = t17982-1.0;
                double t17996 = kab1*t17983;
                double t17984 = kab0-t17996;
                double t17985 = 1.0/c;
                double t17986 = 1.0/pow(rho_a,7.0/3.0);
                double t17987 = a3*gamma_aa*t17974*t17985*t17986;
                double t17988 = t17987+1.0;
                double t17989 = t17976*t17977*t17978*t17979;
                double t17990 = 1.0/pow(rho_a,8.0/3.0);
                double t17991 = a1*gamma_aa*t17990*t17974;
                double t17992 = t17991+t17989+1.0;
                double t17993 = t17976*t17977*t17978*t17979*2.0;
                double t17994 = exp(t17993);
                double t17997 = 1.0/(t17984*t17984);
                double t17998 = 1.0/(t17992*t17992);
                double t17999 = c*c0*t17975*t17994*t17988*t17997*t17998*(2.0/3.0);
                double t18000 = sqrt(t17999);
                double t18001 = 1.0/pow(rho_a,1.1E1/3.0);
                double t18002 = 1.0/pow(rho_a,4.0/3.0);
                double t18003;
                double t18004;
                if (t17999 > expei_cutoff) {
                    t18003 = 1.0;
                    t18004 = expei(t17999);
                } else {
                    t18003 = exp(t17999);
                    t18004 = Ei(-t17999);
                }
                double t18005 = c*c0*t17994*t17988*t17997*t17998*t18002*(2.0/9.0);
                double t18006 = 1.0/pow(rho_a,1.9E1/3.0);
                double t18007 = t17976*t17977*t17979*t18006*(1.6E1/3.0);
                double t18008 = a1*gamma_aa*t17974*t18001*(8.0/3.0);
                double t18009 = t18007+t18008;
                double t18010 = 1.0/(t17992*t17992*t17992);
                double t18011 = a3*c0*gamma_aa*t17974*t17994*t17997*t17998*t18001*(1.4E1/9.0);
                double t18012 = 1.0/pow(rho_a,2.0E1/3.0);
                double t18013 = c*c0*t17976*t17994*t17977*t17979*t17988*t17997*t17998*t18012*(6.4E1/9.0);
                double t18014 = c*c;
                double t18015 = 1.0/pow(rho_a,5.0/3.0);
                double t18016 = 1.0/(t17984*t17984*t17984);
                double t18017 = 1.0/pow(t17980,1.0/5.0);
                double t18023 = c*c0*t17975*t17994*t17988*t17997*t18010*t18009*(4.0/3.0);
                double t18024 = c0*kab1*rab1*t17982*t17994*t17988*t17998*t18014*t18015*t18016*t18017*(1.6E1/4.5E1);
                double t18018 = t18011+t18013+t18005-t18023-t18024;
                double t18019 = t18000*6.0;
                double t18020 = c*c0*t17975*t17994*t17988*t17997*t17998*4.0;
                double t18021 = t18020+t18019+3.0;
                double t18022 = 1.0/t18021;
                double t18025 = c*c0*t17975*t17994*t17988*t17997*t17998*t18003*t18004*(2.0/3.0);
                double t18026 = t18025+1.0;
                double t18027 = t18000*4.0;
                double t18028 = c*c0*t17975*t17994*t17988*t17997*t17998*(8.0/3.0);
                double t18029 = t18027+t18028+6.0;
                double t18030 = 1.0/sqrt(t17999);
                double t18031 = 1.0/pow(rho_b,1.0/3.0);
                double t18032 = gamma_bb*gamma_bb;
                double t18033 = 1.0/pow(rho_b,1.6E1/3.0);
                double t18034 = c*t18031;
                double t18035 = pow(t18034,4.0/5.0);
                double t18048 = rab1*t18035;
                double t18036 = exp(-t18048);
                double t18037 = t18036-1.0;
                double t18049 = kab1*t18037;
                double t18038 = kab0-t18049;
                double t18039 = 1.0/pow(rho_b,7.0/3.0);
                double t18040 = a3*gamma_bb*t17974*t17985*t18039;
                double t18041 = t18040+1.0;
                double t18042 = t17976*t17979*t18032*t18033;
                double t18043 = 1.0/pow(rho_b,8.0/3.0);
                double t18044 = a1*gamma_bb*t17974*t18043;
                double t18045 = t18042+t18044+1.0;
                double t18046 = t17976*t17979*t18032*t18033*2.0;
                double t18047 = exp(t18046);
                double t18050 = 1.0/(t18038*t18038);
                double t18051 = 1.0/(t18045*t18045);
                double t18052 = c*c0*t18031*t18041*t18050*t18051*t18047*(2.0/3.0);
                double t18053 = sqrt(t18052);
                double t18054 = 1.0/pow(rho_b,1.1E1/3.0);
                double t18055 = 1.0/pow(rho_b,4.0/3.0);
                double t18056;
                double t18057;
                if (t18052 > expei_cutoff) {
                    t18056 = 1.0;
                    t18057 = expei(t18052);
                } else {
                    t18056 = exp(t18052);
                    t18057 = Ei(-t18052);
                }
                double t18058 = c*c0*t18041*t18050*t18051*t18055*t18047*(2.0/9.0);
                double t18059 = 1.0/pow(rho_b,1.9E1/3.0);
                double t18060 = t17976*t17979*t18032*t18059*(1.6E1/3.0);
                double t18061 = a1*gamma_bb*t17974*t18054*(8.0/3.0);
                double t18062 = t18060+t18061;
                double t18063 = 1.0/(t18045*t18045*t18045);
                double t18064 = a3*c0*gamma_bb*t17974*t18050*t18051*t18054*t18047*(1.4E1/9.0);
                double t18065 = 1.0/pow(rho_b,2.0E1/3.0);
                double t18066 = c*c0*t17976*t17979*t18032*t18041*t18050*t18051*t18047*t18065*(6.4E1/9.0);
                double t18067 = 1.0/pow(rho_b,5.0/3.0);
                double t18068 = 1.0/(t18038*t18038*t18038);
                double t18069 = 1.0/pow(t18034,1.0/5.0);
                double t18075 = c*c0*t18031*t18041*t18050*t18062*t18063*t18047*(4.0/3.0);
                double t18076 = c0*kab1*rab1*t18014*t18041*t18051*t18036*t18047*t18067*t18068*t18069*(1.6E1/4.5E1);
                double t18070 = t18064+t18066-t18075+t18058-t18076;
                double t18071 = t18053*6.0;
                double t18072 = c*c0*t18031*t18041*t18050*t18051*t18047*4.0;
                double t18073 = t18071+t18072+3.0;
                double t18074 = 1.0/t18073;
                double t18077 = c*c0*t18031*t18041*t18050*t18051*t18047*t18056*t18057*(2.0/3.0);
                double t18078 = t18077+1.0;
                double t18079 = t18053*4.0;
                double t18080 = c*c0*t18031*t18041*t18050*t18051*t18047*(8.0/3.0);
                double t18081 = t18080+t18079+6.0;
                double t18082 = 1.0/sqrt(t18052);
                double t18083 = 1.0/c0;
                double t18084 = exp(-t17989);
                double t18085 = 1.0/sqrt(t17988);
                double t18086 = t17992*t17984*t18084*t18085*1.0E5;
                double t18087 = exp(-t18042);
                double t18088 = 1.0/sqrt(t18041);
                double t18089 = t18045*t18038*t18087*t18088*1.0E5;
                v_rho_a_rho_b[Q] += scale * (c0*heaviside(t18086-c*c0*t17975*(2.0/3.0))*(t18022*t18026*(t18030*t18018*2.0+c*c0*t17994*t17988*t17997*t17998*t18002*(8.0/9.0)+a3*c0*gamma_aa*t17974*t17994*t17997*t17998*t18001*(5.6E1/9.0)-c*c0*t17975*t17994*t17988*t17997*t18010*t18009*(1.6E1/3.0)+c*c0*t17976*t17994*t17977*t17979*t17988*t17997*t17998*t18012*(2.56E2/9.0)-c0*kab1*rab1*t17982*t17994*t17988*t17998*t18014*t18015*t18016*t18017*(6.4E1/4.5E1))+t18022*t18029*(t18011+t18013+t18005-t18023-t18024+c*c0*t17994*t17988*t17997*t17998*t18002*t18003*t18004*(2.0/9.0)+a3*c0*gamma_aa*t17974*t17994*t17997*t17998*t18001*t18003*t18004*(1.4E1/9.0)+c*c0*t17975*t17994*t17988*t17997*t17998*t18003*t18004*t18018*(2.0/3.0)-c*c0*t17975*t17994*t17988*t17997*t18010*t18003*t18004*t18009*(4.0/3.0)+c*c0*t17976*t17994*t17977*t17979*t17988*t17997*t17998*t18003*t18012*t18004*(6.4E1/9.0)-c0*kab1*rab1*t17982*t17994*t17988*t17998*t18003*t18004*t18014*t18015*t18016*t18017*(1.6E1/4.5E1))+t18003*t18004*t18018-1.0/(t18021*t18021)*t18026*t18029*(t18030*t18018*3.0+c*c0*t17994*t17988*t17997*t17998*t18002*(4.0/3.0)+a3*c0*gamma_aa*t17974*t17994*t17997*t17998*t18001*(2.8E1/3.0)-c*c0*t17975*t17994*t17988*t17997*t18010*t18009*8.0+c*c0*t17976*t17994*t17977*t17979*t17988*t17997*t17998*t18012*(1.28E2/3.0)-c0*kab1*rab1*t17982*t17994*t17988*t17998*t18014*t18015*t18016*t18017*(3.2E1/1.5E1))+(pow(rho_a,1.0/3.0)*(t17992*t17992)*(t17984*t17984)*t17985*t18018*t18083*exp(-t17993)*(3.0/2.0))/t17988)*(-1.0/2.0)-c0*heaviside(t18089-c*c0*t18031*(2.0/3.0))*(t18074*t18078*(t18070*t18082*2.0+c*c0*t18041*t18050*t18051*t18055*t18047*(8.0/9.0)+a3*c0*gamma_bb*t17974*t18050*t18051*t18054*t18047*(5.6E1/9.0)-c*c0*t18031*t18041*t18050*t18062*t18063*t18047*(1.6E1/3.0)+c*c0*t17976*t17979*t18032*t18041*t18050*t18051*t18047*t18065*(2.56E2/9.0)-c0*kab1*rab1*t18014*t18041*t18051*t18036*t18047*t18067*t18068*t18069*(6.4E1/4.5E1))+t18081*t18074*(t18064+t18066-t18075+t18058-t18076+c*c0*t18041*t18050*t18051*t18055*t18047*t18056*t18057*(2.0/9.0)+a3*c0*gamma_bb*t17974*t18050*t18051*t18054*t18047*t18056*t18057*(1.4E1/9.0)+c*c0*t18031*t18041*t18050*t18051*t18070*t18047*t18056*t18057*(2.0/3.0)-c*c0*t18031*t18041*t18050*t18062*t18063*t18047*t18056*t18057*(4.0/3.0)+c*c0*t17976*t17979*t18032*t18041*t18050*t18051*t18047*t18056*t18065*t18057*(6.4E1/9.0)-c0*kab1*rab1*t18014*t18041*t18051*t18036*t18047*t18056*t18057*t18067*t18068*t18069*(1.6E1/4.5E1))+t18070*t18056*t18057-t18081*1.0/(t18073*t18073)*t18078*(t18070*t18082*3.0+c*c0*t18041*t18050*t18051*t18055*t18047*(4.0/3.0)+a3*c0*gamma_bb*t17974*t18050*t18051*t18054*t18047*(2.8E1/3.0)-c*c0*t18031*t18041*t18050*t18062*t18063*t18047*8.0+c*c0*t17976*t17979*t18032*t18041*t18050*t18051*t18047*t18065*(1.28E2/3.0)-c0*kab1*rab1*t18014*t18041*t18051*t18036*t18047*t18067*t18068*t18069*(3.2E1/1.5E1))+(pow(rho_b,1.0/3.0)*t17985*t18070*(t18045*t18045)*(t18038*t18038)*t18083*exp(-t18046)*(3.0/2.0))/t18041)*(1.0/2.0)+c0*dirac(-t18086+c*c0*t17975*(2.0/3.0))*(t18003*t18004+t18022*t18026*t18029)*(c*c0*t18002*(2.0/9.0)-t17984*t18009*t18084*t18085*1.0E5+t17992*t17984*t17976*t17977*t17979*t18006*t18084*t18085*(1.6E6/3.0)+a3*gamma_aa*1.0/pow(rho_a,1.0E1/3.0)*t17974*t17992*t17984*t17985*1.0/pow(t17988,3.0/2.0)*t18084*(3.5E5/3.0)-c*kab1*rab1*t17982*t17992*t18002*t18017*t18084*t18085*(8.0E4/3.0))*(1.0/2.0)+c0*dirac(-t18089+c*c0*t18031*(2.0/3.0))*(t18056*t18057+t18081*t18074*t18078)*(c*c0*t18055*(2.0/9.0)-t18062*t18038*t18087*t18088*1.0E5+t17976*t17979*t18032*t18045*t18038*t18059*t18087*t18088*(1.6E6/3.0)+a3*gamma_bb*1.0/pow(rho_b,1.0E1/3.0)*t17974*t17985*1.0/pow(t18041,3.0/2.0)*t18045*t18038*t18087*(3.5E5/3.0)-c*kab1*rab1*t18036*t18045*t18055*t18069*t18087*t18088*(8.0E4/3.0))*(1.0/2.0));
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t18091 = tspi_m13*tspi_m13;
                double t18092 = 1.0/pow(rho_b,1.0/3.0);
                double t18093 = a2*a2;
                double t18094 = gamma_bb*gamma_bb;
                double t18095 = 1.0/pow(rho_b,1.6E1/3.0);
                double t18096 = t18091*t18091;
                double t18097 = c*t18092;
                double t18098 = a4*a4;
                double t18099 = 1.0/c;
                double t18100 = 1.0/pow(rho_b,7.0/3.0);
                double t18101 = sqrt(t18097);
                double t18102 = t18094*t18095*t18096*t18098*2.0;
                double t18103 = exp(t18102);
                double t18104 = t18094*t18095*t18096*t18098;
                double t18105 = t18104+1.0;
                double t18106 = a5*gamma_bb*t18100*t18091*t18099;
                double t18107 = t18106+1.0;
                double t18118 = raa2*t18101;
                double t18108 = exp(-t18118);
                double t18109 = t18108-1.0;
                double t18110 = kaa2*t18109;
                double t18111 = pow(t18097,2.0/5.0);
                double t18119 = raa1*t18111;
                double t18112 = exp(-t18119);
                double t18113 = t18112-1.0;
                double t18114 = kaa1*t18113;
                double t18115 = -kaa0+t18110+t18114;
                double t18116 = 1.0/(t18115*t18115);
                double t18117 = 1.0/t18105;
                double t18120 = 1.0/pow(rho_b,4.0/3.0);
                double t18121 = c*c0*t18103*t18107*t18116*t18092*(2.0/3.0);
                double t18122 = 1.0/(t18105*t18105);
                double t18123 = sqrt(t18121);
                double t18124 = 1.0/pow(rho_b,1.1E1/3.0);
                double t18125 = 1.0/sqrt(t18097);
                double t18126 = c*kaa2*raa2*t18120*t18125*t18108*(1.0/6.0);
                double t18127 = 1.0/pow(t18097,3.0/5.0);
                double t18128 = c*kaa1*raa1*t18120*t18112*t18127*(2.0/1.5E1);
                double t18129 = t18126+t18128;
                double t18130 = 1.0/(t18115*t18115*t18115);
                double t18131 = 1.0/pow(rho_b,2.0E1/3.0);
                double t18132 = c*c0*t18103*t18122*t18107*t18116*t18092*(2.0/3.0);
                double t18133;
                double t18134;
                if (t18132 > expei_cutoff) {
                    t18133 = 1.0;
                    t18134 = expei(t18132);
                } else {
                    t18133 = exp(t18132);
                    t18134 = Ei(-t18132);
                }
                double t18135 = c*c0*t18120*t18103*t18122*t18107*t18116*(2.0/9.0);
                double t18136 = a5*c0*gamma_bb*t18103*t18122*t18124*t18116*t18091*(1.4E1/9.0);
                double t18137 = c*c0*t18103*t18130*t18122*t18107*t18092*t18129*(4.0/3.0);
                double t18138 = 1.0/(t18105*t18105*t18105);
                double t18139 = c*c0*t18103*t18122*t18131*t18107*t18116*t18094*t18096*t18098*(6.4E1/9.0);
                double t18140 = t18123*t18117*6.0;
                double t18141 = c*c0*t18103*t18122*t18107*t18116*t18092*4.0;
                double t18142 = t18140+t18141+3.0;
                double t18143 = 1.0/t18142;
                double t18144 = 1.0/sqrt(t18121);
                double t18145 = c*c;
                double t18146 = 1.0/pow(rho_b,8.0/3.0);
                double t18147 = c*c0*t18120*t18103*t18107*t18116*(2.0/9.0);
                double t18148 = a5*c0*gamma_bb*t18103*t18124*t18116*t18091*(1.4E1/9.0);
                double t18149 = c*c0*t18103*t18130*t18107*t18092*t18129*(4.0/3.0);
                double t18150 = c*c0*t18103*t18131*t18107*t18116*t18094*t18096*t18098*(6.4E1/9.0);
                double t18151 = t18150+t18147+t18148+t18149;
                double t18152 = t18098*t18098;
                double t18153 = t18094*t18094;
                double t18154 = t18096*t18096;
                double t18155 = 1.0/pow(rho_b,1.9E1/3.0);
                double t18156 = raa2*raa2;
                double t18157 = c*kaa2*t18100*t18108*t18156*(1.0/3.6E1);
                double t18158 = raa1*raa1;
                double t18159 = 1.0/pow(t18097,6.0/5.0);
                double t18160 = kaa1*t18112*t18145*t18146*t18158*t18159*(4.0/2.25E2);
                double t18161 = 1.0/pow(t18097,3.0/2.0);
                double t18162 = kaa2*raa2*t18161*t18108*t18145*t18146*(1.0/3.6E1);
                double t18163 = 1.0/pow(t18097,8.0/5.0);
                double t18164 = kaa1*raa1*t18112*t18145*t18163*t18146*(2.0/7.5E1);
                double t18174 = c*kaa2*raa2*t18100*t18125*t18108*(2.0/9.0);
                double t18175 = c*kaa1*raa1*t18100*t18112*t18127*(8.0/4.5E1);
                double t18165 = t18160+t18162+t18164-t18174+t18157-t18175;
                double t18166 = t18129*t18129;
                double t18167 = 1.0/(t18115*t18115*t18115*t18115);
                double t18168 = 1.0/pow(rho_b,1.4E1/3.0);
                double t18169 = 1.0/pow(rho_b,1.0E1);
                double t18170 = 1.0/pow(rho_b,1.3E1);
                double t18171 = 1.0/pow(rho_b,2.3E1/3.0);
                double t18173 = c*c0*t18103*t18131*t18107*t18116*t18138*t18094*t18096*t18098*(6.4E1/9.0);
                double t18172 = t18135+t18136+t18137-t18173+t18139;
                double t18176 = t18172*t18172;
                double t18177 = 1.0/(t18105*t18105*t18105*t18105);
                double t18178 = c*c0*t18100*t18103*t18122*t18107*t18116*(8.0/2.7E1);
                double t18179 = c*c0*t18103*t18122*t18107*t18092*t18166*t18167*4.0;
                double t18180 = a5*c0*gamma_bb*t18103*t18122*t18116*t18091*t18168*(5.6E1/9.0);
                double t18181 = c*c0*t18120*t18103*t18130*t18122*t18107*t18129*(8.0/9.0);
                double t18182 = a5*c0*gamma_bb*t18103*t18130*t18122*t18124*t18091*t18129*(5.6E1/9.0);
                double t18183 = a5*c0*gamma_bb*t18103*t18122*t18116*t18091*t18094*t18096*t18169*t18098*(8.96E2/2.7E1);
                double t18184 = c*c0*t18103*t18122*t18107*t18116*t18152*t18170*t18153*t18154*(2.048E3/2.7E1);
                double t18185 = c*c0*t18103*t18107*t18116*t18152*t18170*t18153*t18154*t18177*(1.024E3/9.0);
                double t18186 = c*c0*t18103*t18122*t18107*t18116*t18171*t18094*t18096*t18098*(4.48E2/9.0);
                double t18187 = c*c0*t18103*t18130*t18122*t18131*t18107*t18129*t18094*t18096*t18098*(2.56E2/9.0);
                double t18235 = c*c0*t18103*t18130*t18122*t18107*t18092*t18165*(4.0/3.0);
                double t18236 = a5*c0*gamma_bb*t18103*t18116*t18091*t18138*t18094*t18096*t18169*t18098*(8.96E2/2.7E1);
                double t18237 = c*c0*t18103*t18107*t18116*t18152*t18170*t18153*t18154*t18138*(4.096E3/2.7E1);
                double t18238 = c*c0*t18103*t18107*t18116*t18171*t18138*t18094*t18096*t18098*(4.48E2/9.0);
                double t18239 = c*c0*t18103*t18130*t18131*t18107*t18129*t18138*t18094*t18096*t18098*(2.56E2/9.0);
                double t18188 = t18180+t18181-t18235+t18182-t18236+t18183-t18237+t18184-t18238+t18185-t18239+t18186+t18178+t18187+t18179;
                double t18189 = c*c0*t18103*t18122*t18133*t18107*t18116*t18134*t18092*(2.0/3.0);
                double t18190 = t18189+1.0;
                double t18191 = t18151*t18117*t18144*2.0;
                double t18192 = c*c0*t18120*t18103*t18122*t18107*t18116*(8.0/9.0);
                double t18193 = a5*c0*gamma_bb*t18103*t18122*t18124*t18116*t18091*(5.6E1/9.0);
                double t18194 = c*c0*t18103*t18130*t18122*t18107*t18092*t18129*(1.6E1/3.0);
                double t18195 = c*c0*t18103*t18122*t18131*t18107*t18116*t18094*t18096*t18098*(2.56E2/9.0);
                double t18257 = t18122*t18123*t18155*t18094*t18096*t18098*(6.4E1/3.0);
                double t18258 = c*c0*t18103*t18131*t18107*t18116*t18138*t18094*t18096*t18098*(2.56E2/9.0);
                double t18196 = t18191+t18192+t18193+t18194-t18257+t18195-t18258;
                double t18197 = 1.0/(t18142*t18142);
                double t18198 = t18123*t18117*4.0;
                double t18199 = c*c0*t18103*t18122*t18107*t18116*t18092*(8.0/3.0);
                double t18200 = t18198+t18199+6.0;
                double t18201 = t18151*t18117*t18144*3.0;
                double t18202 = c*c0*t18120*t18103*t18122*t18107*t18116*(4.0/3.0);
                double t18203 = a5*c0*gamma_bb*t18103*t18122*t18124*t18116*t18091*(2.8E1/3.0);
                double t18204 = c*c0*t18103*t18130*t18122*t18107*t18092*t18129*8.0;
                double t18205 = c*c0*t18103*t18122*t18131*t18107*t18116*t18094*t18096*t18098*(1.28E2/3.0);
                double t18226 = t18122*t18123*t18155*t18094*t18096*t18098*3.2E1;
                double t18227 = c*c0*t18103*t18131*t18107*t18116*t18138*t18094*t18096*t18098*(1.28E2/3.0);
                double t18206 = t18201+t18202+t18203+t18204+t18205-t18226-t18227;
                double t18207 = c*c0*t18120*t18103*t18122*t18133*t18107*t18116*t18134*(2.0/9.0);
                double t18208 = c*c0*t18103*t18130*t18122*t18133*t18107*t18134*t18092*t18129*(4.0/3.0);
                double t18209 = c*c0*t18103*t18122*t18133*t18107*t18116*t18134*t18172*t18092*(2.0/3.0);
                double t18210 = a5*c0*gamma_bb*t18103*t18122*t18124*t18133*t18116*t18134*t18091*(1.4E1/9.0);
                double t18211 = c*c0*t18103*t18122*t18131*t18133*t18107*t18116*t18134*t18094*t18096*t18098*(6.4E1/9.0);
                double t18212 = c*c0*t18100*t18103*t18107*t18116*(8.0/2.7E1);
                double t18213 = a5*c0*gamma_bb*t18103*t18116*t18091*t18168*(5.6E1/9.0);
                double t18214 = c*c0*t18120*t18103*t18130*t18107*t18129*(8.0/9.0);
                double t18215 = c*c0*t18103*t18107*t18092*t18166*t18167*4.0;
                double t18216 = a5*c0*gamma_bb*t18103*t18116*t18091*t18094*t18096*t18169*t18098*(8.96E2/2.7E1);
                double t18217 = a5*c0*gamma_bb*t18103*t18130*t18124*t18091*t18129*(5.6E1/9.0);
                double t18218 = c*c0*t18103*t18107*t18116*t18152*t18170*t18153*t18154*(2.048E3/2.7E1);
                double t18219 = c*c0*t18103*t18107*t18116*t18171*t18094*t18096*t18098*(4.48E2/9.0);
                double t18220 = c*c0*t18103*t18130*t18131*t18107*t18129*t18094*t18096*t18098*(2.56E2/9.0);
                double t18221 = t18220+t18212+t18213+t18214+t18215+t18216+t18217+t18218+t18219-c*c0*t18103*t18130*t18107*t18092*t18165*(4.0/3.0);
                double t18222 = t18151*t18151;
                double t18223 = 1.0/pow(t18121,3.0/2.0);
                double t18224 = 1.0/pow(rho_b,2.2E1/3.0);
                double t18225 = 1.0/pow(rho_b,3.8E1/3.0);
                double t18228 = 1.0/pow(rho_b,2.0/3.0);
                double t18229 = 1.0/t18107;
                double t18230 = 1.0/c0;
                double t18231 = exp(-t18102);
                double t18232 = t18105*t18105;
                double t18233 = t18115*t18115;
                double t18234 = pow(rho_b,1.0/3.0);
                double t18240 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b*rho_b);
                double t18259 = c*c0*t18103*t18131*t18133*t18107*t18116*t18134*t18138*t18094*t18096*t18098*(6.4E1/9.0);
                double t18241 = t18210+t18211+t18135+t18207+t18136+t18208+t18137-t18173+t18209+t18139-t18259;
                double t18242 = t18093*t18094*t18095*t18096;
                double t18243 = pow(t18097,4.0/5.0);
                double t18261 = rab1*t18243;
                double t18244 = exp(-t18261);
                double t18245 = t18244-1.0;
                double t18262 = kab1*t18245;
                double t18246 = kab0-t18262;
                double t18247 = a3*gamma_bb*t18100*t18091*t18099;
                double t18248 = t18247+1.0;
                double t18249 = 1.0/sqrt(t18248);
                double t18250 = a1*gamma_bb*t18091*t18146;
                double t18251 = t18250+t18242+1.0;
                double t18252 = k1*t18101;
                double t18253 = c*k2*t18092;
                double t18254 = t18252+t18253;
                double t18255 = 1.0/(t18254*t18254);
                double t18264 = t18145*t18228*t18255;
                double t18256 = exp(-t18264);
                double t18260 = exp(-t18242);
                double t18263 = t18251*t18260*t18246*t18249*1.0E5;
                double t18265 = t18143*t18190*t18196;
                double t18266 = t18133*t18134*t18172;
                double t18267 = t18200*t18241*t18143;
                double t18268 = t18230*t18231*t18232*t18233*t18234*t18172*t18229*t18099*(3.0/2.0);
                double t18406 = t18200*t18206*t18190*t18197;
                double t18269 = -t18406+t18265+t18266+t18267+t18268;
                double t18270 = 1.0/pow(rho_b,1.0E1/3.0);
                double t18271 = c*c0*t18092*(2.0/3.0);
                double t18272 = 1.0/(t18254*t18254*t18254);
                double t18273 = c*k2*t18120*(1.0/3.0);
                double t18274 = c*k1*t18120*t18125*(1.0/6.0);
                double t18275 = t18273+t18274;
                double t18276 = 1.0/pow(rho_b,5.0/3.0);
                double t18277 = -t18271+t18263;
                double t18278 = heaviside(t18277);
                double t18279 = t18133*t18134;
                double t18280 = t18200*t18143*t18190;
                double t18281 = t18280+t18279;
                double t18282 = t18145*t18255*t18276*(2.0/3.0);
                double t18296 = t18145*t18272*t18228*t18275*2.0;
                double t18283 = t18282-t18296;
                double t18284 = t18271-t18263;
                double t18285 = c*c0*t18120*(2.0/9.0);
                double t18286 = t18155*t18093*t18094*t18096*(1.6E1/3.0);
                double t18287 = a1*gamma_bb*t18124*t18091*(8.0/3.0);
                double t18288 = t18286+t18287;
                double t18289 = t18251*t18260*t18155*t18093*t18246*t18094*t18096*t18249*(1.6E6/3.0);
                double t18290 = 1.0/pow(t18097,1.0/5.0);
                double t18291 = 1.0/pow(t18248,3.0/2.0);
                double t18292 = a3*gamma_bb*t18251*t18260*t18270*t18091*t18246*t18291*t18099*(3.5E5/3.0);
                double t18297 = t18260*t18246*t18249*t18288*1.0E5;
                double t18298 = c*kab1*rab1*t18120*t18251*t18260*t18244*t18290*t18249*(8.0E4/3.0);
                double t18293 = t18292+t18285-t18297+t18289-t18298;
                double t18294 = dirac(t18284);
                double t18295 = 1.0/(c*c);
                double t18299 = t18093*t18094*t18095*t18096*2.0;
                double t18300 = exp(t18299);
                double t18301 = 1.0/(t18246*t18246);
                double t18302 = 1.0/(t18251*t18251);
                double t18303 = c*c0*t18300*t18301*t18302*t18092*t18248*(2.0/3.0);
                double t18304 = sqrt(t18303);
                double t18305 = 1.0/rho_b;
                double t18306 = c*c0*t18120*t18300*t18301*t18302*t18248*(2.0/9.0);
                double t18307 = 1.0/(t18251*t18251*t18251);
                double t18308 = a3*c0*gamma_bb*t18300*t18301*t18302*t18124*t18091*(1.4E1/9.0);
                double t18309 = c*c0*t18300*t18301*t18131*t18302*t18093*t18094*t18248*t18096*(6.4E1/9.0);
                double t18310 = 1.0/(t18246*t18246*t18246);
                double t18312 = c*c0*t18300*t18301*t18307*t18092*t18248*t18288*(4.0/3.0);
                double t18313 = c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18290*t18248*t18276*(1.6E1/4.5E1);
                double t18311 = -t18312-t18313+t18306+t18308+t18309;
                double t18314 = t18224*t18093*t18094*t18096*(3.04E2/9.0);
                double t18315 = a1*gamma_bb*t18091*t18168*(8.8E1/9.0);
                double t18316 = t18314+t18315;
                double t18317;
                double t18318;
                if (t18303 > expei_cutoff) {
                    t18317 = 1.0;
                    t18318 = expei(t18303);
                } else {
                    t18317 = exp(t18303);
                    t18318 = Ei(-t18303);
                }
                double t18319 = c*c0*t18100*t18300*t18301*t18302*t18248*(8.0/2.7E1);
                double t18320 = t18288*t18288;
                double t18321 = 1.0/(t18251*t18251*t18251*t18251);
                double t18322 = c*c0*t18300*t18301*t18320*t18321*t18092*t18248*4.0;
                double t18323 = a3*c0*gamma_bb*t18300*t18301*t18302*t18091*t18168*(5.6E1/9.0);
                double t18324 = a3*c0*gamma_bb*t18300*t18301*t18302*t18091*t18093*t18094*t18096*t18169*(8.96E2/2.7E1);
                double t18325 = t18093*t18093;
                double t18326 = rab1*rab1;
                double t18327 = 1.0/(rho_b*rho_b*rho_b);
                double t18328 = 1.0/pow(t18097,2.0/5.0);
                double t18329 = t18311*t18311;
                double t18330 = c*c0*t18300*t18301*t18302*t18170*t18153*t18154*t18325*t18248*(2.048E3/2.7E1);
                double t18331 = c*c0*t18300*t18301*t18302*t18171*t18093*t18094*t18248*t18096*(4.48E2/9.0);
                double t18332 = c*c0*kab1*t18300*t18310*t18302*t18145*t18244*t18326*t18327*t18328*t18248*(6.4E1/6.75E2);
                double t18333 = kab1*kab1;
                double t18340 = rab1*t18243*2.0;
                double t18334 = exp(-t18340);
                double t18335 = 1.0/(t18246*t18246*t18246*t18246);
                double t18336 = c*c0*t18300*t18302*t18333*t18145*t18334*t18326*t18335*t18327*t18328*t18248*(6.4E1/2.25E2);
                double t18337 = c*c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18327*t18248*t18159*(1.6E1/6.75E2);
                double t18338 = c0*kab1*rab1*t18300*t18310*t18145*t18244*t18307*t18290*t18248*t18276*t18288*(6.4E1/4.5E1);
                double t18339 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b);
                double t18341 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b);
                double t18342 = t18304*6.0;
                double t18343 = c*c0*t18300*t18301*t18302*t18092*t18248*4.0;
                double t18344 = t18342+t18343+3.0;
                double t18345 = 1.0/t18344;
                double t18347 = c*c0*t18120*t18300*t18301*t18307*t18248*t18288*(8.0/9.0);
                double t18348 = c*c0*t18300*t18301*t18307*t18316*t18092*t18248*(4.0/3.0);
                double t18349 = a3*c0*gamma_bb*t18300*t18301*t18124*t18091*t18307*t18288*(5.6E1/9.0);
                double t18350 = c*c0*t18300*t18301*t18131*t18307*t18093*t18094*t18248*t18096*t18288*(2.56E2/9.0);
                double t18351 = c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18146*t18290*t18248*(3.2E1/4.5E1);
                double t18352 = a3*c*c0*gamma_bb*kab1*rab1*t18300*t18310*t18302*t18091*t18244*t18290*t18339*(2.24E2/1.35E2);
                double t18353 = c0*kab1*rab1*t18300*t18310*t18302*t18341*t18145*t18244*t18290*t18093*t18094*t18248*t18096*(1.024E3/1.35E2);
                double t18346 = t18330+t18322+t18331+t18323+t18332-t18350+t18324-t18351-t18352-t18353+t18336+t18319+t18337+t18338-t18347-t18348-t18349;
                double t18354 = 1.0/sqrt(t18303);
                double t18355 = t18304*4.0;
                double t18356 = c*c0*t18300*t18301*t18302*t18092*t18248*(8.0/3.0);
                double t18357 = t18355+t18356+6.0;
                double t18358 = c*c0*t18120*t18300*t18301*t18302*t18317*t18318*t18248*(2.0/9.0);
                double t18359 = a3*c0*gamma_bb*t18300*t18301*t18302*t18124*t18091*t18317*t18318*(1.4E1/9.0);
                double t18360 = c*c0*t18300*t18301*t18302*t18311*t18092*t18317*t18318*t18248*(2.0/3.0);
                double t18361 = c*c0*t18300*t18301*t18131*t18302*t18317*t18093*t18318*t18094*t18248*t18096*(6.4E1/9.0);
                double t18383 = c*c0*t18300*t18301*t18307*t18092*t18317*t18318*t18248*t18288*(4.0/3.0);
                double t18384 = c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18290*t18317*t18318*t18248*t18276*(1.6E1/4.5E1);
                double t18362 = -t18312-t18313+t18306+t18360+t18361+t18308+t18309-t18383-t18384+t18358+t18359;
                double t18363 = c*c0*t18300*t18301*t18302*t18092*t18317*t18318*t18248*(2.0/3.0);
                double t18364 = t18363+1.0;
                double t18365 = 1.0/(t18344*t18344);
                double t18366 = 1.0/pow(t18303,3.0/2.0);
                double t18367 = t18311*t18354*3.0;
                double t18368 = c*c0*t18120*t18300*t18301*t18302*t18248*(4.0/3.0);
                double t18369 = a3*c0*gamma_bb*t18300*t18301*t18302*t18124*t18091*(2.8E1/3.0);
                double t18370 = c*c0*t18300*t18301*t18131*t18302*t18093*t18094*t18248*t18096*(1.28E2/3.0);
                double t18372 = c*c0*t18300*t18301*t18307*t18092*t18248*t18288*8.0;
                double t18373 = c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18290*t18248*t18276*(3.2E1/1.5E1);
                double t18371 = t18370-t18372-t18373+t18367+t18368+t18369;
                double t18374 = t18311*t18354*2.0;
                double t18375 = c*c0*t18120*t18300*t18301*t18302*t18248*(8.0/9.0);
                double t18376 = a3*c0*gamma_bb*t18300*t18301*t18302*t18124*t18091*(5.6E1/9.0);
                double t18377 = c*c0*t18300*t18301*t18131*t18302*t18093*t18094*t18248*t18096*(2.56E2/9.0);
                double t18385 = c*c0*t18300*t18301*t18307*t18092*t18248*t18288*(1.6E1/3.0);
                double t18386 = c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18290*t18248*t18276*(6.4E1/4.5E1);
                double t18378 = t18374+t18375+t18376-t18385+t18377-t18386;
                double t18379 = 1.0/t18248;
                double t18380 = exp(-t18299);
                double t18381 = t18246*t18246;
                double t18382 = t18251*t18251;
                double t18387 = dirac(t18284,1.0);
                double t18388 = t18293*t18293;
                double t18389 = t18317*t18318;
                double t18390 = t18345*t18364*t18357;
                double t18391 = t18390+t18389;
                double t18392 = c*c0*t18100*(8.0/2.7E1);
                double t18393 = t18224*t18251*t18260*t18093*t18246*t18094*t18096*t18249*(3.04E7/9.0);
                double t18394 = t18260*t18155*t18093*t18246*t18094*t18096*t18249*t18288*(3.2E6/3.0);
                double t18395 = kab1*t18251*t18260*t18145*t18244*t18146*t18326*t18328*t18249*(6.4E4/9.0);
                double t18396 = a3*a3;
                double t18397 = 1.0/pow(t18248,5.0/2.0);
                double t18398 = 1.0/pow(rho_b,1.3E1/3.0);
                double t18399 = a3*gamma_bb*t18251*t18260*t18091*t18246*t18291*t18099*t18398*(3.5E6/9.0);
                double t18400 = a3*gamma_bb*t18260*t18270*t18091*t18246*t18291*t18099*t18288*(7.0E5/3.0);
                double t18401 = kab1*rab1*t18251*t18260*t18145*t18244*t18146*t18159*t18249*(1.6E4/9.0);
                double t18402 = 1.0/pow(rho_b,2.9E1/3.0);
                double t18403 = a3*gamma_bb*kab1*rab1*t18251*t18260*t18091*t18244*t18290*t18291*t18168*(5.6E5/9.0);
                double t18404 = c*kab1*rab1*t18251*t18260*t18171*t18244*t18290*t18093*t18094*t18096*t18249*(2.56E6/9.0);
                double t18405 = t18400+t18401+t18403+t18404+t18392+t18393+t18394+t18395+t18399-t18260*t18316*t18246*t18249*1.0E5-t18251*t18260*t18153*t18225*t18154*t18325*t18246*t18249*(2.56E7/9.0)-c*kab1*rab1*t18100*t18251*t18260*t18244*t18290*t18249*(3.2E5/9.0)-c*kab1*rab1*t18120*t18260*t18244*t18290*t18249*t18288*(1.6E5/3.0)-t18131*t18251*t18260*t18246*t18094*t18096*t18295*t18396*t18397*(1.225E6/3.0)-a3*gamma_bb*t18402*t18251*t18260*t18091*t18093*t18246*t18291*t18094*t18096*t18099*(1.12E7/9.0);
                v_rho_b_rho_b[Q] += scale * (-rho_a*(c0*t18278*(t18345*t18357*(t18330+t18322+t18331+t18323+t18332-t18350+t18324-t18351-t18352-t18353+t18336+t18319+t18337+t18329+t18338-t18347-t18348-t18349+t18311*t18305*(1.0/3.0)-(t18311*t18288*2.0)/t18251+t18311*t18155*t18093*t18094*t18096*(3.2E1/3.0)+a3*gamma_bb*t18311*t18270*t18091*t18099*t18379*(7.0/3.0)-(c*kab1*rab1*t18120*t18311*t18244*t18290*(8.0/1.5E1))/t18246+c*c0*t18100*t18300*t18301*t18302*t18317*t18318*t18248*(8.0/2.7E1)+a3*c0*gamma_bb*t18300*t18301*t18302*t18091*t18317*t18318*t18168*(5.6E1/9.0)+c*c0*t18120*t18300*t18301*t18302*t18311*t18317*t18318*t18248*(4.0/9.0)+c*c0*t18300*t18301*t18320*t18321*t18092*t18317*t18318*t18248*4.0+c*c0*t18300*t18301*t18302*t18092*t18317*t18318*t18346*t18248*(2.0/3.0)+c*c0*t18300*t18301*t18302*t18092*t18317*t18318*t18248*t18329*(2.0/3.0)-c*c0*t18120*t18300*t18301*t18307*t18317*t18318*t18248*t18288*(8.0/9.0)-c*c0*t18300*t18301*t18307*t18316*t18092*t18317*t18318*t18248*(4.0/3.0)+a3*c0*gamma_bb*t18300*t18301*t18302*t18311*t18124*t18091*t18317*t18318*(2.8E1/9.0)-a3*c0*gamma_bb*t18300*t18301*t18124*t18091*t18307*t18317*t18318*t18288*(5.6E1/9.0)-c*c0*t18300*t18301*t18311*t18307*t18092*t18317*t18318*t18248*t18288*(8.0/3.0)+c*c0*t18300*t18301*t18302*t18170*t18153*t18154*t18325*t18317*t18318*t18248*(2.048E3/2.7E1)+c*c0*t18300*t18301*t18302*t18171*t18317*t18093*t18318*t18094*t18248*t18096*(4.48E2/9.0)+a3*c0*gamma_bb*t18300*t18301*t18302*t18091*t18317*t18093*t18318*t18094*t18096*t18169*(8.96E2/2.7E1)+c*c0*t18300*t18301*t18131*t18302*t18311*t18317*t18093*t18318*t18094*t18248*t18096*(1.28E2/9.0)-c*c0*t18300*t18301*t18131*t18307*t18317*t18093*t18318*t18094*t18248*t18096*t18288*(2.56E2/9.0)-c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18146*t18290*t18317*t18318*t18248*(3.2E1/4.5E1)+c*c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18317*t18318*t18327*t18248*t18159*(1.6E1/6.75E2)+c*c0*kab1*t18300*t18310*t18302*t18145*t18244*t18317*t18326*t18318*t18327*t18328*t18248*(6.4E1/6.75E2)+c*c0*t18300*t18302*t18333*t18145*t18334*t18317*t18326*t18335*t18318*t18327*t18328*t18248*(6.4E1/2.25E2)-c0*kab1*rab1*t18300*t18310*t18302*t18311*t18145*t18244*t18290*t18317*t18318*t18248*t18276*(3.2E1/4.5E1)+c0*kab1*rab1*t18300*t18310*t18145*t18244*t18307*t18290*t18317*t18318*t18248*t18276*t18288*(6.4E1/4.5E1)-a3*c*c0*gamma_bb*kab1*rab1*t18300*t18310*t18302*t18091*t18244*t18290*t18317*t18318*t18339*(2.24E2/1.35E2)-c0*kab1*rab1*t18300*t18310*t18302*t18341*t18145*t18244*t18290*t18317*t18093*t18318*t18094*t18248*t18096*(1.024E3/1.35E2))+t18317*t18318*t18346+t18317*t18318*t18329+t18362*t18345*t18378*2.0+t18345*t18364*(t18354*t18346*2.0-t18329*t18366+c*c0*t18100*t18300*t18301*t18302*t18248*(3.2E1/2.7E1)+a3*c0*gamma_bb*t18300*t18301*t18302*t18091*t18168*(2.24E2/9.0)+c*c0*t18300*t18301*t18320*t18321*t18092*t18248*1.6E1-c*c0*t18120*t18300*t18301*t18307*t18248*t18288*(3.2E1/9.0)-c*c0*t18300*t18301*t18307*t18316*t18092*t18248*(1.6E1/3.0)-a3*c0*gamma_bb*t18300*t18301*t18124*t18091*t18307*t18288*(2.24E2/9.0)+c*c0*t18300*t18301*t18302*t18170*t18153*t18154*t18325*t18248*(8.192E3/2.7E1)+c*c0*t18300*t18301*t18302*t18171*t18093*t18094*t18248*t18096*(1.792E3/9.0)+a3*c0*gamma_bb*t18300*t18301*t18302*t18091*t18093*t18094*t18096*t18169*(3.584E3/2.7E1)-c*c0*t18300*t18301*t18131*t18307*t18093*t18094*t18248*t18096*t18288*(1.024E3/9.0)-c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18146*t18290*t18248*(1.28E2/4.5E1)+c*c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18327*t18248*t18159*(6.4E1/6.75E2)+c*c0*kab1*t18300*t18310*t18302*t18145*t18244*t18326*t18327*t18328*t18248*(2.56E2/6.75E2)+c*c0*t18300*t18302*t18333*t18145*t18334*t18326*t18335*t18327*t18328*t18248*(2.56E2/2.25E2)+c0*kab1*rab1*t18300*t18310*t18145*t18244*t18307*t18290*t18248*t18276*t18288*(2.56E2/4.5E1)-a3*c*c0*gamma_bb*kab1*rab1*t18300*t18310*t18302*t18091*t18244*t18290*t18339*(8.96E2/1.35E2)-c0*kab1*rab1*t18300*t18310*t18302*t18341*t18145*t18244*t18290*t18093*t18094*t18248*t18096*(4.096E3/1.35E2))-t18362*t18371*t18365*t18357*2.0-t18371*t18364*t18365*t18378*2.0-t18364*t18365*t18357*(t18354*t18346*3.0-t18329*t18366*(3.0/2.0)+c*c0*t18100*t18300*t18301*t18302*t18248*(1.6E1/9.0)+a3*c0*gamma_bb*t18300*t18301*t18302*t18091*t18168*(1.12E2/3.0)+c*c0*t18300*t18301*t18320*t18321*t18092*t18248*2.4E1-c*c0*t18120*t18300*t18301*t18307*t18248*t18288*(1.6E1/3.0)-c*c0*t18300*t18301*t18307*t18316*t18092*t18248*8.0-a3*c0*gamma_bb*t18300*t18301*t18124*t18091*t18307*t18288*(1.12E2/3.0)+c*c0*t18300*t18301*t18302*t18170*t18153*t18154*t18325*t18248*(4.096E3/9.0)+c*c0*t18300*t18301*t18302*t18171*t18093*t18094*t18248*t18096*(8.96E2/3.0)+a3*c0*gamma_bb*t18300*t18301*t18302*t18091*t18093*t18094*t18096*t18169*(1.792E3/9.0)-c*c0*t18300*t18301*t18131*t18307*t18093*t18094*t18248*t18096*t18288*(5.12E2/3.0)-c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18146*t18290*t18248*(6.4E1/1.5E1)+c*c0*kab1*rab1*t18300*t18310*t18302*t18145*t18244*t18327*t18248*t18159*(3.2E1/2.25E2)+c*c0*kab1*t18300*t18310*t18302*t18145*t18244*t18326*t18327*t18328*t18248*(1.28E2/2.25E2)+c*c0*t18300*t18302*t18333*t18145*t18334*t18326*t18335*t18327*t18328*t18248*(1.28E2/7.5E1)+c0*kab1*rab1*t18300*t18310*t18145*t18244*t18307*t18290*t18248*t18276*t18288*(1.28E2/1.5E1)-a3*c*c0*gamma_bb*kab1*rab1*t18300*t18310*t18302*t18091*t18244*t18290*t18339*(4.48E2/4.5E1)-c0*kab1*rab1*t18300*t18310*t18302*t18341*t18145*t18244*t18290*t18093*t18094*t18248*t18096*(2.048E3/4.5E1))+1.0/(t18344*t18344*t18344)*(t18371*t18371)*t18364*t18357*2.0-t18230*t18311*t18380*t18228*t18381*t18382*t18099*t18379*(1.0/2.0)+t18230*t18234*t18380*t18381*t18346*t18382*t18099*t18379*(3.0/2.0)+t18230*t18234*t18380*t18381*t18382*t18329*t18099*t18379*(3.0/2.0)+t18230*t18311*t18251*t18234*t18380*t18381*t18099*t18288*t18379*3.0+kab1*rab1*t18230*t18311*t18305*t18244*t18290*t18380*t18246*t18382*t18379*(4.0/5.0)-t18230*t18311*t18240*t18380*t18093*t18381*t18094*t18382*t18096*t18099*t18379*1.6E1-a3*gamma_bb*t18230*t18311*t18091*t18380*t18327*t18381*t18382*1.0/(t18248*t18248)*t18295*(7.0/2.0))*(-1.0/2.0)+c0*t18405*t18391*t18294*(1.0/2.0)+c0*t18391*t18387*t18388*(1.0/2.0)+c0*t18293*t18294*(t18311*t18317*t18318+t18362*t18345*t18357+t18345*t18364*t18378-t18371*t18364*t18365*t18357+t18230*t18311*t18234*t18380*t18381*t18382*t18099*t18379*(3.0/2.0)))-rho_b*(c0*t18256*t18278*(t18143*t18190*(-t18222*t18223*t18117+t18221*t18117*t18144*2.0+t18123*t18152*t18153*t18225*t18154*t18138*(2.048E3/9.0)-t18122*t18123*t18224*t18094*t18096*t18098*(1.216E3/9.0)+c*c0*t18100*t18103*t18122*t18107*t18116*(3.2E1/2.7E1)-t18122*t18151*t18144*t18155*t18094*t18096*t18098*(6.4E1/3.0)+a5*c0*gamma_bb*t18103*t18122*t18116*t18091*t18168*(2.24E2/9.0)+c*c0*t18120*t18103*t18130*t18122*t18107*t18129*(3.2E1/9.0)-c*c0*t18103*t18130*t18122*t18107*t18092*t18165*(1.6E1/3.0)+c*c0*t18103*t18122*t18107*t18092*t18166*t18167*1.6E1+a5*c0*gamma_bb*t18103*t18130*t18122*t18124*t18091*t18129*(2.24E2/9.0)+c*c0*t18103*t18122*t18107*t18116*t18152*t18170*t18153*t18154*(8.192E3/2.7E1)-c*c0*t18103*t18107*t18116*t18152*t18170*t18153*t18154*t18138*(1.6384E4/2.7E1)+c*c0*t18103*t18107*t18116*t18152*t18170*t18153*t18154*t18177*(4.096E3/9.0)+c*c0*t18103*t18122*t18107*t18116*t18171*t18094*t18096*t18098*(1.792E3/9.0)-c*c0*t18103*t18107*t18116*t18171*t18138*t18094*t18096*t18098*(1.792E3/9.0)+a5*c0*gamma_bb*t18103*t18122*t18116*t18091*t18094*t18096*t18169*t18098*(3.584E3/2.7E1)-a5*c0*gamma_bb*t18103*t18116*t18091*t18138*t18094*t18096*t18169*t18098*(3.584E3/2.7E1)+c*c0*t18103*t18130*t18122*t18131*t18107*t18129*t18094*t18096*t18098*(1.024E3/9.0)-c*c0*t18103*t18130*t18131*t18107*t18129*t18138*t18094*t18096*t18098*(1.024E3/9.0))+t18133*t18134*t18176+t18241*t18143*t18196*2.0+t18133*t18134*t18188+t18200*t18143*(t18180+t18181-t18235+t18182-t18236+t18183-t18237+t18184-t18238+t18176+t18185-t18239+t18186+t18178+t18187+t18179+t18305*t18172*(1.0/3.0)+(t18172*t18129*2.0)/t18115+t18172*t18155*t18094*t18096*t18098*(3.2E1/3.0)-t18117*t18172*t18155*t18094*t18096*t18098*(3.2E1/3.0)+a5*gamma_bb*t18270*t18091*t18172*t18229*t18099*(7.0/3.0)+c*c0*t18100*t18103*t18122*t18133*t18107*t18116*t18134*(8.0/2.7E1)+a5*c0*gamma_bb*t18103*t18122*t18133*t18116*t18134*t18091*t18168*(5.6E1/9.0)+c*c0*t18120*t18103*t18130*t18122*t18133*t18107*t18134*t18129*(8.0/9.0)+c*c0*t18120*t18103*t18122*t18133*t18107*t18116*t18134*t18172*(4.0/9.0)-c*c0*t18103*t18130*t18122*t18133*t18107*t18134*t18092*t18165*(4.0/3.0)+c*c0*t18103*t18122*t18133*t18107*t18116*t18134*t18092*t18176*(2.0/3.0)+c*c0*t18103*t18122*t18133*t18107*t18116*t18134*t18092*t18188*(2.0/3.0)+c*c0*t18103*t18122*t18133*t18107*t18134*t18092*t18166*t18167*4.0+a5*c0*gamma_bb*t18103*t18130*t18122*t18124*t18133*t18134*t18091*t18129*(5.6E1/9.0)+a5*c0*gamma_bb*t18103*t18122*t18124*t18133*t18116*t18134*t18091*t18172*(2.8E1/9.0)+c*c0*t18103*t18130*t18122*t18133*t18107*t18134*t18172*t18092*t18129*(8.0/3.0)+c*c0*t18103*t18122*t18133*t18107*t18116*t18134*t18152*t18170*t18153*t18154*(2.048E3/2.7E1)-c*c0*t18103*t18133*t18107*t18116*t18134*t18152*t18170*t18153*t18154*t18138*(4.096E3/2.7E1)+c*c0*t18103*t18133*t18107*t18116*t18134*t18152*t18170*t18153*t18154*t18177*(1.024E3/9.0)+c*c0*t18103*t18122*t18133*t18107*t18116*t18134*t18171*t18094*t18096*t18098*(4.48E2/9.0)-c*c0*t18103*t18133*t18107*t18116*t18134*t18171*t18138*t18094*t18096*t18098*(4.48E2/9.0)+a5*c0*gamma_bb*t18103*t18122*t18133*t18116*t18134*t18091*t18094*t18096*t18169*t18098*(8.96E2/2.7E1)-a5*c0*gamma_bb*t18103*t18133*t18116*t18134*t18091*t18138*t18094*t18096*t18169*t18098*(8.96E2/2.7E1)+c*c0*t18103*t18130*t18122*t18131*t18133*t18107*t18134*t18129*t18094*t18096*t18098*(2.56E2/9.0)+c*c0*t18103*t18122*t18131*t18133*t18107*t18116*t18134*t18172*t18094*t18096*t18098*(1.28E2/9.0)-c*c0*t18103*t18130*t18131*t18133*t18107*t18134*t18129*t18138*t18094*t18096*t18098*(2.56E2/9.0)-c*c0*t18103*t18131*t18133*t18107*t18116*t18134*t18172*t18138*t18094*t18096*t18098*(1.28E2/9.0))-t18200*t18190*t18197*(t18222*t18223*t18117*(-3.0/2.0)+t18221*t18117*t18144*3.0+t18123*t18152*t18153*t18225*t18154*t18138*(1.024E3/3.0)-t18122*t18123*t18224*t18094*t18096*t18098*(6.08E2/3.0)+c*c0*t18100*t18103*t18122*t18107*t18116*(1.6E1/9.0)-t18122*t18151*t18144*t18155*t18094*t18096*t18098*3.2E1+a5*c0*gamma_bb*t18103*t18122*t18116*t18091*t18168*(1.12E2/3.0)+c*c0*t18120*t18103*t18130*t18122*t18107*t18129*(1.6E1/3.0)-c*c0*t18103*t18130*t18122*t18107*t18092*t18165*8.0+c*c0*t18103*t18122*t18107*t18092*t18166*t18167*2.4E1+a5*c0*gamma_bb*t18103*t18130*t18122*t18124*t18091*t18129*(1.12E2/3.0)+c*c0*t18103*t18122*t18107*t18116*t18152*t18170*t18153*t18154*(4.096E3/9.0)-c*c0*t18103*t18107*t18116*t18152*t18170*t18153*t18154*t18138*(8.192E3/9.0)+c*c0*t18103*t18107*t18116*t18152*t18170*t18153*t18154*t18177*(2.048E3/3.0)+c*c0*t18103*t18122*t18107*t18116*t18171*t18094*t18096*t18098*(8.96E2/3.0)-c*c0*t18103*t18107*t18116*t18171*t18138*t18094*t18096*t18098*(8.96E2/3.0)+a5*c0*gamma_bb*t18103*t18122*t18116*t18091*t18094*t18096*t18169*t18098*(1.792E3/9.0)-a5*c0*gamma_bb*t18103*t18116*t18091*t18138*t18094*t18096*t18169*t18098*(1.792E3/9.0)+c*c0*t18103*t18130*t18122*t18131*t18107*t18129*t18094*t18096*t18098*(5.12E2/3.0)-c*c0*t18103*t18130*t18131*t18107*t18129*t18138*t18094*t18096*t18098*(5.12E2/3.0))-t18200*t18241*t18206*t18197*2.0-t18206*t18190*t18196*t18197*2.0+t18200*1.0/(t18142*t18142*t18142)*(t18206*t18206)*t18190*2.0-t18230*t18231*t18232*t18233*t18172*t18228*t18229*t18099*(1.0/2.0)+t18230*t18231*t18232*t18233*t18234*t18229*t18176*t18099*(3.0/2.0)+t18230*t18231*t18232*t18233*t18234*t18229*t18188*t18099*(3.0/2.0)-t18230*t18231*t18115*t18232*t18234*t18172*t18129*t18229*t18099*3.0+t18230*t18105*t18231*t18240*t18233*t18172*t18094*t18229*t18096*t18098*t18099*1.6E1-t18230*t18231*t18240*t18232*t18233*t18172*t18094*t18229*t18096*t18098*t18099*1.6E1-a5*gamma_bb*t18230*t18231*t18232*1.0/(t18107*t18107)*t18233*t18091*t18172*t18327*t18295*(7.0/2.0))*(-1.0/2.0)+c0*t18281*t18256*t18278*(t18145*t18146*t18255*(1.0E1/9.0)-t18145*t18272*t18275*t18276*(8.0/3.0)+t18145*1.0/(t18254*t18254*t18254*t18254)*t18228*(t18275*t18275)*6.0-t18145*t18272*t18228*(c*k2*t18100*(4.0/9.0)+c*k1*t18100*t18125*(2.0/9.0)-k1*t18161*t18145*t18146*(1.0/3.6E1))*2.0)*(1.0/2.0)+c0*t18405*t18281*t18256*t18294*(1.0/2.0)+c0*t18256*t18293*t18294*t18269+c0*t18256*t18283*t18269*t18278+c0*t18281*t18256*t18387*t18388*(1.0/2.0)-c0*t18281*t18256*(t18283*t18283)*t18278*(1.0/2.0)-c0*t18281*t18256*t18283*t18293*t18294)-c0*t18256*t18269*t18278+c0*t18281*t18256*t18293*t18294+c0*t18281*t18256*t18283*t18278);
            }
            
            // v_gamma_aa_gamma_aa
            if (deriv >= 2) {
                double t19160 = tspi_m13*tspi_m13;
                double t19161 = 1.0/pow(rho_a,1.0/3.0);
                double t19162 = a2*a2;
                double t19163 = gamma_aa*gamma_aa;
                double t19164 = 1.0/pow(rho_a,1.6E1/3.0);
                double t19165 = t19160*t19160;
                double t19166 = c*t19161;
                double t19167 = pow(t19166,4.0/5.0);
                double t19181 = rab1*t19167;
                double t19168 = exp(-t19181);
                double t19169 = t19168-1.0;
                double t19182 = kab1*t19169;
                double t19170 = kab0-t19182;
                double t19171 = 1.0/c;
                double t19172 = 1.0/pow(rho_a,7.0/3.0);
                double t19173 = a3*gamma_aa*t19160*t19171*t19172;
                double t19174 = t19173+1.0;
                double t19175 = t19162*t19163*t19164*t19165;
                double t19176 = 1.0/pow(rho_a,8.0/3.0);
                double t19177 = a1*gamma_aa*t19160*t19176;
                double t19178 = t19175+t19177+1.0;
                double t19179 = t19162*t19163*t19164*t19165*2.0;
                double t19180 = exp(t19179);
                double t19183 = 1.0/(t19170*t19170);
                double t19184 = 1.0/(t19178*t19178);
                double t19185 = c*c0*t19161*t19180*t19174*t19183*t19184*(2.0/3.0);
                double t19186 = a1*t19160*t19176;
                double t19187 = gamma_aa*t19162*t19164*t19165*2.0;
                double t19188 = t19186+t19187;
                double t19189 = 1.0/pow(rho_a,1.7E1/3.0);
                double t19190 = 1.0/(t19178*t19178*t19178);
                double t19191 = sqrt(t19185);
                double t19193 = a3*c0*t19160*t19180*t19183*t19184*t19176*(2.0/3.0);
                double t19194 = c*c0*t19161*t19180*t19190*t19174*t19183*t19188*(4.0/3.0);
                double t19195 = c*c0*gamma_aa*t19162*t19180*t19165*t19174*t19183*t19184*t19189*(8.0/3.0);
                double t19192 = t19193-t19194+t19195;
                double t19196 = t19188*t19188;
                double t19197 = 1.0/(t19178*t19178*t19178*t19178);
                double t19198 = c*c0*t19161*t19180*t19174*t19183*t19196*t19197*4.0;
                double t19199 = c*c0*t19162*t19180*t19165*t19174*t19183*t19184*t19189*(8.0/3.0);
                double t19200 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a);
                double t19201 = a3*c0*gamma_aa*t19200*t19160*t19162*t19180*t19165*t19183*t19184*(1.6E1/3.0);
                double t19202;
                double t19203;
                if (t19185 > expei_cutoff) {
                    t19202 = 1.0;
                    t19203 = expei(t19185);
                } else {
                    t19202 = exp(t19185);
                    t19203 = Ei(-t19185);
                }
                double t19204 = t19162*t19162;
                double t19205 = 1.0/pow(rho_a,1.1E1);
                double t19206 = t19165*t19165;
                double t19207 = c*c0*t19204*t19205*t19206*t19180*t19163*t19174*t19183*t19184*(3.2E1/3.0);
                double t19210 = c*c0*gamma_aa*t19162*t19180*t19190*t19165*t19174*t19183*t19188*t19189*(3.2E1/3.0);
                double t19215 = a3*c0*t19160*t19180*t19190*t19183*t19176*t19188*(8.0/3.0);
                double t19216 = c*c0*t19162*t19180*t19190*t19165*t19174*t19183*t19189*(8.0/3.0);
                double t19208 = t19201-t19210-t19215+t19207-t19216+t19198+t19199;
                double t19209 = t19192*t19192;
                double t19211 = t19191*6.0;
                double t19212 = c*c0*t19161*t19180*t19174*t19183*t19184*4.0;
                double t19213 = t19211+t19212+3.0;
                double t19214 = 1.0/t19213;
                double t19217 = 1.0/sqrt(t19185);
                double t19218 = c*c0*t19202*t19203*t19161*t19180*t19174*t19183*t19184*(2.0/3.0);
                double t19219 = t19218+1.0;
                double t19220 = t19217*t19192*2.0;
                double t19221 = a3*c0*t19160*t19180*t19183*t19184*t19176*(8.0/3.0);
                double t19222 = c*c0*gamma_aa*t19162*t19180*t19165*t19174*t19183*t19184*t19189*(3.2E1/3.0);
                double t19259 = c*c0*t19161*t19180*t19190*t19174*t19183*t19188*(1.6E1/3.0);
                double t19223 = t19220+t19221+t19222-t19259;
                double t19224 = 1.0/(t19213*t19213);
                double t19225 = t19191*4.0;
                double t19226 = c*c0*t19161*t19180*t19174*t19183*t19184*(8.0/3.0);
                double t19227 = t19225+t19226+6.0;
                double t19228 = 1.0/pow(t19185,3.0/2.0);
                double t19229 = t19217*t19192*3.0;
                double t19230 = a3*c0*t19160*t19180*t19183*t19184*t19176*4.0;
                double t19231 = c*c0*gamma_aa*t19162*t19180*t19165*t19174*t19183*t19184*t19189*1.6E1;
                double t19233 = c*c0*t19161*t19180*t19190*t19174*t19183*t19188*8.0;
                double t19232 = t19230+t19231-t19233+t19229;
                double t19234 = a3*c0*t19202*t19203*t19160*t19180*t19183*t19184*t19176*(2.0/3.0);
                double t19235 = c*c0*t19202*t19203*t19161*t19180*t19174*t19183*t19192*t19184*(2.0/3.0);
                double t19236 = c*c0*gamma_aa*t19202*t19203*t19162*t19180*t19165*t19174*t19183*t19184*t19189*(8.0/3.0);
                double t19260 = c*c0*t19202*t19203*t19161*t19180*t19190*t19174*t19183*t19188*(4.0/3.0);
                double t19237 = -t19260+t19234+t19235+t19236+t19193-t19194+t19195;
                double t19238 = 1.0/t19174;
                double t19239 = 1.0/c0;
                double t19240 = pow(rho_a,1.0/3.0);
                double t19241 = exp(-t19179);
                double t19242 = t19170*t19170;
                double t19243 = t19178*t19178;
                double t19244 = 1.0/sqrt(t19174);
                double t19245 = exp(-t19175);
                double t19252 = 1.0/pow(t19174,3.0/2.0);
                double t19256 = t19170*t19244*t19245*t19188*1.0E5;
                double t19257 = a3*t19160*t19170*t19171*t19252*t19172*t19245*t19178*5.0E4;
                double t19258 = gamma_aa*t19170*t19162*t19244*t19164*t19245*t19165*t19178*2.0E5;
                double t19246 = -t19256+t19257+t19258;
                double t19247 = c*c0*t19161*(2.0/3.0);
                double t19254 = t19170*t19244*t19245*t19178*1.0E5;
                double t19248 = -t19254+t19247;
                double t19249 = t19202*t19203;
                double t19250 = t19214*t19227*t19219;
                double t19251 = t19250+t19249;
                double t19253 = 1.0/(c*c);
                double t19255 = dirac(t19248);
                double t19261 = a4*a4;
                double t19262 = t19261*t19163*t19164*t19165*2.0;
                double t19263 = exp(t19262);
                double t19264 = sqrt(t19166);
                double t19265 = t19261*t19163*t19164*t19165;
                double t19266 = t19265+1.0;
                double t19267 = 1.0/(t19266*t19266);
                double t19268 = a5*gamma_aa*t19160*t19171*t19172;
                double t19269 = t19268+1.0;
                double t19280 = raa2*t19264;
                double t19270 = exp(-t19280);
                double t19271 = t19270-1.0;
                double t19272 = kaa2*t19271;
                double t19273 = pow(t19166,2.0/5.0);
                double t19281 = raa1*t19273;
                double t19274 = exp(-t19281);
                double t19275 = t19274-1.0;
                double t19276 = kaa1*t19275;
                double t19277 = -kaa0+t19272+t19276;
                double t19278 = 1.0/(t19277*t19277);
                double t19279 = c*c0*t19161*t19263*t19267*t19269*t19278*(2.0/3.0);
                double t19282 = 1.0/t19266;
                double t19283 = c*c0*t19161*t19263*t19269*t19278*(2.0/3.0);
                double t19288 = a5*c0*t19160*t19263*t19176*t19278*(2.0/3.0);
                double t19289 = c*c0*gamma_aa*t19261*t19263*t19165*t19269*t19278*t19189*(8.0/3.0);
                double t19284 = t19288+t19289;
                double t19285 = sqrt(t19283);
                double t19286 = t19261*t19261;
                double t19287 = 1.0/pow(rho_a,3.2E1/3.0);
                double t19290 = 1.0/sqrt(t19283);
                double t19291 = 1.0/(t19266*t19266*t19266);
                double t19292 = t19282*t19285*6.0;
                double t19293 = c*c0*t19161*t19263*t19267*t19269*t19278*4.0;
                double t19294 = t19292+t19293+3.0;
                double t19295 = 1.0/t19294;
                double t19297 = a5*c0*t19160*t19263*t19176*t19267*t19278*(2.0/3.0);
                double t19298 = c*c0*gamma_aa*t19261*t19263*t19165*t19267*t19269*t19278*t19189*(8.0/3.0);
                double t19299 = c*c0*gamma_aa*t19261*t19263*t19165*t19291*t19269*t19278*t19189*(8.0/3.0);
                double t19296 = t19297+t19298-t19299;
                double t19300;
                double t19301;
                if (t19279 > expei_cutoff) {
                    t19300 = 1.0;
                    t19301 = expei(t19279);
                } else {
                    t19300 = exp(t19279);
                    t19301 = Ei(-t19279);
                }
                double t19302 = t19296*t19296;
                double t19303 = 1.0/(t19266*t19266*t19266*t19266);
                double t19304 = a5*c0*gamma_aa*t19200*t19160*t19261*t19263*t19165*t19267*t19278*(1.6E1/3.0);
                double t19305 = c*c0*t19261*t19263*t19165*t19267*t19269*t19278*t19189*(8.0/3.0);
                double t19306 = c*c0*t19205*t19206*t19163*t19263*t19267*t19286*t19269*t19278*(3.2E1/3.0);
                double t19307 = c*c0*t19303*t19205*t19206*t19163*t19263*t19286*t19269*t19278*1.6E1;
                double t19339 = a5*c0*gamma_aa*t19200*t19160*t19261*t19263*t19165*t19291*t19278*(1.6E1/3.0);
                double t19340 = c*c0*t19261*t19263*t19165*t19291*t19269*t19278*t19189*(8.0/3.0);
                double t19341 = c*c0*t19205*t19206*t19163*t19263*t19291*t19286*t19269*t19278*(6.4E1/3.0);
                double t19308 = t19304-t19340+t19305-t19341+t19306+t19307-t19339;
                double t19309 = t19282*t19285*4.0;
                double t19310 = c*c0*t19161*t19263*t19267*t19269*t19278*(8.0/3.0);
                double t19311 = t19310+t19309+6.0;
                double t19312 = a5*c0*t19300*t19301*t19160*t19263*t19176*t19267*t19278*(2.0/3.0);
                double t19313 = c*c0*t19300*t19301*t19161*t19263*t19267*t19269*t19278*t19296*(2.0/3.0);
                double t19314 = c*c0*gamma_aa*t19300*t19301*t19261*t19263*t19165*t19267*t19269*t19278*t19189*(8.0/3.0);
                double t19356 = c*c0*gamma_aa*t19300*t19301*t19261*t19263*t19165*t19291*t19269*t19278*t19189*(8.0/3.0);
                double t19315 = t19312+t19313+t19314-t19356+t19297+t19298-t19299;
                double t19316 = c*c0*t19300*t19301*t19161*t19263*t19267*t19269*t19278*(2.0/3.0);
                double t19317 = t19316+1.0;
                double t19318 = t19290*t19282*t19284*3.0;
                double t19319 = a5*c0*t19160*t19263*t19176*t19267*t19278*4.0;
                double t19320 = c*c0*gamma_aa*t19261*t19263*t19165*t19267*t19269*t19278*t19189*1.6E1;
                double t19323 = gamma_aa*t19261*t19164*t19165*t19267*t19285*1.2E1;
                double t19324 = c*c0*gamma_aa*t19261*t19263*t19165*t19291*t19269*t19278*t19189*1.6E1;
                double t19321 = t19320-t19323-t19324+t19318+t19319;
                double t19322 = 1.0/(t19294*t19294);
                double t19325 = t19290*t19282*t19284*2.0;
                double t19326 = a5*c0*t19160*t19263*t19176*t19267*t19278*(8.0/3.0);
                double t19327 = c*c0*gamma_aa*t19261*t19263*t19165*t19267*t19269*t19278*t19189*(3.2E1/3.0);
                double t19354 = gamma_aa*t19261*t19164*t19165*t19267*t19285*8.0;
                double t19355 = c*c0*gamma_aa*t19261*t19263*t19165*t19291*t19269*t19278*t19189*(3.2E1/3.0);
                double t19328 = t19325+t19326+t19327-t19354-t19355;
                double t19329 = a5*c0*gamma_aa*t19200*t19160*t19261*t19263*t19165*t19278*(1.6E1/3.0);
                double t19330 = c*c0*t19261*t19263*t19165*t19269*t19278*t19189*(8.0/3.0);
                double t19331 = c*c0*t19205*t19206*t19163*t19263*t19286*t19269*t19278*(3.2E1/3.0);
                double t19332 = t19330+t19331+t19329;
                double t19333 = t19284*t19284;
                double t19334 = 1.0/pow(t19283,3.0/2.0);
                double t19335 = 1.0/t19269;
                double t19336 = exp(-t19262);
                double t19337 = t19266*t19266;
                double t19338 = t19277*t19277;
                double t19342 = 1.0/(rho_a*rho_a);
                double t19343 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a);
                double t19344 = c*c;
                double t19345 = 1.0/pow(rho_a,2.0/3.0);
                double t19346 = k1*t19264;
                double t19347 = c*k2*t19161;
                double t19348 = t19346+t19347;
                double t19349 = 1.0/(t19348*t19348);
                double t19353 = t19344*t19345*t19349;
                double t19350 = exp(-t19353);
                double t19351 = dirac(t19248,1.0);
                double t19352 = t19246*t19246;
                double t19357 = t19300*t19301;
                double t19358 = t19311*t19317*t19295;
                double t19359 = t19357+t19358;
                double t19360 = t19170*t19162*t19244*t19164*t19245*t19165*2.0E5;
                double t19361 = a3*a3;
                double t19362 = 1.0/pow(rho_a,1.4E1/3.0);
                double t19363 = 1.0/pow(t19174,5.0/2.0);
                double t19364 = t19170*t19253*t19361*t19245*t19362*t19165*t19363*t19178*7.5E4;
                double t19365 = t19204*t19170*t19206*t19163*t19244*t19245*t19178*t19287*4.0E5;
                double t19366 = 1.0/pow(rho_a,2.3E1/3.0);
                double t19367 = a3*gamma_aa*t19160*t19170*t19162*t19171*t19252*t19245*t19165*t19366*t19178*2.0E5;
                double t19368 = t19360+t19364+t19365+t19367-t19170*t19162*t19244*t19164*t19245*t19165*t19178*2.0E5-a3*t19160*t19170*t19171*t19252*t19172*t19245*t19188*1.0E5-gamma_aa*t19170*t19162*t19244*t19164*t19245*t19165*t19188*4.0E5;
                v_gamma_aa_gamma_aa[Q] += scale * (rho_b*(c0*heaviside(c*c0*t19161*(-2.0/3.0)+t19170*t19244*t19178*exp(-t19162*t19163*t19164*t19165)*1.0E5)*(t19214*t19219*(t19208*t19217*2.0-t19209*t19228-a3*c0*t19160*t19180*t19190*t19183*t19176*t19188*(3.2E1/3.0)+c*c0*t19161*t19180*t19174*t19183*t19196*t19197*1.6E1-c*c0*t19162*t19180*t19190*t19165*t19174*t19183*t19189*(3.2E1/3.0)+c*c0*t19162*t19180*t19165*t19174*t19183*t19184*t19189*(3.2E1/3.0)+a3*c0*gamma_aa*t19200*t19160*t19162*t19180*t19165*t19183*t19184*(6.4E1/3.0)+c*c0*t19204*t19205*t19206*t19180*t19163*t19174*t19183*t19184*(1.28E2/3.0)-c*c0*gamma_aa*t19162*t19180*t19190*t19165*t19174*t19183*t19188*t19189*(1.28E2/3.0))+t19214*t19227*(t19201-t19210+t19207+t19209+t19198+t19199-(t19192*t19188*2.0)/t19178+gamma_aa*t19162*t19164*t19165*t19192*4.0+a3*t19160*t19171*t19172*t19192*t19238-a3*c0*t19160*t19180*t19190*t19183*t19176*t19188*(8.0/3.0)-c*c0*t19162*t19180*t19190*t19165*t19174*t19183*t19189*(8.0/3.0)+a3*c0*t19202*t19203*t19160*t19180*t19183*t19192*t19184*t19176*(4.0/3.0)-a3*c0*t19202*t19203*t19160*t19180*t19190*t19183*t19176*t19188*(8.0/3.0)+c*c0*t19202*t19203*t19161*t19180*t19208*t19174*t19183*t19184*(2.0/3.0)+c*c0*t19202*t19203*t19161*t19180*t19209*t19174*t19183*t19184*(2.0/3.0)+c*c0*t19202*t19203*t19161*t19180*t19174*t19183*t19196*t19197*4.0-c*c0*t19202*t19203*t19161*t19180*t19190*t19174*t19183*t19192*t19188*(8.0/3.0)-c*c0*t19202*t19203*t19162*t19180*t19190*t19165*t19174*t19183*t19189*(8.0/3.0)+c*c0*t19202*t19203*t19162*t19180*t19165*t19174*t19183*t19184*t19189*(8.0/3.0)+a3*c0*gamma_aa*t19200*t19202*t19203*t19160*t19162*t19180*t19165*t19183*t19184*(1.6E1/3.0)+c*c0*t19202*t19203*t19204*t19205*t19206*t19180*t19163*t19174*t19183*t19184*(3.2E1/3.0)+c*c0*gamma_aa*t19202*t19203*t19162*t19180*t19165*t19174*t19183*t19192*t19184*t19189*(1.6E1/3.0)-c*c0*gamma_aa*t19202*t19203*t19162*t19180*t19190*t19165*t19174*t19183*t19188*t19189*(3.2E1/3.0))+t19202*t19203*t19208+t19202*t19203*t19209+t19214*t19223*t19237*2.0-t19223*t19232*t19224*t19219*2.0-t19232*t19224*t19227*t19237*2.0-t19224*t19227*t19219*(t19208*t19217*3.0-t19209*t19228*(3.0/2.0)-a3*c0*t19160*t19180*t19190*t19183*t19176*t19188*1.6E1+c*c0*t19161*t19180*t19174*t19183*t19196*t19197*2.4E1-c*c0*t19162*t19180*t19190*t19165*t19174*t19183*t19189*1.6E1+c*c0*t19162*t19180*t19165*t19174*t19183*t19184*t19189*1.6E1+a3*c0*gamma_aa*t19200*t19160*t19162*t19180*t19165*t19183*t19184*3.2E1+c*c0*t19204*t19205*t19206*t19180*t19163*t19174*t19183*t19184*6.4E1-c*c0*gamma_aa*t19162*t19180*t19190*t19165*t19174*t19183*t19188*t19189*6.4E1)+1.0/(t19213*t19213*t19213)*(t19232*t19232)*t19227*t19219*2.0+t19240*t19241*t19242*t19171*t19243*t19208*t19238*t19239*(3.0/2.0)+t19240*t19241*t19242*t19171*t19243*t19209*t19238*t19239*(3.0/2.0)+t19240*t19241*t19242*t19171*t19192*t19238*t19239*t19178*t19188*3.0-a3*t19160*t19241*t19242*t19243*t19342*t19253*1.0/(t19174*t19174)*t19192*t19239*(3.0/2.0)-gamma_aa*t19241*t19242*t19162*t19171*t19243*t19343*t19165*t19192*t19238*t19239*6.0)*(1.0/2.0)-c0*t19251*t19351*t19352*(1.0/2.0)+c0*t19251*t19255*t19368*(1.0/2.0)-c0*t19246*t19255*(t19202*t19203*t19192+t19214*t19223*t19219+t19214*t19227*t19237-t19232*t19224*t19227*t19219+t19240*t19241*t19242*t19171*t19243*t19192*t19238*t19239*(3.0/2.0)))+rho_a*(c0*t19350*heaviside(t19254-t19247)*(t19311*t19295*(t19302+t19304+t19305+t19306+t19307+gamma_aa*t19261*t19164*t19165*t19296*4.0+a5*t19160*t19171*t19172*t19335*t19296-gamma_aa*t19261*t19164*t19165*t19282*t19296*4.0-c*c0*t19261*t19263*t19165*t19291*t19269*t19278*t19189*(8.0/3.0)-a5*c0*gamma_aa*t19200*t19160*t19261*t19263*t19165*t19291*t19278*(1.6E1/3.0)+a5*c0*t19300*t19301*t19160*t19263*t19176*t19267*t19278*t19296*(4.0/3.0)+c*c0*t19300*t19301*t19302*t19161*t19263*t19267*t19269*t19278*(2.0/3.0)+c*c0*t19300*t19301*t19161*t19263*t19308*t19267*t19269*t19278*(2.0/3.0)-c*c0*t19205*t19206*t19163*t19263*t19291*t19286*t19269*t19278*(6.4E1/3.0)-c*c0*t19300*t19301*t19261*t19263*t19165*t19291*t19269*t19278*t19189*(8.0/3.0)+c*c0*t19300*t19301*t19261*t19263*t19165*t19267*t19269*t19278*t19189*(8.0/3.0)-a5*c0*gamma_aa*t19200*t19300*t19301*t19160*t19261*t19263*t19165*t19291*t19278*(1.6E1/3.0)+a5*c0*gamma_aa*t19200*t19300*t19301*t19160*t19261*t19263*t19165*t19267*t19278*(1.6E1/3.0)+c*c0*t19300*t19301*t19303*t19205*t19206*t19163*t19263*t19286*t19269*t19278*1.6E1-c*c0*t19300*t19301*t19205*t19206*t19163*t19263*t19291*t19286*t19269*t19278*(6.4E1/3.0)+c*c0*t19300*t19301*t19205*t19206*t19163*t19263*t19267*t19286*t19269*t19278*(3.2E1/3.0)-c*c0*gamma_aa*t19300*t19301*t19261*t19263*t19165*t19291*t19269*t19278*t19296*t19189*(1.6E1/3.0)+c*c0*gamma_aa*t19300*t19301*t19261*t19263*t19165*t19267*t19269*t19278*t19296*t19189*(1.6E1/3.0))+t19300*t19301*t19302+t19300*t19301*t19308+t19315*t19328*t19295*2.0+t19317*t19295*(t19332*t19290*t19282*2.0-t19333*t19334*t19282-t19261*t19164*t19165*t19267*t19285*8.0+t19206*t19163*t19291*t19285*t19286*t19287*3.2E1-gamma_aa*t19261*t19164*t19290*t19165*t19284*t19267*8.0-c*c0*t19261*t19263*t19165*t19291*t19269*t19278*t19189*(3.2E1/3.0)+c*c0*t19261*t19263*t19165*t19267*t19269*t19278*t19189*(3.2E1/3.0)-a5*c0*gamma_aa*t19200*t19160*t19261*t19263*t19165*t19291*t19278*(6.4E1/3.0)+a5*c0*gamma_aa*t19200*t19160*t19261*t19263*t19165*t19267*t19278*(6.4E1/3.0)+c*c0*t19303*t19205*t19206*t19163*t19263*t19286*t19269*t19278*6.4E1-c*c0*t19205*t19206*t19163*t19263*t19291*t19286*t19269*t19278*(2.56E2/3.0)+c*c0*t19205*t19206*t19163*t19263*t19267*t19286*t19269*t19278*(1.28E2/3.0))-t19311*t19321*t19322*t19315*2.0-t19321*t19322*t19317*t19328*2.0-t19311*t19322*t19317*(t19332*t19290*t19282*3.0-t19333*t19334*t19282*(3.0/2.0)-t19261*t19164*t19165*t19267*t19285*1.2E1+t19206*t19163*t19291*t19285*t19286*t19287*4.8E1-gamma_aa*t19261*t19164*t19290*t19165*t19284*t19267*1.2E1-c*c0*t19261*t19263*t19165*t19291*t19269*t19278*t19189*1.6E1+c*c0*t19261*t19263*t19165*t19267*t19269*t19278*t19189*1.6E1-a5*c0*gamma_aa*t19200*t19160*t19261*t19263*t19165*t19291*t19278*3.2E1+a5*c0*gamma_aa*t19200*t19160*t19261*t19263*t19165*t19267*t19278*3.2E1+c*c0*t19303*t19205*t19206*t19163*t19263*t19286*t19269*t19278*9.6E1-c*c0*t19205*t19206*t19163*t19263*t19291*t19286*t19269*t19278*1.28E2+c*c0*t19205*t19206*t19163*t19263*t19267*t19286*t19269*t19278*6.4E1)+t19311*(t19321*t19321)*t19317*1.0/(t19294*t19294*t19294)*2.0+t19302*t19240*t19171*t19335*t19336*t19337*t19239*t19338*(3.0/2.0)+t19240*t19171*t19308*t19335*t19336*t19337*t19239*t19338*(3.0/2.0)-a5*t19160*t19342*t19253*t19336*t19337*t19239*t19338*1.0/(t19269*t19269)*t19296*(3.0/2.0)-gamma_aa*t19171*t19261*t19343*t19335*t19165*t19336*t19337*t19239*t19338*t19296*6.0+gamma_aa*t19171*t19261*t19343*t19335*t19165*t19336*t19239*t19266*t19338*t19296*6.0)*(1.0/2.0)-c0*t19350*t19351*t19352*t19359*(1.0/2.0)+c0*t19350*t19255*t19359*t19368*(1.0/2.0)-c0*t19350*t19246*t19255*(t19300*t19301*t19296+t19311*t19315*t19295+t19317*t19328*t19295-t19311*t19321*t19322*t19317+t19240*t19171*t19335*t19336*t19337*t19239*t19338*t19296*(3.0/2.0))));
            }
            
            // v_gamma_bb_gamma_bb
            if (deriv >= 2) {
                double t19374 = tspi_m13*tspi_m13;
                double t19375 = 1.0/pow(rho_b,1.0/3.0);
                double t19376 = a2*a2;
                double t19377 = gamma_bb*gamma_bb;
                double t19378 = 1.0/pow(rho_b,1.6E1/3.0);
                double t19379 = t19374*t19374;
                double t19380 = c*t19375;
                double t19381 = pow(t19380,4.0/5.0);
                double t19395 = rab1*t19381;
                double t19382 = exp(-t19395);
                double t19383 = t19382-1.0;
                double t19396 = kab1*t19383;
                double t19384 = kab0-t19396;
                double t19385 = 1.0/c;
                double t19386 = 1.0/pow(rho_b,7.0/3.0);
                double t19387 = a3*gamma_bb*t19374*t19385*t19386;
                double t19388 = t19387+1.0;
                double t19389 = t19376*t19377*t19378*t19379;
                double t19390 = 1.0/pow(rho_b,8.0/3.0);
                double t19391 = a1*gamma_bb*t19390*t19374;
                double t19392 = t19391+t19389+1.0;
                double t19393 = t19376*t19377*t19378*t19379*2.0;
                double t19394 = exp(t19393);
                double t19397 = 1.0/(t19384*t19384);
                double t19398 = 1.0/(t19392*t19392);
                double t19399 = c*c0*t19375*t19394*t19388*t19397*t19398*(2.0/3.0);
                double t19400 = a1*t19390*t19374;
                double t19401 = gamma_bb*t19376*t19378*t19379*2.0;
                double t19402 = t19400+t19401;
                double t19403 = 1.0/pow(rho_b,1.7E1/3.0);
                double t19404 = 1.0/(t19392*t19392*t19392);
                double t19405 = sqrt(t19399);
                double t19407 = a3*c0*t19390*t19374*t19394*t19397*t19398*(2.0/3.0);
                double t19408 = c*c0*t19402*t19404*t19375*t19394*t19388*t19397*(4.0/3.0);
                double t19409 = c*c0*gamma_bb*t19403*t19376*t19394*t19379*t19388*t19397*t19398*(8.0/3.0);
                double t19406 = t19407-t19408+t19409;
                double t19410 = t19402*t19402;
                double t19411 = 1.0/(t19392*t19392*t19392*t19392);
                double t19412 = c*c0*t19410*t19411*t19375*t19394*t19388*t19397*4.0;
                double t19413 = c*c0*t19403*t19376*t19394*t19379*t19388*t19397*t19398*(8.0/3.0);
                double t19414 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b);
                double t19415 = a3*c0*gamma_bb*t19414*t19374*t19376*t19394*t19379*t19397*t19398*(1.6E1/3.0);
                double t19416;
                double t19417;
                if (t19399 > expei_cutoff) {
                    t19416 = 1.0;
                    t19417 = expei(t19399);
                } else {
                    t19416 = exp(t19399);
                    t19417 = Ei(-t19399);
                }
                double t19418 = t19376*t19376;
                double t19419 = 1.0/pow(rho_b,1.1E1);
                double t19420 = t19379*t19379;
                double t19421 = c*c0*t19420*t19418*t19419*t19394*t19377*t19388*t19397*t19398*(3.2E1/3.0);
                double t19424 = c*c0*gamma_bb*t19402*t19403*t19404*t19376*t19394*t19379*t19388*t19397*(3.2E1/3.0);
                double t19429 = a3*c0*t19402*t19404*t19390*t19374*t19394*t19397*(8.0/3.0);
                double t19430 = c*c0*t19403*t19404*t19376*t19394*t19379*t19388*t19397*(8.0/3.0);
                double t19422 = t19412+t19421-t19430+t19413+t19415-t19424-t19429;
                double t19423 = t19406*t19406;
                double t19425 = t19405*6.0;
                double t19426 = c*c0*t19375*t19394*t19388*t19397*t19398*4.0;
                double t19427 = t19425+t19426+3.0;
                double t19428 = 1.0/t19427;
                double t19431 = 1.0/sqrt(t19399);
                double t19432 = c*c0*t19416*t19417*t19375*t19394*t19388*t19397*t19398*(2.0/3.0);
                double t19433 = t19432+1.0;
                double t19434 = t19431*t19406*2.0;
                double t19435 = a3*c0*t19390*t19374*t19394*t19397*t19398*(8.0/3.0);
                double t19436 = c*c0*gamma_bb*t19403*t19376*t19394*t19379*t19388*t19397*t19398*(3.2E1/3.0);
                double t19473 = c*c0*t19402*t19404*t19375*t19394*t19388*t19397*(1.6E1/3.0);
                double t19437 = t19434+t19435+t19436-t19473;
                double t19438 = 1.0/(t19427*t19427);
                double t19439 = t19405*4.0;
                double t19440 = c*c0*t19375*t19394*t19388*t19397*t19398*(8.0/3.0);
                double t19441 = t19440+t19439+6.0;
                double t19442 = 1.0/pow(t19399,3.0/2.0);
                double t19443 = t19431*t19406*3.0;
                double t19444 = a3*c0*t19390*t19374*t19394*t19397*t19398*4.0;
                double t19445 = c*c0*gamma_bb*t19403*t19376*t19394*t19379*t19388*t19397*t19398*1.6E1;
                double t19447 = c*c0*t19402*t19404*t19375*t19394*t19388*t19397*8.0;
                double t19446 = t19443+t19444+t19445-t19447;
                double t19448 = a3*c0*t19416*t19390*t19417*t19374*t19394*t19397*t19398*(2.0/3.0);
                double t19449 = c*c0*t19406*t19416*t19417*t19375*t19394*t19388*t19397*t19398*(2.0/3.0);
                double t19450 = c*c0*gamma_bb*t19403*t19416*t19417*t19376*t19394*t19379*t19388*t19397*t19398*(8.0/3.0);
                double t19474 = c*c0*t19402*t19404*t19416*t19417*t19375*t19394*t19388*t19397*(4.0/3.0);
                double t19451 = t19450+t19407-t19408+t19409-t19474+t19448+t19449;
                double t19452 = 1.0/t19388;
                double t19453 = 1.0/c0;
                double t19454 = pow(rho_b,1.0/3.0);
                double t19455 = exp(-t19393);
                double t19456 = t19384*t19384;
                double t19457 = t19392*t19392;
                double t19458 = 1.0/sqrt(t19388);
                double t19459 = exp(-t19389);
                double t19466 = 1.0/pow(t19388,3.0/2.0);
                double t19470 = t19402*t19384*t19458*t19459*1.0E5;
                double t19471 = a3*t19374*t19392*t19384*t19385*t19466*t19386*t19459*5.0E4;
                double t19472 = gamma_bb*t19392*t19384*t19376*t19458*t19378*t19459*t19379*2.0E5;
                double t19460 = -t19470+t19471+t19472;
                double t19461 = c*c0*t19375*(2.0/3.0);
                double t19468 = t19392*t19384*t19458*t19459*1.0E5;
                double t19462 = t19461-t19468;
                double t19463 = t19416*t19417;
                double t19464 = t19441*t19433*t19428;
                double t19465 = t19463+t19464;
                double t19467 = 1.0/(c*c);
                double t19469 = dirac(t19462);
                double t19475 = a4*a4;
                double t19476 = t19475*t19377*t19378*t19379*2.0;
                double t19477 = exp(t19476);
                double t19478 = sqrt(t19380);
                double t19479 = t19475*t19377*t19378*t19379;
                double t19480 = t19479+1.0;
                double t19481 = 1.0/(t19480*t19480);
                double t19482 = a5*gamma_bb*t19374*t19385*t19386;
                double t19483 = t19482+1.0;
                double t19494 = raa2*t19478;
                double t19484 = exp(-t19494);
                double t19485 = t19484-1.0;
                double t19486 = kaa2*t19485;
                double t19487 = pow(t19380,2.0/5.0);
                double t19495 = raa1*t19487;
                double t19488 = exp(-t19495);
                double t19489 = t19488-1.0;
                double t19490 = kaa1*t19489;
                double t19491 = -kaa0+t19490+t19486;
                double t19492 = 1.0/(t19491*t19491);
                double t19493 = c*c0*t19481*t19375*t19483*t19492*t19477*(2.0/3.0);
                double t19496 = 1.0/t19480;
                double t19497 = c*c0*t19375*t19483*t19492*t19477*(2.0/3.0);
                double t19502 = a5*c0*t19390*t19374*t19492*t19477*(2.0/3.0);
                double t19503 = c*c0*gamma_bb*t19403*t19483*t19492*t19475*t19477*t19379*(8.0/3.0);
                double t19498 = t19502+t19503;
                double t19499 = sqrt(t19497);
                double t19500 = t19475*t19475;
                double t19501 = 1.0/pow(rho_b,3.2E1/3.0);
                double t19504 = 1.0/sqrt(t19497);
                double t19505 = 1.0/(t19480*t19480*t19480);
                double t19506 = t19496*t19499*6.0;
                double t19507 = c*c0*t19481*t19375*t19483*t19492*t19477*4.0;
                double t19508 = t19506+t19507+3.0;
                double t19509 = 1.0/t19508;
                double t19511 = a5*c0*t19390*t19481*t19374*t19492*t19477*(2.0/3.0);
                double t19512 = c*c0*gamma_bb*t19403*t19481*t19483*t19492*t19475*t19477*t19379*(8.0/3.0);
                double t19513 = c*c0*gamma_bb*t19403*t19505*t19483*t19492*t19475*t19477*t19379*(8.0/3.0);
                double t19510 = t19511+t19512-t19513;
                double t19514;
                double t19515;
                if (t19493 > expei_cutoff) {
                    t19514 = 1.0;
                    t19515 = expei(t19493);
                } else {
                    t19514 = exp(t19493);
                    t19515 = Ei(-t19493);
                }
                double t19516 = t19510*t19510;
                double t19517 = 1.0/(t19480*t19480*t19480*t19480);
                double t19518 = a5*c0*gamma_bb*t19414*t19481*t19374*t19492*t19475*t19477*t19379*(1.6E1/3.0);
                double t19519 = c*c0*t19403*t19481*t19483*t19492*t19475*t19477*t19379*(8.0/3.0);
                double t19520 = c*c0*t19500*t19420*t19481*t19419*t19483*t19492*t19377*t19477*(3.2E1/3.0);
                double t19521 = c*c0*t19500*t19420*t19517*t19419*t19483*t19492*t19377*t19477*1.6E1;
                double t19553 = a5*c0*gamma_bb*t19414*t19505*t19374*t19492*t19475*t19477*t19379*(1.6E1/3.0);
                double t19554 = c*c0*t19403*t19505*t19483*t19492*t19475*t19477*t19379*(8.0/3.0);
                double t19555 = c*c0*t19500*t19420*t19505*t19419*t19483*t19492*t19377*t19477*(6.4E1/3.0);
                double t19522 = t19520+t19521-t19553+t19518-t19554+t19519-t19555;
                double t19523 = t19496*t19499*4.0;
                double t19524 = c*c0*t19481*t19375*t19483*t19492*t19477*(8.0/3.0);
                double t19525 = t19523+t19524+6.0;
                double t19526 = a5*c0*t19514*t19515*t19390*t19481*t19374*t19492*t19477*(2.0/3.0);
                double t19527 = c*c0*t19510*t19514*t19515*t19481*t19375*t19483*t19492*t19477*(2.0/3.0);
                double t19528 = c*c0*gamma_bb*t19403*t19514*t19515*t19481*t19483*t19492*t19475*t19477*t19379*(8.0/3.0);
                double t19570 = c*c0*gamma_bb*t19403*t19505*t19514*t19515*t19483*t19492*t19475*t19477*t19379*(8.0/3.0);
                double t19529 = t19511+t19512-t19513-t19570+t19526+t19527+t19528;
                double t19530 = c*c0*t19514*t19515*t19481*t19375*t19483*t19492*t19477*(2.0/3.0);
                double t19531 = t19530+1.0;
                double t19532 = t19504*t19496*t19498*3.0;
                double t19533 = a5*c0*t19390*t19481*t19374*t19492*t19477*4.0;
                double t19534 = c*c0*gamma_bb*t19403*t19481*t19483*t19492*t19475*t19477*t19379*1.6E1;
                double t19537 = gamma_bb*t19481*t19475*t19378*t19379*t19499*1.2E1;
                double t19538 = c*c0*gamma_bb*t19403*t19505*t19483*t19492*t19475*t19477*t19379*1.6E1;
                double t19535 = t19532+t19533+t19534-t19537-t19538;
                double t19536 = 1.0/(t19508*t19508);
                double t19539 = t19504*t19496*t19498*2.0;
                double t19540 = a5*c0*t19390*t19481*t19374*t19492*t19477*(8.0/3.0);
                double t19541 = c*c0*gamma_bb*t19403*t19481*t19483*t19492*t19475*t19477*t19379*(3.2E1/3.0);
                double t19568 = gamma_bb*t19481*t19475*t19378*t19379*t19499*8.0;
                double t19569 = c*c0*gamma_bb*t19403*t19505*t19483*t19492*t19475*t19477*t19379*(3.2E1/3.0);
                double t19542 = t19540+t19541+t19539-t19568-t19569;
                double t19543 = a5*c0*gamma_bb*t19414*t19374*t19492*t19475*t19477*t19379*(1.6E1/3.0);
                double t19544 = c*c0*t19403*t19483*t19492*t19475*t19477*t19379*(8.0/3.0);
                double t19545 = c*c0*t19500*t19420*t19419*t19483*t19492*t19377*t19477*(3.2E1/3.0);
                double t19546 = t19543+t19544+t19545;
                double t19547 = t19498*t19498;
                double t19548 = 1.0/pow(t19497,3.0/2.0);
                double t19549 = 1.0/t19483;
                double t19550 = exp(-t19476);
                double t19551 = t19480*t19480;
                double t19552 = t19491*t19491;
                double t19556 = 1.0/(rho_b*rho_b);
                double t19557 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b);
                double t19558 = c*c;
                double t19559 = 1.0/pow(rho_b,2.0/3.0);
                double t19560 = k1*t19478;
                double t19561 = c*k2*t19375;
                double t19562 = t19560+t19561;
                double t19563 = 1.0/(t19562*t19562);
                double t19567 = t19563*t19558*t19559;
                double t19564 = exp(-t19567);
                double t19565 = dirac(t19462,1.0);
                double t19566 = t19460*t19460;
                double t19571 = t19514*t19515;
                double t19572 = t19531*t19525*t19509;
                double t19573 = t19571+t19572;
                double t19574 = t19384*t19376*t19458*t19378*t19459*t19379*2.0E5;
                double t19575 = a3*a3;
                double t19576 = 1.0/pow(rho_b,1.4E1/3.0);
                double t19577 = 1.0/pow(t19388,5.0/2.0);
                double t19578 = t19392*t19384*t19467*t19575*t19459*t19576*t19379*t19577*7.5E4;
                double t19579 = t19420*t19501*t19418*t19392*t19384*t19377*t19458*t19459*4.0E5;
                double t19580 = 1.0/pow(rho_b,2.3E1/3.0);
                double t19581 = a3*gamma_bb*t19580*t19374*t19392*t19384*t19376*t19385*t19466*t19459*t19379*2.0E5;
                double t19582 = t19581+t19574+t19578+t19579-t19392*t19384*t19376*t19458*t19378*t19459*t19379*2.0E5-a3*t19402*t19374*t19384*t19385*t19466*t19386*t19459*1.0E5-gamma_bb*t19402*t19384*t19376*t19458*t19378*t19459*t19379*4.0E5;
                v_gamma_bb_gamma_bb[Q] += scale * (rho_a*(c0*heaviside(c*c0*t19375*(-2.0/3.0)+t19392*t19384*t19458*exp(-t19376*t19377*t19378*t19379)*1.0E5)*(t19433*t19428*(t19422*t19431*2.0-t19423*t19442-a3*c0*t19402*t19404*t19390*t19374*t19394*t19397*(3.2E1/3.0)+c*c0*t19410*t19411*t19375*t19394*t19388*t19397*1.6E1-c*c0*t19403*t19404*t19376*t19394*t19379*t19388*t19397*(3.2E1/3.0)+c*c0*t19403*t19376*t19394*t19379*t19388*t19397*t19398*(3.2E1/3.0)+a3*c0*gamma_bb*t19414*t19374*t19376*t19394*t19379*t19397*t19398*(6.4E1/3.0)+c*c0*t19420*t19418*t19419*t19394*t19377*t19388*t19397*t19398*(1.28E2/3.0)-c*c0*gamma_bb*t19402*t19403*t19404*t19376*t19394*t19379*t19388*t19397*(1.28E2/3.0))+t19441*t19428*(t19412+t19421+t19413+t19423+t19415-t19424-(t19402*t19406*2.0)/t19392+gamma_bb*t19406*t19376*t19378*t19379*4.0+a3*t19406*t19452*t19374*t19385*t19386-a3*c0*t19402*t19404*t19390*t19374*t19394*t19397*(8.0/3.0)-c*c0*t19403*t19404*t19376*t19394*t19379*t19388*t19397*(8.0/3.0)-a3*c0*t19402*t19404*t19416*t19390*t19417*t19374*t19394*t19397*(8.0/3.0)+a3*c0*t19406*t19416*t19390*t19417*t19374*t19394*t19397*t19398*(4.0/3.0)+c*c0*t19410*t19411*t19416*t19417*t19375*t19394*t19388*t19397*4.0+c*c0*t19422*t19416*t19417*t19375*t19394*t19388*t19397*t19398*(2.0/3.0)+c*c0*t19423*t19416*t19417*t19375*t19394*t19388*t19397*t19398*(2.0/3.0)-c*c0*t19402*t19404*t19406*t19416*t19417*t19375*t19394*t19388*t19397*(8.0/3.0)-c*c0*t19403*t19404*t19416*t19417*t19376*t19394*t19379*t19388*t19397*(8.0/3.0)+c*c0*t19403*t19416*t19417*t19376*t19394*t19379*t19388*t19397*t19398*(8.0/3.0)+a3*c0*gamma_bb*t19414*t19416*t19417*t19374*t19376*t19394*t19379*t19397*t19398*(1.6E1/3.0)+c*c0*t19420*t19416*t19417*t19418*t19419*t19394*t19377*t19388*t19397*t19398*(3.2E1/3.0)-c*c0*gamma_bb*t19402*t19403*t19404*t19416*t19417*t19376*t19394*t19379*t19388*t19397*(3.2E1/3.0)+c*c0*gamma_bb*t19403*t19406*t19416*t19417*t19376*t19394*t19379*t19388*t19397*t19398*(1.6E1/3.0))+t19422*t19416*t19417+t19423*t19416*t19417+t19451*t19428*t19437*2.0-t19441*t19451*t19446*t19438*2.0-t19433*t19437*t19446*t19438*2.0-t19441*t19433*t19438*(t19422*t19431*3.0-t19423*t19442*(3.0/2.0)-a3*c0*t19402*t19404*t19390*t19374*t19394*t19397*1.6E1+c*c0*t19410*t19411*t19375*t19394*t19388*t19397*2.4E1-c*c0*t19403*t19404*t19376*t19394*t19379*t19388*t19397*1.6E1+c*c0*t19403*t19376*t19394*t19379*t19388*t19397*t19398*1.6E1+a3*c0*gamma_bb*t19414*t19374*t19376*t19394*t19379*t19397*t19398*3.2E1+c*c0*t19420*t19418*t19419*t19394*t19377*t19388*t19397*t19398*6.4E1-c*c0*gamma_bb*t19402*t19403*t19404*t19376*t19394*t19379*t19388*t19397*6.4E1)+t19441*t19433*1.0/(t19427*t19427*t19427)*(t19446*t19446)*2.0+t19422*t19452*t19453*t19454*t19455*t19456*t19385*t19457*(3.0/2.0)+t19423*t19452*t19453*t19454*t19455*t19456*t19385*t19457*(3.0/2.0)+t19402*t19406*t19452*t19453*t19454*t19392*t19455*t19456*t19385*3.0-a3*t19406*t19453*t19374*t19455*t19456*t19457*t19556*t19467*1.0/(t19388*t19388)*(3.0/2.0)-gamma_bb*t19406*t19452*t19453*t19455*t19456*t19376*t19385*t19457*t19557*t19379*6.0)*(1.0/2.0)-c0*t19465*t19565*t19566*(1.0/2.0)+c0*t19465*t19582*t19469*(1.0/2.0)-c0*t19460*t19469*(t19441*t19451*t19428+t19406*t19416*t19417+t19433*t19428*t19437-t19441*t19433*t19446*t19438+t19406*t19452*t19453*t19454*t19455*t19456*t19385*t19457*(3.0/2.0)))+rho_b*(c0*t19564*heaviside(-t19461+t19468)*(t19525*t19509*(t19520+t19521+t19516+t19518+t19519+gamma_bb*t19510*t19475*t19378*t19379*4.0+a5*t19510*t19374*t19385*t19386*t19549-gamma_bb*t19510*t19475*t19378*t19379*t19496*4.0-c*c0*t19403*t19505*t19483*t19492*t19475*t19477*t19379*(8.0/3.0)-a5*c0*gamma_bb*t19414*t19505*t19374*t19492*t19475*t19477*t19379*(1.6E1/3.0)+a5*c0*t19510*t19514*t19515*t19390*t19481*t19374*t19492*t19477*(4.0/3.0)-c*c0*t19500*t19420*t19505*t19419*t19483*t19492*t19377*t19477*(6.4E1/3.0)+c*c0*t19522*t19514*t19515*t19481*t19375*t19483*t19492*t19477*(2.0/3.0)+c*c0*t19514*t19515*t19516*t19481*t19375*t19483*t19492*t19477*(2.0/3.0)-c*c0*t19403*t19505*t19514*t19515*t19483*t19492*t19475*t19477*t19379*(8.0/3.0)+c*c0*t19403*t19514*t19515*t19481*t19483*t19492*t19475*t19477*t19379*(8.0/3.0)-a5*c0*gamma_bb*t19414*t19505*t19514*t19515*t19374*t19492*t19475*t19477*t19379*(1.6E1/3.0)+a5*c0*gamma_bb*t19414*t19514*t19515*t19481*t19374*t19492*t19475*t19477*t19379*(1.6E1/3.0)-c*c0*t19500*t19420*t19505*t19514*t19515*t19419*t19483*t19492*t19377*t19477*(6.4E1/3.0)+c*c0*t19500*t19420*t19514*t19515*t19481*t19419*t19483*t19492*t19377*t19477*(3.2E1/3.0)+c*c0*t19500*t19420*t19514*t19515*t19517*t19419*t19483*t19492*t19377*t19477*1.6E1-c*c0*gamma_bb*t19510*t19403*t19505*t19514*t19515*t19483*t19492*t19475*t19477*t19379*(1.6E1/3.0)+c*c0*gamma_bb*t19510*t19403*t19514*t19515*t19481*t19483*t19492*t19475*t19477*t19379*(1.6E1/3.0))+t19522*t19514*t19515+t19514*t19515*t19516+t19542*t19509*t19529*2.0+t19531*t19509*(t19504*t19546*t19496*2.0-t19547*t19548*t19496-t19481*t19475*t19378*t19379*t19499*8.0+t19500*t19420*t19501*t19505*t19377*t19499*3.2E1-gamma_bb*t19504*t19481*t19475*t19378*t19379*t19498*8.0-c*c0*t19403*t19505*t19483*t19492*t19475*t19477*t19379*(3.2E1/3.0)+c*c0*t19403*t19481*t19483*t19492*t19475*t19477*t19379*(3.2E1/3.0)-a5*c0*gamma_bb*t19414*t19505*t19374*t19492*t19475*t19477*t19379*(6.4E1/3.0)+a5*c0*gamma_bb*t19414*t19481*t19374*t19492*t19475*t19477*t19379*(6.4E1/3.0)-c*c0*t19500*t19420*t19505*t19419*t19483*t19492*t19377*t19477*(2.56E2/3.0)+c*c0*t19500*t19420*t19481*t19419*t19483*t19492*t19377*t19477*(1.28E2/3.0)+c*c0*t19500*t19420*t19517*t19419*t19483*t19492*t19377*t19477*6.4E1)-t19531*t19542*t19535*t19536*2.0-t19525*t19535*t19536*t19529*2.0-t19531*t19525*t19536*(t19504*t19546*t19496*3.0-t19547*t19548*t19496*(3.0/2.0)-t19481*t19475*t19378*t19379*t19499*1.2E1+t19500*t19420*t19501*t19505*t19377*t19499*4.8E1-gamma_bb*t19504*t19481*t19475*t19378*t19379*t19498*1.2E1-c*c0*t19403*t19505*t19483*t19492*t19475*t19477*t19379*1.6E1+c*c0*t19403*t19481*t19483*t19492*t19475*t19477*t19379*1.6E1-a5*c0*gamma_bb*t19414*t19505*t19374*t19492*t19475*t19477*t19379*3.2E1+a5*c0*gamma_bb*t19414*t19481*t19374*t19492*t19475*t19477*t19379*3.2E1-c*c0*t19500*t19420*t19505*t19419*t19483*t19492*t19377*t19477*1.28E2+c*c0*t19500*t19420*t19481*t19419*t19483*t19492*t19377*t19477*6.4E1+c*c0*t19500*t19420*t19517*t19419*t19483*t19492*t19377*t19477*9.6E1)+t19531*t19525*1.0/(t19508*t19508*t19508)*(t19535*t19535)*2.0+t19522*t19550*t19551*t19453*t19552*t19454*t19385*t19549*(3.0/2.0)+t19550*t19551*t19453*t19516*t19552*t19454*t19385*t19549*(3.0/2.0)-a5*t19510*t19550*t19551*t19453*t19552*t19374*1.0/(t19483*t19483)*t19556*t19467*(3.0/2.0)-gamma_bb*t19510*t19550*t19551*t19453*t19552*t19385*t19475*t19557*t19549*t19379*6.0+gamma_bb*t19510*t19550*t19453*t19480*t19552*t19385*t19475*t19557*t19549*t19379*6.0)*(1.0/2.0)-c0*t19564*t19573*t19565*t19566*(1.0/2.0)+c0*t19564*t19573*t19582*t19469*(1.0/2.0)-c0*t19460*t19564*t19469*(t19510*t19514*t19515+t19531*t19542*t19509+t19525*t19509*t19529-t19531*t19525*t19535*t19536+t19510*t19550*t19551*t19453*t19552*t19454*t19385*t19549*(3.0/2.0))));
            }
            
            // v_rho_a_gamma_aa
            if (deriv >= 2) {
                double t18408 = tspi_m13*tspi_m13;
                double t18409 = 1.0/pow(rho_a,1.0/3.0);
                double t18410 = a2*a2;
                double t18411 = gamma_aa*gamma_aa;
                double t18412 = 1.0/pow(rho_a,1.6E1/3.0);
                double t18413 = t18408*t18408;
                double t18414 = c*t18409;
                double t18415 = a4*a4;
                double t18416 = 1.0/c;
                double t18417 = 1.0/pow(rho_a,7.0/3.0);
                double t18418 = sqrt(t18414);
                double t18419 = t18411*t18412*t18413*t18415*2.0;
                double t18420 = exp(t18419);
                double t18421 = t18411*t18412*t18413*t18415;
                double t18422 = t18421+1.0;
                double t18423 = 1.0/(t18422*t18422);
                double t18424 = a5*gamma_aa*t18416*t18408*t18417;
                double t18425 = t18424+1.0;
                double t18436 = raa2*t18418;
                double t18426 = exp(-t18436);
                double t18427 = t18426-1.0;
                double t18428 = kaa2*t18427;
                double t18429 = pow(t18414,2.0/5.0);
                double t18437 = raa1*t18429;
                double t18430 = exp(-t18437);
                double t18431 = t18430-1.0;
                double t18432 = kaa1*t18431;
                double t18433 = -kaa0+t18432+t18428;
                double t18434 = 1.0/(t18433*t18433);
                double t18435 = c*c0*t18420*t18423*t18425*t18434*t18409*(2.0/3.0);
                double t18438 = 1.0/pow(rho_a,8.0/3.0);
                double t18439 = 1.0/pow(rho_a,4.0/3.0);
                double t18440 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a);
                double t18441 = 1.0/pow(rho_a,2.0E1/3.0);
                double t18442 = 1.0/(t18422*t18422*t18422);
                double t18443 = t18415*t18415;
                double t18444 = 1.0/pow(rho_a,1.2E1);
                double t18445 = t18413*t18413;
                double t18446 = 1.0/sqrt(t18414);
                double t18447 = c*kaa2*raa2*t18426*t18446*t18439*(1.0/6.0);
                double t18448 = 1.0/pow(t18414,3.0/5.0);
                double t18449 = c*kaa1*raa1*t18430*t18439*t18448*(2.0/1.5E1);
                double t18450 = t18447+t18449;
                double t18451 = 1.0/(t18433*t18433*t18433);
                double t18452 = 1.0/pow(rho_a,1.7E1/3.0);
                double t18453 = 1.0/t18422;
                double t18454 = c*c0*t18420*t18425*t18434*t18409*(2.0/3.0);
                double t18455 = sqrt(t18454);
                double t18456 = 1.0/pow(rho_a,1.1E1/3.0);
                double t18457;
                double t18458;
                if (t18435 > expei_cutoff) {
                    t18457 = 1.0;
                    t18458 = expei(t18435);
                } else {
                    t18457 = exp(t18435);
                    t18458 = Ei(-t18435);
                }
                double t18459 = c*c0*t18420*t18423*t18425*t18434*t18439*(2.0/9.0);
                double t18460 = a5*c0*gamma_aa*t18420*t18423*t18434*t18408*t18456*(1.4E1/9.0);
                double t18461 = c*c0*t18420*t18423*t18450*t18451*t18425*t18409*(4.0/3.0);
                double t18462 = c*c0*t18411*t18420*t18413*t18423*t18441*t18415*t18425*t18434*(6.4E1/9.0);
                double t18463 = t18453*t18455*6.0;
                double t18464 = c*c0*t18420*t18423*t18425*t18434*t18409*4.0;
                double t18465 = t18463+t18464+3.0;
                double t18466 = 1.0/t18465;
                double t18467 = a5*c0*t18420*t18423*t18434*t18408*t18438*(2.0/3.0);
                double t18468 = c*c0*gamma_aa*t18420*t18413*t18423*t18415*t18425*t18434*t18452*(8.0/3.0);
                double t18469 = 1.0/sqrt(t18454);
                double t18472 = c*c0*gamma_aa*t18420*t18413*t18415*t18442*t18425*t18434*t18452*(8.0/3.0);
                double t18470 = -t18472+t18467+t18468;
                double t18483 = c*c0*t18411*t18420*t18413*t18441*t18415*t18442*t18425*t18434*(6.4E1/9.0);
                double t18471 = t18460+t18461+t18462-t18483+t18459;
                double t18473 = 1.0/pow(rho_a,1.9E1/3.0);
                double t18474 = a5*c0*t18420*t18423*t18434*t18408*t18456*(1.6E1/9.0);
                double t18475 = a5*c0*t18420*t18423*t18450*t18451*t18408*t18438*(4.0/3.0);
                double t18476 = a5*c0*t18411*t18420*t18413*t18440*t18423*t18415*t18434*t18408*(4.0E1/3.0);
                double t18477 = c*c0*gamma_aa*t18420*t18413*t18423*t18441*t18415*t18425*t18434*(1.36E2/9.0);
                double t18478 = c*c0*gamma_aa*t18411*t18420*t18423*t18425*t18434*t18443*t18444*t18445*(2.56E2/9.0);
                double t18479 = 1.0/(t18422*t18422*t18422*t18422);
                double t18480 = c*c0*gamma_aa*t18411*t18420*t18425*t18434*t18443*t18444*t18445*t18479*(1.28E2/3.0);
                double t18481 = c*c0*gamma_aa*t18420*t18413*t18423*t18450*t18415*t18451*t18425*t18452*(1.6E1/3.0);
                double t18484 = c*c0*gamma_aa*t18420*t18413*t18441*t18415*t18442*t18425*t18434*(1.36E2/9.0);
                double t18485 = c*c0*gamma_aa*t18411*t18420*t18442*t18425*t18434*t18443*t18444*t18445*(5.12E2/9.0);
                double t18486 = c*c0*gamma_aa*t18420*t18413*t18450*t18415*t18442*t18451*t18425*t18452*(1.6E1/3.0);
                double t18543 = a5*c0*t18411*t18420*t18413*t18440*t18415*t18442*t18434*t18408*(4.0E1/3.0);
                double t18482 = t18480-t18543+t18481+t18474+t18475-t18484+t18476-t18485+t18477-t18486+t18478;
                double t18487 = a5*c0*t18420*t18434*t18408*t18438*(2.0/3.0);
                double t18488 = c*c0*gamma_aa*t18420*t18413*t18415*t18425*t18434*t18452*(8.0/3.0);
                double t18489 = t18487+t18488;
                double t18490 = c*c0*t18420*t18425*t18434*t18439*(2.0/9.0);
                double t18491 = a5*c0*gamma_aa*t18420*t18434*t18408*t18456*(1.4E1/9.0);
                double t18492 = c*c0*t18420*t18450*t18451*t18425*t18409*(4.0/3.0);
                double t18493 = c*c0*t18411*t18420*t18413*t18441*t18415*t18425*t18434*(6.4E1/9.0);
                double t18494 = t18490+t18491+t18492+t18493;
                double t18495 = t18453*t18455*4.0;
                double t18496 = c*c0*t18420*t18423*t18425*t18434*t18409*(8.0/3.0);
                double t18497 = t18495+t18496+6.0;
                double t18498 = a5*c0*t18420*t18423*t18434*t18408*t18438*t18457*t18458*(2.0/3.0);
                double t18499 = c*c0*t18420*t18423*t18425*t18434*t18470*t18409*t18457*t18458*(2.0/3.0);
                double t18500 = c*c0*gamma_aa*t18420*t18413*t18423*t18415*t18425*t18434*t18452*t18457*t18458*(8.0/3.0);
                double t18501 = c*c0*t18420*t18423*t18425*t18434*t18409*t18457*t18458*(2.0/3.0);
                double t18502 = t18501+1.0;
                double t18503 = 1.0/(t18465*t18465);
                double t18504 = a5*c0*t18420*t18434*t18408*t18456*(1.6E1/9.0);
                double t18505 = a5*c0*t18420*t18450*t18451*t18408*t18438*(4.0/3.0);
                double t18506 = a5*c0*t18411*t18420*t18413*t18440*t18415*t18434*t18408*(4.0E1/3.0);
                double t18507 = c*c0*gamma_aa*t18420*t18413*t18441*t18415*t18425*t18434*(1.36E2/9.0);
                double t18508 = c*c0*gamma_aa*t18411*t18420*t18425*t18434*t18443*t18444*t18445*(2.56E2/9.0);
                double t18509 = c*c0*gamma_aa*t18420*t18413*t18450*t18415*t18451*t18425*t18452*(1.6E1/3.0);
                double t18510 = t18504+t18505+t18506+t18507+t18508+t18509;
                double t18511 = 1.0/pow(t18454,3.0/2.0);
                double t18512 = 1.0/pow(rho_a,3.5E1/3.0);
                double t18513 = t18453*t18469*t18489*2.0;
                double t18514 = a5*c0*t18420*t18423*t18434*t18408*t18438*(8.0/3.0);
                double t18515 = c*c0*gamma_aa*t18420*t18413*t18423*t18415*t18425*t18434*t18452*(3.2E1/3.0);
                double t18577 = gamma_aa*t18412*t18413*t18423*t18415*t18455*8.0;
                double t18578 = c*c0*gamma_aa*t18420*t18413*t18415*t18442*t18425*t18434*t18452*(3.2E1/3.0);
                double t18516 = t18513+t18514+t18515-t18577-t18578;
                double t18517 = t18453*t18494*t18469*3.0;
                double t18518 = c*c0*t18420*t18423*t18425*t18434*t18439*(4.0/3.0);
                double t18519 = a5*c0*gamma_aa*t18420*t18423*t18434*t18408*t18456*(2.8E1/3.0);
                double t18520 = c*c0*t18420*t18423*t18450*t18451*t18425*t18409*8.0;
                double t18521 = c*c0*t18411*t18420*t18413*t18423*t18441*t18415*t18425*t18434*(1.28E2/3.0);
                double t18540 = t18411*t18413*t18423*t18415*t18455*t18473*3.2E1;
                double t18541 = c*c0*t18411*t18420*t18413*t18441*t18415*t18442*t18425*t18434*(1.28E2/3.0);
                double t18522 = t18520+t18521-t18540-t18541+t18517+t18518+t18519;
                double t18523 = t18453*t18494*t18469*2.0;
                double t18524 = c*c0*t18420*t18423*t18425*t18434*t18439*(8.0/9.0);
                double t18525 = a5*c0*gamma_aa*t18420*t18423*t18434*t18408*t18456*(5.6E1/9.0);
                double t18526 = c*c0*t18420*t18423*t18450*t18451*t18425*t18409*(1.6E1/3.0);
                double t18527 = c*c0*t18411*t18420*t18413*t18423*t18441*t18415*t18425*t18434*(2.56E2/9.0);
                double t18572 = t18411*t18413*t18423*t18415*t18455*t18473*(6.4E1/3.0);
                double t18573 = c*c0*t18411*t18420*t18413*t18441*t18415*t18442*t18425*t18434*(2.56E2/9.0);
                double t18528 = t18523+t18524+t18525+t18526+t18527-t18572-t18573;
                double t18529 = t18453*t18469*t18489*3.0;
                double t18530 = a5*c0*t18420*t18423*t18434*t18408*t18438*4.0;
                double t18531 = c*c0*gamma_aa*t18420*t18413*t18423*t18415*t18425*t18434*t18452*1.6E1;
                double t18538 = gamma_aa*t18412*t18413*t18423*t18415*t18455*1.2E1;
                double t18539 = c*c0*gamma_aa*t18420*t18413*t18415*t18442*t18425*t18434*t18452*1.6E1;
                double t18532 = t18530+t18531+t18529-t18538-t18539;
                double t18533 = c*c0*t18420*t18423*t18425*t18434*t18439*t18457*t18458*(2.0/9.0);
                double t18534 = c*c0*t18420*t18423*t18450*t18451*t18425*t18409*t18457*t18458*(4.0/3.0);
                double t18535 = c*c0*t18420*t18423*t18425*t18434*t18471*t18409*t18457*t18458*(2.0/3.0);
                double t18536 = a5*c0*gamma_aa*t18420*t18423*t18434*t18408*t18456*t18457*t18458*(1.4E1/9.0);
                double t18537 = c*c0*t18411*t18420*t18413*t18423*t18441*t18415*t18425*t18434*t18457*t18458*(6.4E1/9.0);
                double t18542 = 1.0/t18425;
                double t18544 = 1.0/c0;
                double t18545 = pow(rho_a,1.0/3.0);
                double t18546 = exp(-t18419);
                double t18547 = t18422*t18422;
                double t18548 = t18433*t18433;
                double t18549 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a);
                double t18574 = c*c0*t18411*t18420*t18413*t18441*t18415*t18442*t18425*t18434*t18457*t18458*(6.4E1/9.0);
                double t18550 = t18460+t18461+t18533+t18462+t18534+t18535+t18536-t18483+t18537-t18574+t18459;
                double t18579 = c*c0*gamma_aa*t18420*t18413*t18415*t18442*t18425*t18434*t18452*t18457*t18458*(8.0/3.0);
                double t18551 = t18500-t18472+t18467+t18468+t18498-t18579+t18499;
                double t18552 = t18410*t18411*t18412*t18413;
                double t18553 = pow(t18414,4.0/5.0);
                double t18570 = rab1*t18553;
                double t18554 = exp(-t18570);
                double t18555 = t18554-1.0;
                double t18571 = kab1*t18555;
                double t18556 = kab0-t18571;
                double t18557 = a3*gamma_aa*t18416*t18408*t18417;
                double t18558 = t18557+1.0;
                double t18559 = 1.0/sqrt(t18558);
                double t18560 = a1*gamma_aa*t18408*t18438;
                double t18561 = t18560+t18552+1.0;
                double t18562 = c*c;
                double t18563 = 1.0/pow(rho_a,2.0/3.0);
                double t18564 = k1*t18418;
                double t18565 = c*k2*t18409;
                double t18566 = t18564+t18565;
                double t18567 = 1.0/(t18566*t18566);
                double t18576 = t18562*t18563*t18567;
                double t18568 = exp(-t18576);
                double t18569 = exp(-t18552);
                double t18575 = c*c0*t18409*(2.0/3.0);
                double t18580 = t18561*t18556*t18559*t18569*1.0E5;
                double t18581 = 1.0/pow(rho_a,1.0E1/3.0);
                double t18582 = 1.0/pow(t18558,3.0/2.0);
                double t18583 = t18410*t18411*t18413*t18473*(1.6E1/3.0);
                double t18584 = a1*gamma_aa*t18408*t18456*(8.0/3.0);
                double t18585 = t18583+t18584;
                double t18586 = a1*t18408*t18438;
                double t18587 = gamma_aa*t18410*t18412*t18413*2.0;
                double t18588 = t18586+t18587;
                double t18589 = 1.0/(c*c);
                double t18590 = 1.0/pow(t18414,1.0/5.0);
                double t18591 = -t18580+t18575;
                double t18592 = dirac(t18591);
                double t18593 = t18470*t18457*t18458;
                double t18594 = t18502*t18516*t18466;
                double t18595 = t18551*t18466*t18497;
                double t18596 = t18416*t18470*t18542*t18544*t18545*t18546*t18547*t18548*(3.0/2.0);
                double t18730 = t18502*t18503*t18532*t18497;
                double t18597 = -t18730+t18593+t18594+t18595+t18596;
                double t18598 = t18457*t18458;
                double t18599 = t18502*t18466*t18497;
                double t18600 = t18598+t18599;
                double t18601 = 1.0/pow(rho_a,5.0/3.0);
                double t18602 = t18601*t18562*t18567*(2.0/3.0);
                double t18603 = 1.0/(t18566*t18566*t18566);
                double t18604 = c*k2*t18439*(1.0/3.0);
                double t18605 = c*k1*t18446*t18439*(1.0/6.0);
                double t18606 = t18604+t18605;
                double t18607 = t18602-t18603*t18606*t18562*t18563*2.0;
                double t18608 = a3*t18416*t18408*t18417*t18561*t18582*t18556*t18569*5.0E4;
                double t18609 = gamma_aa*t18410*t18412*t18413*t18561*t18556*t18559*t18569*2.0E5;
                double t18611 = t18556*t18559*t18569*t18588*1.0E5;
                double t18610 = -t18611+t18608+t18609;
                double t18612 = c*c0*t18439*(2.0/9.0);
                double t18613 = t18410*t18411*t18413*t18561*t18473*t18556*t18559*t18569*(1.6E6/3.0);
                double t18614 = a3*gamma_aa*t18416*t18408*t18561*t18581*t18582*t18556*t18569*(3.5E5/3.0);
                double t18724 = t18556*t18585*t18559*t18569*1.0E5;
                double t18725 = c*kab1*rab1*t18561*t18554*t18590*t18439*t18559*t18569*(8.0E4/3.0);
                double t18615 = t18612+t18613+t18614-t18724-t18725;
                double t18616 = t18580-t18575;
                double t18617 = heaviside(t18616);
                double t18618 = t18410*t18411*t18412*t18413*2.0;
                double t18619 = exp(t18618);
                double t18620 = 1.0/(t18556*t18556);
                double t18621 = 1.0/(t18561*t18561);
                double t18622 = c*c0*t18620*t18621*t18409*t18619*t18558*(2.0/3.0);
                double t18623 = 1.0/(t18561*t18561*t18561);
                double t18624;
                double t18625;
                if (t18622 > expei_cutoff) {
                    t18624 = 1.0;
                    t18625 = expei(t18622);
                } else {
                    t18624 = exp(t18622);
                    t18625 = Ei(-t18622);
                }
                double t18626 = c*c0*t18620*t18621*t18439*t18619*t18558*(2.0/9.0);
                double t18627 = a3*c0*gamma_aa*t18620*t18621*t18408*t18456*t18619*(1.4E1/9.0);
                double t18628 = c*c0*t18410*t18411*t18413*t18620*t18441*t18621*t18619*t18558*(6.4E1/9.0);
                double t18629 = 1.0/(t18556*t18556*t18556);
                double t18630 = sqrt(t18622);
                double t18631 = t18630*6.0;
                double t18632 = c*c0*t18620*t18621*t18409*t18619*t18558*4.0;
                double t18633 = t18631+t18632+3.0;
                double t18634 = 1.0/t18633;
                double t18657 = c*c0*t18620*t18623*t18409*t18619*t18558*t18585*(4.0/3.0);
                double t18658 = c0*kab1*rab1*t18601*t18621*t18562*t18554*t18590*t18619*t18629*t18558*(1.6E1/4.5E1);
                double t18635 = t18626+t18627+t18628-t18657-t18658;
                double t18636 = 1.0/sqrt(t18622);
                double t18637 = a3*c0*t18620*t18621*t18408*t18438*t18619*(2.0/3.0);
                double t18638 = c*c0*gamma_aa*t18410*t18413*t18620*t18621*t18452*t18619*t18558*(8.0/3.0);
                double t18640 = c*c0*t18620*t18623*t18409*t18619*t18558*t18588*(4.0/3.0);
                double t18639 = -t18640+t18637+t18638;
                double t18641 = a1*t18408*t18456*(8.0/3.0);
                double t18642 = gamma_aa*t18410*t18413*t18473*(3.2E1/3.0);
                double t18643 = t18641+t18642;
                double t18644 = t18410*t18410;
                double t18645 = a3*c0*t18620*t18623*t18408*t18438*t18619*t18585*(4.0/3.0);
                double t18646 = c*c0*t18620*t18623*t18439*t18619*t18558*t18588*(4.0/9.0);
                double t18647 = c*c0*t18620*t18623*t18409*t18643*t18619*t18558*(4.0/3.0);
                double t18648 = 1.0/(t18561*t18561*t18561*t18561);
                double t18649 = a3*c0*gamma_aa*t18620*t18623*t18408*t18456*t18619*t18588*(2.8E1/9.0);
                double t18650 = c*c0*gamma_aa*t18410*t18413*t18620*t18452*t18623*t18619*t18558*t18585*(1.6E1/3.0);
                double t18651 = c*c0*t18410*t18411*t18413*t18620*t18441*t18623*t18619*t18558*t18588*(1.28E2/9.0);
                double t18652 = 1.0/(rho_a*rho_a*rho_a*rho_a);
                double t18653 = a3*c*c0*kab1*rab1*t18621*t18408*t18652*t18554*t18590*t18619*t18629*(1.6E1/4.5E1);
                double t18654 = 1.0/(rho_a*rho_a*rho_a*rho_a*rho_a*rho_a*rho_a);
                double t18655 = c0*gamma_aa*kab1*rab1*t18410*t18413*t18621*t18562*t18554*t18590*t18654*t18619*t18629*t18558*(6.4E1/4.5E1);
                double t18660 = a3*c0*t18620*t18621*t18408*t18456*t18619*(1.6E1/9.0);
                double t18661 = c*c0*t18620*t18409*t18619*t18558*t18585*t18648*t18588*4.0;
                double t18662 = a3*c0*t18410*t18411*t18413*t18440*t18620*t18621*t18408*t18619*(4.0E1/3.0);
                double t18663 = c*c0*gamma_aa*t18410*t18413*t18620*t18441*t18621*t18619*t18558*(1.36E2/9.0);
                double t18664 = c*c0*gamma_aa*t18411*t18620*t18621*t18444*t18445*t18644*t18619*t18558*(2.56E2/9.0);
                double t18665 = c0*kab1*rab1*t18601*t18623*t18562*t18554*t18590*t18619*t18629*t18558*t18588*(3.2E1/4.5E1);
                double t18656 = t18650+t18651-t18660-t18661+t18653-t18662+t18645-t18663+t18646+t18655-t18664+t18647-t18665+t18649;
                double t18659 = 1.0/rho_a;
                double t18666 = c*c0*t18620*t18621*t18624*t18409*t18625*t18619*t18558*(2.0/3.0);
                double t18667 = t18666+1.0;
                double t18668 = t18636*t18639*2.0;
                double t18669 = a3*c0*t18620*t18621*t18408*t18438*t18619*(8.0/3.0);
                double t18670 = c*c0*gamma_aa*t18410*t18413*t18620*t18621*t18452*t18619*t18558*(3.2E1/3.0);
                double t18722 = c*c0*t18620*t18623*t18409*t18619*t18558*t18588*(1.6E1/3.0);
                double t18671 = -t18722+t18670+t18668+t18669;
                double t18672 = 1.0/(t18633*t18633);
                double t18673 = t18635*t18636*2.0;
                double t18674 = c*c0*t18620*t18621*t18439*t18619*t18558*(8.0/9.0);
                double t18675 = a3*c0*gamma_aa*t18620*t18621*t18408*t18456*t18619*(5.6E1/9.0);
                double t18676 = c*c0*t18410*t18411*t18413*t18620*t18441*t18621*t18619*t18558*(2.56E2/9.0);
                double t18708 = c*c0*t18620*t18623*t18409*t18619*t18558*t18585*(1.6E1/3.0);
                double t18709 = c0*kab1*rab1*t18601*t18621*t18562*t18554*t18590*t18619*t18629*t18558*(6.4E1/4.5E1);
                double t18677 = -t18708+t18673-t18709+t18674+t18675+t18676;
                double t18678 = t18630*4.0;
                double t18679 = c*c0*t18620*t18621*t18409*t18619*t18558*(8.0/3.0);
                double t18680 = t18678+t18679+6.0;
                double t18681 = t18636*t18639*3.0;
                double t18682 = a3*c0*t18620*t18621*t18408*t18438*t18619*4.0;
                double t18683 = c*c0*gamma_aa*t18410*t18413*t18620*t18621*t18452*t18619*t18558*1.6E1;
                double t18699 = c*c0*t18620*t18623*t18409*t18619*t18558*t18588*8.0;
                double t18684 = t18681+t18682+t18683-t18699;
                double t18685 = c*c0*t18620*t18621*t18624*t18625*t18439*t18619*t18558*(2.0/9.0);
                double t18686 = a3*c0*gamma_aa*t18620*t18621*t18408*t18624*t18625*t18456*t18619*(1.4E1/9.0);
                double t18687 = c*c0*t18620*t18621*t18624*t18409*t18625*t18635*t18619*t18558*(2.0/3.0);
                double t18688 = c*c0*t18410*t18411*t18413*t18620*t18441*t18621*t18624*t18625*t18619*t18558*(6.4E1/9.0);
                double t18689 = t18635*t18636*3.0;
                double t18690 = c*c0*t18620*t18621*t18439*t18619*t18558*(4.0/3.0);
                double t18691 = a3*c0*gamma_aa*t18620*t18621*t18408*t18456*t18619*(2.8E1/3.0);
                double t18692 = c*c0*t18410*t18411*t18413*t18620*t18441*t18621*t18619*t18558*(1.28E2/3.0);
                double t18700 = c*c0*t18620*t18623*t18409*t18619*t18558*t18585*8.0;
                double t18701 = c0*kab1*rab1*t18601*t18621*t18562*t18554*t18590*t18619*t18629*t18558*(3.2E1/1.5E1);
                double t18693 = -t18700-t18701+t18690+t18691+t18692+t18689;
                double t18694 = a3*c0*t18620*t18621*t18408*t18624*t18625*t18438*t18619*(2.0/3.0);
                double t18695 = c*c0*t18620*t18621*t18624*t18409*t18625*t18619*t18558*t18639*(2.0/3.0);
                double t18696 = c*c0*gamma_aa*t18410*t18413*t18620*t18621*t18452*t18624*t18625*t18619*t18558*(8.0/3.0);
                double t18723 = c*c0*t18620*t18623*t18624*t18409*t18625*t18619*t18558*t18588*(4.0/3.0);
                double t18697 = -t18640-t18723+t18637+t18638+t18694+t18695+t18696;
                double t18698 = 1.0/pow(t18622,3.0/2.0);
                double t18702 = 1.0/t18558;
                double t18703 = exp(-t18618);
                double t18704 = t18556*t18556;
                double t18705 = t18561*t18561;
                double t18706 = 1.0/(rho_a*rho_a);
                double t18707 = t18626+t18627+t18628-t18657-t18658+t18685+t18686+t18687+t18688-c*c0*t18620*t18623*t18624*t18409*t18625*t18619*t18558*t18585*(4.0/3.0)-c0*kab1*rab1*t18601*t18621*t18624*t18562*t18625*t18554*t18590*t18619*t18629*t18558*(1.6E1/4.5E1);
                double t18710 = a3*t18416*t18408*t18561*t18581*t18582*t18556*t18569*(3.5E5/3.0);
                double t18711 = a3*t18416*t18408*t18417*t18582*t18556*t18585*t18569*5.0E4;
                double t18712 = gamma_aa*t18410*t18413*t18561*t18473*t18556*t18559*t18569*(3.2E6/3.0);
                double t18713 = gamma_aa*t18410*t18412*t18413*t18556*t18585*t18559*t18569*2.0E5;
                double t18714 = t18410*t18411*t18413*t18473*t18556*t18559*t18569*t18588*(1.6E6/3.0);
                double t18715 = a3*a3;
                double t18716 = 1.0/pow(t18558,5.0/2.0);
                double t18717 = a3*gamma_aa*t18416*t18408*t18581*t18582*t18556*t18569*t18588*(3.5E5/3.0);
                double t18718 = a3*kab1*rab1*t18408*t18561*t18554*t18590*t18456*t18582*t18569*(4.0E4/3.0);
                double t18719 = 1.0/pow(rho_a,2.6E1/3.0);
                double t18720 = c*gamma_aa*kab1*rab1*t18410*t18413*t18441*t18561*t18554*t18590*t18559*t18569*(1.6E5/3.0);
                double t18721 = t18710+t18711+t18720+t18712+t18713+t18714+t18717+t18718-t18643*t18556*t18559*t18569*1.0E5-c*kab1*rab1*t18554*t18590*t18439*t18559*t18569*t18588*(8.0E4/3.0)-gamma_aa*t18411*t18512*t18561*t18445*t18644*t18556*t18559*t18569*(3.2E6/3.0)-gamma_aa*t18413*t18452*t18561*t18715*t18716*t18556*t18569*t18589*1.75E5-a3*t18410*t18411*t18413*t18416*t18408*t18561*t18582*t18556*t18719*t18569*5.0E5;
                double t18726 = dirac(t18591,1.0);
                double t18727 = t18624*t18625;
                double t18728 = t18634*t18680*t18667;
                double t18729 = t18727+t18728;
                v_rho_a_gamma_aa[Q] += scale * (rho_a*(c0*t18568*heaviside(c*c0*t18409*(-2.0/3.0)+t18561*t18556*t18559*exp(-t18410*t18411*t18412*t18413)*1.0E5)*(t18502*t18466*(t18510*t18453*t18469*2.0-t18511*t18453*t18494*t18489-gamma_aa*t18413*t18423*t18415*t18455*t18473*(1.28E2/3.0)+a5*c0*t18420*t18423*t18434*t18408*t18456*(6.4E1/9.0)+gamma_aa*t18411*t18512*t18442*t18443*t18445*t18455*(2.56E2/3.0)-gamma_aa*t18412*t18413*t18423*t18415*t18494*t18469*4.0-t18411*t18413*t18423*t18415*t18473*t18469*t18489*(3.2E1/3.0)+a5*c0*t18420*t18423*t18450*t18451*t18408*t18438*(1.6E1/3.0)+c*c0*gamma_aa*t18420*t18413*t18423*t18441*t18415*t18425*t18434*(5.44E2/9.0)-c*c0*gamma_aa*t18420*t18413*t18441*t18415*t18442*t18425*t18434*(5.44E2/9.0)+a5*c0*t18411*t18420*t18413*t18440*t18423*t18415*t18434*t18408*(1.6E2/3.0)-a5*c0*t18411*t18420*t18413*t18440*t18415*t18442*t18434*t18408*(1.6E2/3.0)+c*c0*gamma_aa*t18420*t18413*t18423*t18450*t18415*t18451*t18425*t18452*(6.4E1/3.0)-c*c0*gamma_aa*t18420*t18413*t18450*t18415*t18442*t18451*t18425*t18452*(6.4E1/3.0)+c*c0*gamma_aa*t18411*t18420*t18423*t18425*t18434*t18443*t18444*t18445*(1.024E3/9.0)-c*c0*gamma_aa*t18411*t18420*t18442*t18425*t18434*t18443*t18444*t18445*(2.048E3/9.0)+c*c0*gamma_aa*t18411*t18420*t18425*t18434*t18443*t18444*t18445*t18479*(5.12E2/3.0))+t18466*t18497*(t18480-t18543+t18481+t18474+t18475-t18484+t18476-t18485+t18477-t18486+t18478+t18470*t18471+t18470*t18659*(1.0/3.0)+(t18450*t18470*2.0)/t18433+t18411*t18413*t18415*t18470*t18473*(3.2E1/3.0)-t18411*t18413*t18415*t18470*t18453*t18473*(3.2E1/3.0)+a5*gamma_aa*t18416*t18470*t18542*t18408*t18581*(7.0/3.0)+a5*c0*t18420*t18423*t18434*t18408*t18456*t18457*t18458*(1.6E1/9.0)+a5*c0*t18420*t18423*t18450*t18451*t18408*t18438*t18457*t18458*(4.0/3.0)+a5*c0*t18420*t18423*t18434*t18408*t18471*t18438*t18457*t18458*(2.0/3.0)+c*c0*t18420*t18423*t18425*t18434*t18470*t18439*t18457*t18458*(2.0/9.0)+c*c0*t18420*t18423*t18425*t18434*t18409*t18482*t18457*t18458*(2.0/3.0)+a5*c0*gamma_aa*t18420*t18423*t18434*t18470*t18408*t18456*t18457*t18458*(1.4E1/9.0)+c*c0*t18420*t18423*t18450*t18451*t18425*t18470*t18409*t18457*t18458*(4.0/3.0)+c*c0*t18420*t18423*t18425*t18434*t18470*t18471*t18409*t18457*t18458*(2.0/3.0)+c*c0*gamma_aa*t18420*t18413*t18423*t18441*t18415*t18425*t18434*t18457*t18458*(1.36E2/9.0)-c*c0*gamma_aa*t18420*t18413*t18441*t18415*t18442*t18425*t18434*t18457*t18458*(1.36E2/9.0)+a5*c0*t18411*t18420*t18413*t18440*t18423*t18415*t18434*t18408*t18457*t18458*(4.0E1/3.0)-a5*c0*t18411*t18420*t18413*t18440*t18415*t18442*t18434*t18408*t18457*t18458*(4.0E1/3.0)+c*c0*gamma_aa*t18420*t18413*t18423*t18450*t18415*t18451*t18425*t18452*t18457*t18458*(1.6E1/3.0)-c*c0*gamma_aa*t18420*t18413*t18450*t18415*t18442*t18451*t18425*t18452*t18457*t18458*(1.6E1/3.0)+c*c0*gamma_aa*t18420*t18413*t18423*t18415*t18425*t18434*t18452*t18471*t18457*t18458*(8.0/3.0)+c*c0*gamma_aa*t18411*t18420*t18423*t18425*t18434*t18443*t18444*t18445*t18457*t18458*(2.56E2/9.0)-c*c0*gamma_aa*t18420*t18413*t18415*t18442*t18425*t18434*t18452*t18471*t18457*t18458*(8.0/3.0)-c*c0*gamma_aa*t18411*t18420*t18442*t18425*t18434*t18443*t18444*t18445*t18457*t18458*(5.12E2/9.0)+c*c0*gamma_aa*t18411*t18420*t18425*t18434*t18443*t18444*t18445*t18457*t18458*t18479*(1.28E2/3.0)+c*c0*t18411*t18420*t18413*t18423*t18441*t18415*t18425*t18434*t18470*t18457*t18458*(6.4E1/9.0)-c*c0*t18411*t18420*t18413*t18441*t18415*t18442*t18425*t18434*t18470*t18457*t18458*(6.4E1/9.0))+t18550*t18516*t18466+t18551*t18528*t18466+t18482*t18457*t18458-t18502*t18503*t18522*t18516-t18502*t18503*t18532*t18528-t18503*t18522*t18551*t18497-t18503*t18532*t18550*t18497+t18470*t18471*t18457*t18458-t18502*t18503*t18497*(t18510*t18453*t18469*3.0-t18511*t18453*t18494*t18489*(3.0/2.0)-gamma_aa*t18413*t18423*t18415*t18455*t18473*6.4E1+a5*c0*t18420*t18423*t18434*t18408*t18456*(3.2E1/3.0)+gamma_aa*t18411*t18512*t18442*t18443*t18445*t18455*1.28E2-gamma_aa*t18412*t18413*t18423*t18415*t18494*t18469*6.0-t18411*t18413*t18423*t18415*t18473*t18469*t18489*1.6E1+a5*c0*t18420*t18423*t18450*t18451*t18408*t18438*8.0+c*c0*gamma_aa*t18420*t18413*t18423*t18441*t18415*t18425*t18434*(2.72E2/3.0)-c*c0*gamma_aa*t18420*t18413*t18441*t18415*t18442*t18425*t18434*(2.72E2/3.0)+a5*c0*t18411*t18420*t18413*t18440*t18423*t18415*t18434*t18408*8.0E1-a5*c0*t18411*t18420*t18413*t18440*t18415*t18442*t18434*t18408*8.0E1+c*c0*gamma_aa*t18420*t18413*t18423*t18450*t18415*t18451*t18425*t18452*3.2E1-c*c0*gamma_aa*t18420*t18413*t18450*t18415*t18442*t18451*t18425*t18452*3.2E1+c*c0*gamma_aa*t18411*t18420*t18423*t18425*t18434*t18443*t18444*t18445*(5.12E2/3.0)-c*c0*gamma_aa*t18411*t18420*t18442*t18425*t18434*t18443*t18444*t18445*(1.024E3/3.0)+c*c0*gamma_aa*t18411*t18420*t18425*t18434*t18443*t18444*t18445*t18479*2.56E2)+t18502*t18522*t18532*1.0/(t18465*t18465*t18465)*t18497*2.0+t18416*t18542*t18544*t18482*t18545*t18546*t18547*t18548*(3.0/2.0)+t18416*t18470*t18542*t18471*t18544*t18545*t18546*t18547*t18548*(3.0/2.0)-a5*1.0/(t18425*t18425)*t18408*t18471*t18544*t18706*t18546*t18547*t18548*t18589*(3.0/2.0)+gamma_aa*t18413*t18422*t18415*t18416*t18542*t18471*t18544*t18546*t18548*t18549*6.0-gamma_aa*t18413*t18415*t18416*t18542*t18471*t18544*t18546*t18547*t18548*t18549*6.0)*(-1.0/2.0)+c0*t18600*t18721*t18592*t18568*(1.0/2.0)+c0*t18607*t18617*t18568*t18597*(1.0/2.0)+c0*t18615*t18592*t18568*t18597*(1.0/2.0)+c0*t18610*t18568*dirac(t18575-t18561*t18556*t18559*t18569*1.0E5)*(t18502*t18528*t18466+t18471*t18457*t18458+t18550*t18466*t18497-t18502*t18503*t18522*t18497+t18416*t18542*t18471*t18544*t18545*t18546*t18547*t18548*(3.0/2.0))*(1.0/2.0)+c0*t18600*t18610*t18615*t18726*t18568*(1.0/2.0)-c0*t18600*t18610*t18607*t18592*t18568*(1.0/2.0))+rho_b*(c0*t18617*(t18634*t18667*(t18636*t18656*2.0+t18635*t18639*t18698-a3*c0*t18620*t18621*t18408*t18456*t18619*(6.4E1/9.0)+a3*c0*t18620*t18623*t18408*t18438*t18619*t18585*(1.6E1/3.0)+c*c0*t18620*t18623*t18409*t18643*t18619*t18558*(1.6E1/3.0)+c*c0*t18620*t18623*t18439*t18619*t18558*t18588*(1.6E1/9.0)+a3*c0*gamma_aa*t18620*t18623*t18408*t18456*t18619*t18588*(1.12E2/9.0)-c*c0*t18620*t18409*t18619*t18558*t18585*t18648*t18588*1.6E1-c*c0*gamma_aa*t18410*t18413*t18620*t18441*t18621*t18619*t18558*(5.44E2/9.0)-a3*c0*t18410*t18411*t18413*t18440*t18620*t18621*t18408*t18619*(1.6E2/3.0)+c*c0*gamma_aa*t18410*t18413*t18620*t18452*t18623*t18619*t18558*t18585*(6.4E1/3.0)-c*c0*gamma_aa*t18411*t18620*t18621*t18444*t18445*t18644*t18619*t18558*(1.024E3/9.0)+c*c0*t18410*t18411*t18413*t18620*t18441*t18623*t18619*t18558*t18588*(5.12E2/9.0)+a3*c*c0*kab1*rab1*t18621*t18408*t18652*t18554*t18590*t18619*t18629*(6.4E1/4.5E1)-c0*kab1*rab1*t18601*t18623*t18562*t18554*t18590*t18619*t18629*t18558*t18588*(1.28E2/4.5E1)+c0*gamma_aa*kab1*rab1*t18410*t18413*t18621*t18562*t18554*t18590*t18654*t18619*t18629*t18558*(2.56E2/4.5E1))+t18634*t18680*(t18650+t18651-t18660-t18661+t18653-t18662+t18645-t18663+t18646+t18655-t18664+t18647-t18665+t18649-t18635*t18639-t18639*t18659*(1.0/3.0)+(t18585*t18639*2.0)/t18561-t18410*t18411*t18413*t18473*t18639*(3.2E1/3.0)-a3*gamma_aa*t18702*t18416*t18408*t18581*t18639*(7.0/3.0)+(c*kab1*rab1*t18554*t18590*t18439*t18639*(8.0/1.5E1))/t18556-a3*c0*t18620*t18621*t18408*t18624*t18625*t18456*t18619*(1.6E1/9.0)-a3*c0*t18620*t18621*t18408*t18624*t18625*t18635*t18438*t18619*(2.0/3.0)+a3*c0*t18620*t18623*t18408*t18624*t18625*t18438*t18619*t18585*(4.0/3.0)+c*c0*t18620*t18623*t18624*t18409*t18625*t18643*t18619*t18558*(4.0/3.0)+c*c0*t18620*t18621*t18624*t18409*t18625*t18619*t18656*t18558*(2.0/3.0)-c*c0*t18620*t18621*t18624*t18625*t18439*t18619*t18558*t18639*(2.0/9.0)+c*c0*t18620*t18623*t18624*t18625*t18439*t18619*t18558*t18588*(4.0/9.0)-a3*c0*gamma_aa*t18620*t18621*t18408*t18624*t18625*t18456*t18619*t18639*(1.4E1/9.0)+a3*c0*gamma_aa*t18620*t18623*t18408*t18624*t18625*t18456*t18619*t18588*(2.8E1/9.0)-c*c0*t18620*t18621*t18624*t18409*t18625*t18635*t18619*t18558*t18639*(2.0/3.0)+c*c0*t18620*t18623*t18624*t18409*t18625*t18635*t18619*t18558*t18588*(4.0/3.0)+c*c0*t18620*t18623*t18624*t18409*t18625*t18619*t18558*t18585*t18639*(4.0/3.0)-c*c0*t18620*t18624*t18409*t18625*t18619*t18558*t18585*t18648*t18588*4.0-c*c0*gamma_aa*t18410*t18413*t18620*t18441*t18621*t18624*t18625*t18619*t18558*(1.36E2/9.0)-a3*c0*t18410*t18411*t18413*t18440*t18620*t18621*t18408*t18624*t18625*t18619*(4.0E1/3.0)-c*c0*gamma_aa*t18410*t18413*t18620*t18621*t18452*t18624*t18625*t18635*t18619*t18558*(8.0/3.0)+c*c0*gamma_aa*t18410*t18413*t18620*t18452*t18623*t18624*t18625*t18619*t18558*t18585*(1.6E1/3.0)-c*c0*gamma_aa*t18411*t18620*t18621*t18444*t18624*t18445*t18625*t18644*t18619*t18558*(2.56E2/9.0)-c*c0*t18410*t18411*t18413*t18620*t18441*t18621*t18624*t18625*t18619*t18558*t18639*(6.4E1/9.0)+c*c0*t18410*t18411*t18413*t18620*t18441*t18623*t18624*t18625*t18619*t18558*t18588*(1.28E2/9.0)+a3*c*c0*kab1*rab1*t18621*t18408*t18624*t18625*t18652*t18554*t18590*t18619*t18629*(1.6E1/4.5E1)+c0*kab1*rab1*t18601*t18621*t18624*t18562*t18625*t18554*t18590*t18619*t18629*t18558*t18639*(1.6E1/4.5E1)-c0*kab1*rab1*t18601*t18623*t18624*t18562*t18625*t18554*t18590*t18619*t18629*t18558*t18588*(3.2E1/4.5E1)+c0*gamma_aa*kab1*rab1*t18410*t18413*t18621*t18624*t18562*t18625*t18554*t18590*t18654*t18619*t18629*t18558*(6.4E1/4.5E1))+t18624*t18625*t18656-t18634*t18677*t18697-t18634*t18671*(t18626+t18627+t18628+t18685+t18686+t18687+t18688-c*c0*t18620*t18623*t18409*t18619*t18558*t18585*(4.0/3.0)-c*c0*t18620*t18623*t18624*t18409*t18625*t18619*t18558*t18585*(4.0/3.0)-c0*kab1*rab1*t18601*t18621*t18562*t18554*t18590*t18619*t18629*t18558*(1.6E1/4.5E1)-c0*kab1*rab1*t18601*t18621*t18624*t18562*t18625*t18554*t18590*t18619*t18629*t18558*(1.6E1/4.5E1))-t18624*t18625*t18635*t18639+t18680*t18707*t18672*t18684+t18671*t18672*t18693*t18667+t18680*t18672*t18693*t18697+t18672*t18684*t18667*t18677-t18680*t18672*t18667*(t18636*t18656*3.0+t18635*t18639*t18698*(3.0/2.0)-a3*c0*t18620*t18621*t18408*t18456*t18619*(3.2E1/3.0)+a3*c0*t18620*t18623*t18408*t18438*t18619*t18585*8.0+c*c0*t18620*t18623*t18409*t18643*t18619*t18558*8.0+c*c0*t18620*t18623*t18439*t18619*t18558*t18588*(8.0/3.0)+a3*c0*gamma_aa*t18620*t18623*t18408*t18456*t18619*t18588*(5.6E1/3.0)-c*c0*t18620*t18409*t18619*t18558*t18585*t18648*t18588*2.4E1-c*c0*gamma_aa*t18410*t18413*t18620*t18441*t18621*t18619*t18558*(2.72E2/3.0)-a3*c0*t18410*t18411*t18413*t18440*t18620*t18621*t18408*t18619*8.0E1+c*c0*gamma_aa*t18410*t18413*t18620*t18452*t18623*t18619*t18558*t18585*3.2E1-c*c0*gamma_aa*t18411*t18620*t18621*t18444*t18445*t18644*t18619*t18558*(5.12E2/3.0)+c*c0*t18410*t18411*t18413*t18620*t18441*t18623*t18619*t18558*t18588*(2.56E2/3.0)+a3*c*c0*kab1*rab1*t18621*t18408*t18652*t18554*t18590*t18619*t18629*(3.2E1/1.5E1)-c0*kab1*rab1*t18601*t18623*t18562*t18554*t18590*t18619*t18629*t18558*t18588*(6.4E1/1.5E1)+c0*gamma_aa*kab1*rab1*t18410*t18413*t18621*t18562*t18554*t18590*t18654*t18619*t18629*t18558*(1.28E2/1.5E1))-1.0/(t18633*t18633*t18633)*t18680*t18684*t18693*t18667*2.0+t18702*t18703*t18416*t18704*t18705*t18544*t18545*t18656*(3.0/2.0)-t18702*t18703*t18416*t18704*t18705*t18544*t18545*t18635*t18639*(3.0/2.0)-t18702*t18703*t18416*t18704*t18561*t18544*t18545*t18635*t18588*3.0+a3*t18703*t18704*t18408*t18705*t18544*t18706*t18635*1.0/(t18558*t18558)*t18589*(3.0/2.0)+gamma_aa*t18410*t18413*t18702*t18703*t18416*t18704*t18705*t18544*t18635*t18549*6.0)*(1.0/2.0)+c0*t18721*t18592*t18729*(1.0/2.0)+c0*t18610*t18592*(t18624*t18625*t18635+t18634*t18680*t18707+t18634*t18667*t18677-t18680*t18672*t18693*t18667+t18702*t18703*t18416*t18704*t18705*t18544*t18545*t18635*(3.0/2.0))*(1.0/2.0)+c0*t18615*t18592*(t18624*t18625*t18639+t18634*t18671*t18667+t18634*t18680*t18697-t18680*t18672*t18684*t18667+t18702*t18703*t18416*t18704*t18705*t18544*t18545*t18639*(3.0/2.0))*(1.0/2.0)+c0*t18610*t18615*t18726*t18729*(1.0/2.0))+c0*t18617*t18568*t18597*(1.0/2.0)-c0*t18600*t18610*t18592*t18568*(1.0/2.0));
            }
            
            // v_rho_a_gamma_bb
            if (deriv >= 2) {
                double t18733 = tspi_m13*tspi_m13;
                double t18734 = 1.0/pow(rho_b,1.0/3.0);
                double t18735 = a2*a2;
                double t18736 = gamma_bb*gamma_bb;
                double t18737 = 1.0/pow(rho_b,1.6E1/3.0);
                double t18738 = t18733*t18733;
                double t18739 = t18735*t18736*t18737*t18738*2.0;
                double t18740 = exp(t18739);
                double t18741 = c*t18734;
                double t18742 = pow(t18741,4.0/5.0);
                double t18754 = rab1*t18742;
                double t18743 = exp(-t18754);
                double t18744 = t18743-1.0;
                double t18755 = kab1*t18744;
                double t18745 = kab0-t18755;
                double t18746 = 1.0/c;
                double t18747 = 1.0/pow(rho_b,7.0/3.0);
                double t18748 = a3*gamma_bb*t18733*t18746*t18747;
                double t18749 = t18748+1.0;
                double t18750 = t18735*t18736*t18737*t18738;
                double t18751 = 1.0/pow(rho_b,8.0/3.0);
                double t18752 = a1*gamma_bb*t18733*t18751;
                double t18753 = t18750+t18752+1.0;
                double t18756 = 1.0/(t18745*t18745);
                double t18757 = 1.0/(t18753*t18753);
                double t18758 = c*c0*t18740*t18734*t18756*t18757*t18749*(2.0/3.0);
                double t18759 = a1*t18733*t18751;
                double t18760 = gamma_bb*t18735*t18737*t18738*2.0;
                double t18761 = t18760+t18759;
                double t18762 = 1.0/(t18753*t18753*t18753);
                double t18763 = 1.0/pow(rho_b,1.7E1/3.0);
                double t18764;
                double t18765;
                if (t18758 > expei_cutoff) {
                    t18764 = 1.0;
                    t18765 = expei(t18758);
                } else {
                    t18764 = exp(t18758);
                    t18765 = Ei(-t18758);
                }
                double t18766 = a3*c0*t18740*t18733*t18751*t18756*t18757*(2.0/3.0);
                double t18767 = c*c0*gamma_bb*t18740*t18735*t18763*t18738*t18756*t18757*t18749*(8.0/3.0);
                double t18774 = c*c0*t18740*t18734*t18761*t18762*t18756*t18749*(4.0/3.0);
                double t18768 = -t18774+t18766+t18767;
                double t18769 = sqrt(t18758);
                double t18770 = t18769*6.0;
                double t18771 = c*c0*t18740*t18734*t18756*t18757*t18749*4.0;
                double t18772 = t18770+t18771+3.0;
                double t18773 = 1.0/t18772;
                double t18775 = c*c0*t18740*t18734*t18764*t18756*t18765*t18757*t18749*(2.0/3.0);
                double t18776 = t18775+1.0;
                double t18777 = t18769*4.0;
                double t18778 = c*c0*t18740*t18734*t18756*t18757*t18749*(8.0/3.0);
                double t18779 = t18777+t18778+6.0;
                double t18780 = 1.0/sqrt(t18758);
                double t18781 = 1.0/sqrt(t18749);
                double t18782 = exp(-t18750);
                v_rho_a_gamma_bb[Q] += scale * (c0*heaviside(c*c0*t18734*(-2.0/3.0)+t18753*t18745*t18781*exp(-t18735*t18736*t18737*t18738)*1.0E5)*(t18764*t18765*t18768+t18773*t18779*(-t18774+t18766+t18767+a3*c0*t18740*t18733*t18751*t18764*t18756*t18765*t18757*(2.0/3.0)-c*c0*t18740*t18734*t18761*t18762*t18764*t18756*t18765*t18749*(4.0/3.0)+c*c0*t18740*t18734*t18764*t18756*t18765*t18757*t18749*t18768*(2.0/3.0)+c*c0*gamma_bb*t18740*t18735*t18763*t18764*t18738*t18756*t18765*t18757*t18749*(8.0/3.0))+t18773*t18776*(t18780*t18768*2.0+a3*c0*t18740*t18733*t18751*t18756*t18757*(8.0/3.0)-c*c0*t18740*t18734*t18761*t18762*t18756*t18749*(1.6E1/3.0)+c*c0*gamma_bb*t18740*t18735*t18763*t18738*t18756*t18757*t18749*(3.2E1/3.0))-1.0/(t18772*t18772)*t18776*t18779*(t18780*t18768*3.0+a3*c0*t18740*t18733*t18751*t18756*t18757*4.0-c*c0*t18740*t18734*t18761*t18762*t18756*t18749*8.0+c*c0*gamma_bb*t18740*t18735*t18763*t18738*t18756*t18757*t18749*1.6E1)+(pow(rho_b,1.0/3.0)*(t18753*t18753)*(t18745*t18745)*t18746*t18768*exp(-t18739)*(3.0/2.0))/(c0*t18749))*(1.0/2.0)-c0*dirac(c*c0*t18734*(2.0/3.0)-t18753*t18745*t18781*t18782*1.0E5)*(t18764*t18765+t18773*t18776*t18779)*(t18761*t18745*t18781*t18782*-1.0E5+a3*t18733*t18753*t18745*t18746*t18782*t18747*1.0/pow(t18749,3.0/2.0)*5.0E4+gamma_bb*t18735*t18753*t18745*t18781*t18737*t18782*t18738*2.0E5)*(1.0/2.0));
            }
            
            // v_rho_b_gamma_aa
            if (deriv >= 2) {
                double t18784 = tspi_m13*tspi_m13;
                double t18785 = 1.0/pow(rho_a,1.0/3.0);
                double t18786 = a2*a2;
                double t18787 = gamma_aa*gamma_aa;
                double t18788 = 1.0/pow(rho_a,1.6E1/3.0);
                double t18789 = t18784*t18784;
                double t18790 = t18786*t18787*t18788*t18789*2.0;
                double t18791 = exp(t18790);
                double t18792 = c*t18785;
                double t18793 = pow(t18792,4.0/5.0);
                double t18805 = rab1*t18793;
                double t18794 = exp(-t18805);
                double t18795 = t18794-1.0;
                double t18806 = kab1*t18795;
                double t18796 = kab0-t18806;
                double t18797 = 1.0/c;
                double t18798 = 1.0/pow(rho_a,7.0/3.0);
                double t18799 = a3*gamma_aa*t18784*t18797*t18798;
                double t18800 = t18799+1.0;
                double t18801 = t18786*t18787*t18788*t18789;
                double t18802 = 1.0/pow(rho_a,8.0/3.0);
                double t18803 = a1*gamma_aa*t18802*t18784;
                double t18804 = t18801+t18803+1.0;
                double t18807 = 1.0/(t18796*t18796);
                double t18808 = 1.0/(t18804*t18804);
                double t18809 = c*c0*t18800*t18807*t18808*t18791*t18785*(2.0/3.0);
                double t18810 = a1*t18802*t18784;
                double t18811 = gamma_aa*t18786*t18788*t18789*2.0;
                double t18812 = t18810+t18811;
                double t18813 = 1.0/(t18804*t18804*t18804);
                double t18814 = 1.0/pow(rho_a,1.7E1/3.0);
                double t18815;
                double t18816;
                if (t18809 > expei_cutoff) {
                    t18815 = 1.0;
                    t18816 = expei(t18809);
                } else {
                    t18815 = exp(t18809);
                    t18816 = Ei(-t18809);
                }
                double t18817 = a3*c0*t18802*t18807*t18808*t18791*t18784*(2.0/3.0);
                double t18818 = c*c0*gamma_aa*t18800*t18814*t18807*t18808*t18791*t18786*t18789*(8.0/3.0);
                double t18825 = c*c0*t18800*t18812*t18813*t18807*t18791*t18785*(4.0/3.0);
                double t18819 = -t18825+t18817+t18818;
                double t18820 = sqrt(t18809);
                double t18821 = t18820*6.0;
                double t18822 = c*c0*t18800*t18807*t18808*t18791*t18785*4.0;
                double t18823 = t18821+t18822+3.0;
                double t18824 = 1.0/t18823;
                double t18826 = c*c0*t18800*t18815*t18807*t18816*t18808*t18791*t18785*(2.0/3.0);
                double t18827 = t18826+1.0;
                double t18828 = t18820*4.0;
                double t18829 = c*c0*t18800*t18807*t18808*t18791*t18785*(8.0/3.0);
                double t18830 = t18828+t18829+6.0;
                double t18831 = 1.0/sqrt(t18809);
                double t18832 = 1.0/sqrt(t18800);
                double t18833 = exp(-t18801);
                v_rho_b_gamma_aa[Q] += scale * (c0*heaviside(c*c0*t18785*(-2.0/3.0)+t18804*t18832*t18796*exp(-t18786*t18787*t18788*t18789)*1.0E5)*(t18815*t18816*t18819+t18830*t18824*(-t18825+t18817+t18818+a3*c0*t18802*t18815*t18807*t18816*t18808*t18791*t18784*(2.0/3.0)-c*c0*t18800*t18812*t18813*t18815*t18807*t18816*t18791*t18785*(4.0/3.0)+c*c0*t18800*t18815*t18807*t18816*t18808*t18791*t18819*t18785*(2.0/3.0)+c*c0*gamma_aa*t18800*t18814*t18815*t18807*t18816*t18808*t18791*t18786*t18789*(8.0/3.0))+t18824*t18827*(t18831*t18819*2.0+a3*c0*t18802*t18807*t18808*t18791*t18784*(8.0/3.0)-c*c0*t18800*t18812*t18813*t18807*t18791*t18785*(1.6E1/3.0)+c*c0*gamma_aa*t18800*t18814*t18807*t18808*t18791*t18786*t18789*(3.2E1/3.0))-t18830*1.0/(t18823*t18823)*t18827*(t18831*t18819*3.0+a3*c0*t18802*t18807*t18808*t18791*t18784*4.0-c*c0*t18800*t18812*t18813*t18807*t18791*t18785*8.0+c*c0*gamma_aa*t18800*t18814*t18807*t18808*t18791*t18786*t18789*1.6E1)+(pow(rho_a,1.0/3.0)*(t18804*t18804)*t18819*(t18796*t18796)*t18797*exp(-t18790)*(3.0/2.0))/(c0*t18800))*(1.0/2.0)-c0*dirac(c*c0*t18785*(2.0/3.0)-t18804*t18832*t18833*t18796*1.0E5)*(t18815*t18816+t18830*t18824*t18827)*(t18812*t18832*t18833*t18796*-1.0E5+a3*1.0/pow(t18800,3.0/2.0)*t18804*t18833*t18784*t18796*t18797*t18798*5.0E4+gamma_aa*t18804*t18832*t18833*t18786*t18796*t18788*t18789*2.0E5)*(1.0/2.0));
            }
            
            // v_rho_b_gamma_bb
            if (deriv >= 2) {
                double t18836 = tspi_m13*tspi_m13;
                double t18837 = 1.0/pow(rho_b,1.0/3.0);
                double t18838 = a2*a2;
                double t18839 = gamma_bb*gamma_bb;
                double t18840 = 1.0/pow(rho_b,1.6E1/3.0);
                double t18841 = t18836*t18836;
                double t18842 = c*t18837;
                double t18843 = a4*a4;
                double t18844 = 1.0/c;
                double t18845 = 1.0/pow(rho_b,7.0/3.0);
                double t18846 = sqrt(t18842);
                double t18847 = t18840*t18841*t18843*t18839*2.0;
                double t18848 = exp(t18847);
                double t18849 = t18840*t18841*t18843*t18839;
                double t18850 = t18849+1.0;
                double t18851 = 1.0/(t18850*t18850);
                double t18852 = a5*gamma_bb*t18844*t18836*t18845;
                double t18853 = t18852+1.0;
                double t18864 = raa2*t18846;
                double t18854 = exp(-t18864);
                double t18855 = t18854-1.0;
                double t18856 = kaa2*t18855;
                double t18857 = pow(t18842,2.0/5.0);
                double t18865 = raa1*t18857;
                double t18858 = exp(-t18865);
                double t18859 = t18858-1.0;
                double t18860 = kaa1*t18859;
                double t18861 = -kaa0+t18860+t18856;
                double t18862 = 1.0/(t18861*t18861);
                double t18863 = c*c0*t18851*t18853*t18862*t18837*t18848*(2.0/3.0);
                double t18866 = 1.0/pow(rho_b,8.0/3.0);
                double t18867 = 1.0/pow(rho_b,4.0/3.0);
                double t18868 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b);
                double t18869 = 1.0/pow(rho_b,2.0E1/3.0);
                double t18870 = 1.0/(t18850*t18850*t18850);
                double t18871 = t18843*t18843;
                double t18872 = 1.0/pow(rho_b,1.2E1);
                double t18873 = t18841*t18841;
                double t18874 = 1.0/sqrt(t18842);
                double t18875 = c*kaa2*raa2*t18854*t18874*t18867*(1.0/6.0);
                double t18876 = 1.0/pow(t18842,3.0/5.0);
                double t18877 = c*kaa1*raa1*t18858*t18867*t18876*(2.0/1.5E1);
                double t18878 = t18875+t18877;
                double t18879 = 1.0/(t18861*t18861*t18861);
                double t18880 = 1.0/pow(rho_b,1.7E1/3.0);
                double t18881 = 1.0/t18850;
                double t18882 = c*c0*t18853*t18862*t18837*t18848*(2.0/3.0);
                double t18883 = sqrt(t18882);
                double t18884 = 1.0/pow(rho_b,1.1E1/3.0);
                double t18885;
                double t18886;
                if (t18863 > expei_cutoff) {
                    t18885 = 1.0;
                    t18886 = expei(t18863);
                } else {
                    t18885 = exp(t18863);
                    t18886 = Ei(-t18863);
                }
                double t18887 = c*c0*t18851*t18853*t18862*t18848*t18867*(2.0/9.0);
                double t18888 = a5*c0*gamma_bb*t18851*t18862*t18836*t18848*t18884*(1.4E1/9.0);
                double t18889 = c*c0*t18851*t18853*t18837*t18848*t18878*t18879*(4.0/3.0);
                double t18890 = c*c0*t18841*t18851*t18843*t18853*t18862*t18839*t18848*t18869*(6.4E1/9.0);
                double t18891 = t18881*t18883*6.0;
                double t18892 = c*c0*t18851*t18853*t18862*t18837*t18848*4.0;
                double t18893 = t18891+t18892+3.0;
                double t18894 = 1.0/t18893;
                double t18895 = a5*c0*t18851*t18862*t18836*t18848*t18866*(2.0/3.0);
                double t18896 = c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18880*t18848*(8.0/3.0);
                double t18897 = 1.0/sqrt(t18882);
                double t18900 = c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18880*t18848*(8.0/3.0);
                double t18898 = -t18900+t18895+t18896;
                double t18911 = c*c0*t18841*t18843*t18870*t18853*t18862*t18839*t18848*t18869*(6.4E1/9.0);
                double t18899 = -t18911+t18890+t18887+t18888+t18889;
                double t18901 = 1.0/pow(rho_b,1.9E1/3.0);
                double t18902 = a5*c0*t18851*t18862*t18836*t18848*t18884*(1.6E1/9.0);
                double t18903 = a5*c0*t18851*t18836*t18848*t18866*t18878*t18879*(4.0/3.0);
                double t18904 = a5*c0*t18841*t18851*t18843*t18862*t18836*t18839*t18848*t18868*(4.0E1/3.0);
                double t18905 = c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18848*t18869*(1.36E2/9.0);
                double t18906 = c*c0*gamma_bb*t18851*t18853*t18862*t18871*t18872*t18873*t18839*t18848*(2.56E2/9.0);
                double t18907 = 1.0/(t18850*t18850*t18850*t18850);
                double t18908 = c*c0*gamma_bb*t18853*t18862*t18871*t18907*t18872*t18873*t18839*t18848*(1.28E2/3.0);
                double t18909 = c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18880*t18848*t18878*t18879*(1.6E1/3.0);
                double t18912 = c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18848*t18869*(1.36E2/9.0);
                double t18913 = c*c0*gamma_bb*t18870*t18853*t18862*t18871*t18872*t18873*t18839*t18848*(5.12E2/9.0);
                double t18914 = c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18880*t18848*t18878*t18879*(1.6E1/3.0);
                double t18971 = a5*c0*t18841*t18843*t18870*t18862*t18836*t18839*t18848*t18868*(4.0E1/3.0);
                double t18910 = t18902+t18903-t18912+t18904-t18913+t18905-t18914+t18906+t18908-t18971+t18909;
                double t18915 = a5*c0*t18862*t18836*t18848*t18866*(2.0/3.0);
                double t18916 = c*c0*gamma_bb*t18841*t18843*t18853*t18862*t18880*t18848*(8.0/3.0);
                double t18917 = t18915+t18916;
                double t18918 = c*c0*t18853*t18862*t18848*t18867*(2.0/9.0);
                double t18919 = a5*c0*gamma_bb*t18862*t18836*t18848*t18884*(1.4E1/9.0);
                double t18920 = c*c0*t18853*t18837*t18848*t18878*t18879*(4.0/3.0);
                double t18921 = c*c0*t18841*t18843*t18853*t18862*t18839*t18848*t18869*(6.4E1/9.0);
                double t18922 = t18920+t18921+t18918+t18919;
                double t18923 = t18881*t18883*4.0;
                double t18924 = c*c0*t18851*t18853*t18862*t18837*t18848*(8.0/3.0);
                double t18925 = t18923+t18924+6.0;
                double t18926 = a5*c0*t18851*t18862*t18836*t18848*t18866*t18885*t18886*(2.0/3.0);
                double t18927 = c*c0*t18851*t18853*t18862*t18837*t18848*t18885*t18886*t18898*(2.0/3.0);
                double t18928 = c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18880*t18848*t18885*t18886*(8.0/3.0);
                double t18929 = c*c0*t18851*t18853*t18862*t18837*t18848*t18885*t18886*(2.0/3.0);
                double t18930 = t18929+1.0;
                double t18931 = 1.0/(t18893*t18893);
                double t18932 = a5*c0*t18862*t18836*t18848*t18884*(1.6E1/9.0);
                double t18933 = a5*c0*t18836*t18848*t18866*t18878*t18879*(4.0/3.0);
                double t18934 = a5*c0*t18841*t18843*t18862*t18836*t18839*t18848*t18868*(4.0E1/3.0);
                double t18935 = c*c0*gamma_bb*t18841*t18843*t18853*t18862*t18848*t18869*(1.36E2/9.0);
                double t18936 = c*c0*gamma_bb*t18853*t18862*t18871*t18872*t18873*t18839*t18848*(2.56E2/9.0);
                double t18937 = c*c0*gamma_bb*t18841*t18843*t18853*t18880*t18848*t18878*t18879*(1.6E1/3.0);
                double t18938 = t18932+t18933+t18934+t18935+t18936+t18937;
                double t18939 = 1.0/pow(t18882,3.0/2.0);
                double t18940 = 1.0/pow(rho_b,3.5E1/3.0);
                double t18941 = t18881*t18917*t18897*2.0;
                double t18942 = a5*c0*t18851*t18862*t18836*t18848*t18866*(8.0/3.0);
                double t18943 = c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18880*t18848*(3.2E1/3.0);
                double t19005 = gamma_bb*t18840*t18841*t18851*t18843*t18883*8.0;
                double t19006 = c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18880*t18848*(3.2E1/3.0);
                double t18944 = t18941+t18942+t18943-t19005-t19006;
                double t18945 = t18922*t18881*t18897*3.0;
                double t18946 = c*c0*t18851*t18853*t18862*t18848*t18867*(4.0/3.0);
                double t18947 = a5*c0*gamma_bb*t18851*t18862*t18836*t18848*t18884*(2.8E1/3.0);
                double t18948 = c*c0*t18851*t18853*t18837*t18848*t18878*t18879*8.0;
                double t18949 = c*c0*t18841*t18851*t18843*t18853*t18862*t18839*t18848*t18869*(1.28E2/3.0);
                double t18968 = t18901*t18841*t18851*t18843*t18883*t18839*3.2E1;
                double t18969 = c*c0*t18841*t18843*t18870*t18853*t18862*t18839*t18848*t18869*(1.28E2/3.0);
                double t18950 = t18945+t18946+t18947+t18948+t18949-t18968-t18969;
                double t18951 = t18922*t18881*t18897*2.0;
                double t18952 = c*c0*t18851*t18853*t18862*t18848*t18867*(8.0/9.0);
                double t18953 = a5*c0*gamma_bb*t18851*t18862*t18836*t18848*t18884*(5.6E1/9.0);
                double t18954 = c*c0*t18851*t18853*t18837*t18848*t18878*t18879*(1.6E1/3.0);
                double t18955 = c*c0*t18841*t18851*t18843*t18853*t18862*t18839*t18848*t18869*(2.56E2/9.0);
                double t19000 = t18901*t18841*t18851*t18843*t18883*t18839*(6.4E1/3.0);
                double t19001 = c*c0*t18841*t18843*t18870*t18853*t18862*t18839*t18848*t18869*(2.56E2/9.0);
                double t18956 = t18951+t18952+t18953+t18954+t18955-t19000-t19001;
                double t18957 = t18881*t18917*t18897*3.0;
                double t18958 = a5*c0*t18851*t18862*t18836*t18848*t18866*4.0;
                double t18959 = c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18880*t18848*1.6E1;
                double t18966 = gamma_bb*t18840*t18841*t18851*t18843*t18883*1.2E1;
                double t18967 = c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18880*t18848*1.6E1;
                double t18960 = t18957-t18966+t18958-t18967+t18959;
                double t18961 = c*c0*t18851*t18853*t18862*t18848*t18867*t18885*t18886*(2.0/9.0);
                double t18962 = c*c0*t18851*t18853*t18837*t18848*t18885*t18886*t18878*t18879*(4.0/3.0);
                double t18963 = c*c0*t18851*t18853*t18862*t18837*t18848*t18885*t18886*t18899*(2.0/3.0);
                double t18964 = a5*c0*gamma_bb*t18851*t18862*t18836*t18848*t18884*t18885*t18886*(1.4E1/9.0);
                double t18965 = c*c0*t18841*t18851*t18843*t18853*t18862*t18839*t18848*t18885*t18886*t18869*(6.4E1/9.0);
                double t18970 = 1.0/t18853;
                double t18972 = 1.0/c0;
                double t18973 = pow(rho_b,1.0/3.0);
                double t18974 = exp(-t18847);
                double t18975 = t18850*t18850;
                double t18976 = t18861*t18861;
                double t18977 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b);
                double t19002 = c*c0*t18841*t18843*t18870*t18853*t18862*t18839*t18848*t18885*t18886*t18869*(6.4E1/9.0);
                double t18978 = -t18911+t18961+t18890+t18962+t18963+t18964+t18965+t18887+t18888+t18889-t19002;
                double t19007 = c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18880*t18848*t18885*t18886*(8.0/3.0);
                double t18979 = -t18900+t18926+t18927+t18928+t18895+t18896-t19007;
                double t18980 = t18840*t18841*t18838*t18839;
                double t18981 = pow(t18842,4.0/5.0);
                double t18998 = rab1*t18981;
                double t18982 = exp(-t18998);
                double t18983 = t18982-1.0;
                double t18999 = kab1*t18983;
                double t18984 = kab0-t18999;
                double t18985 = a3*gamma_bb*t18844*t18836*t18845;
                double t18986 = t18985+1.0;
                double t18987 = 1.0/sqrt(t18986);
                double t18988 = a1*gamma_bb*t18836*t18866;
                double t18989 = t18980+t18988+1.0;
                double t18990 = c*c;
                double t18991 = 1.0/pow(rho_b,2.0/3.0);
                double t18992 = k1*t18846;
                double t18993 = c*k2*t18837;
                double t18994 = t18992+t18993;
                double t18995 = 1.0/(t18994*t18994);
                double t19004 = t18990*t18991*t18995;
                double t18996 = exp(-t19004);
                double t18997 = exp(-t18980);
                double t19003 = c*c0*t18837*(2.0/3.0);
                double t19008 = t18984*t18987*t18997*t18989*1.0E5;
                double t19009 = 1.0/pow(rho_b,1.0E1/3.0);
                double t19010 = 1.0/pow(t18986,3.0/2.0);
                double t19011 = t18901*t18841*t18838*t18839*(1.6E1/3.0);
                double t19012 = a1*gamma_bb*t18836*t18884*(8.0/3.0);
                double t19013 = t19011+t19012;
                double t19014 = a1*t18836*t18866;
                double t19015 = gamma_bb*t18840*t18841*t18838*2.0;
                double t19016 = t19014+t19015;
                double t19017 = 1.0/(c*c);
                double t19018 = 1.0/pow(t18842,1.0/5.0);
                double t19019 = t19003-t19008;
                double t19020 = dirac(t19019);
                double t19021 = t18885*t18886*t18898;
                double t19022 = t18930*t18944*t18894;
                double t19023 = t18925*t18894*t18979;
                double t19024 = t18844*t18970*t18972*t18973*t18974*t18975*t18976*t18898*(3.0/2.0);
                double t19158 = t18930*t18931*t18960*t18925;
                double t19025 = t19021+t19022+t19023+t19024-t19158;
                double t19026 = t18885*t18886;
                double t19027 = t18930*t18925*t18894;
                double t19028 = t19026+t19027;
                double t19029 = 1.0/pow(rho_b,5.0/3.0);
                double t19030 = t18990*t18995*t19029*(2.0/3.0);
                double t19031 = 1.0/(t18994*t18994*t18994);
                double t19032 = c*k2*t18867*(1.0/3.0);
                double t19033 = c*k1*t18874*t18867*(1.0/6.0);
                double t19034 = t19032+t19033;
                double t19035 = t19030-t18990*t18991*t19031*t19034*2.0;
                double t19036 = a3*t18844*t18836*t18845*t18984*t18997*t18989*t19010*5.0E4;
                double t19037 = gamma_bb*t18840*t18841*t18838*t18984*t18987*t18997*t18989*2.0E5;
                double t19039 = t18984*t18987*t18997*t19016*1.0E5;
                double t19038 = t19036+t19037-t19039;
                double t19040 = c*c0*t18867*(2.0/9.0);
                double t19041 = t18901*t18841*t18838*t18839*t18984*t18987*t18997*t18989*(1.6E6/3.0);
                double t19042 = a3*gamma_bb*t18844*t18836*t18984*t18997*t18989*t19010*t19009*(3.5E5/3.0);
                double t19152 = t18984*t18987*t18997*t19013*1.0E5;
                double t19153 = c*kab1*rab1*t18982*t18867*t18987*t18997*t18989*t19018*(8.0E4/3.0);
                double t19043 = t19040+t19041+t19042-t19152-t19153;
                double t19044 = -t19003+t19008;
                double t19045 = heaviside(t19044);
                double t19046 = t18840*t18841*t18838*t18839*2.0;
                double t19047 = exp(t19046);
                double t19048 = 1.0/(t18984*t18984);
                double t19049 = 1.0/(t18989*t18989);
                double t19050 = c*c0*t18837*t18986*t19047*t19048*t19049*(2.0/3.0);
                double t19051 = 1.0/(t18989*t18989*t18989);
                double t19052;
                double t19053;
                if (t19050 > expei_cutoff) {
                    t19052 = 1.0;
                    t19053 = expei(t19050);
                } else {
                    t19052 = exp(t19050);
                    t19053 = Ei(-t19050);
                }
                double t19054 = c*c0*t18867*t18986*t19047*t19048*t19049*(2.0/9.0);
                double t19055 = a3*c0*gamma_bb*t18836*t18884*t19047*t19048*t19049*(1.4E1/9.0);
                double t19056 = c*c0*t18841*t18838*t18839*t18869*t18986*t19047*t19048*t19049*(6.4E1/9.0);
                double t19057 = 1.0/(t18984*t18984*t18984);
                double t19058 = sqrt(t19050);
                double t19059 = t19058*6.0;
                double t19060 = c*c0*t18837*t18986*t19047*t19048*t19049*4.0;
                double t19061 = t19060+t19059+3.0;
                double t19062 = 1.0/t19061;
                double t19085 = c*c0*t18837*t18986*t19013*t19051*t19047*t19048*(4.0/3.0);
                double t19086 = c0*kab1*rab1*t18990*t18982*t18986*t19018*t19029*t19047*t19057*t19049*(1.6E1/4.5E1);
                double t19063 = t19054+t19055+t19056-t19085-t19086;
                double t19064 = 1.0/sqrt(t19050);
                double t19065 = a3*c0*t18836*t18866*t19047*t19048*t19049*(2.0/3.0);
                double t19066 = c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19047*t19048*t19049*(8.0/3.0);
                double t19068 = c*c0*t18837*t18986*t19051*t19016*t19047*t19048*(4.0/3.0);
                double t19067 = t19065+t19066-t19068;
                double t19069 = a1*t18836*t18884*(8.0/3.0);
                double t19070 = gamma_bb*t18901*t18841*t18838*(3.2E1/3.0);
                double t19071 = t19070+t19069;
                double t19072 = t18838*t18838;
                double t19073 = a3*c0*t18836*t18866*t19013*t19051*t19047*t19048*(4.0/3.0);
                double t19074 = c*c0*t18867*t18986*t19051*t19016*t19047*t19048*(4.0/9.0);
                double t19075 = c*c0*t18837*t18986*t19051*t19071*t19047*t19048*(4.0/3.0);
                double t19076 = 1.0/(t18989*t18989*t18989*t18989);
                double t19077 = a3*c0*gamma_bb*t18836*t18884*t19051*t19016*t19047*t19048*(2.8E1/9.0);
                double t19078 = c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19013*t19051*t19047*t19048*(1.6E1/3.0);
                double t19079 = c*c0*t18841*t18838*t18839*t18869*t18986*t19051*t19016*t19047*t19048*(1.28E2/9.0);
                double t19080 = 1.0/(rho_b*rho_b*rho_b*rho_b);
                double t19081 = a3*c*c0*kab1*rab1*t18836*t18982*t19080*t19018*t19047*t19057*t19049*(1.6E1/4.5E1);
                double t19082 = 1.0/(rho_b*rho_b*rho_b*rho_b*rho_b*rho_b*rho_b);
                double t19083 = c0*gamma_bb*kab1*rab1*t18841*t18990*t18838*t18982*t18986*t19018*t19082*t19047*t19057*t19049*(6.4E1/4.5E1);
                double t19088 = a3*c0*t18836*t18884*t19047*t19048*t19049*(1.6E1/9.0);
                double t19089 = c*c0*t18837*t18986*t19013*t19016*t19047*t19048*t19076*4.0;
                double t19090 = a3*c0*t18841*t18836*t18838*t18839*t18868*t19047*t19048*t19049*(4.0E1/3.0);
                double t19091 = c*c0*gamma_bb*t18841*t18838*t18869*t18986*t19047*t19048*t19049*(1.36E2/9.0);
                double t19092 = c*c0*gamma_bb*t18872*t18873*t18839*t18986*t19072*t19047*t19048*t19049*(2.56E2/9.0);
                double t19093 = c0*kab1*rab1*t18990*t18982*t18986*t19051*t19016*t19018*t19029*t19047*t19057*(3.2E1/4.5E1);
                double t19084 = t19081-t19090+t19073-t19091+t19074+t19083-t19092+t19075-t19093+t19077+t19078+t19079-t19088-t19089;
                double t19087 = 1.0/rho_b;
                double t19094 = c*c0*t18837*t18986*t19052*t19053*t19047*t19048*t19049*(2.0/3.0);
                double t19095 = t19094+1.0;
                double t19096 = t19064*t19067*2.0;
                double t19097 = a3*c0*t18836*t18866*t19047*t19048*t19049*(8.0/3.0);
                double t19098 = c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19047*t19048*t19049*(3.2E1/3.0);
                double t19150 = c*c0*t18837*t18986*t19051*t19016*t19047*t19048*(1.6E1/3.0);
                double t19099 = -t19150+t19096+t19097+t19098;
                double t19100 = 1.0/(t19061*t19061);
                double t19101 = t19063*t19064*2.0;
                double t19102 = c*c0*t18867*t18986*t19047*t19048*t19049*(8.0/9.0);
                double t19103 = a3*c0*gamma_bb*t18836*t18884*t19047*t19048*t19049*(5.6E1/9.0);
                double t19104 = c*c0*t18841*t18838*t18839*t18869*t18986*t19047*t19048*t19049*(2.56E2/9.0);
                double t19136 = c*c0*t18837*t18986*t19013*t19051*t19047*t19048*(1.6E1/3.0);
                double t19137 = c0*kab1*rab1*t18990*t18982*t18986*t19018*t19029*t19047*t19057*t19049*(6.4E1/4.5E1);
                double t19105 = t19101+t19102+t19103+t19104-t19136-t19137;
                double t19106 = t19058*4.0;
                double t19107 = c*c0*t18837*t18986*t19047*t19048*t19049*(8.0/3.0);
                double t19108 = t19106+t19107+6.0;
                double t19109 = t19064*t19067*3.0;
                double t19110 = a3*c0*t18836*t18866*t19047*t19048*t19049*4.0;
                double t19111 = c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19047*t19048*t19049*1.6E1;
                double t19127 = c*c0*t18837*t18986*t19051*t19016*t19047*t19048*8.0;
                double t19112 = t19110+t19111+t19109-t19127;
                double t19113 = c*c0*t18867*t18986*t19052*t19053*t19047*t19048*t19049*(2.0/9.0);
                double t19114 = a3*c0*gamma_bb*t18836*t18884*t19052*t19053*t19047*t19048*t19049*(1.4E1/9.0);
                double t19115 = c*c0*t18837*t18986*t19052*t19053*t19063*t19047*t19048*t19049*(2.0/3.0);
                double t19116 = c*c0*t18841*t18838*t18839*t18869*t18986*t19052*t19053*t19047*t19048*t19049*(6.4E1/9.0);
                double t19117 = t19063*t19064*3.0;
                double t19118 = c*c0*t18867*t18986*t19047*t19048*t19049*(4.0/3.0);
                double t19119 = a3*c0*gamma_bb*t18836*t18884*t19047*t19048*t19049*(2.8E1/3.0);
                double t19120 = c*c0*t18841*t18838*t18839*t18869*t18986*t19047*t19048*t19049*(1.28E2/3.0);
                double t19128 = c*c0*t18837*t18986*t19013*t19051*t19047*t19048*8.0;
                double t19129 = c0*kab1*rab1*t18990*t18982*t18986*t19018*t19029*t19047*t19057*t19049*(3.2E1/1.5E1);
                double t19121 = t19120+t19117+t19118+t19119-t19128-t19129;
                double t19122 = a3*c0*t18836*t18866*t19052*t19053*t19047*t19048*t19049*(2.0/3.0);
                double t19123 = c*c0*t18837*t18986*t19052*t19053*t19047*t19048*t19049*t19067*(2.0/3.0);
                double t19124 = c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19052*t19053*t19047*t19048*t19049*(8.0/3.0);
                double t19151 = c*c0*t18837*t18986*t19051*t19016*t19052*t19053*t19047*t19048*(4.0/3.0);
                double t19125 = t19122+t19123+t19124-t19151+t19065+t19066-t19068;
                double t19126 = 1.0/pow(t19050,3.0/2.0);
                double t19130 = 1.0/t18986;
                double t19131 = exp(-t19046);
                double t19132 = t18984*t18984;
                double t19133 = t18989*t18989;
                double t19134 = 1.0/(rho_b*rho_b);
                double t19135 = t19113+t19114+t19115+t19116+t19054+t19055+t19056-t19085-t19086-c*c0*t18837*t18986*t19013*t19051*t19052*t19053*t19047*t19048*(4.0/3.0)-c0*kab1*rab1*t18990*t18982*t18986*t19052*t19053*t19018*t19029*t19047*t19057*t19049*(1.6E1/4.5E1);
                double t19138 = a3*t18844*t18836*t18984*t18997*t18989*t19010*t19009*(3.5E5/3.0);
                double t19139 = a3*t18844*t18836*t18845*t18984*t18997*t19010*t19013*5.0E4;
                double t19140 = gamma_bb*t18901*t18841*t18838*t18984*t18987*t18997*t18989*(3.2E6/3.0);
                double t19141 = gamma_bb*t18840*t18841*t18838*t18984*t18987*t18997*t19013*2.0E5;
                double t19142 = t18901*t18841*t18838*t18839*t18984*t18987*t18997*t19016*(1.6E6/3.0);
                double t19143 = a3*a3;
                double t19144 = 1.0/pow(t18986,5.0/2.0);
                double t19145 = a3*gamma_bb*t18844*t18836*t18984*t18997*t19010*t19016*t19009*(3.5E5/3.0);
                double t19146 = a3*kab1*rab1*t18836*t18982*t18884*t18997*t18989*t19010*t19018*(4.0E4/3.0);
                double t19147 = 1.0/pow(rho_b,2.6E1/3.0);
                double t19148 = c*gamma_bb*kab1*rab1*t18841*t18838*t18982*t18869*t18987*t18997*t18989*t19018*(1.6E5/3.0);
                double t19149 = t19140+t19141+t19142+t19145+t19146+t19138+t19139+t19148-t18984*t18987*t18997*t19071*1.0E5-c*kab1*rab1*t18982*t18867*t18987*t18997*t19016*t19018*(8.0E4/3.0)-gamma_bb*t18940*t18873*t18839*t18984*t18987*t18997*t18989*t19072*(3.2E6/3.0)-gamma_bb*t18841*t18880*t18984*t18997*t18989*t19017*t19143*t19144*1.75E5-a3*t18841*t18844*t18836*t18838*t18839*t18984*t18997*t18989*t19010*t19147*5.0E5;
                double t19154 = dirac(t19019,1.0);
                double t19155 = t19052*t19053;
                double t19156 = t19062*t19108*t19095;
                double t19157 = t19155+t19156;
                v_rho_b_gamma_bb[Q] += scale * (rho_b*(c0*t18996*heaviside(c*c0*t18837*(-2.0/3.0)+t18984*t18987*t18989*exp(-t18840*t18841*t18838*t18839)*1.0E5)*(t18930*t18894*(t18881*t18938*t18897*2.0-t18922*t18881*t18917*t18939-gamma_bb*t18901*t18841*t18851*t18843*t18883*(1.28E2/3.0)+a5*c0*t18851*t18862*t18836*t18848*t18884*(6.4E1/9.0)-gamma_bb*t18840*t18841*t18922*t18851*t18843*t18897*4.0+gamma_bb*t18940*t18870*t18871*t18873*t18883*t18839*(2.56E2/3.0)-t18901*t18841*t18851*t18843*t18917*t18839*t18897*(3.2E1/3.0)+a5*c0*t18851*t18836*t18848*t18866*t18878*t18879*(1.6E1/3.0)+c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18848*t18869*(5.44E2/9.0)-c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18848*t18869*(5.44E2/9.0)+a5*c0*t18841*t18851*t18843*t18862*t18836*t18839*t18848*t18868*(1.6E2/3.0)-a5*c0*t18841*t18843*t18870*t18862*t18836*t18839*t18848*t18868*(1.6E2/3.0)+c*c0*gamma_bb*t18851*t18853*t18862*t18871*t18872*t18873*t18839*t18848*(1.024E3/9.0)-c*c0*gamma_bb*t18870*t18853*t18862*t18871*t18872*t18873*t18839*t18848*(2.048E3/9.0)+c*c0*gamma_bb*t18853*t18862*t18871*t18907*t18872*t18873*t18839*t18848*(5.12E2/3.0)+c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18880*t18848*t18878*t18879*(6.4E1/3.0)-c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18880*t18848*t18878*t18879*(6.4E1/3.0))+t18925*t18894*(t18902+t18903-t18912+t18904-t18913+t18905-t18914+t18906+t18908-t18971+t18909+t18898*t18899+t18898*t19087*(1.0/3.0)+(t18878*t18898*2.0)/t18861+t18901*t18841*t18843*t18839*t18898*(3.2E1/3.0)-t18901*t18841*t18843*t18881*t18839*t18898*(3.2E1/3.0)+a5*gamma_bb*t18844*t18970*t18836*t18898*t19009*(7.0/3.0)+a5*c0*t18851*t18862*t18836*t18848*t18884*t18885*t18886*(1.6E1/9.0)+a5*c0*t18851*t18862*t18836*t18848*t18866*t18885*t18886*t18899*(2.0/3.0)+a5*c0*t18851*t18836*t18848*t18866*t18885*t18886*t18878*t18879*(4.0/3.0)+c*c0*t18910*t18851*t18853*t18862*t18837*t18848*t18885*t18886*(2.0/3.0)+c*c0*t18851*t18853*t18862*t18848*t18867*t18885*t18886*t18898*(2.0/9.0)+a5*c0*gamma_bb*t18851*t18862*t18836*t18848*t18884*t18885*t18886*t18898*(1.4E1/9.0)+c*c0*t18851*t18853*t18862*t18837*t18848*t18885*t18886*t18898*t18899*(2.0/3.0)+c*c0*t18851*t18853*t18837*t18848*t18885*t18886*t18878*t18879*t18898*(4.0/3.0)+c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18848*t18885*t18886*t18869*(1.36E2/9.0)-c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18848*t18885*t18886*t18869*(1.36E2/9.0)+a5*c0*t18841*t18851*t18843*t18862*t18836*t18839*t18848*t18885*t18868*t18886*(4.0E1/3.0)-a5*c0*t18841*t18843*t18870*t18862*t18836*t18839*t18848*t18885*t18868*t18886*(4.0E1/3.0)+c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18880*t18848*t18885*t18886*t18899*(8.0/3.0)-c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18880*t18848*t18885*t18886*t18899*(8.0/3.0)+c*c0*gamma_bb*t18851*t18853*t18862*t18871*t18872*t18873*t18839*t18848*t18885*t18886*(2.56E2/9.0)-c*c0*gamma_bb*t18870*t18853*t18862*t18871*t18872*t18873*t18839*t18848*t18885*t18886*(5.12E2/9.0)+c*c0*gamma_bb*t18853*t18862*t18871*t18907*t18872*t18873*t18839*t18848*t18885*t18886*(1.28E2/3.0)+c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18880*t18848*t18885*t18886*t18878*t18879*(1.6E1/3.0)-c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18880*t18848*t18885*t18886*t18878*t18879*(1.6E1/3.0)+c*c0*t18841*t18851*t18843*t18853*t18862*t18839*t18848*t18885*t18886*t18869*t18898*(6.4E1/9.0)-c*c0*t18841*t18843*t18870*t18853*t18862*t18839*t18848*t18885*t18886*t18869*t18898*(6.4E1/9.0))+t18910*t18885*t18886+t18944*t18894*t18978+t18956*t18894*t18979-t18930*t18931*t18950*t18944-t18930*t18931*t18960*t18956-t18931*t18950*t18925*t18979-t18931*t18960*t18925*t18978+t18885*t18886*t18898*t18899-t18930*t18931*t18925*(t18881*t18938*t18897*3.0-t18922*t18881*t18917*t18939*(3.0/2.0)-gamma_bb*t18901*t18841*t18851*t18843*t18883*6.4E1+a5*c0*t18851*t18862*t18836*t18848*t18884*(3.2E1/3.0)-gamma_bb*t18840*t18841*t18922*t18851*t18843*t18897*6.0+gamma_bb*t18940*t18870*t18871*t18873*t18883*t18839*1.28E2-t18901*t18841*t18851*t18843*t18917*t18839*t18897*1.6E1+a5*c0*t18851*t18836*t18848*t18866*t18878*t18879*8.0+c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18862*t18848*t18869*(2.72E2/3.0)-c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18862*t18848*t18869*(2.72E2/3.0)+a5*c0*t18841*t18851*t18843*t18862*t18836*t18839*t18848*t18868*8.0E1-a5*c0*t18841*t18843*t18870*t18862*t18836*t18839*t18848*t18868*8.0E1+c*c0*gamma_bb*t18851*t18853*t18862*t18871*t18872*t18873*t18839*t18848*(5.12E2/3.0)-c*c0*gamma_bb*t18870*t18853*t18862*t18871*t18872*t18873*t18839*t18848*(1.024E3/3.0)+c*c0*gamma_bb*t18853*t18862*t18871*t18907*t18872*t18873*t18839*t18848*2.56E2+c*c0*gamma_bb*t18841*t18851*t18843*t18853*t18880*t18848*t18878*t18879*3.2E1-c*c0*gamma_bb*t18841*t18843*t18870*t18853*t18880*t18848*t18878*t18879*3.2E1)+t18930*t18950*t18960*t18925*1.0/(t18893*t18893*t18893)*2.0+t18910*t18844*t18970*t18972*t18973*t18974*t18975*t18976*(3.0/2.0)+t18844*t18970*t18972*t18973*t18974*t18975*t18976*t18898*t18899*(3.0/2.0)-a5*1.0/(t18853*t18853)*t18836*t18972*t18974*t18975*t18976*t18899*t19017*t19134*(3.0/2.0)+gamma_bb*t18841*t18850*t18843*t18844*t18970*t18972*t18974*t18976*t18977*t18899*6.0-gamma_bb*t18841*t18843*t18844*t18970*t18972*t18974*t18975*t18976*t18977*t18899*6.0)*(-1.0/2.0)+c0*t18996*t19020*t19025*t19043*(1.0/2.0)+c0*t18996*t19025*t19035*t19045*(1.0/2.0)+c0*t18996*t19020*t19028*t19149*(1.0/2.0)+c0*t18996*t19038*dirac(t19003-t18984*t18987*t18997*t18989*1.0E5)*(t18930*t18956*t18894+t18925*t18894*t18978+t18885*t18886*t18899-t18930*t18931*t18950*t18925+t18844*t18970*t18972*t18973*t18974*t18975*t18976*t18899*(3.0/2.0))*(1.0/2.0)-c0*t18996*t19020*t19035*t19028*t19038*(1.0/2.0)+c0*t18996*t19043*t19028*t19154*t19038*(1.0/2.0))+rho_a*(c0*t19045*(t19062*t19095*(t19064*t19084*2.0+t19063*t19126*t19067-a3*c0*t18836*t18884*t19047*t19048*t19049*(6.4E1/9.0)+a3*c0*t18836*t18866*t19013*t19051*t19047*t19048*(1.6E1/3.0)+c*c0*t18837*t18986*t19051*t19071*t19047*t19048*(1.6E1/3.0)+c*c0*t18867*t18986*t19051*t19016*t19047*t19048*(1.6E1/9.0)+a3*c0*gamma_bb*t18836*t18884*t19051*t19016*t19047*t19048*(1.12E2/9.0)-c*c0*t18837*t18986*t19013*t19016*t19047*t19048*t19076*1.6E1-c*c0*gamma_bb*t18841*t18838*t18869*t18986*t19047*t19048*t19049*(5.44E2/9.0)-a3*c0*t18841*t18836*t18838*t18839*t18868*t19047*t19048*t19049*(1.6E2/3.0)+c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19013*t19051*t19047*t19048*(6.4E1/3.0)-c*c0*gamma_bb*t18872*t18873*t18839*t18986*t19072*t19047*t19048*t19049*(1.024E3/9.0)+c*c0*t18841*t18838*t18839*t18869*t18986*t19051*t19016*t19047*t19048*(5.12E2/9.0)+a3*c*c0*kab1*rab1*t18836*t18982*t19080*t19018*t19047*t19057*t19049*(6.4E1/4.5E1)-c0*kab1*rab1*t18990*t18982*t18986*t19051*t19016*t19018*t19029*t19047*t19057*(1.28E2/4.5E1)+c0*gamma_bb*kab1*rab1*t18841*t18990*t18838*t18982*t18986*t19018*t19082*t19047*t19057*t19049*(2.56E2/4.5E1))+t19062*t19108*(t19081-t19090+t19073-t19091+t19074+t19083-t19092+t19075-t19093+t19077+t19078+t19079-t19088-t19089-t19063*t19067-t19067*t19087*(1.0/3.0)+(t19013*t19067*2.0)/t18989-t18901*t18841*t18838*t18839*t19067*(3.2E1/3.0)-a3*gamma_bb*t18844*t18836*t19130*t19009*t19067*(7.0/3.0)+(c*kab1*rab1*t18982*t18867*t19018*t19067*(8.0/1.5E1))/t18984-a3*c0*t18836*t18884*t19052*t19053*t19047*t19048*t19049*(1.6E1/9.0)+a3*c0*t18836*t18866*t19013*t19051*t19052*t19053*t19047*t19048*(4.0/3.0)-a3*c0*t18836*t18866*t19052*t19053*t19063*t19047*t19048*t19049*(2.0/3.0)+c*c0*t18837*t18986*t19051*t19052*t19053*t19071*t19047*t19048*(4.0/3.0)+c*c0*t18867*t18986*t19051*t19016*t19052*t19053*t19047*t19048*(4.0/9.0)+c*c0*t18837*t18986*t19052*t19053*t19047*t19048*t19084*t19049*(2.0/3.0)-c*c0*t18867*t18986*t19052*t19053*t19047*t19048*t19049*t19067*(2.0/9.0)+a3*c0*gamma_bb*t18836*t18884*t19051*t19016*t19052*t19053*t19047*t19048*(2.8E1/9.0)-a3*c0*gamma_bb*t18836*t18884*t19052*t19053*t19047*t19048*t19049*t19067*(1.4E1/9.0)+c*c0*t18837*t18986*t19051*t19016*t19052*t19053*t19063*t19047*t19048*(4.0/3.0)+c*c0*t18837*t18986*t19013*t19051*t19052*t19053*t19047*t19048*t19067*(4.0/3.0)-c*c0*t18837*t18986*t19013*t19016*t19052*t19053*t19047*t19048*t19076*4.0-c*c0*t18837*t18986*t19052*t19053*t19063*t19047*t19048*t19049*t19067*(2.0/3.0)-c*c0*gamma_bb*t18841*t18838*t18869*t18986*t19052*t19053*t19047*t19048*t19049*(1.36E2/9.0)-a3*c0*t18841*t18836*t18838*t18839*t18868*t19052*t19053*t19047*t19048*t19049*(4.0E1/3.0)+c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19013*t19051*t19052*t19053*t19047*t19048*(1.6E1/3.0)-c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19052*t19053*t19063*t19047*t19048*t19049*(8.0/3.0)-c*c0*gamma_bb*t18872*t18873*t18839*t18986*t19052*t19053*t19072*t19047*t19048*t19049*(2.56E2/9.0)+c*c0*t18841*t18838*t18839*t18869*t18986*t19051*t19016*t19052*t19053*t19047*t19048*(1.28E2/9.0)-c*c0*t18841*t18838*t18839*t18869*t18986*t19052*t19053*t19047*t19048*t19049*t19067*(6.4E1/9.0)+a3*c*c0*kab1*rab1*t18836*t18982*t19052*t19053*t19080*t19018*t19047*t19057*t19049*(1.6E1/4.5E1)-c0*kab1*rab1*t18990*t18982*t18986*t19051*t19016*t19052*t19053*t19018*t19029*t19047*t19057*(3.2E1/4.5E1)+c0*kab1*rab1*t18990*t18982*t18986*t19052*t19053*t19018*t19029*t19047*t19057*t19049*t19067*(1.6E1/4.5E1)+c0*gamma_bb*kab1*rab1*t18841*t18990*t18838*t18982*t18986*t19052*t19053*t19018*t19082*t19047*t19057*t19049*(6.4E1/4.5E1))-t19105*t19062*t19125+t19052*t19053*t19084-t19062*t19099*(t19113+t19114+t19115+t19116+t19054+t19055+t19056-c*c0*t18837*t18986*t19013*t19051*t19047*t19048*(4.0/3.0)-c*c0*t18837*t18986*t19013*t19051*t19052*t19053*t19047*t19048*(4.0/3.0)-c0*kab1*rab1*t18990*t18982*t18986*t19018*t19029*t19047*t19057*t19049*(1.6E1/4.5E1)-c0*kab1*rab1*t18990*t18982*t18986*t19052*t19053*t19018*t19029*t19047*t19057*t19049*(1.6E1/4.5E1))+t19100*t19121*t19125*t19108+t19100*t19112*t19108*t19135+t19100*t19112*t19105*t19095+t19100*t19121*t19095*t19099-t19052*t19053*t19063*t19067-t19100*t19108*t19095*(t19064*t19084*3.0+t19063*t19126*t19067*(3.0/2.0)-a3*c0*t18836*t18884*t19047*t19048*t19049*(3.2E1/3.0)+a3*c0*t18836*t18866*t19013*t19051*t19047*t19048*8.0+c*c0*t18837*t18986*t19051*t19071*t19047*t19048*8.0+c*c0*t18867*t18986*t19051*t19016*t19047*t19048*(8.0/3.0)+a3*c0*gamma_bb*t18836*t18884*t19051*t19016*t19047*t19048*(5.6E1/3.0)-c*c0*t18837*t18986*t19013*t19016*t19047*t19048*t19076*2.4E1-c*c0*gamma_bb*t18841*t18838*t18869*t18986*t19047*t19048*t19049*(2.72E2/3.0)-a3*c0*t18841*t18836*t18838*t18839*t18868*t19047*t19048*t19049*8.0E1+c*c0*gamma_bb*t18841*t18880*t18838*t18986*t19013*t19051*t19047*t19048*3.2E1-c*c0*gamma_bb*t18872*t18873*t18839*t18986*t19072*t19047*t19048*t19049*(5.12E2/3.0)+c*c0*t18841*t18838*t18839*t18869*t18986*t19051*t19016*t19047*t19048*(2.56E2/3.0)+a3*c*c0*kab1*rab1*t18836*t18982*t19080*t19018*t19047*t19057*t19049*(3.2E1/1.5E1)-c0*kab1*rab1*t18990*t18982*t18986*t19051*t19016*t19018*t19029*t19047*t19057*(6.4E1/1.5E1)+c0*gamma_bb*kab1*rab1*t18841*t18990*t18838*t18982*t18986*t19018*t19082*t19047*t19057*t19049*(1.28E2/1.5E1))-t19112*t19121*1.0/(t19061*t19061*t19061)*t19108*t19095*2.0+t18844*t18972*t18973*t19130*t19131*t19132*t19133*t19084*(3.0/2.0)-t18844*t18972*t18973*t18989*t19130*t19131*t19132*t19016*t19063*3.0-t18844*t18972*t18973*t19130*t19131*t19132*t19133*t19063*t19067*(3.0/2.0)+a3*t18836*t18972*1.0/(t18986*t18986)*t19131*t19132*t19133*t19017*t19134*t19063*(3.0/2.0)+gamma_bb*t18841*t18844*t18972*t18838*t18977*t19130*t19131*t19132*t19133*t19063*6.0)*(1.0/2.0)+c0*t19020*t19157*t19149*(1.0/2.0)+c0*t19020*t19038*(t19052*t19053*t19063+t19062*t19108*t19135+t19105*t19062*t19095-t19100*t19121*t19108*t19095+t18844*t18972*t18973*t19130*t19131*t19132*t19133*t19063*(3.0/2.0))*(1.0/2.0)+c0*t19020*t19043*(t19062*t19125*t19108+t19052*t19053*t19067+t19062*t19095*t19099-t19100*t19112*t19108*t19095+t18844*t18972*t18973*t19130*t19131*t19132*t19133*t19067*(3.0/2.0))*(1.0/2.0)+c0*t19043*t19154*t19038*t19157*(1.0/2.0))+c0*t18996*t19025*t19045*(1.0/2.0)-c0*t18996*t19020*t19028*t19038*(1.0/2.0));
            }
            
        }
    }
}

}
